/*
 * Kylin-video
 *
 * Copyright (C) 2021, Tianjin KYLIN Information Technology Co., Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * Authors: Liu Cong <liucong1@kylinos.cn>
 *
 */

#include "previewwidget.h"
#include "ui_previewwidget.h"

#include <QGSettings>
#include <QPainter>
#include <QThread>
#include <QDebug>
#include <QTime>

#include "core/util.h"
#include "global/paths.h"
#include "global/functions.h"

#include <syslog.h>

using namespace Global;

PreviewWidget::PreviewWidget(QWidget *parent) :
    QWidget(parent),
    ui(new Ui::PreviewWidget)
{
    ui->setupUi(this);

    setAttribute(Qt::WA_TranslucentBackground);
    m_videoTbr = new VideoThumbnailer;
    m_videoTbr->setThumbnailSize(176);
    currentFile = "";
    // 根据主题设置样式
    setBlackTheme();

    connect(g_core_signal, &GlobalCoreSignal::sigFileInfoChange, this, &PreviewWidget::slotFileInfoChange);
    connect(this, &PreviewWidget::updatePreview, this, [this](){
        updatePos();
        QString time_str = Functions::timeToStr((double)previewTime);
        ui->label_time->setText(time_str);
        ui->label_preview->setStyleSheet(QString("border-radius:8px;border-image:url(%1);")
                                         .arg(QDir::homePath().append("/.cache/kylin-video/tmp-preview.png")));
        show();
    });

    QFont f("Noto Sans CJK SC Regular");
    f.setPixelSize(16);
    ui->label_time->setFont(f);
    ui->label_preview->setFixedSize(192, 108);
    ui->frame->setFixedSize(192, 108);
}

PreviewWidget::~PreviewWidget()
{
    delete ui;
}

void PreviewWidget::setHide()
{
    previewTime = -1;
    hide();
}

void PreviewWidget::setBlackTheme()
{
    ui->widget->setStyleSheet("QWidget#widget{background-color:rgba(28,28,28,0);}");
    ui->label_time->setStyleSheet("background-color:rgba(0,0,0,0);color:rgb(222,222,222)");
    ui->frame->setStyleSheet("background-color:rgba(1,1,1,110);border-radius:8px;");
}

void PreviewWidget::setLightTheme()
{
    ui->widget->setStyleSheet("QWidget#widget{background-color:rgba(223,223,223,0);}QLabel{background-color:rgb(223,223,223);}");
    ui->label_time->setStyleSheet("background-color:rgba(0,0,0,0);color:rgb(56,56,56)");
    ui->frame->setStyleSheet("background-color:rgba(1,1,1,110);border-radius:8px;");
}

void PreviewWidget::setPreview(int time)
{
    if (time == previewTime) {
        return;
    }
    previewTime = time;
    QString time_str = Functions::timeToStr((double)previewTime);
    QThread::create([this, time_str, time](){
        // 应该放到线程中去做
        QString tmp_path = QDir::homePath().append("/.cache/kylin-video/tmp-preview.png");
        // 如果文件不存在返回
        QFileInfo fi(currentFile);
        if (!fi.exists()) {
            QImage img(":/ico/no-preview.png");
            img.save(tmp_path);
        }
        else {
            m_videoTbr->setSeekTime(time_str.toStdString());
            m_videoTbr->generateThumbnail(currentFile.toStdString(), Png, tmp_path.toStdString());
        }

        if (previewTime == time)
            emit updatePreview();
    })->start();
}

void PreviewWidget::slotFileInfoChange(Mpv::FileInfo info)
{
    currentFile = info.file_path;
    duration = info.length;
    previewTime = 0;
}

void PreviewWidget::updatePos()
{
    // 更新位置，跟着鼠标走
    if(previewTime == 0)
        previewTime = 1;
    int x = (double)previewTime * (double)moveRange / (double)duration - (ui->label_preview->width() / 2) + 240;
    int y = height() - 75 - ui->label_preview->height();
    ui->frame->move(x, y);
}
