/*
 *  Copyright 1994-2013 Olivier Girondel
 *
 *  This file is part of lebiniou.
 *
 *  lebiniou is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  lebiniou is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with lebiniou. If not, see <http://www.gnu.org/licenses/>.
 */

#include <GL/gl.h>
#include <GL/glu.h>
#include <ft2build.h>
#include <freetype/freetype.h>
#include <freetype/ftglyph.h>
#include "utils.h"
#include "ttf.h"
#include "constants.h"

#define SQUARE_TEXT

//This holds all of the information related to any
//freetype font that we want to create.  
struct font_data {
  float  h;		///< Holds the height of the font.
  GLuint *textures;	///< Holds the texture id's 
  GLuint list_base;	///< Holds the first display list id
};

static struct font_data fdata;
u_short fontlineskip; // Needed for the OSD to work

#ifdef SQUARE_TEXT
///This function gets the first power of 2 >= the
///int that we pass it.
static inline int
next_p2(const int a)
{
  int rval = 1;
  while (rval<a) rval <<= 1;
  return rval;
}
#endif


///Create a display list coresponding to the give character.
static void
make_dlist(FT_Face face, unsigned char ch, GLuint list_base, GLuint *tex_base)
{
  //The first thing we do is get FreeType to render our character
  //into a bitmap.  This actually requires a couple of FreeType commands:
  
  //Load the Glyph for our character.
  if (FT_Load_Glyph(face, FT_Get_Char_Index(face, ch), FT_LOAD_DEFAULT))
    xerror("FT_Load_Glyph failed\n");
  
  //Move the face's glyph into a Glyph object.
  FT_Glyph glyph;
  if (FT_Get_Glyph(face->glyph, &glyph))
    xerror("FT_Get_Glyph failed\n");
  
  //Convert the glyph to a bitmap.
  FT_Glyph_To_Bitmap(&glyph, FT_RENDER_MODE_NORMAL, 0, 1);
  FT_BitmapGlyph bitmap_glyph = (FT_BitmapGlyph)glyph;
  
  //This reference will make accessing the bitmap easier
  FT_Bitmap *bitmap = &bitmap_glyph->bitmap;

#ifdef SQUARE_TEXT
  //Use our helper function to get the widths of
  //the bitmap data that we will need in order to create
  //our texture.

  int width = next_p2(bitmap->width);
  int height = next_p2(bitmap->rows);
  
  //Allocate memory for the texture data.
  GLubyte* expanded_data = xmalloc(2*width*height*sizeof(GLubyte));
  
  //Here we fill in the data for the expanded bitmap.
  //Notice that we are using two channel bitmap (one for
  //luminocity and one for alpha), but we assign
  //both luminocity and alpha to the value that we
  //find in the FreeType bitmap. 
  //We use the ?: operator so that value which we use
  //will be 0 if we are in the padding zone, and whatever
  //is the the Freetype bitmap otherwise.
  int i, j;
  for (j = 0; j < height; j++) {
    for (i = 0; i < width; i++) {
      expanded_data[2*(i+j*width)] = 255;
      expanded_data[2*(i+j*width)+1] =
	((i >= bitmap->width) || (j >= bitmap->rows)) ?
	0 : bitmap->buffer[i + bitmap->width * j];
    }
  }
#endif
  
  //Now we just setup some texture paramaters.
  glBindTexture(GL_TEXTURE_2D, tex_base[ch]);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

  //Here we actually create the texture itself, notice
  //that we are using GL_LUMINANCE_ALPHA to indicate that
  //we are using 2 channel data.
#ifdef SQUARE_TEXT
  glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, width, height,
	       0, GL_LUMINANCE_ALPHA, GL_UNSIGNED_BYTE, expanded_data);
  //With the texture created, we don't need to expanded data anymore
  xfree(expanded_data);
#else
  glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, bitmap->width, bitmap->rows,
	       0, GL_LUMINANCE_ALPHA, GL_UNSIGNED_BYTE, bitmap->buffer);
#endif
  
  //So now we can create the display list
  glNewList(list_base+ch, GL_COMPILE);

  glBindTexture(GL_TEXTURE_2D, tex_base[ch]);
  
  //first we need to move over a little so that
  //the character has the right amount of space
  //between it and the one before it.
  glTranslatef(bitmap_glyph->left, 0, 0);
  
  //Now we move down a little in the case that the
  //bitmap extends past the bottom of the line 
  //(this is only true for characters like 'g' or 'y'.
  glPushMatrix();
  glTranslatef(0, bitmap_glyph->top-bitmap->rows, 0);

#ifdef SQUARE_TEXT
  //Now we need to account for the fact that many of
  //our textures are filled with empty padding space.
  //We figure what portion of the texture is used by 
  //the actual character and store that information in 
  //the x and y variables, then when we draw the
  //quad, we will only reference the parts of the texture
  //that we contain the character itself.
  float	__xx__ = (float)bitmap->width / (float)width,
    __yy__ = (float)bitmap->rows / (float)height;
#else
#define __xx__ bitmap->width
#define __yy__ bitmap->rows
#endif

  //Here we draw the texturemaped quads.
  //The bitmap that we got from FreeType was not 
  //oriented quite like we would like it to be,
  //so we need to link the texture to the quad
  //so that the result will be properly aligned.
  glBegin(GL_QUADS);
  glTexCoord2d(0, 0); glVertex2f(0, bitmap->rows);
  glTexCoord2d(0, __yy__); glVertex2f(0, 0);
  glTexCoord2d(__xx__, __yy__); glVertex2f(bitmap->width, 0);
  glTexCoord2d(__xx__, 0); glVertex2f(bitmap->width, bitmap->rows);
  glEnd();
  glPopMatrix();
  glTranslatef(face->glyph->advance.x >> 6, 0, 0);

  //increment the raster position as if we were a bitmap font.
  //(only needed if you want to calculate text length)
  //we do :)
  glBitmap(0, 0, 0, 0, face->glyph->advance.x >> 6, 0, NULL);
  
  //Finish the display list
  glEndList();
}


void
ttf_init()
{
  const char *fname = OSD_FONT;
  unsigned int h = OSD_PTSIZE;
  struct font_data *data = &fdata;

  //Allocate some memory to store the texture ids.
  data->textures = xmalloc(128 * sizeof(GLuint));
  
  data->h = fontlineskip = h;
  
  //Create and initilize a freetype font library.
  FT_Library library;
  if (FT_Init_FreeType(&library)) 
    xerror("FT_Init_FreeType failed\n");
  
  //The object in which Freetype holds information on a given
  //font is called a "face".
  FT_Face face;
  
  //This is where we load in the font information from the file.
  //Of all the places where the code might die, this is the most likely,
  //as FT_New_Face will die if the font file does not exist or is somehow broken.
  if (FT_New_Face(library, fname, 0, &face)) 
    xerror("FT_New_Face failed (there is probably a problem with your font file)\n");
  
  //For some twisted reason, Freetype measures font size
  //in terms of 1/64ths of pixels.  Thus, to make a font
  //h pixels high, we need to request a size of h*64.
  //(h << 6 is just a prettier way of writting h*64)
  FT_Set_Char_Size(face, h << 6, h << 6, 96, 96);
  
  //Here we ask opengl to allocate resources for
  //all the textures and displays lists which we
  //are about to create.  
  data->list_base = glGenLists(128);
  glGenTextures(128, data->textures);
  
  //This is where we actually create each of the fonts display lists.
  unsigned char i;
  for (i = 0; i < 128; i++)
    make_dlist(face, i, data->list_base, data->textures);
  
  //We don't need the face information now that the display
  //lists have been created, so we free the associated resources.
  FT_Done_Face(face);
  
  //Ditto for the library.
  FT_Done_FreeType(library);
}


void
ttf_quit()
{
  struct font_data *data = &fdata;

  glDeleteLists(data->list_base, 128);
  glDeleteTextures(128, data->textures);
  xfree(data->textures);
}


/// A fairly straight forward function that pushes
/// a projection matrix that will make object world 
/// coordinates identical to window coordinates.
static inline void
pushScreenCoordinateMatrix()
{
  glPushAttrib(GL_TRANSFORM_BIT);
  GLint	viewport[4];
  glGetIntegerv(GL_VIEWPORT, viewport);
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  gluOrtho2D(viewport[0], viewport[2], viewport[1], viewport[3]);
  glPopAttrib();
}


/// Pops the projection matrix without changing the current
/// MatrixMode.
static inline void
pop_projection_matrix()
{
  glPushAttrib(GL_TRANSFORM_BIT);
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glPopAttrib();
}


///Much like Nehe's glPrint function, but modified to work
///with freetype fonts.
static void
print(float x, float y, const char *text, float *len)
{
  const struct font_data *ft_font = &fdata;

  // We want a coordinate system where things coresponding to window pixels.
  pushScreenCoordinateMatrix();					
  
  GLuint font = ft_font->list_base;
  //  float h = ft_font->h / .63f; //We make the height about 1.5* that of
  
  glPushAttrib(GL_LIST_BIT|GL_CURRENT_BIT|GL_ENABLE_BIT|GL_TRANSFORM_BIT);
  glMatrixMode(GL_MODELVIEW);
  glDisable(GL_LIGHTING);
  glEnable(GL_TEXTURE_2D);
  glDisable(GL_DEPTH_TEST);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);	
  
  glListBase(font);
  //printf("list base: %d\n", font);
  
  float modelview_matrix[16];	
  glGetFloatv(GL_MODELVIEW_MATRIX, modelview_matrix);
  
  glPushMatrix();
  glLoadIdentity();
  glTranslatef(x, y, 0);
  glMultMatrixf(modelview_matrix);

  //  The commented out raster position stuff can be useful if you need to
  //  know the length of the text that you are creating.
  //  If you decide to use it make sure to also uncomment the glBitmap command
  //  in make_dlist().
  if (NULL != len) {
    glRasterPos2f(0, 0);
    glCallLists(strlen(text), GL_UNSIGNED_BYTE, text);
    float rpos[4];
    glGetFloatv(GL_CURRENT_RASTER_POSITION, rpos);
    //    float d = x - rpos[0];
    //assert(d >= 0.0);
    //*len = (uint16_t)d;
    *len = rpos[0];
    // printf("len(%s): %d rpos[0]: %f\n", text, *len, rpos[0]);
  } else
    glCallLists(strlen(text), GL_UNSIGNED_BYTE, text);

  glPopMatrix();
  glPopAttrib();
  pop_projection_matrix();
}


u_short
osd_print(const u_short x, u_short y,
	  const u_char rev_x, const u_char rev_y,
	  __attribute__ ((unused)) const u_char mode, const int disabled, const char *fmt, ...)
{
  char str[OSD_BUFFLEN+1];
  va_list ap;
  //SDL_Surface *text = NULL;
  //  SDL_Rect    dstrect;
  //  SDL_Color   fg_color;

  memset((void *)str, '\0', (OSD_BUFFLEN+1)*sizeof(char));
  assert(NULL != fmt);
  va_start(ap, fmt);
  vsprintf(str, fmt, ap); /* TODO vsnprintf */
  va_end(ap);

  /* mode 1 or 2: transparent background */
  /* mode 3: solid black background */

  /* manque plus que le petit printf qui va bien :) */
  /* plus la gestion des rev_x/y */
  //  glPushMatrix();
  //glLoadIdentity();
  //if (!rev_x) { // && !rev_y) { // && !rev_y) { // temp
    // printf("text @%dx%d (rev: %d %d) (mode: %d) (disabled: %d) (fmt: '%s')\n>>> %s <<<\n",
    // x, y, rev_x, rev_y, mode, disabled, fmt, str);
  uint16_t dst_x;
  if (rev_x) {
    float len = 0;

    glEnable(GL_COLOR_LOGIC_OP);
    glLogicOp(GL_NOOP);
    print(0, 0, str, &len);
    glPopMatrix();
    dst_x = MAXX - (x + len);
  } else
    dst_x = x;
  glDisable(GL_COLOR_LOGIC_OP);

  uint16_t dst_y = (rev_y) ? (MAXY - y) : (y + fontlineskip);
  dst_y = MAXY - dst_y; // OpenGL has inversed coordinates

  if (disabled)
    glColor3ub(255, 0, 0);
  else
    glColor3ub(255, 255, 255);

  print(dst_x, dst_y, str, NULL);
  y += fontlineskip; // - 1; /* FIXME why -1 ?? --oliv3 */

#if 0
  fg_color = (disabled) ? red : white;

  text = ((mode == 1) || (mode == 2)) ? TTF_RenderText_Blended(font, str, black)
    : TTF_RenderText_Shaded(font, str, fg_color, black);

  if (text != NULL) {
    if (mode == 3) {
      dstrect.x = (rev_x) ? (out_width - x - text->w) : x;
      dstrect.y = (rev_y) ? (out_height - y - text->h) : y;
      dstrect.w = text->w;
      dstrect.h = text->h;
      SDL_BlitSurface(text, NULL, drv.sc, &dstrect);
      SDL_FreeSurface(text);
    } else {
      int dx, dy;
      
      dstrect.w = text->w;
      dstrect.h = text->h;
      for (dx = -2; dx <= 2; dx ++) {
	for (dy = -2; dy <= 2; dy ++) {
	  dstrect.x = (rev_x) ? (out_width - x - text->w) : x;
	  dstrect.y = (rev_y) ? (out_height - y - text->h) : y;
	  dstrect.x += dx;
	  dstrect.y += dy;
	  SDL_BlitSurface(text, NULL, drv.sc, &dstrect);
	}
      }
      SDL_FreeSurface(text);

      text = TTF_RenderText_Blended(font, str, fg_color);
      dstrect.x = (rev_x) ? (out_width - x - text->w) : x;
      dstrect.y = (rev_y) ? (out_height - y - text->h) : y;
      SDL_BlitSurface(text, NULL, drv.sc, &dstrect);
      SDL_FreeSurface(text);
    }
    y += TTF_FontLineSkip(font) - 1; /* FIXME why -1 ?? --oliv3 */
  }
#endif /* 0liv3 */

  return y;
}
