/*!
 * \file
 * \brief Filter design test program
 * \author Tony Ottosson and Adam Piatyszek
 *
 * -------------------------------------------------------------------------
 *
 * Copyright (C) 1995-2013  (see AUTHORS file for a list of contributors)
 *
 * This file is part of IT++ - a C++ library of mathematical, signal
 * processing, speech processing, and communications classes and functions.
 *
 * IT++ is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * IT++ is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with IT++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * -------------------------------------------------------------------------
 */

#include <itpp/itsignal.h>
#include "gtest/gtest.h"

using namespace itpp;
using namespace std;

TEST (FilterDesign, All)
{
  const double eps = 1e-5;
  int i;

  RNG_reset(0);
  {
    //Stabilisation of real filters
    vec p = "0.7 3.0 -0.4";
    vec p2 = polystab(p);
    vec p2_expect = "0.7 0.067948 -0.0205197";
    for (i = 0; i < p2.length(); ++i) {
      ASSERT_NEAR(p2_expect[i], p2[i], eps);
    }

    p = randn(7);
    p2 = polystab(p);
    p2_expect = "-0.538679 0.177994 0.0682446 0.00533096 -0.423768 0.283801 0.140756";
    for (i = 0; i < p2.length(); ++i) {
      ASSERT_NEAR(p2_expect[i], p2[i], eps);
    }

    p = poly(vec("1.1 0.7 0.2"));
    p2 = polystab(p);
    p2_expect = "1 -1.80909 0.958182 -0.127273";
    for (i = 0; i < p2.length(); ++i) {
      ASSERT_NEAR(p2_expect[i], p2[i], eps);
    }
  }

  {
    // Stabilisation of complex filters
    cvec p = "0.7 3.0 -0.4";
    cvec p2 = polystab(p);
    cvec p2_expect = "0.7+0i 0.067948+0i -0.0205197+0i";
    for (i = 0; i < p2.length(); ++i) {
      ASSERT_NEAR(p2_expect[i].real(), p2[i].real(), eps);
      ASSERT_NEAR(p2_expect[i].imag(), p2[i].imag(), eps);
    }

    p = randn_c(7);
    p2 = polystab(p);
    p2_expect = "-0.477568-0.495879i -0.0943114+0.0400239i -0.302978-0.102541i -0.0391834+0.072045i "
    "-0.0104879+0.17661i -0.00760533-0.0478387i 0.0397027+0.0173413i";
    for (i = 0; i < p2.length(); ++i) {
      ASSERT_NEAR(p2_expect[i].real(), p2[i].real(), eps);
      ASSERT_NEAR(p2_expect[i].imag(), p2[i].imag(), eps);
    }

    p = poly(cvec("1.1 0.7 0.2"));
    p2 = polystab(p);
    p2_expect = "1+0i -1.80909+0i 0.958182+0i -0.127273+0i";
    for (i = 0; i < p2.length(); ++i) {
      ASSERT_NEAR(p2_expect[i].real(), p2[i].real(), eps);
      ASSERT_NEAR(p2_expect[i].imag(), p2[i].imag(), eps);
    }

    cvec a = randn_c(4);
    cvec b = randn_c(6);
    cvec h = freqz(b, a, 32);
    cvec h_expect = "-0.624626-1.17758i -0.723594-0.875573i -0.785197-0.579523i -0.809466-0.307105i "
    "-0.795505-0.0763535i -0.741218+0.0967196i -0.644514+0.200455i -0.505709+0.228512i -0.329798+0.178836i "
    "-0.127347+0.0513423i 0.0860505-0.154368i 0.290704-0.440555i 0.462875-0.812365i 0.572569-1.279i "
    "0.576688-1.85548i 0.401261-2.56385i -0.105448-3.42346i -1.29433-4.36244i -3.84967-4.66146i "
    "-6.82565-1.62413i -4.85021+2.58433i -1.83063+2.79958i -0.451672+1.85213i 0.050022+0.993934i "
    "0.16639+0.356617i 0.101157-0.0921887i -0.0539619-0.390646i -0.250759-0.566939i -0.45974-0.641763i "
    "-0.660596-0.631587i -0.838317-0.550787i -0.981551-0.412832i";
    for (i = 0; i < h.length(); ++i) {
      ASSERT_NEAR(h_expect[i].real(), h[i].real(), eps);
      ASSERT_NEAR(h_expect[i].imag(), h[i].imag(), eps);
    }
  }
  {
    // Yulewalk filter design
    vec f = "0 0.5 0.6 1";
    vec m = "1 1 0 0";
    vec a, b, R;
    filter_design_autocorrelation(256, f, m, R);
    vec R_expect = "0.532559 0.315821 -0.031951 -0.098821 0.030194 0.052117 -0.027468 -0.030472 0.024049 "
    "0.017893 -0.020260 -0.010026 0.016429 0.005118 -0.012847 -0.002225 0.009732 0.000713 -0.007211 -0.000100 "
    "0.005318 0.000009 -0.004008 -0.000158 0.003178 0.000354 -0.002694 -0.000487 0.002425 0.000517 -0.002257 "
    "-0.000453 0.002112 0.000332 -0.001947 -0.000198 0.001752 0.000087 -0.001541 -0.000020 0.001334 0.000000 "
    "-0.001154 -0.000016 0.001015 0.000050 -0.000918 -0.000083 0.000856 0.000102 -0.000817 -0.000102 0.000786 "
    "0.000084 -0.000751 -0.000057 0.000707 0.000029 -0.000654 -0.000009 0.000596 0.000000 -0.000542 -0.000003 "
    "0.000495 0.000013 -0.000460 -0.000026 0.000436 0.000035 -0.000420 -0.000037 0.000409 0.000033 -0.000396 "
    "-0.000024 0.000381 0.000014 -0.000361 -0.000005 0.000338 0.000001 -0.000315 -0.000001 0.000294 0.000005 "
    "-0.000277 -0.000010 0.000265 0.000015 -0.000257 -0.000017 0.000251 0.000017 -0.000246 -0.000013 0.000239 "
    "0.000008 -0.000230 -0.000003 0.000219 0.000001 -0.000207 -0.000000 0.000196 0.000002 -0.000186 -0.000005 "
    "0.000179 0.000008 -0.000174 -0.000009 0.000171 0.000009 -0.000168 -0.000008 0.000165 0.000005 -0.000160 "
    "-0.000003 0.000154 0.000001 -0.000147 -0.000000 0.000140 0.000001 -0.000135 -0.000002 0.000130 0.000004 "
    "-0.000127 -0.000005 0.000125 0.000006 -0.000123 -0.000005 0.000121 0.000004 -0.000118 -0.000002 0.000115 "
    "0.000001 -0.000111 -0.000000 0.000106 0.000000 -0.000103 -0.000001 0.000099 0.000002 -0.000097 -0.000003 "
    "0.000096 0.000004 -0.000095 -0.000003 0.000093 0.000003 -0.000092 -0.000002 0.000089 0.000001 -0.000087 "
    "-0.000000 0.000084 0.000000 -0.000081 -0.000001 0.000079 0.000001 -0.000077 -0.000002 0.000076 0.000003 "
    "-0.000075 -0.000002 0.000075 0.000002 -0.000074 -0.000001 0.000072 0.000001 -0.000070 -0.000000 0.000068 "
    "0.000000 -0.000066 -0.000000 0.000065 0.000001 -0.000063 -0.000001 0.000063 0.000002 -0.000062 -0.000002 "
    "0.000061 0.000002 -0.000061 -0.000001 0.000060 0.000000 -0.000059 -0.000000 0.000057 0.000000 -0.000056 "
    "-0.000000 0.000054 0.000001 -0.000053 -0.000001 0.000053 0.000001 -0.000052 -0.000001 0.000052 0.000001 "
    "-0.000051 -0.000001 0.000051 0.000000 -0.000050 -0.000000 0.000049 0.000000 -0.000048 -0.000000 0.000047 "
    "0.000000 -0.000046 -0.000001 0.000045 0.000001 -0.000045 -0.000001 0.000045 0.000001 -0.000044 -0.000001 "
    "0.000044 0.000000 -0.000043 -0.000000 0.000042 0.000000 -0.000042 -0.000000 0.000041 0.000000 -0.000040 "
    "-0.000000 0.000040 0.000001 -0.000039 -0.000001";
    for (i = 0; i < R.length(); ++i) {
      ASSERT_NEAR(R_expect[i], R[i], eps);
    }

    arma_estimator(8, 8, R, b, a);
    vec a_expect = "1.000000 0.674659 1.903750 0.910429 1.202849 0.356944 0.264436 0.033088 0.009850";
    vec b_expect = "0.098191 0.427406 0.944264 1.365558 1.419767 1.112417 0.654916 0.271520 0.062529";
    for (i = 0; i < b.length(); ++i) {
      ASSERT_NEAR(b_expect[i], b[i], eps);
      ASSERT_NEAR(a_expect[i], a[i], eps);
    }

    vec n = "0:1:256";
    double fd = 0.1;
    R = besselj(0, 2 * pi * fd * n);
    R_expect = "1.000000 0.903713 0.642512 0.290564 -0.054960 -0.304242 -0.401986 -0.342615 -0.168862 0.045176 0.220277 "
    "0.298483 0.260759 0.130387 -0.038298 -0.181211 -0.247891 -0.218682 -0.109979 0.033729 0.157507 0.216539 0.192023 "
    "0.096868 -0.030454 -0.141182 -0.194690 -0.173208 -0.087544 0.027970 0.129064 0.178348 0.159020 0.080477 -0.026004 "
    "-0.119609 -0.165531 -0.147829 -0.074884 0.024400 0.111968 0.155131 0.138710 0.070314 -0.023059 -0.105625 -0.146473 "
    "-0.131094 -0.066490 0.021917 0.100251 0.139119 0.124608 0.063229 -0.020929 -0.095621 -0.132772 -0.118998 -0.060405 "
    "0.020063 0.091579 0.127221 0.114084 0.057928 -0.019296 -0.088009 -0.122313 -0.109732 -0.055733 0.018611 0.084827 "
    "0.117932 0.105843 0.053770 -0.017994 -0.081967 -0.113991 -0.102340 -0.052000 0.017434 0.079377 0.110420 0.099163 "
    "0.050395 -0.016923 -0.077019 -0.107165 -0.096265 -0.048929 0.016455 0.074859 0.104181 0.093607 0.047585 -0.016024 "
    "-0.072871 -0.101434 -0.091157 -0.046345 0.015625 0.071033 0.098893 0.088890 0.045197 -0.015254 -0.069328 -0.096534 "
    "-0.086785 -0.044131 0.014908 0.067740 0.094336 0.084822 0.043137 -0.014585 -0.066256 -0.092282 -0.082987 -0.042207 "
    "0.014282 0.064866 0.090356 0.081266 0.041335 -0.013997 -0.063560 -0.088546 -0.079647 -0.040514 0.013729 0.062330 "
    "0.086841 0.078122 0.039741 -0.013475 -0.061168 -0.085231 -0.076681 -0.039010 0.013235 0.060069 0.083707 0.075317 "
    "0.038319 -0.013007 -0.059027 -0.082261 -0.074023 -0.037662 0.012791 0.058038 0.080889 0.072794 0.037039 -0.012585 "
    "-0.057096 -0.079582 -0.071624 -0.036445 0.012389 0.056199 0.078337 0.070509 0.035879 -0.012202 -0.055343 -0.077149 "
    "-0.069444 -0.035338 0.012023 0.054525 0.076013 0.068426 0.034822 -0.011851 -0.053743 -0.074926 -0.067451 -0.034327 "
    "0.011687 0.052993 0.073884 0.066517 0.033853 -0.011529 -0.052273 -0.072885 -0.065621 -0.033398 0.011378 0.051582 "
    "0.071925 0.064760 0.032961 -0.011232 -0.050918 -0.071002 -0.063932 -0.032540 0.011092 0.050279 0.070114 0.063135 "
    "0.032135 -0.010957 -0.049663 -0.069258 -0.062367 -0.031745 0.010827 0.049070 0.068432 0.061627 0.031369 -0.010701 "
    "-0.048497 -0.067636 -0.060912 -0.031006 0.010579 0.047943 0.066867 0.060221 0.030655 -0.010462 -0.047409 -0.066123 "
    "-0.059554 -0.030316 0.010348 0.046891 0.065404 0.058908 0.029988 -0.010238 -0.046391 -0.064708 -0.058283 -0.029670 "
    "0.010132 0.045906 0.064033 0.057677 0.029362 -0.010029 -0.045436 -0.063379 -0.057090 -0.029064 0.009929 0.044980 "
    "0.062745 0.056520 0.028774 -0.009831 -0.044537 -0.062129";
    for (i = 0; i < R.length(); ++i) {
      ASSERT_NEAR(R_expect[i], R[i], eps);
    }
    arma_estimator(8, 8, R, b, a);
    a_expect = "1.000000 -5.639923 14.887872 -23.713094 24.788487 -17.355262 7.935089 -2.164767 0.269989";
    b_expect = "0.117196 -0.457317 0.947020 -1.258532 1.157363 -0.723747 0.290453 -0.062104 0.004607";
    for (i = 0; i < b.length(); ++i) {
      ASSERT_NEAR(b_expect[i], b[i], eps);
      ASSERT_NEAR(a_expect[i], a[i], eps);
    }
  }
}
