/* ----- core/inv3sntrup653, derived from supercop/crypto_core/try.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ntruprime.h>
#include "ntruprime_test.h"

#define fail ((ok = 0),printf)
static const char *core_inv3sntrup653_checksums[] = {
  "aa5ad48e98fd02d0029bae0563ff23e304b66d09714ad3a50b7eaa0c39d49de1",
  "07515a32ef8e6bfbc348fd7b45480a3a87632fc4d277c9ec04d8984eae369a17",
} ;

static void (*crypto_core)(unsigned char *,const unsigned char *,const unsigned char *,const unsigned char *);
#define crypto_core_OUTPUTBYTES ntruprime_core_inv3sntrup653_OUTPUTBYTES
#define crypto_core_INPUTBYTES ntruprime_core_inv3sntrup653_INPUTBYTES
#define crypto_core_KEYBYTES ntruprime_core_inv3sntrup653_KEYBYTES
#define crypto_core_CONSTBYTES ntruprime_core_inv3sntrup653_CONSTBYTES

static void *storage_core_inv3sntrup653_h;
static unsigned char *test_core_inv3sntrup653_h;
static void *storage_core_inv3sntrup653_n;
static unsigned char *test_core_inv3sntrup653_n;
static void *storage_core_inv3sntrup653_k;
static unsigned char *test_core_inv3sntrup653_k;
static void *storage_core_inv3sntrup653_c;
static unsigned char *test_core_inv3sntrup653_c;
static void *storage_core_inv3sntrup653_h2;
static unsigned char *test_core_inv3sntrup653_h2;
static void *storage_core_inv3sntrup653_n2;
static unsigned char *test_core_inv3sntrup653_n2;
static void *storage_core_inv3sntrup653_k2;
static unsigned char *test_core_inv3sntrup653_k2;
static void *storage_core_inv3sntrup653_c2;
static unsigned char *test_core_inv3sntrup653_c2;

#define precomputed_core_inv3sntrup653_NUM 5

static const unsigned char precomputed_core_inv3sntrup653_h[precomputed_core_inv3sntrup653_NUM][crypto_core_OUTPUTBYTES] = {
  {255,1,0,1,0,0,0,1,0,255,255,1,255,0,0,255,0,1,255,1,1,1,0,1,255,255,1,0,255,1,255,255,255,255,255,1,255,1,0,1,1,0,0,1,1,1,0,1,255,0,1,255,1,1,255,1,1,1,0,0,0,255,0,1,255,255,1,0,1,0,0,0,0,0,1,255,255,255,0,1,1,255,1,255,255,1,255,0,0,0,255,1,0,0,0,1,0,0,255,0,0,1,0,0,255,1,0,1,1,1,1,0,1,255,0,0,255,255,0,0,1,255,255,255,255,1,0,1,255,1,255,255,0,255,1,255,255,255,255,255,1,1,0,1,1,0,0,255,0,255,255,1,0,1,0,0,0,0,0,1,0,255,1,255,1,1,0,0,255,0,1,0,1,0,0,0,0,1,0,1,255,0,0,1,1,255,255,1,1,255,255,255,1,255,1,255,1,0,255,255,1,255,0,0,0,1,1,1,0,1,1,255,1,0,1,255,0,1,1,255,0,0,0,0,0,255,1,255,255,1,1,255,0,255,1,0,1,1,1,1,0,0,1,255,1,255,255,255,1,0,1,255,255,1,0,0,1,255,0,255,255,1,255,0,0,255,1,0,1,1,1,1,255,255,255,255,255,1,255,255,255,0,0,255,0,0,1,255,0,255,255,0,255,0,0,0,255,1,1,0,255,255,0,255,1,0,255,1,255,255,255,255,255,0,0,1,1,0,255,1,255,0,0,1,0,0,0,1,1,0,1,255,1,255,1,0,0,0,255,1,0,255,1,255,255,255,255,1,255,0,255,255,255,1,255,1,0,0,255,1,255,1,255,255,1,255,255,0,1,1,255,255,255,0,1,1,255,1,1,1,1,1,255,0,1,1,255,1,0,1,255,1,0,1,0,255,1,255,1,0,255,0,1,1,1,0,1,0,1,255,1,1,1,255,0,255,255,255,0,1,0,255,0,1,255,1,255,0,0,255,0,255,0,255,1,1,1,0,1,1,0,255,255,0,0,0,0,255,0,0,0,1,255,255,1,1,0,0,1,1,1,255,0,255,1,1,0,1,1,1,1,255,1,1,1,0,0,1,1,0,255,255,0,1,1,0,1,255,0,0,255,0,1,255,0,1,1,0,255,255,1,0,255,1,255,1,0,0,255,1,1,1,1,1,255,0,0,0,0,255,0,0,255,255,0,255,0,255,0,255,0,0,255,1,1,255,1,0,1,255,0,0,255,1,0,255,1,255,255,1,0,0,0,255,1,1,1,1,255,1,0,0,255,255,0,1,255,1,0,0,255,0,0,1,255,0,1,0,1,1,0,255,1,255,255,1,1,1,1,0,255,1,255,255,0,255,0,255,0,0,255,255,255,255,0,255,1,1,0,0,0,255,0,0,0,0,0,255,0,0,255,255,255,255,255,0,1,255,1,255,0,255,1,1,255,255,255,0,255,255,255,255,1,255,255,0,255,255,1,0,0,255,255,0},
  {255,255,1,0,0,1,255,255,1,0,1,255,1,1,255,1,255,255,255,0,0,255,0,0,1,255,1,255,255,1,255,0,0,1,1,0,255,255,255,0,0,1,255,0,0,255,0,1,1,1,0,1,1,1,1,0,0,1,1,1,1,0,1,0,255,0,1,1,255,0,0,255,1,0,0,0,1,0,0,1,1,255,1,255,1,1,0,255,255,255,1,0,0,0,255,0,0,1,1,0,255,1,1,0,255,0,255,255,0,255,255,1,255,0,1,1,0,1,255,1,255,0,0,255,255,255,255,1,255,255,0,255,1,1,1,0,0,1,0,1,0,255,1,0,255,0,255,255,255,255,0,255,255,1,0,255,1,1,0,0,255,1,1,255,0,255,0,0,1,0,1,0,0,0,0,1,1,0,1,1,0,1,255,0,255,255,1,255,0,0,255,0,0,0,255,255,255,1,1,0,0,1,1,1,255,255,1,1,0,1,1,1,1,0,255,1,0,1,1,255,255,0,0,0,255,0,0,0,255,1,0,255,1,255,255,0,1,0,1,1,255,1,1,0,1,1,0,1,255,255,1,255,255,255,0,255,1,1,255,255,255,0,1,255,0,0,1,0,1,255,255,1,255,0,255,0,0,1,0,255,1,1,1,1,0,1,255,1,0,1,255,0,1,0,0,255,1,0,255,0,0,255,255,0,0,1,1,1,255,0,1,1,0,1,1,0,0,0,0,1,0,255,0,0,255,0,0,255,1,0,1,255,1,1,255,0,255,1,1,255,255,0,1,255,1,0,0,255,255,255,255,1,255,255,0,0,255,1,1,1,1,255,1,255,255,255,1,1,255,1,1,0,255,1,1,0,0,255,0,1,0,255,255,255,255,255,255,255,1,0,0,1,0,255,255,255,255,255,0,255,0,1,1,1,0,0,1,255,0,0,1,0,0,255,255,1,255,0,255,1,255,255,0,1,1,1,0,255,0,0,1,0,1,1,255,0,0,1,255,1,255,0,1,0,255,1,1,255,0,0,0,255,0,255,0,0,255,0,0,255,0,0,255,1,255,0,255,1,1,1,0,0,0,255,255,0,0,1,1,1,0,0,1,0,255,1,0,0,1,1,0,0,1,0,1,0,0,0,0,1,1,1,255,255,255,0,0,0,0,0,0,255,0,1,0,0,1,0,255,255,0,255,0,255,255,255,255,0,255,1,255,255,255,0,1,255,1,255,0,255,1,0,0,255,255,0,1,255,0,1,255,255,0,1,255,1,1,0,1,0,1,0,0,1,255,1,0,0,255,0,1,255,1,0,1,1,0,0,1,255,255,1,0,255,255,255,1,255,1,255,1,255,255,0,255,1,255,255,255,0,1,1,0,0,255,1,255,1,0,255,1,0,1,1,0,255,1,255,255,1,255,255,0,255,0,255,1,255,255,255,0,0,0,1,255,1,255,1,255,1,1,0,255,0,255,0,1,255,0,0,0,1,255,0},
  {255,1,255,0,1,255,1,1,1,0,1,0,0,1,255,255,0,0,255,1,1,0,1,1,0,0,255,255,0,1,1,0,255,0,1,1,1,1,1,1,255,1,0,1,0,255,1,0,1,255,255,1,0,0,0,255,1,255,255,0,0,255,1,1,255,1,255,1,0,1,0,1,0,1,1,1,0,255,255,255,255,0,0,0,255,1,1,1,1,1,1,0,1,255,0,0,1,255,255,255,255,255,255,1,0,255,1,255,255,1,0,1,0,255,255,0,0,0,255,1,0,0,1,1,255,1,255,0,0,0,255,255,0,1,0,255,255,0,0,1,0,1,0,1,1,1,0,255,0,1,0,1,1,1,0,0,255,0,0,0,0,255,0,0,0,255,255,0,1,0,1,255,1,255,1,255,255,0,0,255,255,0,255,0,0,255,255,255,255,255,0,255,1,255,255,1,255,0,1,1,0,255,0,0,0,1,1,255,1,0,0,255,255,0,0,255,1,1,0,1,1,1,255,0,255,0,0,1,255,0,0,255,0,0,1,255,255,1,255,255,1,255,0,1,1,255,0,1,255,1,0,255,1,255,0,255,255,0,1,0,1,1,1,0,1,1,0,0,255,0,255,0,255,255,0,1,255,0,1,0,0,1,1,255,1,255,1,0,0,255,0,255,1,255,255,255,0,1,0,0,255,1,255,0,1,0,1,1,0,0,1,255,255,1,1,255,255,255,1,255,1,255,1,0,1,0,1,255,1,0,1,255,255,1,1,0,255,255,0,1,0,0,1,1,255,0,0,255,1,0,255,1,255,255,255,1,255,1,0,0,255,0,1,255,255,0,0,255,1,255,255,255,1,1,255,1,255,1,255,0,0,255,0,1,255,255,0,1,0,1,255,0,1,1,0,255,1,255,1,1,1,255,0,0,0,1,1,255,255,1,0,255,255,0,0,1,0,1,255,0,1,0,0,255,255,0,255,0,0,0,1,1,0,255,1,255,0,1,255,0,0,1,1,255,0,1,0,1,1,255,1,0,1,255,1,1,255,255,0,0,255,0,255,0,255,255,255,0,0,0,0,255,255,1,255,0,1,1,1,0,1,1,0,255,255,0,255,255,0,1,0,255,255,0,255,0,0,255,255,1,255,255,0,0,0,0,255,1,1,0,255,1,1,1,0,255,255,1,255,255,255,255,255,0,0,0,0,0,255,0,255,0,0,0,1,1,1,255,255,1,1,255,255,255,0,0,255,0,0,1,255,255,1,1,1,255,0,255,255,0,255,1,1,1,0,1,1,255,255,1,255,1,0,255,0,255,255,0,1,1,255,0,1,0,1,1,0,1,255,1,0,0,1,1,1,255,1,255,1,255,255,1,0,0,255,0,0,0,255,0,255,1,255,255,1,255,0,0,255,0,1,255,1,0,1,0,255,1,255,255,0,1,0,1,255,0,255,1,0,255,255,255,1,1,1,1,255,1,1,255,1,1,255,0},
  {0,0,255,1,0,1,1,255,1,1,0,1,1,1,0,0,255,255,0,255,0,0,0,1,0,0,0,255,255,1,1,255,255,0,1,1,255,0,0,255,1,1,1,255,255,0,1,0,255,255,1,255,1,0,0,1,1,1,255,1,0,1,1,0,0,255,255,0,0,255,1,1,0,0,1,1,255,0,1,0,1,1,255,255,1,1,1,0,255,255,1,1,0,0,0,1,1,255,255,0,1,0,255,1,0,0,0,255,255,255,1,255,0,1,255,0,1,1,0,255,0,255,1,0,0,255,255,255,1,255,0,1,0,0,255,1,0,255,1,1,1,255,1,255,255,0,1,1,1,1,255,1,255,1,0,1,1,0,255,0,1,0,1,0,1,255,255,1,255,1,1,255,0,1,0,1,1,0,0,1,255,255,255,1,0,255,1,0,1,0,1,0,0,1,1,255,255,1,255,0,1,255,0,255,1,1,0,255,255,0,1,255,255,1,1,1,0,0,0,255,0,0,1,1,1,255,0,255,0,255,0,0,0,0,255,1,1,255,255,0,255,1,1,1,0,255,0,1,255,255,1,255,255,0,1,0,0,1,1,1,255,1,1,255,0,1,255,1,255,255,0,1,1,255,1,0,255,1,255,0,0,1,255,1,1,255,1,0,1,1,255,0,255,1,0,255,0,255,1,255,1,1,0,0,255,255,0,1,255,0,0,1,255,0,0,0,1,0,1,1,1,0,255,255,255,0,0,0,0,0,1,1,255,1,0,255,0,1,1,0,255,255,1,255,1,255,0,255,255,1,1,1,0,0,255,255,1,1,1,1,0,255,0,1,255,0,255,0,0,255,0,0,255,0,1,0,1,255,1,255,255,0,1,255,1,0,1,1,1,255,1,0,1,1,255,0,255,0,0,255,255,0,1,255,255,255,255,255,0,255,1,1,255,0,0,0,0,0,1,0,0,1,255,0,0,0,0,1,255,1,0,1,255,1,0,0,255,1,255,1,0,0,1,0,1,1,1,0,1,1,255,255,1,0,255,255,0,255,0,0,0,1,1,1,1,255,1,255,1,255,1,0,0,0,0,255,1,0,255,255,0,255,1,255,1,1,255,0,1,255,1,0,255,1,255,0,255,1,1,255,0,255,1,1,255,0,0,0,255,0,0,0,1,0,1,255,255,1,1,1,255,255,255,255,1,255,1,255,1,0,1,0,255,255,255,0,1,255,255,255,1,255,0,1,1,0,1,0,0,1,1,1,255,255,1,0,0,255,1,1,0,1,1,0,1,1,1,1,255,255,0,255,1,255,0,1,255,1,1,1,1,0,255,255,255,0,1,1,1,1,0,0,255,255,0,1,1,1,0,255,255,255,1,1,0,255,1,1,255,255,1,0,1,1,1,0,0,0,0,255,255,1,255,0,255,1,255,1,1,255,255,255,0,0,255,1,1,255,0,255,1,0,0,0,255,1,1,1,255,0,1,255,0,0},
  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255},
} ;

static const unsigned char precomputed_core_inv3sntrup653_n[precomputed_core_inv3sntrup653_NUM][crypto_core_INPUTBYTES] = {
  {143,212,18,81,211,14,122,14,247,132,221,205,61,159,13,229,135,142,121,208,178,17,125,28,171,142,208,178,170,171,119,246,151,113,37,154,178,172,45,29,69,156,68,100,58,13,111,73,241,44,5,142,0,40,172,149,53,170,49,181,218,104,155,17,246,249,104,40,220,231,221,63,67,24,94,113,53,246,201,238,109,40,218,37,154,162,91,191,159,178,162,22,170,255,183,175,84,183,234,120,223,47,199,74,233,135,178,43,158,101,123,232,4,246,51,150,128,22,170,198,84,212,41,237,43,75,148,221,74,81,245,30,64,94,128,146,190,126,104,33,209,198,223,39,178,132,46,104,113,101,244,128,98,23,252,183,203,217,120,88,13,53,203,103,22,252,43,97,99,0,41,169,79,22,89,236,90,97,136,152,41,217,30,47,156,170,135,217,197,51,58,17,116,168,56,213,10,163,110,13,209,234,237,119,202,14,84,155,181,87,250,226,61,72,64,135,199,181,75,170,89,226,115,171,141,93,138,91,2,222,158,202,145,24,185,59,226,67,132,163,120,61,226,167,118,196,112,36,123,144,129,70,220,157,33,233,147,237,90,86,199,137,61,158,41,185,117,201,138,69,13,25,158,247,60,129,253,64,44,251,169,89,93,111,152,216,114,91,155,176,181,241,87,159,42,49,125,234,222,235,86,139,86,179,185,82,11,92,102,170,219,118,41,144,162,114,238,211,192,217,166,15,25,103,223,67,95,27,167,253,227,29,222,116,38,44,12,183,185,142,120,28,107,154,132,93,70,145,71,248,215,25,118,144,129,118,62,166,16,61,244,62,6,139,136,91,7,126,29,218,69,234,101,142,249,205,106,140,141,161,216,211,138,218,114,198,229,208,63,74,98,184,18,140,243,46,147,221,7,85,97,134,132,83,14,139,4,213,3,217,238,22,125,0,116,107,243,225,110,139,109,171,181,225,19,2,231,247,254,101,234,25,183,111,97,65,189,222,15,111,25,114,239,91,85,253,23,163,165,74,57,88,98,117,204,229,213,156,138,99,168,82,52,118,196,32,48,52,142,12,93,190,81,198,255,179,123,181,218,54,219,20,189,92,175,16,44,97,176,48,16,134,1,187,213,221,80,0,204,195,55,133,40,24,238,83,130,94,200,194,139,116,231,2,20,34,66,188,87,178,25,199,63,56,34,221,213,147,160,116,56,5,235,29,143,189,160,170,78,140,242,177,104,219,66,182,139,24,159,204,240,103,47,74,157,119,47,226,136,158,191,111,77,1,232,125,165,18,142,120,134,190,219,57,225,244,163,236,18,151,53,133,163,8,250,22,157,230,247,195,122,31,192,36,39,143,9,15,238,183,54,253,127,31,125,229,78,150,210,212,110,71,149,218,191,199,193,82,58,47,181,98,176,243,238,49,198,8,183,50,40,8,65,128,116,246,12,96,220,44,169,107,34,22,191,82,157,242,12,127,245,223,138},
  {100,61,62,13,119,214,210,98,90,231,132,235,230,119,183,10,122,47,228,230,5,104,220,122,13,24,42,96,217,67,64,57,255,165,189,10,110,166,13,194,244,166,83,224,254,85,45,132,43,62,111,151,73,249,82,180,142,213,92,88,165,18,17,36,179,71,10,159,122,3,162,40,242,22,77,77,16,61,150,104,112,169,39,110,3,37,227,31,249,168,236,212,158,164,202,52,52,210,212,158,189,80,181,124,106,121,140,42,94,63,163,79,166,206,204,11,118,19,190,109,164,142,148,25,175,194,186,27,217,152,179,114,18,141,32,202,191,138,254,238,203,71,79,5,157,135,250,238,146,207,146,162,199,214,119,85,52,115,211,227,114,122,177,188,86,18,252,10,251,32,182,22,88,63,88,135,145,253,3,123,228,11,68,249,133,90,115,182,244,2,202,191,134,217,186,74,108,101,129,13,210,48,154,131,215,59,137,253,237,242,55,56,182,16,147,185,49,192,31,241,22,146,232,177,99,194,242,52,182,177,150,28,13,240,145,238,46,74,25,117,174,247,49,176,216,54,71,162,23,231,116,63,198,159,204,74,206,73,58,245,226,121,202,69,109,45,234,236,123,200,35,177,54,167,221,149,172,19,172,45,148,3,175,188,154,10,69,213,145,184,15,253,6,145,125,72,76,251,80,55,64,174,130,149,20,162,84,88,103,92,202,243,223,244,66,117,237,201,46,214,14,222,77,211,231,163,46,149,216,53,194,172,152,192,201,129,50,113,113,223,198,126,183,127,54,33,226,41,96,185,241,179,0,104,243,119,131,213,238,98,100,35,178,77,68,58,209,220,147,190,124,155,33,143,123,160,81,90,181,233,144,96,81,101,63,249,211,87,129,32,206,232,30,125,145,98,170,197,99,4,131,76,36,200,103,200,130,244,129,83,65,211,202,122,2,70,229,22,192,79,247,191,33,220,157,245,30,41,253,88,189,170,223,24,195,192,221,119,216,66,39,0,27,29,24,72,136,251,155,111,209,90,213,117,203,107,130,35,254,230,227,77,79,135,251,251,71,41,114,7,236,217,4,163,118,104,190,60,112,67,159,239,185,165,248,12,121,12,211,59,229,227,38,40,245,75,248,242,223,80,161,111,254,195,173,115,34,75,32,246,150,134,123,173,161,205,86,161,27,211,150,112,168,231,105,136,242,59,2,34,138,61,198,132,63,102,77,110,10,117,190,255,207,32,186,223,21,169,81,64,42,211,8,70,35,163,119,46,238,106,21,173,12,242,71,230,241,170,65,152,24,211,156,142,169,175,117,17,130,84,207,5,173,106,129,245,236,238,7,113,18,76,179,210,44,148,175,22,116,5,170,138,253,152,132,138,43,68,63,50,198,25,146,142,52,123,45,93,133,155,67,7,199,210,218,47,41,117,100,254,203,126,175,169,23,96,116,104,140,161,31,178,202,68,65,120,116,92,24,75,218,29,42},
  {54,54,128,108,129,144,224,29,239,228,161,105,251,250,226,54,188,150,245,230,240,206,250,94,96,112,28,131,220,248,235,247,180,24,135,79,241,37,11,32,45,92,43,201,250,24,40,45,199,96,133,92,38,233,237,179,103,31,195,130,227,123,181,159,42,97,35,246,61,94,48,84,74,109,85,106,161,27,156,121,135,51,124,208,98,245,166,95,119,100,48,234,185,243,164,81,139,125,196,112,11,191,237,81,4,106,81,43,53,13,106,223,225,153,59,23,88,69,97,193,236,203,38,189,229,123,150,223,61,111,167,208,221,193,138,204,119,128,236,216,92,202,240,9,196,202,166,31,176,94,225,233,82,25,14,113,57,254,24,39,93,9,75,108,159,144,195,219,27,247,182,35,151,160,237,20,248,16,145,12,150,109,173,220,236,70,20,123,103,214,209,146,115,76,181,60,241,91,82,226,53,43,203,178,13,11,59,122,207,134,66,100,238,34,112,252,70,26,136,105,53,220,137,169,69,221,42,172,251,83,130,1,26,119,9,176,109,101,10,11,146,155,166,235,171,189,202,60,122,89,84,199,180,237,31,200,132,118,129,146,118,23,1,175,243,185,160,250,184,116,147,21,55,95,73,141,126,164,52,27,135,118,187,176,36,69,116,248,131,123,195,69,78,62,240,140,93,2,3,6,219,54,28,156,148,225,132,125,63,222,158,33,221,177,159,103,131,138,252,140,135,101,24,84,147,154,59,177,67,143,149,0,200,48,11,112,17,91,182,40,162,215,44,83,87,59,62,49,217,74,234,242,63,226,101,37,168,231,73,178,250,45,144,64,80,126,206,220,78,230,67,140,7,151,145,1,12,166,243,107,178,85,16,253,248,101,252,30,114,83,95,24,44,66,10,246,43,247,217,148,40,40,172,85,192,67,24,9,97,205,231,168,103,244,3,255,54,32,86,169,60,228,21,114,148,230,35,15,68,103,147,127,244,0,108,177,83,70,40,96,134,168,38,96,72,34,57,25,150,132,195,138,199,1,135,190,84,89,46,137,150,208,2,190,225,51,11,11,16,61,6,185,3,221,91,96,141,239,177,143,109,159,57,109,38,49,119,75,30,148,236,144,172,41,85,151,118,110,99,146,195,24,125,5,171,165,190,245,248,66,84,187,153,155,58,20,218,83,242,185,245,91,106,27,178,34,226,52,25,183,76,203,122,76,41,19,100,146,44,5,36,27,130,73,198,126,115,192,51,255,43,0,43,72,255,105,116,42,205,13,225,179,56,100,236,215,243,137,157,241,42,251,125,66,195,191,170,238,124,138,10,40,125,242,56,126,112,162,51,0,34,68,202,129,140,216,244,226,187,160,91,65,148,124,11,169,184,94,113,44,122,1,205,45,150,173,182,94,52,95,0,177,115,4,170,186,58,39,54,223,33,83,139,93,173,68,184,116,20,69,228,254,92,213,231,179,85,189,134,232,176,58,40},
  {164,51,145,9,15,60,140,91,41,47,91,139,53,96,108,33,94,235,229,181,228,14,99,213,117,212,73,249,109,32,197,237,51,50,209,50,253,2,181,225,212,221,253,182,100,154,42,214,228,76,249,213,2,93,128,194,197,187,171,31,229,204,214,222,65,165,242,91,95,15,168,88,105,7,178,65,44,1,105,190,30,88,106,142,200,162,166,80,165,36,4,236,216,236,6,79,144,66,69,226,38,123,137,41,220,206,198,98,70,47,162,51,236,228,223,248,140,31,242,124,174,197,42,185,239,217,146,156,193,151,123,75,59,150,164,194,215,67,67,79,225,2,150,235,98,135,19,167,53,95,15,164,172,13,39,25,46,191,111,64,209,157,3,109,67,123,204,118,51,40,76,187,86,198,137,45,33,90,26,176,146,51,10,95,170,108,121,38,113,94,106,40,97,37,222,90,54,113,12,218,7,75,43,90,238,83,181,7,253,47,223,219,125,6,27,13,189,146,140,20,143,141,18,54,254,174,158,100,59,200,44,162,24,108,139,153,170,5,245,156,182,81,194,58,164,240,166,209,149,247,65,126,231,181,82,166,48,210,191,223,155,144,27,27,210,225,151,135,221,193,44,244,118,78,162,87,97,248,64,39,179,109,10,120,63,231,208,3,71,112,247,25,173,146,103,176,164,27,42,159,149,248,64,129,121,108,240,42,196,211,210,51,231,203,37,181,14,125,138,248,217,79,137,97,172,194,33,245,65,215,201,224,226,218,93,83,152,102,184,226,204,18,82,170,167,84,239,242,229,232,31,173,190,172,228,51,98,163,248,38,21,152,65,27,209,7,187,113,48,3,38,202,200,232,65,102,237,243,194,3,166,161,58,74,210,187,38,52,93,195,230,234,19,36,72,216,232,234,148,58,213,39,126,143,246,31,243,240,167,18,29,190,44,95,94,163,178,113,151,94,190,34,109,147,118,103,209,6,53,47,205,223,204,53,52,230,222,126,193,241,232,153,138,106,160,237,215,123,75,29,217,42,192,15,253,22,51,21,105,192,11,201,152,32,173,93,63,37,154,213,14,134,41,61,102,127,24,227,116,115,108,45,52,112,149,159,103,160,179,213,36,252,161,6,161,18,64,75,66,49,7,143,232,79,86,92,4,245,177,90,117,203,133,212,229,183,225,118,28,105,19,40,224,180,46,232,220,8,149,192,29,40,86,191,94,30,236,44,176,43,68,86,54,84,1,31,19,124,107,49,158,238,0,72,24,73,84,58,247,43,28,200,249,184,27,81,115,159,58,113,36,180,150,112,69,218,253,54,187,113,32,77,199,194,151,174,155,225,116,104,150,148,55,247,89,207,4,136,208,72,201,72,165,200,162,47,98,203,244,239,188,80,6,23,66,19,78,208,194,37,46,247,25,32,222,93,252,172,109,250,142,228,23,33,227,199,227,232,206,140,1,157,223,241,217,253,12,143,195,17,228,1,122},
  {255,255,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
} ;

static const unsigned char precomputed_core_inv3sntrup653_k[precomputed_core_inv3sntrup653_NUM][crypto_core_KEYBYTES] = {
  {},
  {},
  {},
  {},
  {},
} ;

static const unsigned char precomputed_core_inv3sntrup653_c[precomputed_core_inv3sntrup653_NUM][crypto_core_CONSTBYTES] = {
  {},
  {},
  {},
  {},
  {},
} ;

static void test_core_inv3sntrup653_impl(long long impl)
{
  unsigned char *h = test_core_inv3sntrup653_h;
  unsigned char *n = test_core_inv3sntrup653_n;
  unsigned char *k = test_core_inv3sntrup653_k;
  unsigned char *c = test_core_inv3sntrup653_c;
  unsigned char *h2 = test_core_inv3sntrup653_h2;
  unsigned char *n2 = test_core_inv3sntrup653_n2;
  unsigned char *k2 = test_core_inv3sntrup653_k2;
  unsigned char *c2 = test_core_inv3sntrup653_c2;
  long long hlen = crypto_core_OUTPUTBYTES;
  long long nlen = crypto_core_INPUTBYTES;
  long long klen = crypto_core_KEYBYTES;
  long long clen = crypto_core_CONSTBYTES;

  if (targeti && strcmp(targeti,".") && strcmp(targeti,ntruprime_dispatch_core_inv3sntrup653_implementation(impl))) return;
  if (targetn && atol(targetn) != impl) return;
  if (impl >= 0) {
    crypto_core = ntruprime_dispatch_core_inv3sntrup653(impl);
    printf("core_inv3sntrup653 %lld implementation %s compiler %s\n",impl,ntruprime_dispatch_core_inv3sntrup653_implementation(impl),ntruprime_dispatch_core_inv3sntrup653_compiler(impl));
  } else {
    crypto_core = ntruprime_core_inv3sntrup653;
    printf("core_inv3sntrup653 selected implementation %s compiler %s\n",ntruprime_core_inv3sntrup653_implementation(),ntruprime_core_inv3sntrup653_compiler());
  }
  for (long long checksumbig = 0;checksumbig < 2;++checksumbig) {
    long long loops = checksumbig ? 4096 : 512;

    checksum_clear();

    for (long long loop = 0;loop < loops;++loop) {

      output_prepare(h2,h,hlen);
      input_prepare(n2,n,nlen);
      input_prepare(k2,k,klen);
      input_prepare(c2,c,clen);
      secret(n,nlen);
      secret(k,klen);
      secret(c,clen);
      crypto_core(h,n,k,c);
      public(n,nlen);
      public(k,klen);
      public(c,clen);
      public(h,hlen);
      checksum(h,hlen);
      output_compare(h2,h,hlen,"crypto_core");
      input_compare(n2,n,nlen,"crypto_core");
      input_compare(k2,k,klen,"crypto_core");
      input_compare(c2,c,clen,"crypto_core");

      double_canary(h2,h,hlen);
      double_canary(n2,n,nlen);
      double_canary(k2,k,klen);
      double_canary(c2,c,clen);
      secret(n2,nlen);
      secret(k2,klen);
      secret(c2,clen);
      crypto_core(h2,n2,k2,c2);
      public(n2,nlen);
      public(k2,klen);
      public(c2,clen);
      public(h2,hlen);
      if (memcmp(h2,h,hlen) != 0) fail("failure: crypto_core is nondeterministic\n");
    }
    checksum_expected(core_inv3sntrup653_checksums[checksumbig]);
  }
  for (long long precomp = 0;precomp < precomputed_core_inv3sntrup653_NUM;++precomp) {
    output_prepare(h2,h,crypto_core_OUTPUTBYTES);
    input_prepare(n2,n,crypto_core_INPUTBYTES);
    memcpy(n,precomputed_core_inv3sntrup653_n[precomp],crypto_core_INPUTBYTES);
    memcpy(n2,precomputed_core_inv3sntrup653_n[precomp],crypto_core_INPUTBYTES);
    input_prepare(k2,k,crypto_core_KEYBYTES);
    memcpy(k,precomputed_core_inv3sntrup653_k[precomp],crypto_core_KEYBYTES);
    memcpy(k2,precomputed_core_inv3sntrup653_k[precomp],crypto_core_KEYBYTES);
    input_prepare(c2,c,crypto_core_CONSTBYTES);
    memcpy(c,precomputed_core_inv3sntrup653_c[precomp],crypto_core_CONSTBYTES);
    memcpy(c2,precomputed_core_inv3sntrup653_c[precomp],crypto_core_CONSTBYTES);
    crypto_core(h,n,k,c);
    if (memcmp(h,precomputed_core_inv3sntrup653_h[precomp],crypto_core_OUTPUTBYTES)) {
      fail("failure: crypto_core fails precomputed test vectors\n");
      printf("expected h: ");
      for (long long pos = 0;pos < crypto_core_OUTPUTBYTES;++pos) printf("%02x",((unsigned char *) precomputed_core_inv3sntrup653_h[precomp])[pos]);
      printf("\n");
      printf("received h: ");
      for (long long pos = 0;pos < crypto_core_OUTPUTBYTES;++pos) printf("%02x",h[pos]);
      printf("\n");
    }
    output_compare(h2,h,crypto_core_OUTPUTBYTES,"crypto_core");
    input_compare(n2,n,crypto_core_INPUTBYTES,"crypto_core");
    input_compare(k2,k,crypto_core_KEYBYTES,"crypto_core");
    input_compare(c2,c,crypto_core_CONSTBYTES,"crypto_core");
  }
}

void test_core_inv3sntrup653(void)
{
  long long maxalloc = 0;
  if (targeto && strcmp(targeto,"core")) return;
  if (targetp && strcmp(targetp,"inv3sntrup653")) return;
  storage_core_inv3sntrup653_h = callocplus(crypto_core_OUTPUTBYTES);
  test_core_inv3sntrup653_h = aligned(storage_core_inv3sntrup653_h,crypto_core_OUTPUTBYTES);
  if (crypto_core_OUTPUTBYTES > maxalloc) maxalloc = crypto_core_OUTPUTBYTES;
  storage_core_inv3sntrup653_n = callocplus(crypto_core_INPUTBYTES);
  test_core_inv3sntrup653_n = aligned(storage_core_inv3sntrup653_n,crypto_core_INPUTBYTES);
  if (crypto_core_INPUTBYTES > maxalloc) maxalloc = crypto_core_INPUTBYTES;
  storage_core_inv3sntrup653_k = callocplus(crypto_core_KEYBYTES);
  test_core_inv3sntrup653_k = aligned(storage_core_inv3sntrup653_k,crypto_core_KEYBYTES);
  if (crypto_core_KEYBYTES > maxalloc) maxalloc = crypto_core_KEYBYTES;
  storage_core_inv3sntrup653_c = callocplus(crypto_core_CONSTBYTES);
  test_core_inv3sntrup653_c = aligned(storage_core_inv3sntrup653_c,crypto_core_CONSTBYTES);
  if (crypto_core_CONSTBYTES > maxalloc) maxalloc = crypto_core_CONSTBYTES;
  storage_core_inv3sntrup653_h2 = callocplus(maxalloc);
  test_core_inv3sntrup653_h2 = aligned(storage_core_inv3sntrup653_h2,crypto_core_OUTPUTBYTES);
  storage_core_inv3sntrup653_n2 = callocplus(maxalloc);
  test_core_inv3sntrup653_n2 = aligned(storage_core_inv3sntrup653_n2,crypto_core_INPUTBYTES);
  storage_core_inv3sntrup653_k2 = callocplus(maxalloc);
  test_core_inv3sntrup653_k2 = aligned(storage_core_inv3sntrup653_k2,crypto_core_KEYBYTES);
  storage_core_inv3sntrup653_c2 = callocplus(maxalloc);
  test_core_inv3sntrup653_c2 = aligned(storage_core_inv3sntrup653_c2,crypto_core_CONSTBYTES);

  for (long long offset = 0;offset < 2;++offset) {
    if (targetoffset && atol(targetoffset) != offset) continue;
    if (offset && valgrind) break;
    printf("core_inv3sntrup653 offset %lld\n",offset);
    for (long long impl = -1;impl < ntruprime_numimpl_core_inv3sntrup653();++impl)
      forked(test_core_inv3sntrup653_impl,impl);
    ++test_core_inv3sntrup653_h;
    ++test_core_inv3sntrup653_n;
    ++test_core_inv3sntrup653_k;
    ++test_core_inv3sntrup653_c;
    ++test_core_inv3sntrup653_h2;
    ++test_core_inv3sntrup653_n2;
    ++test_core_inv3sntrup653_k2;
    ++test_core_inv3sntrup653_c2;
  }
  free(storage_core_inv3sntrup653_c2);
  free(storage_core_inv3sntrup653_k2);
  free(storage_core_inv3sntrup653_n2);
  free(storage_core_inv3sntrup653_h2);
  free(storage_core_inv3sntrup653_c);
  free(storage_core_inv3sntrup653_k);
  free(storage_core_inv3sntrup653_n);
  free(storage_core_inv3sntrup653_h);
}
#undef crypto_core_OUTPUTBYTES
#undef crypto_core_INPUTBYTES
#undef crypto_core_KEYBYTES
#undef crypto_core_CONSTBYTES

