use ExtUtils::MakeMaker;
use strict;
use warnings;

my $MIN_OIS_VERSION = '1.3.0';

main();
exit 0;

sub main {
    my $fixed_args = fixed_args('OIS');
    my $varied_args = varied_args();
    my $gpp_warn_args = gpp_warn_args();

    WriteMakefile(%$fixed_args, %$varied_args, %$gpp_warn_args);
}

sub fixed_args {
    my ($pkg) = @_;

    return {
        'NAME'          => $pkg,
        'VERSION_FROM'  => "$pkg.pm",
        'ABSTRACT_FROM' => "$pkg.pm",
        'AUTHOR'        => 'Scott Lanning <slanning@cpan.org>',
        'BUILD_REQUIRES' => {
            'ExtUtils::ParseXS' => '2.2206',    # INCLUDE_COMMAND
        },
        'CC'            => $ENV{'CXX'} || 'g++',
        'LD'            => '$(CC)',
        'OBJECT'        => '$(O_FILES)',
        'XSOPT'         => '-C++',
        'TYPEMAPS'      => ['perlobject.map'],
    };
}

sub gpp_warn_args {
    # If you're using gcc >= 4.2, you'll probably
    # get warnings like this during `make` :
    #  OIS.c:1835: warning: deprecated conversion from string constant to 'char*'
    # The C code generated by `xsubpp` uses literal strings (string constants)
    # as args to functions expecting char*. This disables those warnings....

    if (my_compiler() eq 'g++') {
        my $str = `g++ -dumpversion`;
        unless ($?) {
            chomp $str;
            my ($v, $s) = split /\./, $str;
            if (($v == 4 && $s >= 2) || $v > 4) {
                # http://www.nntp.perl.org/group/perl.perl5.porters/2012/03/msg184347.html
                my $ccflags = $Config::Config{ccflags} || '';
                return {'CCFLAGS' => "-Wno-write-strings $ccflags"};
            }
        }
    }

    # there will be no warnings, or we'll just tolerate them
    return {};
}

sub my_compiler {
    return $ENV{'CXX'} || 'g++';
}

sub varied_args {
    # Make sure libs are known by pkg-config
    my @errors = ();
    push @errors, check_pkg_config('OIS', $MIN_OIS_VERSION);
    if (@errors) {
        die(map { "$_$/" } @errors);
    }

    # Get include dirs and defines
    my @cflags = ();
    push @cflags, pkg_config('OIS', 'cflags');

    # Get lib dirs
    my @libs = ();
    push @libs, pkg_config('OIS', 'libs');

    return {
        'PREREQ_PM'         => {
            'Scalar::Util' => 1.18,   # bugfix for looks_like_number
            'Test::More' => 0,
        },
        'INC'               => join(' ', @cflags, '-I.'),
        'LIBS'              => join(' ', @libs),
    };
}

sub check_pkg_config {
    my ($pkg, $required_version) = @_;

    my $pc = 'pkg-config';
    my @errors = ();

    # Check that pkg-config is installed
    my $pcver = `$pc --version`;
    if ($pcver eq '') {
        push @errors, "$pc not found";
    }
    else {
        # Check that pkg-config knows about the package
        my $pkgexists = `$pc --exists $pkg`;
        if ($?) {
            push @errors, "Package $pkg not found by $pc";
        }
        else {
            # Check that the package is the right version
            my $pkgver = `$pc --atleast-version='$required_version' $pkg`;
            if ($?) {
                push @errors, "Package $pkg is not the right version (at least $required_version)";
            }
        }
    }

    return @errors;
}

sub pkg_config {
    my ($pkg, $option) = @_;

    my $pc = 'pkg-config';

    my $str = `$pc --$option $pkg`;
    if ($?) {
        die "$pc --$option $pkg: $str\n";
    }
    else {
        chomp $str;
        return $str;
    }
}
