#ifndef _EFFECTUTIL_TCC
#define _EFFECTUTIL_TCC

#include "osl/effect_util/effectUtil.h"
#include "osl/effect_action/storePiece.h"
#include "osl/move_classifier/kingOpenMove.h"
#include "osl/container/pieceVector.h"
#include <boost/static_assert.hpp>

namespace osl
{
  namespace effect_util
  {
    template <osl::Player P, bool InterestEmpty, Direction Dir>
    struct TestEffectOfMove
    {
      template <class State, class Function>
      static void testShort(const State& s, int mask, Square from, 
			    Function& f)
      {
	BOOST_STATIC_ASSERT(! DirectionTraits<Dir>::isLong);
	if (! (mask & DirectionTraits<Dir>::mask))
	  return;

	const Offset offset = DirectionPlayerTraits<Dir,P>::offset();
	const Square target = from+offset;
	const Piece piece = s.pieceAt(target);
	if (piece.isEdge())
	  return;
	if (InterestEmpty || (! piece.isEmpty()))
	  f(target);
      }
      template <class State, class Function>
      static void testLong(const State& s, int mask, Square from, 
			    Function& f)
      {
	BOOST_STATIC_ASSERT(DirectionTraits<Dir>::isLong);
	if (! (mask & DirectionTraits<Dir>::mask))
	  return;

	const Offset offset = DirectionPlayerTraits<Dir,P>::offset();

	Square target = from+offset;
	Piece piece = s.pieceAt(target);
	while (piece.isEmpty())
	{
	  if (InterestEmpty)
	    f(target);		
	  target = target+offset;
	  piece = s.pieceAt(target);
	}
	if (piece.isPiece())
	{
	  f(target);
	}
      }
    };
  } // namespace effect_util
} // namespace osl

template <osl::Player P, class Function, bool InterestEmpty>
void osl::effect_util::EffectUtil::
forEachEffectOfPtypeO(const NumEffectState& state, Square from, Ptype ptype,
		      Function& f)
{
  const int mask = Ptype_Table.getMoveMask(ptype);
  TestEffectOfMove<P,InterestEmpty,UL>::testShort(state, mask, from, f);
  TestEffectOfMove<P,InterestEmpty,U>::testShort(state, mask, from, f);
  TestEffectOfMove<P,InterestEmpty,UR>::testShort(state, mask, from, f);
  TestEffectOfMove<P,InterestEmpty,L>::testShort(state, mask, from, f);
  TestEffectOfMove<P,InterestEmpty,R>::testShort(state, mask, from, f);
  TestEffectOfMove<P,InterestEmpty,DL>::testShort(state, mask, from, f);
  TestEffectOfMove<P,InterestEmpty,D>::testShort(state, mask, from, f);
  TestEffectOfMove<P,InterestEmpty,DR>::testShort(state, mask, from, f);
  TestEffectOfMove<P,InterestEmpty,UUL>::testShort(state, mask, from, f);
  TestEffectOfMove<P,InterestEmpty,UUR>::testShort(state, mask, from, f);
  TestEffectOfMove<P,InterestEmpty,LONG_UL>::testLong(state, mask, from, f);
  TestEffectOfMove<P,InterestEmpty,LONG_U>::testLong(state, mask, from, f);
  TestEffectOfMove<P,InterestEmpty,LONG_UR>::testLong(state, mask, from, f);
  TestEffectOfMove<P,InterestEmpty,LONG_L>::testLong(state, mask, from, f);
  TestEffectOfMove<P,InterestEmpty,LONG_R>::testLong(state, mask, from, f);
  TestEffectOfMove<P,InterestEmpty,LONG_DL>::testLong(state, mask, from, f);
  TestEffectOfMove<P,InterestEmpty,LONG_D>::testLong(state, mask, from, f);
  TestEffectOfMove<P,InterestEmpty,LONG_DR>::testLong(state, mask, from, f);
}

template <class Function, bool InterestEmpty>
void osl::effect_util::EffectUtil::
forEachEffectOfPtypeO(const NumEffectState& state, Square from, PtypeO ptypeo,
		      Function& f)
{
  const Player P = getOwner(ptypeo);
  if (P == BLACK)
    forEachEffectOfPtypeO<BLACK,Function,InterestEmpty>
      (state, from, getPtype(ptypeo), f);
  else
    forEachEffectOfPtypeO<WHITE,Function,InterestEmpty>
      (state, from, getPtype(ptypeo), f);
}

struct osl::effect_util::EffectUtil::SafeCapture
{
public:
  const NumEffectState& state;
  Piece safe_one;
  SafeCapture(const NumEffectState& s) : state(s), safe_one(Piece::EMPTY())
  {
  }
  template <Player P>
  void doAction(Piece effect_piece, Square target)
  {
    if (move_classifier::KingOpenMove<P>::isMember
	(state, effect_piece.ptype(), effect_piece.square(), target))
      return;
    safe_one = effect_piece;
  }
};

template <osl::Player P>
osl::Piece
osl::effect_util::EffectUtil::safeCaptureNotByKing(const NumEffectState& state, Square target,
						   Piece king)
{
  assert(king.owner() == P);
  assert(king.ptype() == KING);
  PieceMask ignore = state.pin(P);
  ignore.set(king.number());
  const Piece piece = state.findAttackNotBy(P, target, ignore);
  if (piece.isPiece())
    return piece;
  SafeCapture safe_captures(state);
  state.template forEachEffectNotBy<P>(target, king, safe_captures);
  
  return safe_captures.safe_one;
}

template <class EvalT>
struct osl::effect_util::EffectUtil::FindThreat
{
  const NumEffectState& state;
  Player target;
  int attacker_value;
  PieceVector& supported, & unsupported;
  FindThreat(const NumEffectState& st, Player t, int a,
	     PieceVector& s, PieceVector& u)
    : state(st), target(t), attacker_value(a), supported(s), unsupported(u)
  {
  }
  void operator()(Square pos)
  {
    const Piece cur = state.pieceOnBoard(pos);
    assert(cur.isPiece());
    if (cur.owner() != target)
      return;
    if (state.hasEffectAt(target, pos))
    {
      if (abs(EvalT::captureValue(cur.ptypeO()))
	  > attacker_value)
	supported.push_back(cur);
    }
    else
    {
      unsupported.push_back(cur);
    }
  }
};

template <class EvalT>
void osl::EffectUtil::
findThreat(const NumEffectState& state, Square position,
	   PtypeO ptypeo, PieceVector& out)
{
  PieceVector supported, unsupported;
  const int attacker_value = abs(EvalT::captureValue(ptypeo));
  FindThreat<EvalT> f(state, alt(getOwner(ptypeo)), attacker_value, 
		      supported, unsupported);
  forEachEffectOfPtypeO<FindThreat<EvalT>, false>
    (state, position, ptypeo, f);

  unsupported.sortByPtype();
  supported.sortByPtype();
  PieceVector::iterator u=unsupported.begin(), s=supported.begin();

  if (u!=unsupported.end())
  {
    while ((s!=supported.end()) 
	   && ((abs(EvalT::captureValue(s->ptypeO()))
		- attacker_value)
	       > abs(EvalT::captureValue(u->ptypeO()))))
    {
      out.push_back(*s);
      ++s;
    }
  }
  out.push_back(u, unsupported.end());
  out.push_back(s, supported.end());
}

#endif /* _EFFECTUTIL_TCC */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
