/*
 * Copyright © INRIA 2009
 * Brice Goglin <Brice.Goglin@inria.fr>
 *
 * This software is a computer program whose purpose is to provide
 * a fast inter-process communication subsystem.
 *
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided only
 * with a limited warranty  and the software's author,  the holder of the
 * economic rights,  and the successive licensors  have only  limited
 * liability.
 *
 * In this respect, the user's attention is drawn to the risks associated
 * with loading,  using,  modifying and/or developing or reproducing the
 * software by the user in light of its specific status of free software,
 * that may mean  that it is complicated to manipulate,  and  that  also
 * therefore means  that it is reserved for developers  and  experienced
 * professionals having in-depth computer knowledge. Users are therefore
 * encouraged to load and test the software's suitability as regards their
 * requirements in conditions enabling the security of their systems and/or
 * data to be ensured and,  more generally, to use and operate it in the
 * same conditions as regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL-B license and that you accept its terms.
 */

#ifndef __knem_hal_h__
#define __knem_hal_h__

#include <linux/vmalloc.h>
#include <linux/mm.h>

#ifdef KNEM_HAVE_REMAP_VMALLOC_RANGE
#define knem_vmalloc_user vmalloc_user
#define knem_remap_vmalloc_range remap_vmalloc_range
#else /* KNEM_HAVE_REMAP_VMALLOC_RANGE */
static inline void *
knem_vmalloc_user(unsigned long size)
{
	return __vmalloc(size, GFP_KERNEL | __GFP_HIGHMEM | __GFP_ZERO, PAGE_KERNEL);
}

static inline int
knem_remap_vmalloc_range(struct vm_area_struct *vma, void *addr, unsigned long pgoff)
{
	unsigned long uaddr = vma->vm_start;
	unsigned long usize = vma->vm_end - vma->vm_start;
	int ret;

	addr += pgoff << PAGE_SHIFT;
	do {
		struct page *page = vmalloc_to_page(addr);
		ret = vm_insert_page(vma, uaddr, page);
		if (ret)
			return ret;

		uaddr += PAGE_SIZE;
		addr += PAGE_SIZE;
		usize -= PAGE_SIZE;
	} while (usize > 0);

	/* Prevent "things" like memory migration? VM_flags need a cleanup... */
	vma->vm_flags |= VM_RESERVED;

	return ret;
}
#endif /* KNEM_HAVE_REMAP_VMALLOC_RANGE */

#ifdef KNEM_CPUMASK_SCNPRINTF_USES_PTR
#define knem_cpumask_scnprintf(buf, len, mask) cpumask_scnprintf(buf, len, mask)
#else
#define knem_cpumask_scnprintf(buf, len, mask) cpumask_scnprintf(buf, len, *mask)
#endif

#ifdef KNEM_HAVE_OLD_DMA_ENGINE_API

/* kernel <= 2.6.28 with DMA engine support through NET_DMA */
#ifdef CONFIG_NET_DMA
#define KNEM_HAVE_DMA_ENGINE 1
#include <linux/netdevice.h>
#include <net/netdma.h>
static inline struct dma_chan * knem_get_dma_channel(void) { return get_softnet_dma(); }
static inline void knem_put_dma_channel(struct dma_chan *chan) { if (chan) dma_chan_put(chan); }
static inline int knem_dma_channel_avail(void) { return __get_cpu_var(softnet_data).net_dma != NULL; }
#endif

#elif defined KNEM_HAVE_DMA_ENGINE_API

/* kernel >= 2.6.29 with nice DMA engine suport */
#ifdef CONFIG_DMA_ENGINE
#define KNEM_HAVE_DMA_ENGINE 1
#include <linux/dmaengine.h>
static inline struct dma_chan * knem_get_dma_channel(void) { dmaengine_get(); return dma_find_channel(DMA_MEMCPY); }
static inline void knem_put_dma_channel(struct dma_chan *chan) { dmaengine_put(); }
static inline int knem_dma_channel_avail(void) { return dma_find_channel(DMA_MEMCPY) != NULL; }
#endif

#else /* !KNEM_HAVE_{,OLD_}DMA_ENGINE_API */

/* kernel <= 2.6.17 with no DMA engine at all */

#endif /* !KNEM_HAVE_{,OLD_}DMA_ENGINE_API */

#endif /* __knem_hal_h__ */
