///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/animation/AnimManager.h>
#include <core/utilities/ProgressIndicator.h>

#include "DXAFileParser.h"
#include "DXAParserSettingsDialog.h"
#include <atomviz/atoms/AtomsObject.h>
#include <atomviz/atoms/datachannels/AtomTypeDataChannel.h>
#include <atomviz/modifier/analysis/cna/CNAModifier.h>
#include "../CompressedTextParserStream.h"

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(DXAFileParser, MultiFileParser)

/******************************************************************************
* Opens the settings dialog for this parser.
******************************************************************************/
bool DXAFileParser::showSettingsDialog(QWidget* parent)
{
	DXAParserSettingsDialog dialog(this, parent);
	if(dialog.exec() != QDialog::Accepted)
		return false;
	return true;
}

/******************************************************************************
* Checks if the given file has format that can be read by this importer.
******************************************************************************/
bool DXAFileParser::checkFileFormat(const QString& filepath)
{
	// Open the input file for reading.
	CompressedTextParserStream stream(filepath);

	char buffer[1024];
	int count = stream.getline(buffer, sizeof(buffer)/sizeof(buffer[0]));
	if(count < 0 || count >= sizeof(buffer)/sizeof(buffer[0])-1) return false;

	if(count<12) return false;

	return (qstrncmp(buffer, "# DXA LAMMPS", 12) == 0);
}

/******************************************************************************
* Parses the header of the given file and returns the number of data columns contained in the file.
******************************************************************************/
bool DXAFileParser::inspectFileHeader(const QString& filename, int& numberOfColumns, QStringList& columnNames)
{
	numberOfColumns = 5;
	return true;
}

/******************************************************************************
* Reads an atomic data set from the input file.
******************************************************************************/
EvaluationStatus DXAFileParser::loadTimeStep(AtomsObject* destination, int movieFrame, const QString& filename, streampos byteOffset, int lineNumber, bool suppressDialogs)
{
	CHECK_OBJECT_POINTER(destination);

	// Show the progress indicator.
	ProgressIndicator progress(tr("Opening DXA file '%1'").arg(inputFile()), 0, suppressDialogs);

	// Open the input file for reading.
	CompressedTextParserStream stream(filename, true);

	// Seek to the byte offset where the requested movie frame is stored.
	if(byteOffset != streampos(0))
		stream.seek(byteOffset);

	setlocale(LC_NUMERIC, "C");

	// Read first header line.
	stream.readline();
	if(stream.line().compare(0, 12, "# DXA LAMMPS") != 0)
		throw Exception(tr("The file %1 is not a DXA file.").arg(filename));

	// Skip comment line.
	stream.readline();

	// These are the format strings used with the sscanf() parsing function.
	// We have to obey the floating point precision used to compile the program
	// because the destination variables are either single or double precision variables.
#ifdef USE_DOUBLE_PRECISION_FP
	#define FLOAT_SCANF_STRING_1   "%lg"
	#define FLOAT_SCANF_STRING_2   "%lg %lg"
	#define FLOAT_SCANF_STRING_3   "%lg %lg %lg"
#else
	#define FLOAT_SCANF_STRING_1   "%g"
	#define FLOAT_SCANF_STRING_2   "%g %g"
	#define FLOAT_SCANF_STRING_3   "%g %g %g"
#endif

	int numNonbulkAtoms = 0;
	int numDisorderedAtoms = 0;
	int numAtoms = 0;
	int timeStep;
	Box3 simBox;

	while(!stream.eof()) {

		// Parse next line.
		stream.readline();

		do {
			if(stream.line().find("ITEM: TIMESTEP") != string::npos) {
				if(sscanf(stream.readline().c_str(), "%i", &timeStep) != 1)
					throw Exception(tr("DXA file parsing error. Invalid timestep number (line %1): %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
				break;
			}
			else if(stream.line().find("ITEM: NUMBER OF NONBULK ATOMS") != string::npos) {
				// Parse number of non-bulk atoms.
				if(sscanf(stream.readline().c_str(), "%u", &numNonbulkAtoms) != 1 || numNonbulkAtoms < 0 || numNonbulkAtoms > 1e9)
					throw Exception(tr("DXA file parsing error. Invalid number of non-bulk atoms (line %1): %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
				break;
			}
			else if(stream.line().find("ITEM: NUMBER OF DISORDERED ATOMS") != string::npos) {
				// Parse number of disordered atoms.
				if(sscanf(stream.readline().c_str(), "%u", &numDisorderedAtoms) != 1 || numDisorderedAtoms < 0 || numDisorderedAtoms > 1e9)
					throw Exception(tr("DXA file parsing error. Invalid number of disordered atoms (line %1): %2").arg(stream.lineNumber()).arg(stream.line().c_str()));

				numAtoms = numDisorderedAtoms + numNonbulkAtoms;
				progress.setLabelText(tr("Loading DXA file (%1 atoms at time step %2)").arg(numAtoms).arg(timeStep));
				progress.setMaximum(numAtoms);
				break;
			}
			else if(stream.line().find("ITEM: BOX BOUNDS xy xz yz") != string::npos) {
				// Parse triclinic simulation box.
				FloatType tiltFactors[3];
				for(size_t k=0; k<3; k++) {
					if(sscanf(stream.readline().c_str(), FLOAT_SCANF_STRING_3, &simBox.minc[k], &simBox.maxc[k], &tiltFactors[k]) != 3)
						throw Exception(tr("Invalid box size in line %1 of DXA file: %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
				}
				// LAMMPS only stores the outer bounding box of the simulation cell in the dump file.
				// We have to determine the size of the actual triclinic cell.
				simBox.minc.X -= min(min(tiltFactors[0], tiltFactors[1]), (FloatType)0.0);
				simBox.maxc.X -= max(max(tiltFactors[0], tiltFactors[1]), (FloatType)0.0);
				simBox.minc.Y -= min(tiltFactors[2], (FloatType)0.0);
				simBox.maxc.Y -= max(tiltFactors[2], (FloatType)0.0);
				destination->simulationCell()->setCellShape(simBox.minc, Vector3(simBox.sizeX(), 0, 0), Vector3(tiltFactors[0],simBox.sizeY(), 0), Vector3(tiltFactors[1],tiltFactors[2],simBox.sizeZ()));
				break;
			}
			else if(stream.line().find("ITEM: BOX BOUNDS") != string::npos) {
				// Parse orthogonal simulation box size.
				for(size_t k=0; k<3; k++) {
					if(sscanf(stream.readline().c_str(), FLOAT_SCANF_STRING_2, &simBox.minc[k], &simBox.maxc[k]) != 2)
						throw Exception(tr("Invalid box size in line %1 of DXA file: %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
				}
				destination->simulationCell()->setBoxShape(simBox);
				break;
			}
			else if(stream.line().find("ITEM: PERIODIC BOUNDARY CONDITIONS") != string::npos) {
				// Parse PBC flags
				int pbcFlags[3];
				if(sscanf(stream.readline().c_str(), "%u %u %u", &pbcFlags[0], &pbcFlags[1], &pbcFlags[2]) != 3)
					throw Exception(tr("Invalid periodic boundary condition flags in line %1 of dump file: %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
				destination->simulationCell()->setPeriodicity(pbcFlags[0], pbcFlags[1], pbcFlags[2]);
				break;
			}
			else if(stream.line().find("ITEM: ATOMS") != string::npos) {
				// Parse atom coordinates.
				destination->setAtomsCount(numAtoms);

				bool isBinaryFormat = (stream.line().find("binary") != string::npos);
				if(!isBinaryFormat)
					throw Exception(tr("DXA file in ASCII format are not yet supported."));

				// Get the required data channels.
				DataChannel* atomTagChannel = destination->createStandardDataChannel(DataChannel::AtomIndexChannel);
				OVITO_ASSERT(atomTagChannel->size() == numAtoms);

				AtomTypeDataChannel* cnaTypeChannel = static_object_cast<AtomTypeDataChannel>(destination->createStandardDataChannel(DataChannel::CNATypeChannel));
				OVITO_ASSERT(cnaTypeChannel->size() == numAtoms);

				// Remove old atoms types.
				while(!cnaTypeChannel->atomTypes().empty())
					cnaTypeChannel->removeAtomType(cnaTypeChannel->atomTypes().front());

				// Add our atom types.
				AtomType::SmartPtr fccType(new AtomType());
				fccType->setName(tr("FCC"));
				fccType->setColor(Color(0.4f, 1.0f, 0.4f));
				cnaTypeChannel->insertAtomType(fccType);

				AtomType::SmartPtr hcpType(new AtomType());
				hcpType->setName(tr("HCP"));
				hcpType->setColor(Color(1.0f, 0.4f, 0.4f));
				cnaTypeChannel->insertAtomType(hcpType);

				AtomType::SmartPtr bccType(new AtomType());
				bccType->setName(tr("BCC"));
				bccType->setColor(Color(0.4f, 0.4f, 1.0f));
				cnaTypeChannel->insertAtomType(bccType);

				AtomType::SmartPtr icosahedralType(new AtomType());
				icosahedralType->setName(tr("Icosahedral"));
				icosahedralType->setColor(Color(1.0f, 1.0f, 0.7f));
				cnaTypeChannel->insertAtomType(icosahedralType);

				AtomType::SmartPtr noneType(new AtomType());
				noneType->setName(tr("Other"));
				noneType->setColor(Color(1.0f, 1.0f, 1.0f));
				cnaTypeChannel->insertAtomType(noneType);

				DataChannel* posChannel = destination->createStandardDataChannel(DataChannel::PositionChannel);
				OVITO_ASSERT(posChannel->size() == numAtoms);

				DataChannel* colorChannel = destination->createStandardDataChannel(DataChannel::ColorChannel);
				OVITO_ASSERT(colorChannel->size() == numAtoms);

				// Parse non-bulk atoms.
				for(int i = 0; i < numNonbulkAtoms; i++) {
					//MsgLogger() << "Reading atom =" << i << endl;
					int tag;
					short cnaType;
					float pos[3];
					stream.read(&tag, sizeof(tag));
					stream.read(&cnaType, sizeof(cnaType));
					stream.read(pos, sizeof(pos));
					int numNeighbors;
					CommonNeighborAnalysisModifier::CNAAtomType cnaTypeOvito;
					Color color;
					switch(cnaType) {
					case 1: // FCC
						numNeighbors = 12;
						cnaTypeOvito = CommonNeighborAnalysisModifier::FCC;
						color = fccType->color();
						break;
					case 2: // HCP
						numNeighbors = 12;
						cnaTypeOvito = CommonNeighborAnalysisModifier::HCP;
						color = hcpType->color();
						break;
					case 3: // BCC
						numNeighbors = 14;
						cnaTypeOvito = CommonNeighborAnalysisModifier::BCC;
						color = bccType->color();
						break;
					default:
						throw Exception(tr("Unknown crystalline atom type in DXA file: type id %1").arg(cnaType));
					}
					int neighborTag;
					for(int n = 0; n < numNeighbors; n++)
						stream.read(&neighborTag, sizeof(neighborTag));

					float floatNumbers[3][3];
					stream.read(floatNumbers, sizeof(floatNumbers));

					atomTagChannel->setInt(i, tag);
					posChannel->setPoint3(i, Point3(pos[0], pos[1], pos[2]));
					cnaTypeChannel->setInt(i, (int)cnaTypeOvito);
					colorChannel->setVector3(i, color);

					// Update progress indicator.
					if((i % 4096) == 0) {
						progress.setValue(i);
						if(progress.isCanceled())
							return EvaluationStatus(EvaluationStatus::EVALUATION_ERROR);
					}
				}

				// Parse disordered atoms.
				for(int i = 0; i < numDisorderedAtoms; i++) {
					int tag;
					float pos[3];
					stream.read(&tag, sizeof(tag));
					stream.read(pos, sizeof(pos));

					atomTagChannel->setInt(i + numNonbulkAtoms, tag);
					posChannel->setPoint3(i + numNonbulkAtoms, Point3(pos[0], pos[1], pos[2]));
					cnaTypeChannel->setInt(i + numNonbulkAtoms, (int)CommonNeighborAnalysisModifier::OTHER);
					colorChannel->setVector3(i + numNonbulkAtoms, noneType->color());

					// Update progress indicator.
					if((i % 4096) == 0) {
						progress.setValue(i+numNonbulkAtoms);
						if(progress.isCanceled()) return EvaluationStatus(EvaluationStatus::EVALUATION_ERROR);
					}
				}

				destination->invalidate();

				// Stop parsing here
				QString statusMessage = tr("%1 atoms at timestep %2").arg(numAtoms).arg(timeStep);
				return EvaluationStatus(EvaluationStatus::EVALUATION_SUCCESS, statusMessage);
			}
			else if(stream.line().find("ITEM:") != string::npos) {
				// Skip lines up to next ITEM:
				while(!stream.eof()) {
					stream.readline();
					if(stream.line().find("ITEM:") != string::npos) break;
				}
			}
			else if(stream.line().find_first_not_of(" \t\n\r") == string::npos) {
				// Skip empty lines
				break;
			}
			else {
				throw Exception(tr("DXA file parsing error. Invalid line %1 in dump file: %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
			}
		}
		while(!stream.eof());
	}

	destination->invalidate();
	throw Exception(tr("DXA file parsing error. No ATOMS section found in input file."));
}

};	// End of namespace AtomViz
