///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __ATOMS_FILE_WRITER_H
#define __ATOMS_FILE_WRITER_H

#include <core/Core.h>
#include <core/scene/objects/PipelineFlowState.h>
#include <core/data/importexport/ImporterExporter.h>
#include <atomviz/AtomViz.h>

namespace AtomViz {

class AtomsObject;			// defined in AtomsObject.h

/**
 * \brief Abstract base class for all atom file writers.
 *
 * High-level usage:
 *   - Just call ExportToFile().
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT AtomsFileWriter : public ImporterExporter
{
protected:
	/// \brief Constructs a new instance of this class.
	/// \param isLoading Specifies whether the object's data fields will be initialized from the
	///                  data stored in a scene file after the instance has been created.
	AtomsFileWriter(bool isLoading = false) : ImporterExporter(isLoading) {
		INIT_PROPERTY_FIELD(AtomsFileWriter, _outputFilename);
	}

public:

	////////////////////////// from ImporterExporter //////////////////////////

	/// \brief Exports the scene to the given file.
	/// \return \a true if the scene has been exported; \a false if the export has been aborted by the user.
	/// \throws Exception when the export has failed.
	///
	/// This is the high-level method to write an atoms file.
	virtual bool exportToFile(const QString& filePath, DataSet* scene, bool suppressDialogs = false);

	/////////////////////////// Specific methods ///////////////////////////////

	/// \brief Sets the name of the output file produced by this writer.
	/// \param filename The path to the new output file.
	/// \sa outputFile()
	virtual void setOutputFile(const QString& filename) { _outputFilename = filename; }

	/// \brief Returns the path to the output file set by the method setOutputFile().
	/// \sa setOutputFile()
	const QString& outputFile() const { return _outputFilename; }

	/// \brief Prepares the writer based on the scene to be exported.
	/// \param dataset The scene that is going to be exported.
	/// \param suppressDialogs Specifies whether any dialogs or message boxes shown by the writer should be supressed during preparation.
	///                        This parameter is set to true when the parser is invoked from a script to not interrupt the export process.
	/// \throws Exception on error.
	/// \return \a false when the operation has been canceled by the user; \a true on success.
	///
	/// The default implementation of this method just returns \a true.
	virtual bool prepareScene(DataSet* dataset, bool suppressDialogs = false) { return true; }

	/// \brief Opens the settings dialog for this writer.
	/// \param atomsObj The AtomsObject to be exported.
	/// \param parent The parent widget to be used for the settings dialog.
	/// \return \a true if the dialog has been approved by the user; \a false when the user has canceled the operation.
	///
	/// The default implementation of this method does not show any dialog and always returns \a true.
	///
	/// \note The output file name has to be set via setOutputFile() before this method may be called.
	virtual bool showSettingsDialog(AtomsObject* atomsObj, QWidget* parent) { return true; }

	/// \brief Exports the atoms in the given scene to the output file.
	/// \param dataset The scene that contains an AtomsObject.
	/// \param suppressDialogs Specifies whether any dialogs or message boxes shown by the writer should be supressed during preparation.
	///                        This parameter is set to true when the parser is invoked from a script to not interrupt the export process.
	/// \throws Exception on error.
	/// \return \a false when the operation has been canceled by the user; \a true on success.
	///
	/// \note setOutputFile() and prepareScene() must have been called before this method can be used.
	virtual bool exportAtoms(DataSet* dataset, bool suppressDialogs = false) = 0;

protected:

	/// \brief Returns the atoms that will be exported by this exporter at a given animation time.
	/// \param dataset The scene to be exported.
	/// \param time The animation for which the atoms are requested.
	/// \return The returned pipeline result contains the AtomsObject to be exported.
	///         The returned PipelineFlowState might be empty if there is no AtomsObject in the scene.
	///
	/// This method can be used by implementation of exportAtoms() to retrieve the atoms
	/// that should be exported.
	PipelineFlowState retrieveAtoms(DataSet* dataset, TimeTicks time);

private:

	/// The path to the output file created by this writer.
	PropertyField<QString> _outputFilename;

private:
	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(AtomsFileWriter)
	DECLARE_PROPERTY_FIELD(_outputFilename)
};

};	// End of namespace AtomViz

#endif // __ATOMS_FILE_WRITER_H
