///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/mainwnd/MainFrame.h>
#include <core/gui/panels/CommandPanel.h>
#include <core/gui/ApplicationManager.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/input/XFormManager.h>
#include <core/viewport/snapping/SnappingManager.h>
#include <core/scene/SceneNode.h>
#include <core/scene/SceneRoot.h>
#include <core/scene/ObjectNode.h>
#include <core/scene/GroupNode.h>
#include <core/scene/objects/ModifiedObject.h>
#include <core/scene/objects/Modifier.h>
#include <core/data/DataSetManager.h>
#include "DefaultSceneRenderer.h"

namespace Core {

/// The scaling factor that is applied to the bounding box of a selected
/// object to build the selection brackets shown in the viewports.
const FloatType DefaultSceneRenderer::SELECTION_MARKER_SCALING = 1.03f;

/******************************************************************************
* Is called by the Viewport class to let this scene renderer do its job.
******************************************************************************/
void DefaultSceneRenderer::renderScene()
{
	// Render all nodes in the scene recursively.
	SceneRoot* rootNode = DATASET_MANAGER.currentSet()->sceneRoot();
	if(rootNode == NULL) return;

	// In wireframe mode, two passes are needed to render the nodes.
	if(viewport->settings()->shadingMode() == Viewport::SHADING_WIREFRAME) {
		// First render not selected nodes.
		for(SceneNodesIterator iter(rootNode); !iter.finished(); iter.next()) {
			if(iter.current()->isSelected() == false)
				renderNode(iter.current());
		}
		// Then render selected nodes to make them appear above the unselected nodes.
		for(SceneNodesIterator iter(rootNode); !iter.finished(); iter.next()) {
			if(iter.current()->isSelected())
				renderNode(iter.current());
		}
	}
	else {
		// In shaded mode the rendering order doesn't matter because we are using the z-buffer.
		for(SceneNodesIterator iter(rootNode); !iter.finished(); iter.next())
			renderNode(iter.current());
	}

	// Render tripods for the selected nodes.
	XFORM_MANAGER.renderTripods(viewport);

	// Render snapping marker.
	SNAPPING_MANAGER.lastSnapPoint().render(viewport);
}

/******************************************************************************
* Renders a single node.
******************************************************************************/
void DefaultSceneRenderer::renderNode(SceneNode* node)
{
    CHECK_OBJECT_POINTER(node);

	if(node->isObjectNode()) {
		ObjectNode* objNode = static_object_cast<ObjectNode>(node);

		// Do not render node if it is the view node of the viewport or
		// if it is the target of the view node.
		if(viewport->settings()->viewNode() && (viewport->settings()->viewNode() == objNode || viewport->settings()->viewNode()->targetNode() == objNode)) return;

		// This will render the unselected nodes in the
		// first pass and the selected ones in the second pass.
		bool isSelected = objNode->isSelected();

		// Evaluate geometry pipeline of object node.
		const PipelineFlowState& flowState = objNode->evalPipeline(currentTime);
		if(flowState.result() != NULL) {
			CHECK_OBJECT_POINTER(flowState.result());

			// Setup transformation.
			TimeInterval iv;
			const AffineTransformation& nodeTM = objNode->getWorldTransform(currentTime, iv);
			viewport->setWorldMatrix(objNode->objectTransform() * nodeTM);

			// Render selection marker.
			if(viewport->settings()->shadingMode() != Viewport::SHADING_WIREFRAME
				&& isSelected && flowState.result()->showSelectionMarker())
			{
				renderSelectionMarker(objNode, Viewport::getVPColor(Viewport::COLOR_SELECTION));
			}

			// Prepare interactive material.
			if(viewport->settings()->shadingMode() != Viewport::SHADING_WIREFRAME)
				setupNodeMaterial(objNode);

			// Render object node.
			flowState.result()->renderObject(currentTime, objNode, viewport);
		}

		// Iterate over all modifiers applied to the object.
		ModifiedObject* modObj = dynamic_object_cast<ModifiedObject>(objNode->sceneObject());
		renderModifiedObject(modObj, objNode);

		// Render trajectory.
		if(objNode->showTrajectoryEnabled())
			renderTrajectory(objNode);
	}
	else if(node->isGroupNode()) {
		GroupNode* gn = static_object_cast<GroupNode>(node);

		// render bounding box for group node
		if(gn->isGroupOpen()) {
			TimeInterval interval;
			const AffineTransformation& nodeTM = gn->getWorldTransform(currentTime, interval);
			viewport->setWorldMatrix(nodeTM);
			renderSelectionMarker(gn, Viewport::getVPColor(Viewport::COLOR_SELECTION));
		}
	}
}


/******************************************************************************
* Renders the selection marker around a selected scene node.
******************************************************************************/
void DefaultSceneRenderer::renderSelectionMarker(SceneNode* node, const ColorA& color)
{
	Point3 selMarkerVerts[48];

	// Render brackts around the object.
	Box3 bounding = node->localBoundingBox(currentTime).centerScale(SELECTION_MARKER_SCALING);
	if(bounding.isEmpty()) return;

	// Calculate vertices of the brackets.
	Vector3 of = bounding.size() * (1.0/4.0);
	for(int i=0; i<8; i++) {
		int j = i*6;
		selMarkerVerts[j] = bounding[i];
		for(int k=1; k<6; k++) selMarkerVerts[j+k] = selMarkerVerts[j];
	}
	for(int i=0; i<2; i++, of.Z = -of.Z) {
		int j = i*24;
		selMarkerVerts[j+1].X += of.X;
		selMarkerVerts[j+3].Y += of.Y;
		selMarkerVerts[j+5].Z += of.Z;
		selMarkerVerts[j+6+1].X -= of.X;
		selMarkerVerts[j+6+3].Y += of.Y;
		selMarkerVerts[j+6+5].Z += of.Z;
		selMarkerVerts[j+12+1].X += of.X;
		selMarkerVerts[j+12+3].Y -= of.Y;
		selMarkerVerts[j+12+5].Z += of.Z;
		selMarkerVerts[j+18+1].X -= of.X;
		selMarkerVerts[j+18+3].Y -= of.Y;
		selMarkerVerts[j+18+5].Z += of.Z;
	}

	viewport->setRenderingColor(color);
	viewport->renderLines(48, bounding, selMarkerVerts);
}

/******************************************************************************
* Takes the node's material and creates a viewport material from it.
******************************************************************************/
void DefaultSceneRenderer::setupNodeMaterial(ObjectNode* objNode)
{
	/// The default material to use when no material is assigned to the node.
	static Window3DMaterial defaultMaterial;

	defaultMaterial.diffuse = objNode->displayColor();
	viewport->setMaterialCount(1);
	viewport->setMaterial(0, &defaultMaterial);
}

/******************************************************************************
* Computes the bounding box of the scene. This used used to adjust
* the position of the near and far clipping plane.
******************************************************************************/
Box3 DefaultSceneRenderer::sceneExtents(ViewportRecord* settings, TimeTicks time, SceneExtentsMode mode)
{
	CHECK_POINTER(settings);
    Box3 bb;

	SceneRoot* rootNode = DATASET_MANAGER.currentSet()->sceneRoot();
	if(rootNode == NULL) return bb;

	for(SceneNodesIterator iter(rootNode); !iter.finished(); iter.next()) {
		if(!iter.current()->isObjectNode()) continue;
		ObjectNode* objNode = static_object_cast<ObjectNode>(iter.current());

		if(mode != RENDERABLE_OBJECTS) {
			if(settings->viewNode() && (settings->viewNode() == objNode || settings->viewNode()->targetNode() == objNode))
				continue;
		}

		const PipelineFlowState& flowState = objNode->evalPipeline(time);
		if(flowState.result() != NULL) {
			CHECK_OBJECT_POINTER(flowState.result());

			if(mode == SELECTED_OBJECTS && !objNode->isSelected()) continue;
			if(mode == RENDERABLE_OBJECTS && !flowState.result()->isRenderable()) continue;

			const Box3& objBox = objNode->worldBoundingBox(time);
			bb.addBox(objBox);

			// Include selection marker.
			if(mode != RENDERABLE_OBJECTS) {
				if(settings->shadingMode() != Viewport::SHADING_WIREFRAME && objNode->isSelected() && flowState.result()->showSelectionMarker()) {
					bb.addBox(objBox.centerScale(SELECTION_MARKER_SCALING));
				}
			}
		}
	}
	return bb;
}

/******************************************************************************
* Render the selected modifiers.
******************************************************************************/
void DefaultSceneRenderer::renderModifiedObject(ModifiedObject* modObj, ObjectNode* objNode)
{
	if(!modObj) return;
	OVITO_ASSERT(APPLICATION_MANAGER.guiMode());

	// Retrieve the modifier that is currently selected.
	Modifier* selectedModifier = dynamic_object_cast<Modifier>(MAIN_FRAME->commandPanel()->editObject());
	if(!selectedModifier) return;

	// Iterate over all modifier applications until the selected modifier is found.
	// Then render it.
	Q_FOREACH(ModifierApplication* modApp, modObj->modifierApplications()) {
		if(modApp->modifier() == selectedModifier) {

			// Setup transformation.
			TimeInterval iv;
			const AffineTransformation& nodeTM = objNode->getWorldTransform(currentTime, iv);
			viewport->setWorldMatrix(objNode->objectTransform() * nodeTM);

			// Render selected modifier.
			selectedModifier->renderModifier(currentTime, objNode, modApp, viewport);

			return;
		}
	}

	// Not found yet -> go on with nested modified objects.
	for(int i=0; i<modObj->inputObjectCount(); i++)
		renderModifiedObject(dynamic_object_cast<ModifiedObject>(modObj->inputObject(i)), objNode);
}

/******************************************************************************
* Renders the motion trajectory of a node.
******************************************************************************/
void DefaultSceneRenderer::renderTrajectory(SceneNode* node)
{
	if(!viewport->isRendering()) return;

	TimeInterval iv;
	viewport->setWorldMatrix(IDENTITY);

	// Disable lighting.
	glPushAttrib(GL_LIGHTING_BIT);
	glDisable(GL_LIGHTING);

	Box3 boundingBox;
	glColor3d(0.5, 0.5, 1.0);
	glBegin(GL_LINE_STRIP);
	for(TimeTicks time = ANIM_MANAGER.animationInterval().start(); time <= ANIM_MANAGER.animationInterval().end(); time += ANIM_MANAGER.ticksPerFrame()) {
		Point3 p = ORIGIN + node->getWorldTransform(time, iv).getTranslation();
		glVertex3v(p.constData());
		boundingBox += p;
	}
	glEnd();
	glColor3d(1.0, 1.0, 1.0);
	glBegin(GL_POINTS);
	for(TimeTicks time = ANIM_MANAGER.animationInterval().start(); time <= ANIM_MANAGER.animationInterval().end(); time += ANIM_MANAGER.ticksPerFrame()) {
		Point3 p = ORIGIN + node->getWorldTransform(time, iv).getTranslation();
		glVertex3v(p.constData());
	}
	glEnd();
	glPopAttrib();

	viewport->enlargeSceneExtent(boundingBox);
}

/******************************************************************************
* Performs hit testing on all visible nodes in the scene and returns a list of nodes that
* are within the given pick region.
******************************************************************************/
QVector<SceneNode*> DefaultSceneRenderer::hitTestSceneNodes(const PickRegion& pickRegion)
{
	OVITO_ASSERT(APPLICATION_MANAGER.guiMode());
	CHECK_POINTER(viewport);

	QVector<SceneNode*> resultList;
	QVector<FloatType> distances;

	SceneRoot* rootNode = DATASET_MANAGER.currentSet()->sceneRoot();
	if(rootNode == NULL) return resultList;

	for(SceneNodesIterator iter(rootNode); !iter.finished(); iter.next()) {
		SceneNode* node = iter.current();

		// Do not hit test active camera node.
		if(viewport->settings()->viewNode() && (viewport->settings()->viewNode() == node || viewport->settings()->viewNode()->targetNode() == node))
			continue;

		// Let the node do the actual hit testing.
		FloatType hitValue = node->hitTest(currentTime, viewport, pickRegion);

		// Add the node to the result list if a hit has been recorded.
		if(hitValue != HIT_TEST_NONE) {
			// If the user has clicked on a node that is part of a group then the group is selected
			// instead of the node.
			GroupNode* gn = node->closedParentGroup();
			if(gn != NULL) {
				if(resultList.contains(gn)) continue;
				node = gn;
			}

			if(pickRegion.type() != PickRegion::POINT) {
				// Just append node to list.
				resultList.append(node);
			}
			else {
				// Sort nodes by hit distance.
				int index = 0;
				while(index < distances.size() && hitValue > distances[index])
					index++;
				distances.insert(index, hitValue);
				resultList.insert(index, node);
			}
		}
	}

	return resultList;
}

};
