/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment media rendering library
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Julien Moutte <julien@fluendo.com>
 */

/**
 * SECTION:pgmcanvas
 * @short_description: A virtual positioning class for drawables.
 * @see_also: #PgmViewport, #PgmDrawable.
 *
 * <refsect2>
 * <para>
 * A #PgmCanvas object is used as a virtual positioning system for the
 * #PgmDrawable. It provides Pigment users (application developers using the
 * library) a very simple and flexible way to build their user interface
 * without having to worry about visual rendering problems such as non square
 * pixels, aspect ratio of images or videos, keeping proportional alignment or
 * spacing between objects when resizing, etc.
 * </para>
 * <title>Canvas coordinates</title>
 * <para>
 * The origin of the canvas coordinate system corresponds to the upper-left
 * corner. A canvas can only understand canvas coordinates, and those
 * coordinates have nothing to do with pixels. This is why the methods you
 * can call on a canvas always use coordinates with floating numbers. Let's
 * say you want to draw a user interface with an aspect ratio of 16:9 (which is
 * nice to do on most wide screen TV of the market). In that case you would for
 * example set the canvas size to width 16.0 and height 9.0. Every coordinate
 * and size of the drawables you are going to put on the canvas will use the
 * same coordinate system. So, if you wanted to put an image in the middle of
 * the user interface you would compute its position so that its center is at
 * (8.0, 4.5) on the canvas. You would also define the size of that image using
 * values of the same scale. For example, the width of the image could be 1.0
 * and its height 1.0 thus making appear as a square.
 * </para>
 * <title>Link between canvas and viewport</title>
 * <para>
 * A canvas is never drawn directly which is why you can use abstract
 * coordinates to place objects on it. Also, since it is using floating
 * numbers there is no limit in precision. At some point we still need to
 * have a visual representation of what is on the canvas, this is what
 * #PgmViewport does. A canvas can be projected on 0 to n viewport.
 * </para>
 * <title>Scalable user interface</title>
 * <para>
 * It is entirely the application's responsibility to decide how many drawables
 * are going to be drawn on the canvas and their size. Using the viewport
 * information the application can decide which aspect ratio to use and how
 * many drawables can fit on the canvas while still being easily readable. If
 * for example the end user (user interface user) steps away from the monitor
 * and cannot read the text the application can just make the drawables bigger
 * on the canvas which will automatically get reprojected onto the viewport.
 * </para>
 * <title>Rendering layers</title>
 * <para>
 * When adding a drawable to a canvas you have to choose a #PgmDrawableLayer
 * in which that drawable will be stored. The three layers are used to handle a
 * first control on the drawing order. The first layer drawn is the
 * %PGM_DRAWABLE_FAR one, all the drawables inside this layer will appear behind
 * the two others. The second layer drawn is the %PGM_DRAWABLE_MIDDLE one, the
 * drawables inside this layer will be drawn over the %PGM_DRAWABLE_FAR layer
 * and behind the %PGM_DRAWABLE_NEAR layer. The third layer drawn is the
 * %PGM_DRAWABLE_NEAR one, all the drawables inside this layer will be drawn
 * over the two others. This is useful to make sure that dialog is shown on top
 * of all the other objects for example or to make sure that no object is going
 * to go behind the background. Drawables inside a layer are automatically
 * ordered by Pigment depending on their z coordinate set through
 * pgm_drawable_set_position(). Pigment also provides a way to reorder
 * drawables inside a layer that have the same z coordinate thanks to the
 * pgm_canvas_set_order() function.
 * </para>
 * <title>Drawable reference counting</title>
 * <para>
 * When a drawable is created it has a floating reference, when you add it to
 * the canvas, the canvas takes a reference to that drawable and sink the object
 * reference. That means that the canvas now owns the drawable reference and
 * you do not need to unref the drawable when cleaning up your objects. Just
 * unrefing the canvas is going to cleanup all the drawables that were added to
 * it. Here is an example:
 * <example id="pigment-canvas-cycle">
 * <title>Create a canvas, add drawables, unref the canvas</title>
 * <programlisting language="c">
 * PgmCanvas *canvas = pgm_canvas_new ();
 * PgmDrawable *drb1 = pgm_text_new ("hello world");
 * PgmDrawable *drb2 = pgm_image_new_from_fd (fd, 1024);
 * PgmDrawable *drb3 = pgm_image_new_from_image (drb2);
 * &nbsp;
 * pgm_canvas_add_many (canvas, PGM_DRAWABLE_MIDDLE, drb1, drb2, drb3, NULL);
 * gst_object_unref (canvas); // Unref canvas, drb1, drb2 and drb3
 * </programlisting>
 * </example>
 * </para>
 * <title>Signal connections</title>
 * <para>
 * The <link linkend="PgmCanvas-drawable-added">drawable-added</link> signal is
 * fired whenever a new drawable is added to the canvas. Likewise the <link
 * linkend="PgmCanvas-drawable-removed">drawable-removed</link> signal is fired
 * whenever an drawable is removed from the canvas. The <link
 * linkend="PgmCanvas-drawable-reordered">drawable-reordered</link> signal is
 * fired whenever a drawable is reordered inside the canvas.
 * </para>
 * </refsect2>
 *
 * Last reviewed on 2007-08-03 (0.3.0)
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "pgmcanvas.h"
#include "pgmimage.h"
#include "pgmmarshal.h"
#include "pgmenumtypes.h"

GST_DEBUG_CATEGORY_STATIC (pgm_canvas_debug);
#define GST_CAT_DEFAULT pgm_canvas_debug

/* Canvas signals */
enum {
  DRAWABLE_ADDED,
  DRAWABLE_REMOVED,
  DRAWABLE_REORDERED,
  SIZE_CHANGED,
  REGENERATED,
  LAST_SIGNAL
};

static GstObjectClass *parent_class = NULL;
static guint pgm_canvas_signals[LAST_SIGNAL] = { 0 };

/* Private functions */

/* GCompareFunc used to compare z coordinate */
static gint
z_compare (PgmDrawable *drb1,
           PgmDrawable *drb2)
{
  if (drb1->z < drb2->z)
    return -1;
  else
    return 1;
}

/* Returns whether or not a drawable from a list needs to be reordered
 * depending on its z coordinate. */
static gboolean
need_ordering_update (GList *list,
                      PgmDrawable *drawable)
{
  list = g_list_find (list, drawable);

  if (!list)
    return FALSE;

  if (!((list->next && drawable->z > PGM_DRAWABLE (list->next->data)->z) ||
        (list->prev && drawable->z < PGM_DRAWABLE (list->prev->data)->z)))
    return FALSE;

  return TRUE;
}

/* Update the position and the size if the grid aligned property is set */
static void
update_grid_aligned_drawable (PgmDrawable *drawable)
{
  gboolean aligned;
  gfloat x, y, z, w, h;

  GST_OBJECT_LOCK (drawable);
  aligned = drawable->grid_aligned;
  x = drawable->orig_x;
  y = drawable->orig_y;
  z = drawable->z;
  w = drawable->orig_width;
  h = drawable->orig_height;
  GST_OBJECT_UNLOCK (drawable);

  if (aligned)
    {
      pgm_drawable_set_position (drawable, x, y, z);
      pgm_drawable_set_size (drawable, w, h);
    }
}

/* Calls update_grid_aligned_drawable on a canvas layer */
static void
update_grid_aligned_drawables (PgmCanvas *canvas,
                               GList *layer)
{
  GList *copy = NULL, *walk;

  /* Copy the layer taking ref on drawables with the canvas lock */
  GST_OBJECT_LOCK (canvas);
  walk = layer;
  while (walk)
    {
      copy = g_list_prepend (copy, g_object_ref (walk->data));
      walk = walk->next;
    }
   GST_OBJECT_UNLOCK (canvas);

   /* Call update_grid_aligned_drawables on each drawable */
   walk = copy;
   while (walk)
     {
       update_grid_aligned_drawable ((PgmDrawable*) walk->data);
       g_object_unref (walk->data);
       walk = walk->next;
     }
   g_list_free (copy);
}

/* GObject stuff */

G_DEFINE_TYPE (PgmCanvas, pgm_canvas, GST_TYPE_OBJECT);

static void
pgm_canvas_dispose (GObject *object)
{
  PgmCanvas *canvas = PGM_CANVAS (object);

  while (canvas->far_layer)
    pgm_canvas_remove (canvas, PGM_DRAWABLE (canvas->far_layer->data));

  while (canvas->middle_layer)
    pgm_canvas_remove (canvas, PGM_DRAWABLE (canvas->middle_layer->data));

  while (canvas->near_layer)
    pgm_canvas_remove (canvas, PGM_DRAWABLE (canvas->near_layer->data));

  g_hash_table_unref (canvas->format_counter);
  canvas->format_counter = NULL;

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_canvas_class_init (PgmCanvasClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;

  parent_class = g_type_class_peek_parent (klass);

  /**
   * PgmCanvas::drawable-added:
   * @canvas: the #PgmCanvas
   * @drawable: the #PgmDrawable that was added to @canvas
   * @layer: the #PgmDrawableLayer in which @drawable was added
   * @order: the order in @layer in which @drawable was added
   *
   * Will be emitted after @drawable was added to @canvas.
   */
  pgm_canvas_signals[DRAWABLE_ADDED] =
    g_signal_new ("drawable-added", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_FIRST,
                  G_STRUCT_OFFSET (PgmCanvasClass, drawable_added),
                  NULL, NULL, pgm_marshal_VOID__OBJECT_OBJECT_INT, G_TYPE_NONE,
                  3, PGM_TYPE_DRAWABLE, PGM_TYPE_DRAWABLE_LAYER, G_TYPE_INT);
  /**
   * PgmCanvas::drawable-removed:
   * @canvas: the #PgmCanvas
   * @drawable: the #PgmDrawable that was removed from @canvas
   * @layer: the #PgmDrawableLayer from which @drawable was removed
   *
   * Will be emitted after @drawable was removed from @canvas.
   */
  pgm_canvas_signals[DRAWABLE_REMOVED] =
    g_signal_new ("drawable-removed", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_FIRST,
                  G_STRUCT_OFFSET (PgmCanvasClass, drawable_removed),
                  NULL, NULL, pgm_marshal_VOID__OBJECT_OBJECT, G_TYPE_NONE,
                  2, PGM_TYPE_DRAWABLE, PGM_TYPE_DRAWABLE_LAYER);
  /**
   * PgmCanvas::drawable-reordered:
   * @canvas: the #PgmCanvas
   * @drawable: the #PgmDrawable that was reordered in @canvas
   * @layer: the #PgmDrawableLayer in which @drawable was reordered
   * @order: the new order of @drawable
   *
   * Will be emitted after @drawable was reordered in @canvas.
   */
  pgm_canvas_signals[DRAWABLE_REORDERED] =
    g_signal_new ("drawable-reordered", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_FIRST,
                  G_STRUCT_OFFSET (PgmCanvasClass, drawable_reordered),
                  NULL, NULL, pgm_marshal_VOID__OBJECT_OBJECT_INT, G_TYPE_NONE,
                  3, PGM_TYPE_DRAWABLE, PGM_TYPE_DRAWABLE_LAYER, G_TYPE_INT);
  /**
   * PgmCanvas::size-changed:
   * @canvas: the #PgmCanvas
   *
   * Will be emitted after @canvas size was changed.
   */
  pgm_canvas_signals[SIZE_CHANGED] =
    g_signal_new ("size-changed", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_FIRST,
                  G_STRUCT_OFFSET (PgmCanvasClass, size_changed),
                  NULL, NULL, pgm_marshal_VOID__VOID, G_TYPE_NONE, 0);
  /**
   * PgmCanvas::regenerated:
   * @canvas: the #PgmCanvas
   *
   * Will be emitted after @canvas received a regeneration request.
   */
  pgm_canvas_signals[REGENERATED] =
    g_signal_new ("regenerated", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_FIRST,
                  G_STRUCT_OFFSET (PgmCanvasClass, regenerated),
                  NULL, NULL, pgm_marshal_VOID__VOID, G_TYPE_NONE, 0);

  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_canvas_dispose);

  GST_DEBUG_CATEGORY_INIT (pgm_canvas_debug, "pgm_canvas", 0, "canvas object");
}

static void
pgm_canvas_init (PgmCanvas *canvas)
{
  guint i;

  /* canvas->entered_emission_stopper = NULL; */

  /* Create and insert all the format in the counter hash initialized at 0 */
  canvas->format_counter = g_hash_table_new (NULL, NULL);
  for (i = 0; i < PGM_IMAGE_NB_PIXEL_FORMATS; i++)
    g_hash_table_insert (canvas->format_counter,
                         GINT_TO_POINTER (1 << i),
                         GINT_TO_POINTER (0));
  /* canvas->nb_formats = 0; */

  /* canvas->far_layer = NULL; */
  /* canvas->middle_layer = NULL; */
  /* canvas->near_layer = NULL; */
  canvas->width = 4.0f;
  canvas->height = 3.0f;
  canvas->pixel_offset_x = 0.005f; /* 4/800 */
  canvas->pixel_offset_y = 0.005f; /* 3/600 */
  /* canvas->pixel_formats = 0; */
}

/* Public methods */

/**
 * pgm_canvas_new:
 *
 * Creates a new #PgmCanvas instance.
 *
 * MT safe.
 *
 * Returns: a new #PgmCanvas instance.
 */
PgmCanvas *
pgm_canvas_new (void)
{
  PgmCanvas *canvas;

  canvas = g_object_new (PGM_TYPE_CANVAS, NULL);
  GST_DEBUG_OBJECT (canvas, "created new canvas");

  return canvas;
}

/**
 * pgm_canvas_set_size:
 * @canvas: a #PgmCanvas object.
 * @width: the canvas width.
 * @height: the canvas height.
 *
 * Sets the @width and @height size of @canvas. These values are not supposed
 * to be pixels. You are strongly encouraged to use abstract coordinates such as
 * 16.0x9.0 for a 16:9 interface or 4.0x3.0 for a 4:3 one, etc.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_canvas_set_size (PgmCanvas *canvas,
                     gfloat width,
                     gfloat height)
{
  g_return_val_if_fail (PGM_IS_CANVAS (canvas), PGM_ERROR_X);

  GST_OBJECT_LOCK (canvas);

  canvas->width = MAX (1.0f, width);
  canvas->height = MAX (1.0f, height);

  GST_OBJECT_UNLOCK (canvas);

  g_signal_emit (G_OBJECT (canvas), pgm_canvas_signals[SIZE_CHANGED], 0);

  return PGM_ERROR_OK;
}

/**
 * pgm_canvas_get_size:
 * @canvas: a #PgmCanvas object.
 * @width: a pointer to a #gfloat where the canvas width is going to
 * be stored.
 * @height: a pointer to a #gfloat where the canvas height is going to
 * be stored.
 *
 * Retrieves width and height size of @canvas in @width and @height.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_canvas_get_size (PgmCanvas *canvas,
                     gfloat *width,
                     gfloat *height)
{
  g_return_val_if_fail (PGM_IS_CANVAS (canvas), PGM_ERROR_X);
  g_return_val_if_fail (width != NULL, PGM_ERROR_X);
  g_return_val_if_fail (height != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (canvas);

  *width = canvas->width;
  *height = canvas->height;

  GST_OBJECT_UNLOCK (canvas);

  return PGM_ERROR_OK;
}

/**
 * pgm_canvas_add:
 * @canvas: a #PgmCanvas object.
 * @layer: a #PgmDrawableLayer to add #drawable into.
 * @drawable: the #PgmDrawable object to add.
 *
 * Adds @drawable to @canvas in @layer. Both @canvas and @drawable reference
 * counts will be increased by one as they are linking to each other. @drawable
 * reference will be sink as well so if the object reference was floating it
 * now belongs to @canvas and you don't have to unref @drawable when cleaning
 * up your objects.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_canvas_add (PgmCanvas *canvas,
                PgmDrawableLayer layer,
                PgmDrawable *drawable)
{
  gint order = 0;

  g_return_val_if_fail (PGM_IS_CANVAS (canvas), PGM_ERROR_X);
  g_return_val_if_fail (PGM_IS_DRAWABLE (drawable), PGM_ERROR_X);

  GST_OBJECT_LOCK (canvas);

  /* Adds a reference to drawable, sink floating reference if any and set
   * parent */
  if (G_UNLIKELY (!gst_object_set_parent (GST_OBJECT_CAST (drawable),
                                          GST_OBJECT_CAST (canvas))))
    {
      GST_OBJECT_UNLOCK (canvas);
      GST_WARNING_OBJECT (drawable, "drawable already has parent");
      return PGM_ERROR_X;
    }

  /* Add the drawable to its requested layer at the correct position */
  switch (layer)
    {
    case PGM_DRAWABLE_FAR:
      canvas->far_layer = g_list_insert_sorted (canvas->far_layer, drawable,
                                                (GCompareFunc) z_compare);
      order = g_list_index (canvas->far_layer, drawable);
      break;
    case PGM_DRAWABLE_MIDDLE:
      canvas->middle_layer = g_list_insert_sorted (canvas->middle_layer,
                                                   drawable,
                                                   (GCompareFunc) z_compare);
      order = g_list_index (canvas->middle_layer, drawable);
      break;
    case PGM_DRAWABLE_NEAR:
      canvas->near_layer = g_list_insert_sorted (canvas->near_layer, drawable,
                                                 (GCompareFunc) z_compare);
      order = g_list_index (canvas->near_layer, drawable);
      break;
    default:
      GST_WARNING_OBJECT (drawable, "has not been assigned to any layer");
      goto beach;
    }

  GST_OBJECT_LOCK (drawable);
  drawable->layer = layer;
  GST_OBJECT_UNLOCK (drawable);

  GST_OBJECT_UNLOCK (canvas);

  update_grid_aligned_drawable (drawable);

  g_signal_emit (G_OBJECT (canvas), pgm_canvas_signals[DRAWABLE_ADDED], 0,
                 drawable, layer, order);

  return PGM_ERROR_OK;

 beach:
  GST_OBJECT_UNLOCK (canvas);
  return PGM_ERROR_X;
}

/**
 * pgm_canvas_remove:
 * @canvas: a #PgmCanvas object.
 * @drawable: the #PgmDrawable object to remove.
 *
 * Removes @drawable from @canvas. Both @canvas and @drawable reference counts
 * will be decreased by one as they were referencing each other. In most cases
 * that means that @drawable is going to be destroyed because @canvas owned the
 * only reference to it. If you want to reuse @drawable in another canvas or
 * in another layer you need to take a reference to it before removing it from
 * @canvas.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_canvas_remove (PgmCanvas *canvas,
                   PgmDrawable *drawable)
{
  PgmDrawableLayer layer;

  g_return_val_if_fail (PGM_IS_CANVAS (canvas), PGM_ERROR_X);
  g_return_val_if_fail (PGM_IS_DRAWABLE (drawable), PGM_ERROR_X);

  GST_DEBUG_OBJECT (canvas, "removing drawable %s from canvas",
                    GST_OBJECT_NAME (drawable));

  GST_OBJECT_LOCK (drawable);

  /* Check if the drawable is in the canvas */
  if (G_UNLIKELY (GST_OBJECT_PARENT (drawable) != GST_OBJECT (canvas)))
    {
      GST_OBJECT_UNLOCK (drawable);
      return PGM_ERROR_X;
    }

  /* Check if the drawable is already being removed and immediately
   * return */
  if (G_UNLIKELY (GST_OBJECT_FLAG_IS_SET
                  (drawable, PGM_DRAWABLE_FLAG_UNPARENTING)))
    {
      GST_OBJECT_UNLOCK (drawable);
      return PGM_ERROR_X;
    }

  GST_OBJECT_FLAG_SET (drawable, PGM_DRAWABLE_FLAG_UNPARENTING);

  GST_OBJECT_UNLOCK (drawable);

  GST_OBJECT_LOCK (canvas);

  /* Remove the drawable from its layer */
  switch (drawable->layer)
    {
    case PGM_DRAWABLE_FAR:
      canvas->far_layer = g_list_remove (canvas->far_layer, drawable);
      break;
    case PGM_DRAWABLE_MIDDLE:
      canvas->middle_layer = g_list_remove (canvas->middle_layer, drawable);
      break;
    case PGM_DRAWABLE_NEAR:
      canvas->near_layer = g_list_remove (canvas->near_layer, drawable);
      break;
    default:
      /* We should have a really big problem to take that path */
      GST_WARNING_OBJECT (drawable, "does not seem to be part of any layer");
    }

  GST_OBJECT_UNLOCK (canvas);

  /* we ref here because after the _unparent() the drawable can be disposed
   * and we still need it to reset the UNPARENTING flag. */
  gst_object_ref (drawable);
  gst_object_unparent (GST_OBJECT_CAST (drawable));

  GST_OBJECT_LOCK (drawable);
  layer = drawable->layer;
  drawable->layer = PGM_DRAWABLE_UNBOUND;
  GST_OBJECT_FLAG_UNSET (drawable, PGM_DRAWABLE_FLAG_UNPARENTING);
  GST_OBJECT_UNLOCK (drawable);

  /* Drawables are not visible anymore when removed from the canvas */
  pgm_drawable_hide (drawable);

  g_signal_emit (G_OBJECT (canvas), pgm_canvas_signals[DRAWABLE_REMOVED], 0,
                 drawable, layer);

  /* drawable is really out of our control now */
  gst_object_unref (drawable);

  return PGM_ERROR_OK;
}

/**
 * pgm_canvas_add_many:
 * @canvas: a #PgmCanvas object.
 * @layer: a #PgmDrawableLayer to add the drawables into.
 * @drawable_1: the #PgmDrawable object to add.
 * @...: additional #PgmDrawable objects to add.
 *
 * Adds a NULL-terminated list of drawables to @canvas. This function is
 * equivalent to calling pgm_canvas_add() for each member of the list.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_canvas_add_many (PgmCanvas *canvas,
                     PgmDrawableLayer layer,
                     PgmDrawable *drawable_1,
                     ...)
{
  va_list args;

  g_return_val_if_fail (PGM_IS_CANVAS (canvas), PGM_ERROR_X);

  va_start (args, drawable_1);
  while (drawable_1)
    {
      pgm_canvas_add (canvas, layer, drawable_1);
      drawable_1 = va_arg (args, PgmDrawable *);
    }
  va_end (args);

  return PGM_ERROR_OK;
}

/**
 * pgm_canvas_remove_many:
 * @canvas: a #PgmCanvas object.
 * @drawable_1: the #PgmDrawable object to remove.
 * @...: additional #PgmDrawable objects to remove.
 *
 * Removes a NULL-terminated list of drawables from @canvas. This function is
 * equivalent to calling pgm_canvas_remove() for each member of the list.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 *
 */
PgmError
pgm_canvas_remove_many (PgmCanvas *canvas,
                        PgmDrawable *drawable_1,
                        ...)
{
  va_list args;

  g_return_val_if_fail (PGM_IS_CANVAS (canvas), PGM_ERROR_X);

  va_start (args, drawable_1);
  while (drawable_1)
    {
      pgm_canvas_remove (canvas, drawable_1);
      drawable_1 = va_arg (args, PgmDrawable *);
    }
  va_end (args);

  return PGM_ERROR_OK;
}

/**
 * pgm_canvas_set_order:
 * @canvas: a #PgmCanvas object.
 * @drawable: the #PgmDrawable to reorder.
 * @order: the new position of @drawable.
 *
 * Defines the ordering of @drawable in its layer at position @order. Since
 * drawables are ordered function of their z coordinate, set through
 * pgm_drawable_set_position(), this function is only useful if you want to
 * reorder drawables at the same z coordinate. If you try to reorder a drawable
 * inside a layer at an order where the current drawable doesn't have the same
 * z coordinate, the function will do nothing and return an error.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_canvas_set_order (PgmCanvas *canvas,
                      PgmDrawable *drawable,
                      gint order)
{
  PgmDrawable *data = NULL;
  GList *list;

  g_return_val_if_fail (PGM_IS_CANVAS (canvas), PGM_ERROR_X);
  g_return_val_if_fail (PGM_IS_DRAWABLE (drawable), PGM_ERROR_X);

  GST_OBJECT_LOCK (canvas);
  GST_OBJECT_LOCK (drawable);

  /* Drawable should have canvas as a parent */
  if (PGM_CANVAS (GST_OBJECT_PARENT (drawable)) != canvas)
    {
      GST_DEBUG_OBJECT (canvas, "canvas is not the parent of '%s'",
                        GST_OBJECT_NAME (drawable));
      goto error;
    }

  /* Get the layer GList */
  switch (drawable->layer)
    {
    case PGM_DRAWABLE_NEAR:
      list = canvas->near_layer;
      break;

    case PGM_DRAWABLE_MIDDLE:
      list = canvas->middle_layer;
      break;

    case PGM_DRAWABLE_FAR:
      list = canvas->far_layer;
      break;

    default:
      /* Wrong layer */
      GST_DEBUG_OBJECT (canvas, "'%s' cannot be reordered in this layer",
                        GST_OBJECT_NAME (drawable));
      goto error;
    }

  /* Get the drawable at the given order */
  data = g_list_nth_data (list, order);

  /* Out of range checking */
  if (data == NULL)
    {
      GST_DEBUG_OBJECT (canvas, "order is out of range");
      goto error;
    }
  /* Same position in the list checking */
  if (data == drawable)
    {
      GST_DEBUG_OBJECT (canvas, "'%s' is already at position %d",
                        GST_OBJECT_NAME (drawable), order);
      goto nothing_to_do;
    }
  /* Same z coordinate checking */
  if (data->z != drawable->z)
    {
      GST_DEBUG_OBJECT (canvas, "drawable at the requested position doesn't "
                        "have the same z coordinate");
      goto nothing_to_do;
    }

  /* Reorder the drawable */
  list = g_list_remove (list, drawable);
  list = g_list_insert (list, drawable, order);

  /* Affect the reordered list to the canvas */
  if (drawable->layer == PGM_DRAWABLE_NEAR)
    canvas->near_layer = list;
  else if (drawable->layer == PGM_DRAWABLE_MIDDLE)
    canvas->middle_layer = list;
  else
    canvas->far_layer = list;

  GST_OBJECT_UNLOCK (drawable);
  GST_OBJECT_UNLOCK (canvas);

  g_signal_emit (G_OBJECT (canvas), pgm_canvas_signals[DRAWABLE_REORDERED],
                 0, drawable, drawable->layer, order);

  return PGM_ERROR_OK;

 nothing_to_do:
  GST_OBJECT_UNLOCK (drawable);
  GST_OBJECT_UNLOCK (canvas);
  return PGM_ERROR_OK;

 error:
   GST_OBJECT_UNLOCK (drawable);
   GST_OBJECT_UNLOCK (canvas);
   return PGM_ERROR_X;
}

/**
 * pgm_canvas_get_order:
 * @canvas: a #PgmCanvas object.
 * @drawable: the #PgmDrawable to retrieve the layering.
 * @layer: a pointer to a #PgmDrawableLayer in which @drawable layer is going to
 * be stored.
 * @order: a pointer to a #gint in which the position of @drawable in the @layer
 * is going to be stored.
 *
 * Retrieves the layering of @drawable.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_canvas_get_order (PgmCanvas *canvas,
                      PgmDrawable *drawable,
                      PgmDrawableLayer *layer,
                      gint *order)
{
  g_return_val_if_fail (PGM_IS_CANVAS (canvas), PGM_ERROR_X);
  g_return_val_if_fail (PGM_IS_DRAWABLE (drawable), PGM_ERROR_X);
  g_return_val_if_fail (layer != NULL, PGM_ERROR_X);
  g_return_val_if_fail (order != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (drawable);
  GST_OBJECT_LOCK (canvas);

  if (G_UNLIKELY ((gpointer) GST_OBJECT_PARENT (drawable) != (gpointer) canvas))
    {
      GST_WARNING_OBJECT (canvas, "not the parent of %s",
                          GST_STR_NULL (GST_OBJECT_NAME (drawable)));
      GST_OBJECT_UNLOCK (drawable);
      GST_OBJECT_UNLOCK (canvas);
      return PGM_ERROR_X;
    }

  *layer = drawable->layer;

  GST_OBJECT_UNLOCK (drawable);

  switch (*layer)
    {
    case PGM_DRAWABLE_NEAR:
      *order = g_list_index (canvas->near_layer, drawable);
      break;

    case PGM_DRAWABLE_MIDDLE:
      *order = g_list_index (canvas->middle_layer, drawable);
      break;

    case PGM_DRAWABLE_FAR:
      *order = g_list_index (canvas->far_layer, drawable);
      break;

    default:
      *order = -1;
      break;
    }

  GST_OBJECT_UNLOCK (canvas);

  return PGM_ERROR_OK;
}

/**
 * pgm_canvas_get_layer_count:
 * @canvas: a #PgmCanvas object.
 * @layer: the #PgmDrawableLayer to retrieve the size.
 * @count: a pointer to a #gint in which the number of drawables in @layer is
 * going to be stored.
 *
 * Retrieves the number of drawables in @layer.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_canvas_get_layer_count (PgmCanvas *canvas,
                            PgmDrawableLayer layer,
                            gint *count)
{
  g_return_val_if_fail (PGM_IS_CANVAS (canvas), PGM_ERROR_X);
  g_return_val_if_fail (count != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (canvas);

  switch (layer)
    {
    case PGM_DRAWABLE_NEAR:
      *count = g_list_length (canvas->near_layer);
      break;

    case PGM_DRAWABLE_MIDDLE:
      *count = g_list_length (canvas->middle_layer);
      break;

    case PGM_DRAWABLE_FAR:
      *count = g_list_length (canvas->far_layer);
      break;

    default:
      *count = -1;
      break;
    }

  GST_OBJECT_UNLOCK (canvas);

  return PGM_ERROR_OK;
}

/**
 * pgm_canvas_regenerate:
 * @canvas: a #PgmCanvas object.
 *
 * Affects all the drawables of the #canvas which need to be displayed
 * at a fixed size. It regenerates those rasterized pixmaps which have
 * been affected by the #canvas projection.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_canvas_regenerate (PgmCanvas *canvas)
{
  g_return_val_if_fail (PGM_IS_CANVAS (canvas), PGM_ERROR_X);

  g_signal_emit (G_OBJECT (canvas), pgm_canvas_signals[REGENERATED], 0);

  return PGM_ERROR_OK;
}

/**
 * pgm_canvas_get_pixel_formats:
 * @canvas: a #PgmCanvas object.
 * @pixel_formats: a pointer to a #gulong where the mask of supported
 * #PgmImagePixelFormat is going to be stored.
 *
 * Gets the list of the supported pixel formats by @canvas. This is the
 * intersection of the pixel formats supported by the viewports to which
 * @canvas is bound.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_canvas_get_pixel_formats (PgmCanvas *canvas,
                              gulong *pixel_formats)
{
  g_return_val_if_fail (PGM_IS_CANVAS (canvas), PGM_ERROR_X);
  g_return_val_if_fail (pixel_formats != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (canvas);
  *pixel_formats = canvas->pixel_formats;
  GST_OBJECT_UNLOCK (canvas);

  return PGM_ERROR_OK;
}

/**
 * pgm_canvas_get_pixel_offsets:
 * @canvas: a #PgmCanvas object.
 * @x: a pointer to a #gfloat where the x pixel offset is going to be stored.
 * @y: a pointer to a #gfloat where the y pixel offset is going to be stored.
 *
 * Gets the x and y ratios corresponding to the offsets in canvas coordinates
 * between two pixels at z=0.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_canvas_get_pixel_offsets (PgmCanvas *canvas,
                              gfloat *x,
                              gfloat *y)
{
  g_return_val_if_fail (PGM_IS_CANVAS (canvas), PGM_ERROR_X);
  g_return_val_if_fail (x != NULL, PGM_ERROR_X);
  g_return_val_if_fail (y != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (canvas);
  *x = canvas->pixel_offset_x;
  *y = canvas->pixel_offset_y;
  GST_OBJECT_UNLOCK (canvas);

  return PGM_ERROR_OK;
}

/* Protected methods */

/**
 * _pgm_canvas_update_order:
 * @canvas: the #PgmCanvas object.
 * @drawable: the #PgmDrawable to reorder.
 *
 * This function is only used internally by pgm_drawable_set_position() to
 * update the ordering of @drawable inside its layer when the z coordinate is
 * changed.
 *
 * MT safe.
 */
void
_pgm_canvas_update_order (PgmCanvas *canvas,
                          PgmDrawable *drawable)
{
  gint order;

  g_return_if_fail (PGM_IS_CANVAS (canvas));
  g_return_if_fail (PGM_IS_DRAWABLE (drawable));

  GST_OBJECT_LOCK (canvas);
  GST_OBJECT_LOCK (drawable);

  switch (drawable->layer)
    {
    case PGM_DRAWABLE_NEAR:
      if (!need_ordering_update (canvas->near_layer, drawable))
        goto no_update_needed;
      canvas->near_layer = g_list_remove (canvas->near_layer, drawable);
      canvas->near_layer = g_list_insert_sorted (canvas->near_layer, drawable,
                                                 (GCompareFunc) z_compare);
      order = g_list_index (canvas->near_layer, drawable);
      break;

    case PGM_DRAWABLE_MIDDLE:
      if (!need_ordering_update (canvas->middle_layer, drawable))
        goto no_update_needed;
      canvas->middle_layer = g_list_remove (canvas->middle_layer, drawable);
      canvas->middle_layer = g_list_insert_sorted (canvas->middle_layer,
                                                   drawable,
                                                   (GCompareFunc) z_compare);
      order = g_list_index (canvas->middle_layer, drawable);
      break;

    case PGM_DRAWABLE_FAR:
      if (!need_ordering_update (canvas->far_layer, drawable))
        goto no_update_needed;
      canvas->far_layer = g_list_remove (canvas->far_layer, drawable);
      canvas->far_layer = g_list_insert_sorted (canvas->far_layer, drawable,
                                                (GCompareFunc) z_compare);
      order = g_list_index (canvas->far_layer, drawable);
      break;

    default:
      goto no_update_needed;
      break;
    }

  GST_OBJECT_UNLOCK (drawable);
  GST_OBJECT_UNLOCK (canvas);

  g_signal_emit (G_OBJECT (canvas), pgm_canvas_signals[DRAWABLE_REORDERED],
                 0, drawable, drawable->layer, order);
  return;

 no_update_needed:
  GST_OBJECT_UNLOCK (drawable);
  GST_OBJECT_UNLOCK (canvas);
}

/**
 * _pgm_canvas_update_grid_aligned_drawables:
 * @canvas: a #PgmCanvas object.
 *
 * This function is only used internally by pgm_viewport_update_projection() to
 * realign drawables with the grid aligned property set.
 */
void
_pgm_canvas_update_grid_aligned_drawables (PgmCanvas *canvas)
{
  update_grid_aligned_drawables (canvas, canvas->near_layer);
  update_grid_aligned_drawables (canvas, canvas->middle_layer);
  update_grid_aligned_drawables (canvas, canvas->far_layer);
}

/**
 * _pgm_canvas_add_pixel_formats:
 * @canvas: a #PgmCanvas object.
 * @pixel_formats: the mask of #PgmImagePixelFormat to add.
 *
 * This function is only used internally by pgm_viewport_set_canvas() to
 * update @canvas pixel formats. @canvas pixel formats stores the intersection
 * of all the viewport pixel formats projecting @canvas. This function adds
 * a new pixel formats mask to @canvas pixel formats.
 *
 * MT safe.
 */
void
_pgm_canvas_add_pixel_formats (PgmCanvas *canvas,
                               gulong pixel_formats)
{
  gulong format;
  guint count;
  guint i;

  GST_OBJECT_LOCK (canvas);

  canvas->nb_formats++;
  canvas->pixel_formats = 0;

  for (i = 0; i < PGM_IMAGE_NB_PIXEL_FORMATS; i++)
    {
      format = 1 << i;
      count = 0;

      /* If the pixel format is available we update the format counter hash */
      if (pixel_formats & format)
        {
          count = GPOINTER_TO_INT (g_hash_table_lookup
                                   (canvas->format_counter,
                                    GINT_TO_POINTER (format))) + 1;
          g_hash_table_replace (canvas->format_counter,
                                GINT_TO_POINTER (format),
                                GINT_TO_POINTER (count));
        }

      /* If the number of pixel formats added equals the count of the current
       * format, all the pixel formats added supports the current format, so
       * that we can add the format to the canvas pixel formats mask. */
      if (count == canvas->nb_formats)
        canvas->pixel_formats |= format;
    }

  GST_OBJECT_UNLOCK (canvas);
}

/**
 * _pgm_canvas_remove_pixel_formats:
 * @canvas: a #PgmCanvas object.
 * @pixel_formats: the mask of #PgmImagePixelFormat to remove.
 *
 * This function is only used internally by pgm_viewport_set_canvas() to
 * update @canvas pixel formats. @canvas pixel formats stores the intersection
 * of all the viewport pixel formats projecting @canvas. This function removes
 * a pixel formats mask from the @canvas pixel formats.
 *
 * MT safe.
 */
void
_pgm_canvas_remove_pixel_formats (PgmCanvas *canvas,
                                  gulong pixel_formats)
{
  gulong format;
  guint count;
  guint i;

  GST_OBJECT_LOCK (canvas);

  /* We cannot remove a format not applied */
  if (G_UNLIKELY (canvas->nb_formats == 0))
    {
      GST_OBJECT_UNLOCK (canvas);
      return;
    }

  canvas->nb_formats--;
  canvas->pixel_formats = 0;

  for (i = 0; i < PGM_IMAGE_NB_PIXEL_FORMATS; i++)
    {
      format = 1 << i;

      /* Get the format counter */
      count = GPOINTER_TO_INT (g_hash_table_lookup (canvas->format_counter,
                                                    GINT_TO_POINTER (format)));

      /* If the pixel format is available we update the format counter hash
       * decreasing the count value */
      if (pixel_formats & format)
        {
          count--;
          g_hash_table_replace (canvas->format_counter,
                                GINT_TO_POINTER (format),
                                GINT_TO_POINTER (count));
        }

      /* And update the pixel format accordingly. The check of a non-zero
       * count permits to avoid filling the whole mask with 1 when the number
       * of formats applied is 0. */
      if (count && count == canvas->nb_formats)
        canvas->pixel_formats |= format;
    }

  GST_OBJECT_UNLOCK (canvas);
}


/**
 * _pgm_canvas_get_entered_emission_stopper:
 * @canvas: a #PgmCanvas instance.
 *
 * This function is only used internally by _pgm_drawable_do_motion_event() to
 * get the #PgmDrawable::entered emission stopper, that is, the #PgmDrawable
 * for which the callback connected to #PgmDrawable::entered returned TRUE.
 *
 * Returns: a #PgmDrawable instance, without a reference to it.
 */
PgmDrawable*
_pgm_canvas_get_entered_emission_stopper (PgmCanvas *canvas)
{
  PgmDrawable *entered_emission_stopper;

  g_return_val_if_fail (PGM_IS_CANVAS (canvas), NULL);

  GST_OBJECT_LOCK (canvas);
  entered_emission_stopper = canvas->entered_emission_stopper;
  GST_OBJECT_UNLOCK (canvas);

  return entered_emission_stopper;
}

/**
 * _pgm_canvas_set_entered_emission_stopper:
 * @canvas: a #PgmCanvas instance.
 * @drawable: a #PgmCanvas instance.
 *
 * This function is only used internally by _pgm_drawable_do_motion_event() to
 * set the #PgmDrawable::entered emission stopper, that is, the #PgmDrawable
 * for which the callback connected to #PgmDrawable::entered returned TRUE.
 *
 * MT safe.
 */
void
_pgm_canvas_set_entered_emission_stopper (PgmCanvas *canvas,
                                          PgmDrawable *drawable)
{
  g_return_if_fail (PGM_IS_CANVAS (canvas));

  GST_OBJECT_LOCK (canvas);
  canvas->entered_emission_stopper = drawable;
  GST_OBJECT_UNLOCK (canvas);
}
