/*
 *@BEGIN LICENSE
 *
 * PSI4: an ab initio quantum chemistry software package
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 *@END LICENSE
 */

/*! \file atom_data.cc
    \ingroup optking
    \brief atom_data.cc : initialize static class members in atom_data.h
*/

#include "atom_data.h"

#include <string>
#include <map>

namespace opt {

using std::string;
using std::map;

#define LAST_Z_TO_SYMBOL (118)
const char *Z_to_symbol[] = {
"X","H","HE","LI","BE","B","C","N","O","F","NE","NA","MG","AL","SI",
"P","S","CL","AR","K","CA","SC","TI","V","CR","MN","FE","CO","NI",
"CU","ZN","GA","GE","AS","SE","BR","KR","RB","SR","Y","ZR","NB","MO",
"TC","RU","RH","PD","AG","CD","IN","SN","SB","TE","I","XE","CS","BA",
"LA","CE","PR","ND","PM","SM","EU","GD","TB","DY","HO","ER","TM","YB",
"LU","HF","TA","W","RE","OS","IR","PT","AU","HG","TL","PB","BI","PO",
"AT","RN","FR","RA","AC","TH","PA","U","NP","PU","AM","CM","BK","CF",
"ES","FM","MD","NO","LR","RF","DB","SG","BH","HS","MT","DS","RG",
"UUB","UUT","UUQ","UUP","UUH","UUS","UUO"};

#define LAST_Z_TO_MASS (118)
const double Z_to_mass[] = {
0.,1.00782503207,4.00260325415,7.016004548,9.012182201,11.009305406,
12,14.00307400478,15.99491461956,18.998403224,19.99244017542,
22.98976928087,23.985041699,26.981538627,27.97692653246,30.973761629,
31.972070999,34.968852682,39.96238312251,38.963706679,39.962590983,
44.955911909,47.947946281,50.943959507,51.940507472,54.938045141,
55.934937475,58.933195048,57.935342907,62.929597474,63.929142222,
68.925573587,73.921177767,74.921596478,79.916521271,78.918337087,
85.910610729,84.911789737,87.905612124,88.905848295,89.904704416,
92.906378058,97.905408169,98.906254747,101.904349312,102.905504292,
105.903485715,106.90509682,113.90335854,114.903878484,119.902194676,
120.903815686,129.906224399,126.904472681,131.904153457,132.905451932,
137.905247237,138.906353267,139.905438706,140.907652769,141.907723297,
144.912749023,151.919732425,152.921230339,157.924103912,158.925346757,
163.929174751,164.93032207,165.930293061,168.93421325,173.938862089,
174.940771819,179.946549953,180.947995763,183.950931188,186.955753109,
191.96148069,192.96292643,194.964791134,196.966568662,201.970643011,
204.974427541,207.976652071,208.980398734,208.982430435,210.987496271,
222.017577738,222.01755173,228.031070292,227.027752127,232.038055325,
231.03588399,238.050788247,237.048173444,242.058742611,243.06138108,
247.07035354,247.07030708,251.079586788,252.082978512,257.095104724,
258.098431319,255.093241131,260.105504,263.112547,255.107398,259.114500,
262.122892,263.128558,265.136151,281.162061,272.153615,283.171792,283.176451,
285.183698,287.191186,292.199786,291.206564,293.214670};

map<string,double> create_element_to_Z_map() {
  map<string,double> m;
  m["G"]  =  0.0;  m["GHOST"]     =  0.0;
  m["H"]  =  1.0;  m["HYDROGEN"]  =  1.0;
  m["HE"] =  2.0;  m["HELIUM"]    =  2.0;
  m["LI"] =  3.0;  m["LITHIUM"]   =  3.0;
  m["BE"] =  4.0;  m["BERYLLIUM"] =  4.0;
  m["B"]  =  5.0;  m["BORON"]     =  5.0;
  m["C"]  =  6.0;  m["CARBON"]    =  6.0;
  m["N"]  =  7.0;  m["NITROGEN"]  =  7.0;
  m["O"]  =  8.0;  m["OXYGEN"]    =  8.0;
  m["F"]  =  9.0;  m["FLUORINE"]  =  9.0;
  m["NE"] = 10.0;  m["NEON"]      = 10.0;
  m["NA"] = 11.0;  m["SODIUM"]    = 11.0;
  m["MG"] = 12.0;  m["MAGNESIUM"] = 12.0;
  m["AL"] = 13.0;  m["ALUMINUM"]  = 13.0;
  m["SI"] = 14.0;  m["SILICON"]   = 14.0;
  m["P"]  = 15.0;  m["PHOSPHORUS"]= 15.0;
  m["S"]  = 16.0;  m["SULPHUR"]   = 16.0; m["SULFUR"] = 16.0;
  m["CL"] = 17.0;  m["CHLORINE"]  = 17.0;
  m["AR"] = 18.0;  m["ARGON"]     = 18.0;
  m["K"]  = 19.0;  m["POTASSIUM"] = 19.0;
  m["CA"] = 20.0;  m["CALCIUM"]   = 20.0;
  m["SC"] = 21.0;  m["SCANDIUM"]  = 21.0;
  m["TI"] = 22.0;  m["TITANIUM"]  = 22.0;
  m["V"]  = 23.0;  m["VANADIUM"]  = 23.0;
  m["CR"] = 24.0;  m["CHROMIUM"]  = 24.0;
  m["MN"] = 25.0;  m["MANGANESE"] = 25.0;
  m["FE"] = 26.0;  m["IRON"]      = 26.0;
  m["CO"] = 27.0;  m["COBALT"]    = 27.0;
  m["NI"] = 28.0;  m["NICKEL"]    = 28.0;
  m["CU"] = 29.0;  m["COPPER"]    = 29.0;
  m["ZN"] = 30.0;  m["ZINC"]      = 30.0;
  m["GA"] = 31.0;  m["GALLIUM"]   = 31.0;
  m["GE"] = 32.0;  m["GERMANIUM"] = 32.0;
  m["AS"] = 33.0;  m["ARSENIC"]   = 33.0;
  m["SE"] = 34.0;  m["SELENIUM"]  = 34.0;
  m["BR"] = 35.0;  m["BROMINE"]   = 35.0;
  m["KR"] = 36.0;  m["KRYPTON"]   = 36.0;
  m["RB"] = 37.0;  m["RUBIDIUM"]  = 37.0;
  m["SR"] = 38.0;  m["STRONTIUM"] = 38.0;
  m["Y"]  = 39.0;  m["YTTRIUM"]   = 39.0;
  m["ZR"] = 40.0;  m["ZIRCONIUM"] = 40.0;
  m["NB"] = 41.0;  m["NIOBIUM"]   = 41.0;
  m["MO"] = 42.0;  m["MOLYBDENUM"]= 42.0;
  m["TC"] = 43.0;  m["TECHNETIUM"]= 43.0;
  m["RU"] = 44.0;  m["RUTHENIUM"] = 44.0;
  m["RH"] = 45.0;  m["RHODIUM"]   = 45.0;
  m["PD"] = 46.0;  m["PALLADIUM"] = 46.0;
  m["AG"] = 47.0;  m["SILVER"]    = 47.0;
  m["CD"] = 48.0;  m["CADMIUM"]   = 48.0;
  m["IN"] = 49.0;  m["INDIUM"]    = 49.0;
  m["SN"] = 50.0;  m["TIN"]       = 50.0;
  m["SB"] = 51.0;  m["ANTIMONY"]  = 51.0;
  m["TE"] = 52.0;  m["TELLURIUM"] = 52.0;
  m["I"]  = 53.0;  m["IODINE"]    = 53.0;
  m["XE"] = 54.0;  m["XENON"]     = 54.0;
  m["CS"] = 55.0;  m["CESIUM"]    = 55.0;
  m["BA"] = 56.0;  m["BARIUM"]    = 56.0;
  m["LA"] = 57.0;  m["LANTHANUM"] = 57.0;
  m["CE"] = 58.0;  m["CERIUM"]    = 58.0;
  m["PR"] = 59.0;  m["PRASEODYMIUM"] = 59.0;
  m["ND"] = 60.0;  m["NEODYMIUM"] = 60.0;
  m["PM"] = 61.0;  m["PROMETHIUM"]= 61.0;
  m["SM"] = 62.0;  m["SAMARIUM"]  = 62.0;
  m["EU"] = 63.0;  m["EUROPIUM"]  = 63.0;
  m["GD"] = 64.0;  m["GADOLINIUM"]= 64.0;
  m["TB"] = 65.0;  m["TERBIUM"]   = 65.0;
  m["DY"] = 66.0;  m["DYSPROSIUM"]= 66.0;
  m["HO"] = 67.0;  m["HOLMIUM"]   = 67.0;
  m["ER"] = 68.0;  m["ERBIUM"]    = 68.0;
  m["TM"] = 69.0;  m["THULIUM"]   = 69.0;
  m["YB"] = 70.0;  m["YTTERBIUM"] = 70.0;
  m["LU"] = 71.0;  m["LUTETIUM"]  = 71.0;
  m["HF"] = 72.0;  m["HAFNIUM"]   = 72.0;
  m["TA"] = 73.0;  m["TANTALUM"]  = 73.0;
  m["W"]  = 74.0;  m["TUNGSTEN"]  = 74.0;
  m["RE"] = 75.0;  m["RHENIUM"]   = 75.0;
  m["OS"] = 76.0;  m["OSMIUM"]    = 76.0;
  m["IR"] = 77.0;  m["IRIDIUM"]   = 77.0;
  m["PT"] = 78.0;  m["PLATINUM"]  = 78.0;
  m["AU"] = 79.0;  m["GOLD"]      = 79.0;
  m["HG"] = 80.0;  m["MERCURY"]   = 80.0;
  m["TL"] = 81.0;  m["THALLIUM"]  = 81.0;
  m["PB"] = 82.0;  m["LEAD"]      = 82.0;
  m["BI"] = 83.0;  m["BISMUTH"]   = 83.0;
  m["PO"] = 84.0;  m["POLONIUM"]  = 84.0;
  m["AT"] = 85.0;  m["ASTATINE"]  = 85.0;
  m["RN"] = 86.0;  m["RADON"]     = 86.0;
  m["FR"] = 87.0;  m["FRANCIUM"]  = 87.0;
  m["RA"] = 88.0;  m["RADIUM"]    = 88.0;
  m["AC"] = 89.0;  m["ACTINIUM"]  = 89.0;
  m["TH"] = 90.0;  m["THORIUM"]   = 90.0;
  m["PA"] = 91.0;  m["PROTACTINIUM"]= 91.0;
  m["U"]  = 92.0;  m["URANIUM"]   = 92.0;
  m["NP"] = 93.0;  m["NEPTUNIUM"] = 93.0;
  m["PU"] = 94.0;  m["PLUTONIUM"] = 94.0;
  m["AM"] = 95.0;  m["AMERICIUM"] = 95.0;
  m["CM"] = 96.0;  m["CURIUM"]    = 96.0;
  m["BK"] = 97.0;  m["BERKELIUM"] = 97.0;
  m["CF"] = 98.0;  m["CALIFORNIUM"] = 98.0;
  m["ES"] = 99.0;  m["EINSTEINIUM"] = 99.0;
  m["FM"] = 100.0; m["FERMIUM"]     = 100.0;
  m["MD"] = 101.0; m["MENDELEVIUM"] = 101.0;
  m["NO"] = 102.0; m["NOBELIUM"]    = 102.0;
  m["LR"] = 103.0; m["LAWRENCIUM"]  = 103.0;
  m["UNQ"]= 104.0;
  m["UNP"]= 105.0;
  m["UNH"]= 106.0;
  m["UNS"]= 107.0;
  return m;
}

// Create static map for symbol->Z
static map<string,double> element_to_Z_map = create_element_to_Z_map();

// Define function to access map ; could be eliminated later
double Element_to_Z(std::string lbl) {
  return opt::element_to_Z_map[lbl];
}

/*
map<string,double> create_isotope_to_mass_map() {
  map<string,double> m;

  m[       "H"] =         1.0078250320;
  m[      "H1"] =         1.0078250320;
  m[      "H2"] =         2.0141017780;
  m[       "D"] =         2.0141017780;
  m[      "H3"] =         3.0160492780;
  m[       "T"] =         3.0160492780;
  m[      "H4"] =         4.0278064240;
  m[      "H5"] =         5.0353114880;
  m[      "H6"] =         6.0449425940;
  m[      "H7"] =         7.0527490000;
  m[      "HE"] =         4.0026032540;
  m[     "HE3"] =         3.0160293190;
  m[     "HE4"] =         4.0026032540;
  m[     "HE5"] =         5.0122236240;
  m[     "HE6"] =         6.0188891240;
  m[     "HE7"] =         7.0280206180;
  m[     "HE8"] =         8.0339218970;
  m[     "HE9"] =         9.0439502860;
  m[    "HE10"] =        10.0523988400;
  m[      "LI"] =         7.0160045480;
  m[     "LI3"] =         3.0307750000;
  m[     "LI4"] =         4.0271855580;
  m[     "LI5"] =         5.0125378000;
  m[     "LI6"] =         6.0151227940;
  m[     "LI7"] =         7.0160045480;
  m[     "LI8"] =         8.0224873620;
  m[     "LI9"] =         9.0267895050;
  m[    "LI10"] =        10.0354812600;
  m[    "LI11"] =        11.0437977200;
  m[    "LI12"] =        12.0537800000;
  m[      "BE"] =         9.0121822010;
  m[     "BE5"] =         5.0407900000;
  m[     "BE6"] =         6.0197263170;
  m[     "BE7"] =         7.0169298280;
  m[     "BE8"] =         8.0053051030;
  m[     "BE9"] =         9.0121822010;
  m[    "BE10"] =        10.0135338200;
  m[    "BE11"] =        11.0216577500;
  m[    "BE12"] =        12.0269207400;
  m[    "BE13"] =        13.0356930100;
  m[    "BE14"] =        14.0428929200;
  m[    "BE15"] =        15.0534600000;
  m[    "BE16"] =        16.0619200000;
  m[       "B"] =        11.0093054100;
  m[      "B6"] =         6.0468100000;
  m[      "B7"] =         7.0299179010;
  m[      "B8"] =         8.0246072330;
  m[      "B9"] =         9.0133287820;
  m[     "B10"] =        10.0129369900;
  m[     "B11"] =        11.0093054100;
  m[     "B12"] =        12.0143521000;
  m[     "B13"] =        13.0177802200;
  m[     "B14"] =        14.0254040100;
  m[     "B15"] =        15.0311030200;
  m[     "B16"] =        16.0398088300;
  m[     "B17"] =        17.0469899100;
  m[     "B18"] =        18.0561700000;
  m[     "B19"] =        19.0637300000;
  m[       "C"] =        12.0000000000;
  m[      "C8"] =         8.0376750250;
  m[      "C9"] =         9.0310366890;
  m[     "C10"] =        10.0168532300;
  m[     "C11"] =        11.0114336100;
  m[     "C12"] =        12.0000000000;
  m[     "C13"] =        13.0033548400;
  m[     "C14"] =        14.0032419900;
  m[     "C15"] =        15.0105992600;
  m[     "C16"] =        16.0147012500;
  m[     "C17"] =        17.0225861200;
  m[     "C18"] =        18.0267593500;
  m[     "C19"] =        19.0348050200;
  m[     "C20"] =        20.0403197500;
  m[     "C21"] =        21.0493400000;
  m[     "C22"] =        22.0572000000;
  m[       "N"] =        14.0030740000;
  m[     "N10"] =        10.0416536700;
  m[     "N11"] =        11.0260909600;
  m[     "N12"] =        12.0186132000;
  m[     "N13"] =        13.0057386100;
  m[     "N14"] =        14.0030740000;
  m[     "N15"] =        15.0001089000;
  m[     "N16"] =        16.0061016600;
  m[     "N17"] =        17.0084502600;
  m[     "N18"] =        18.0140789600;
  m[     "N19"] =        19.0170287000;
  m[     "N20"] =        20.0233658100;
  m[     "N21"] =        21.0271082400;
  m[     "N22"] =        22.0343949300;
  m[     "N23"] =        23.0412200000;
  m[     "N24"] =        24.0510400000;
  m[     "N25"] =        25.0606600000;
  m[       "O"] =        15.9949146200;
  m[     "O12"] =        12.0344049000;
  m[     "O13"] =        13.0248122100;
  m[     "O14"] =        14.0085962500;
  m[     "O15"] =        15.0030656200;
  m[     "O16"] =        15.9949146200;
  m[     "O17"] =        16.9991317000;
  m[     "O18"] =        17.9991610000;
  m[     "O19"] =        19.0035801300;
  m[     "O20"] =        20.0040767400;
  m[     "O21"] =        21.0086558900;
  m[     "O22"] =        22.0099669500;
  m[     "O23"] =        23.0156876600;
  m[     "O24"] =        24.0204729200;
  m[     "O25"] =        25.0294600000;
  m[     "O26"] =        26.0383400000;
  m[     "O27"] =        27.0482600000;
  m[     "O28"] =        28.0578100000;
  m[       "F"] =        18.9984032200;
  m[     "F14"] =        14.0350600000;
  m[     "F15"] =        15.0180091000;
  m[     "F16"] =        16.0114657200;
  m[     "F17"] =        17.0020952400;
  m[     "F18"] =        18.0009379600;
  m[     "F19"] =        18.9984032200;
  m[     "F20"] =        19.9999813200;
  m[     "F21"] =        20.9999489500;
  m[     "F22"] =        22.0029988200;
  m[     "F23"] =        23.0035746300;
  m[     "F24"] =        24.0081154900;
  m[     "F25"] =        25.0121017500;
  m[     "F26"] =        26.0196155600;
  m[     "F27"] =        27.0267600900;
  m[     "F28"] =        28.0356700000;
  m[     "F29"] =        29.0432600000;
  m[     "F30"] =        30.0525000000;
  m[     "F31"] =        31.0604290000;
  m[      "NE"] =        19.9924401800;
  m[    "NE16"] =        16.0257612600;
  m[    "NE17"] =        17.0176715000;
  m[    "NE18"] =        18.0057082100;
  m[    "NE19"] =        19.0018802500;
  m[    "NE20"] =        19.9924401800;
  m[    "NE21"] =        20.9938466800;
  m[    "NE22"] =        21.9913851100;
  m[    "NE23"] =        22.9944669000;
  m[    "NE24"] =        23.9936107800;
  m[    "NE25"] =        24.9977368900;
  m[    "NE26"] =        26.0004612100;
  m[    "NE27"] =        27.0075899000;
  m[    "NE28"] =        28.0120715800;
  m[    "NE29"] =        29.0193859300;
  m[    "NE30"] =        30.0248010500;
  m[    "NE31"] =        31.0331100000;
  m[    "NE32"] =        32.0400200000;
  m[    "NE33"] =        33.0493800000;
  m[    "NE34"] =        34.0570280000;
  m[      "NA"] =        22.9897692800;
  m[    "NA18"] =        18.0259690000;
  m[    "NA19"] =        19.0138775000;
  m[    "NA20"] =        20.0073513300;
  m[    "NA21"] =        20.9976552100;
  m[    "NA22"] =        21.9944364300;
  m[    "NA23"] =        22.9897692800;
  m[    "NA24"] =        23.9909627800;
  m[    "NA25"] =        24.9899539700;
  m[    "NA26"] =        25.9926330000;
  m[    "NA27"] =        26.9940767900;
  m[    "NA28"] =        27.9989380000;
  m[    "NA29"] =        29.0028610000;
  m[    "NA30"] =        30.0089760000;
  m[    "NA31"] =        31.0135854500;
  m[    "NA32"] =        32.0204665600;
  m[    "NA33"] =        33.0267197600;
  m[    "NA34"] =        34.0351700000;
  m[    "NA35"] =        35.0424930000;
  m[    "NA36"] =        36.0514800000;
  m[    "NA37"] =        37.0593400000;
  m[      "MG"] =        23.9850417000;
  m[    "MG19"] =        19.0354700000;
  m[    "MG20"] =        20.0188625500;
  m[    "MG21"] =        21.0117129100;
  m[    "MG22"] =        21.9995738400;
  m[    "MG23"] =        22.9941236700;
  m[    "MG24"] =        23.9850417000;
  m[    "MG25"] =        24.9858369200;
  m[    "MG26"] =        25.9825929300;
  m[    "MG27"] =        26.9843405900;
  m[    "MG28"] =        27.9838768300;
  m[    "MG29"] =        28.9886000000;
  m[    "MG30"] =        29.9904340000;
  m[    "MG31"] =        30.9965460000;
  m[    "MG32"] =        31.9989750000;
  m[    "MG33"] =        33.0052540000;
  m[    "MG34"] =        34.0094564200;
  m[    "MG35"] =        35.0173400000;
  m[    "MG36"] =        36.0230000000;
  m[    "MG37"] =        37.0314000000;
  m[    "MG38"] =        38.0375700000;
  m[    "MG39"] =        39.0467720000;
  m[    "MG40"] =        40.0539300000;
  m[      "AL"] =        26.9815386300;
  m[    "AL21"] =        21.0280400000;
  m[    "AL22"] =        22.0195200000;
  m[    "AL23"] =        23.0072674300;
  m[    "AL24"] =        23.9999388700;
  m[    "AL25"] =        24.9904281000;
  m[    "AL26"] =        25.9868916900;
  m[    "AL27"] =        26.9815386300;
  m[    "AL28"] =        27.9819103100;
  m[    "AL29"] =        28.9804450500;
  m[    "AL30"] =        29.9829602600;
  m[    "AL31"] =        30.9839466200;
  m[    "AL32"] =        31.9881244900;
  m[    "AL33"] =        32.9908433400;
  m[    "AL34"] =        33.9968518400;
  m[    "AL35"] =        34.9998602400;
  m[    "AL36"] =        36.0062072000;
  m[    "AL37"] =        37.0106778200;
  m[    "AL38"] =        38.0172310200;
  m[    "AL39"] =        39.0229700000;
  m[    "AL40"] =        40.0314500000;
  m[    "AL41"] =        41.0383300000;
  m[    "AL42"] =        42.0468900000;
  m[      "SI"] =        27.9769265300;
  m[    "SI22"] =        22.0345300000;
  m[    "SI23"] =        23.0255200000;
  m[    "SI24"] =        24.0115456200;
  m[    "SI25"] =        25.0041055700;
  m[    "SI26"] =        25.9923299200;
  m[    "SI27"] =        26.9867049100;
  m[    "SI28"] =        27.9769265300;
  m[    "SI29"] =        28.9764947000;
  m[    "SI30"] =        29.9737701700;
  m[    "SI31"] =        30.9753632300;
  m[    "SI32"] =        31.9741480800;
  m[    "SI33"] =        32.9780002200;
  m[    "SI34"] =        33.9785755200;
  m[    "SI35"] =        34.9845835800;
  m[    "SI36"] =        35.9865994800;
  m[    "SI37"] =        36.9929360800;
  m[    "SI38"] =        37.9956336000;
  m[    "SI39"] =        39.0020700100;
  m[    "SI40"] =        40.0058691200;
  m[    "SI41"] =        41.0145600000;
  m[    "SI42"] =        42.0197900000;
  m[    "SI43"] =        43.0286600000;
  m[    "SI44"] =        44.0352600000;
  m[       "P"] =        30.9737616300;
  m[     "P24"] =        24.0343500000;
  m[     "P25"] =        25.0202600000;
  m[     "P26"] =        26.0117800000;
  m[     "P27"] =        26.9992302400;
  m[     "P28"] =        27.9923147600;
  m[     "P29"] =        28.9818006100;
  m[     "P30"] =        29.9783137900;
  m[     "P31"] =        30.9737616300;
  m[     "P32"] =        31.9739072700;
  m[     "P33"] =        32.9717255400;
  m[     "P34"] =        33.9736362600;
  m[     "P35"] =        34.9733141200;
  m[     "P36"] =        35.9782596800;
  m[     "P37"] =        36.9796089500;
  m[     "P38"] =        37.9841568300;
  m[     "P39"] =        38.9861794800;
  m[     "P40"] =        39.9912969500;
  m[     "P41"] =        40.9943354400;
  m[     "P42"] =        42.0010079100;
  m[     "P43"] =        43.0061900000;
  m[     "P44"] =        44.0129900000;
  m[     "P45"] =        45.0192200000;
  m[     "P46"] =        46.0273800000;
  m[       "S"] =        31.9720710000;
  m[     "S26"] =        26.0278800000;
  m[     "S27"] =        27.0188330000;
  m[     "S28"] =        28.0043727600;
  m[     "S29"] =        28.9966080500;
  m[     "S30"] =        29.9849032500;
  m[     "S31"] =        30.9795547300;
  m[     "S32"] =        31.9720710000;
  m[     "S33"] =        32.9714587600;
  m[     "S34"] =        33.9678669000;
  m[     "S35"] =        34.9690321600;
  m[     "S36"] =        35.9670807600;
  m[     "S37"] =        36.9711255700;
  m[     "S38"] =        37.9711633200;
  m[     "S39"] =        38.9751343100;
  m[     "S40"] =        39.9754517300;
  m[     "S41"] =        40.9795821500;
  m[     "S42"] =        41.9810224200;
  m[     "S43"] =        42.9871547900;
  m[     "S44"] =        43.9902133900;
  m[     "S45"] =        44.9965081100;
  m[     "S46"] =        46.0007500000;
  m[     "S47"] =        47.0085900000;
  m[     "S48"] =        48.0141700000;
  m[     "S49"] =        49.0236190000;
  m[      "CL"] =        34.9688526800;
  m[    "CL28"] =        28.0285100000;
  m[    "CL29"] =        29.0141100000;
  m[    "CL30"] =        30.0047700000;
  m[    "CL31"] =        30.9924130900;
  m[    "CL32"] =        31.9856899000;
  m[    "CL33"] =        32.9774518900;
  m[    "CL34"] =        33.9737628200;
  m[    "CL35"] =        34.9688526800;
  m[    "CL36"] =        35.9683069800;
  m[    "CL37"] =        36.9659025900;
  m[    "CL38"] =        37.9680104300;
  m[    "CL39"] =        38.9680081600;
  m[    "CL40"] =        39.9704154700;
  m[    "CL41"] =        40.9706845300;
  m[    "CL42"] =        41.9732548000;
  m[    "CL43"] =        42.9740544000;
  m[    "CL44"] =        43.9782810700;
  m[    "CL45"] =        44.9802868900;
  m[    "CL46"] =        45.9842100400;
  m[    "CL47"] =        46.9887100000;
  m[    "CL48"] =        47.9949500000;
  m[    "CL49"] =        49.0003200000;
  m[    "CL50"] =        50.0078400000;
  m[    "CL51"] =        51.0144900000;
  m[      "AR"] =        39.9623831200;
  m[    "AR30"] =        30.0215600000;
  m[    "AR31"] =        31.0121230000;
  m[    "AR32"] =        31.9976379800;
  m[    "AR33"] =        32.9899257100;
  m[    "AR34"] =        33.9802712400;
  m[    "AR35"] =        34.9752575900;
  m[    "AR36"] =        35.9675451100;
  m[    "AR37"] =        36.9667763200;
  m[    "AR38"] =        37.9627323900;
  m[    "AR39"] =        38.9643132300;
  m[    "AR40"] =        39.9623831200;
  m[    "AR41"] =        40.9645006100;
  m[    "AR42"] =        41.9630457400;
  m[    "AR43"] =        42.9656360600;
  m[    "AR44"] =        43.9649240300;
  m[    "AR45"] =        44.9680399600;
  m[    "AR46"] =        45.9680941300;
  m[    "AR47"] =        46.9721867900;
  m[    "AR48"] =        47.9745400000;
  m[    "AR49"] =        48.9805200000;
  m[    "AR50"] =        49.9844300000;
  m[    "AR51"] =        50.9916300000;
  m[    "AR52"] =        51.9967800000;
  m[    "AR53"] =        53.0049400000;
  m[       "K"] =        38.9637066800;
  m[     "K32"] =        32.0219200000;
  m[     "K33"] =        33.0072600000;
  m[     "K34"] =        33.9984100000;
  m[     "K35"] =        34.9880096900;
  m[     "K36"] =        35.9812922400;
  m[     "K37"] =        36.9733758900;
  m[     "K38"] =        37.9690811800;
  m[     "K39"] =        38.9637066800;
  m[     "K40"] =        39.9639984800;
  m[     "K41"] =        40.9618257600;
  m[     "K42"] =        41.9624028100;
  m[     "K43"] =        42.9607155400;
  m[     "K44"] =        43.9615568000;
  m[     "K45"] =        44.9606994900;
  m[     "K46"] =        45.9619768600;
  m[     "K47"] =        46.9616784700;
  m[     "K48"] =        47.9655135400;
  m[     "K49"] =        48.9674509300;
  m[     "K50"] =        49.9727833600;
  m[     "K51"] =        50.9763800000;
  m[     "K52"] =        51.9826100000;
  m[     "K53"] =        52.9871200000;
  m[     "K54"] =        53.9942000000;
  m[     "K55"] =        54.9997100000;
  m[      "CA"] =        39.9625909800;
  m[    "CA34"] =        34.0141200000;
  m[    "CA35"] =        35.0049400000;
  m[    "CA36"] =        35.9930870600;
  m[    "CA37"] =        36.9858702700;
  m[    "CA38"] =        37.9763184500;
  m[    "CA39"] =        38.9707197300;
  m[    "CA40"] =        39.9625909800;
  m[    "CA41"] =        40.9622780600;
  m[    "CA42"] =        41.9586180100;
  m[    "CA43"] =        42.9587666300;
  m[    "CA44"] =        43.9554817500;
  m[    "CA45"] =        44.9561865700;
  m[    "CA46"] =        45.9536925900;
  m[    "CA47"] =        46.9545460100;
  m[    "CA48"] =        47.9525341800;
  m[    "CA49"] =        48.9556741500;
  m[    "CA50"] =        49.9575189600;
  m[    "CA51"] =        50.9614992100;
  m[    "CA52"] =        51.9651000000;
  m[    "CA53"] =        52.9700500000;
  m[    "CA54"] =        53.9743500000;
  m[    "CA55"] =        54.9805500000;
  m[    "CA56"] =        55.9855700000;
  m[    "CA57"] =        56.9923560000;
  m[      "SC"] =        44.9559119100;
  m[    "SC36"] =        36.0149200000;
  m[    "SC37"] =        37.0030500000;
  m[    "SC38"] =        37.9947000000;
  m[    "SC39"] =        38.9847900000;
  m[    "SC40"] =        39.9779674100;
  m[    "SC41"] =        40.9692511300;
  m[    "SC42"] =        41.9655164300;
  m[    "SC43"] =        42.9611506600;
  m[    "SC44"] =        43.9594027500;
  m[    "SC45"] =        44.9559119100;
  m[    "SC46"] =        45.9551718900;
  m[    "SC47"] =        46.9524075100;
  m[    "SC48"] =        47.9522314700;
  m[    "SC49"] =        48.9500239800;
  m[    "SC50"] =        49.9521876900;
  m[    "SC51"] =        50.9536033700;
  m[    "SC52"] =        51.9566754700;
  m[    "SC53"] =        52.9596100000;
  m[    "SC54"] =        53.9632645600;
  m[    "SC55"] =        54.9682439500;
  m[    "SC56"] =        55.9728700000;
  m[    "SC57"] =        56.9777900000;
  m[    "SC58"] =        57.9837100000;
  m[    "SC59"] =        58.9892200000;
  m[    "SC60"] =        59.9957100000;
  m[      "TI"] =        47.9479462800;
  m[    "TI38"] =        38.0097700000;
  m[    "TI39"] =        39.0016100000;
  m[    "TI40"] =        39.9904988400;
  m[    "TI41"] =        40.9831450000;
  m[    "TI42"] =        41.9730309000;
  m[    "TI43"] =        42.9685225000;
  m[    "TI44"] =        43.9596900700;
  m[    "TI45"] =        44.9581256200;
  m[    "TI46"] =        45.9526315600;
  m[    "TI47"] =        46.9517630900;
  m[    "TI48"] =        47.9479462800;
  m[    "TI49"] =        48.9478699800;
  m[    "TI50"] =        49.9447911900;
  m[    "TI51"] =        50.9466149600;
  m[    "TI52"] =        51.9468973100;
  m[    "TI53"] =        52.9497271700;
  m[    "TI54"] =        53.9510524000;
  m[    "TI55"] =        54.9552650600;
  m[    "TI56"] =        55.9581996400;
  m[    "TI57"] =        56.9639891400;
  m[    "TI58"] =        57.9669700000;
  m[    "TI59"] =        58.9729300000;
  m[    "TI60"] =        59.9767600000;
  m[    "TI61"] =        60.9832000000;
  m[    "TI62"] =        61.9874900000;
  m[    "TI63"] =        62.9944200000;
  m[       "V"] =        50.9439595100;
  m[     "V40"] =        40.0110900000;
  m[     "V41"] =        40.9997800000;
  m[     "V42"] =        41.9912300000;
  m[     "V43"] =        42.9806500000;
  m[     "V44"] =        43.9741100000;
  m[     "V45"] =        44.9657758100;
  m[     "V46"] =        45.9602004800;
  m[     "V47"] =        46.9549089400;
  m[     "V48"] =        47.9522537100;
  m[     "V49"] =        48.9485161000;
  m[     "V50"] =        49.9471584900;
  m[     "V51"] =        50.9439595100;
  m[     "V52"] =        51.9447754800;
  m[     "V53"] =        52.9443379800;
  m[     "V54"] =        53.9464398500;
  m[     "V55"] =        54.9472337000;
  m[     "V56"] =        55.9505309700;
  m[     "V57"] =        56.9525614300;
  m[     "V58"] =        57.9568341400;
  m[     "V59"] =        58.9602074100;
  m[     "V60"] =        59.9650268600;
  m[     "V61"] =        60.9684800000;
  m[     "V62"] =        61.9737800000;
  m[     "V63"] =        62.9775500000;
  m[     "V64"] =        63.9834700000;
  m[     "V65"] =        64.9879200000;
  m[      "CR"] =        51.9405074700;
  m[    "CR42"] =        42.0064300000;
  m[    "CR43"] =        42.9977100000;
  m[    "CR44"] =        43.9855490000;
  m[    "CR45"] =        44.9796400000;
  m[    "CR46"] =        45.9683586400;
  m[    "CR47"] =        46.9629000500;
  m[    "CR48"] =        47.9540317200;
  m[    "CR49"] =        48.9513357200;
  m[    "CR50"] =        49.9460442100;
  m[    "CR51"] =        50.9447674300;
  m[    "CR52"] =        51.9405074700;
  m[    "CR53"] =        52.9406493900;
  m[    "CR54"] =        53.9388804000;
  m[    "CR55"] =        54.9408396700;
  m[    "CR56"] =        55.9406531400;
  m[    "CR57"] =        56.9436130100;
  m[    "CR58"] =        57.9443531300;
  m[    "CR59"] =        58.9485863700;
  m[    "CR60"] =        59.9500760300;
  m[    "CR61"] =        60.9547172000;
  m[    "CR62"] =        61.9566131900;
  m[    "CR63"] =        62.9618600000;
  m[    "CR64"] =        63.9644100000;
  m[    "CR65"] =        64.9701600000;
  m[    "CR66"] =        65.9733800000;
  m[    "CR67"] =        66.9795500000;
  m[      "MN"] =        54.9380451400;
  m[    "MN44"] =        44.0068700000;
  m[    "MN45"] =        44.9945100000;
  m[    "MN46"] =        45.9867200000;
  m[    "MN47"] =        46.9761000000;
  m[    "MN48"] =        47.9685200000;
  m[    "MN49"] =        48.9596180100;
  m[    "MN50"] =        49.9542382300;
  m[    "MN51"] =        50.9482107900;
  m[    "MN52"] =        51.9455654600;
  m[    "MN53"] =        52.9412901200;
  m[    "MN54"] =        53.9403588500;
  m[    "MN55"] =        54.9380451400;
  m[    "MN56"] =        55.9389049100;
  m[    "MN57"] =        56.9382853800;
  m[    "MN58"] =        57.9399815500;
  m[    "MN59"] =        58.9404402400;
  m[    "MN60"] =        59.9429112500;
  m[    "MN61"] =        60.9446526400;
  m[    "MN62"] =        61.9484282200;
  m[    "MN63"] =        62.9502399900;
  m[    "MN64"] =        63.9542490900;
  m[    "MN65"] =        64.9563360700;
  m[    "MN66"] =        65.9610800000;
  m[    "MN67"] =        66.9641400000;
  m[    "MN68"] =        67.9693000000;
  m[    "MN69"] =        68.9728400000;
  m[      "FE"] =        55.9349374800;
  m[    "FE45"] =        45.0145780000;
  m[    "FE46"] =        46.0008100000;
  m[    "FE47"] =        46.9928900000;
  m[    "FE48"] =        47.9805040000;
  m[    "FE49"] =        48.9736100000;
  m[    "FE50"] =        49.9629889800;
  m[    "FE51"] =        50.9568195400;
  m[    "FE52"] =        51.9481138800;
  m[    "FE53"] =        52.9453079400;
  m[    "FE54"] =        53.9396105000;
  m[    "FE55"] =        54.9382933600;
  m[    "FE56"] =        55.9349374800;
  m[    "FE57"] =        56.9353939700;
  m[    "FE58"] =        57.9332755600;
  m[    "FE59"] =        58.9348754600;
  m[    "FE60"] =        59.9340716800;
  m[    "FE61"] =        60.9367452800;
  m[    "FE62"] =        61.9367674400;
  m[    "FE63"] =        62.9403690900;
  m[    "FE64"] =        63.9412012700;
  m[    "FE65"] =        64.9453802700;
  m[    "FE66"] =        65.9467806400;
  m[    "FE67"] =        66.9509472400;
  m[    "FE68"] =        67.9537000000;
  m[    "FE69"] =        68.9587800000;
  m[    "FE70"] =        69.9614600000;
  m[    "FE71"] =        70.9667200000;
  m[    "FE72"] =        71.9696200000;
  m[      "CO"] =        58.9331950500;
  m[    "CO47"] =        47.0114900000;
  m[    "CO48"] =        48.0017600000;
  m[    "CO49"] =        48.9897200000;
  m[    "CO50"] =        49.9815400000;
  m[    "CO51"] =        50.9707200000;
  m[    "CO52"] =        51.9635900000;
  m[    "CO53"] =        52.9542189000;
  m[    "CO54"] =        53.9484596400;
  m[    "CO55"] =        54.9419990300;
  m[    "CO56"] =        55.9398392800;
  m[    "CO57"] =        56.9362913700;
  m[    "CO58"] =        57.9357528100;
  m[    "CO59"] =        58.9331950500;
  m[    "CO60"] =        59.9338170600;
  m[    "CO61"] =        60.9324757600;
  m[    "CO62"] =        61.9340505600;
  m[    "CO63"] =        62.9336116100;
  m[    "CO64"] =        63.9358099100;
  m[    "CO65"] =        64.9364784600;
  m[    "CO66"] =        65.9397620000;
  m[    "CO67"] =        66.9408895300;
  m[    "CO68"] =        67.9448730600;
  m[    "CO69"] =        68.9463200000;
  m[    "CO70"] =        69.9510000000;
  m[    "CO71"] =        70.9529000000;
  m[    "CO72"] =        71.9578100000;
  m[    "CO73"] =        72.9602400000;
  m[    "CO74"] =        73.9653800000;
  m[    "CO75"] =        74.9683300000;
  m[      "NI"] =        57.9353429100;
  m[    "NI48"] =        48.0197500000;
  m[    "NI49"] =        49.0096600000;
  m[    "NI50"] =        49.9959300000;
  m[    "NI51"] =        50.9877200000;
  m[    "NI52"] =        51.9756800000;
  m[    "NI53"] =        52.9684700000;
  m[    "NI54"] =        53.9579055000;
  m[    "NI55"] =        54.9513302500;
  m[    "NI56"] =        55.9421320200;
  m[    "NI57"] =        56.9397935300;
  m[    "NI58"] =        57.9353429100;
  m[    "NI59"] =        58.9343467100;
  m[    "NI60"] =        59.9307863700;
  m[    "NI61"] =        60.9310560300;
  m[    "NI62"] =        61.9283451200;
  m[    "NI63"] =        62.9296693700;
  m[    "NI64"] =        63.9279659600;
  m[    "NI65"] =        64.9300843000;
  m[    "NI66"] =        65.9291393300;
  m[    "NI67"] =        66.9315694100;
  m[    "NI68"] =        67.9318687900;
  m[    "NI69"] =        68.9356102700;
  m[    "NI70"] =        69.9365000000;
  m[    "NI71"] =        70.9407362800;
  m[    "NI72"] =        71.9420926800;
  m[    "NI73"] =        72.9464700000;
  m[    "NI74"] =        73.9480700000;
  m[    "NI75"] =        74.9528700000;
  m[    "NI76"] =        75.9553300000;
  m[    "NI77"] =        76.9605500000;
  m[    "NI78"] =        77.9631800000;
  m[      "CU"] =        62.9295974700;
  m[    "CU52"] =        51.9971800000;
  m[    "CU53"] =        52.9855500000;
  m[    "CU54"] =        53.9767100000;
  m[    "CU55"] =        54.9660500000;
  m[    "CU56"] =        55.9585600000;
  m[    "CU57"] =        56.9492110800;
  m[    "CU58"] =        57.9445385000;
  m[    "CU59"] =        58.9394980300;
  m[    "CU60"] =        59.9373650300;
  m[    "CU61"] =        60.9334578200;
  m[    "CU62"] =        61.9325837500;
  m[    "CU63"] =        62.9295974700;
  m[    "CU64"] =        63.9297641800;
  m[    "CU65"] =        64.9277894900;
  m[    "CU66"] =        65.9288688100;
  m[    "CU67"] =        66.9277303100;
  m[    "CU68"] =        67.9296108900;
  m[    "CU69"] =        68.9294292700;
  m[    "CU70"] =        69.9323923400;
  m[    "CU71"] =        70.9326768300;
  m[    "CU72"] =        71.9358203100;
  m[    "CU73"] =        72.9366752800;
  m[    "CU74"] =        73.9398748600;
  m[    "CU75"] =        74.9419000000;
  m[    "CU76"] =        75.9452750300;
  m[    "CU77"] =        76.9478500000;
  m[    "CU78"] =        77.9519600000;
  m[    "CU79"] =        78.9545600000;
  m[    "CU80"] =        79.9608700000;
  m[      "ZN"] =        63.9291422200;
  m[    "ZN54"] =        53.9929500000;
  m[    "ZN55"] =        54.9839800000;
  m[    "ZN56"] =        55.9723800000;
  m[    "ZN57"] =        56.9647880000;
  m[    "ZN58"] =        57.9545915600;
  m[    "ZN59"] =        58.9492637600;
  m[    "ZN60"] =        59.9418270400;
  m[    "ZN61"] =        60.9395106400;
  m[    "ZN62"] =        61.9343297600;
  m[    "ZN63"] =        62.9332115700;
  m[    "ZN64"] =        63.9291422200;
  m[    "ZN65"] =        64.9292409800;
  m[    "ZN66"] =        65.9260334200;
  m[    "ZN67"] =        66.9271273500;
  m[    "ZN68"] =        67.9248441500;
  m[    "ZN69"] =        68.9265502800;
  m[    "ZN70"] =        69.9253192700;
  m[    "ZN71"] =        70.9277216000;
  m[    "ZN72"] =        71.9268579500;
  m[    "ZN73"] =        72.9297791000;
  m[    "ZN74"] =        73.9294586100;
  m[    "ZN75"] =        74.9329367400;
  m[    "ZN76"] =        75.9332935700;
  m[    "ZN77"] =        76.9369589700;
  m[    "ZN78"] =        77.9384402200;
  m[    "ZN79"] =        78.9426520000;
  m[    "ZN80"] =        79.9443423500;
  m[    "ZN81"] =        80.9504800000;
  m[    "ZN82"] =        81.9544200000;
  m[    "ZN83"] =        82.9610300000;
  m[      "GA"] =        68.9255735900;
  m[    "GA56"] =        55.9949100000;
  m[    "GA57"] =        56.9829300000;
  m[    "GA58"] =        57.9742500000;
  m[    "GA59"] =        58.9633700000;
  m[    "GA60"] =        59.9570600000;
  m[    "GA61"] =        60.9494462900;
  m[    "GA62"] =        61.9441752400;
  m[    "GA63"] =        62.9392942000;
  m[    "GA64"] =        63.9368387500;
  m[    "GA65"] =        64.9327347500;
  m[    "GA66"] =        65.9315890100;
  m[    "GA67"] =        66.9282017000;
  m[    "GA68"] =        67.9279800800;
  m[    "GA69"] =        68.9255735900;
  m[    "GA70"] =        69.9260219700;
  m[    "GA71"] =        70.9247013500;
  m[    "GA72"] =        71.9263662700;
  m[    "GA73"] =        72.9251746800;
  m[    "GA74"] =        73.9269457600;
  m[    "GA75"] =        74.9265002500;
  m[    "GA76"] =        75.9288276300;
  m[    "GA77"] =        76.9291543000;
  m[    "GA78"] =        77.9316081800;
  m[    "GA79"] =        78.9328932600;
  m[    "GA80"] =        79.9365157800;
  m[    "GA81"] =        80.9377523600;
  m[    "GA82"] =        81.9429900000;
  m[    "GA83"] =        82.9469800000;
  m[    "GA84"] =        83.9526500000;
  m[    "GA85"] =        84.9570000000;
  m[    "GA86"] =        85.9631200000;
  m[      "GE"] =        73.9211777700;
  m[    "GE58"] =        57.9910100000;
  m[    "GE59"] =        58.9817500000;
  m[    "GE60"] =        59.9701900000;
  m[    "GE61"] =        60.9637900000;
  m[    "GE62"] =        61.9546500000;
  m[    "GE63"] =        62.9496400000;
  m[    "GE64"] =        63.9416530000;
  m[    "GE65"] =        64.9394364100;
  m[    "GE66"] =        65.9338434500;
  m[    "GE67"] =        66.9327340700;
  m[    "GE68"] =        67.9280942400;
  m[    "GE69"] =        68.9279645300;
  m[    "GE70"] =        69.9242473800;
  m[    "GE71"] =        70.9249509500;
  m[    "GE72"] =        71.9220758200;
  m[    "GE73"] =        72.9234589500;
  m[    "GE74"] =        73.9211777700;
  m[    "GE75"] =        74.9228589500;
  m[    "GE76"] =        75.9214025600;
  m[    "GE77"] =        76.9235485900;
  m[    "GE78"] =        77.9228527400;
  m[    "GE79"] =        78.9254010000;
  m[    "GE80"] =        79.9253723900;
  m[    "GE81"] =        80.9288204700;
  m[    "GE82"] =        81.9295497300;
  m[    "GE83"] =        82.9346200000;
  m[    "GE84"] =        83.9374700000;
  m[    "GE85"] =        84.9430300000;
  m[    "GE86"] =        85.9464900000;
  m[    "GE87"] =        86.9525100000;
  m[    "GE88"] =        87.9569100000;
  m[    "GE89"] =        88.9638300000;
  m[      "AS"] =        74.9215964800;
  m[    "AS60"] =        59.9931300000;
  m[    "AS61"] =        60.9806200000;
  m[    "AS62"] =        61.9732000000;
  m[    "AS63"] =        62.9636900000;
  m[    "AS64"] =        63.9575720000;
  m[    "AS65"] =        64.9495640000;
  m[    "AS66"] =        65.9447100000;
  m[    "AS67"] =        66.9391860700;
  m[    "AS68"] =        67.9367690700;
  m[    "AS69"] =        68.9322736800;
  m[    "AS70"] =        69.9309248300;
  m[    "AS71"] =        70.9271124300;
  m[    "AS72"] =        71.9267522800;
  m[    "AS73"] =        72.9238248400;
  m[    "AS74"] =        73.9239286900;
  m[    "AS75"] =        74.9215964800;
  m[    "AS76"] =        75.9223940200;
  m[    "AS77"] =        76.9206472900;
  m[    "AS78"] =        77.9218272800;
  m[    "AS79"] =        78.9209479300;
  m[    "AS80"] =        79.9225338200;
  m[    "AS81"] =        80.9221322900;
  m[    "AS82"] =        81.9245040700;
  m[    "AS83"] =        82.9249800200;
  m[    "AS84"] =        83.9290580000;
  m[    "AS85"] =        84.9320200000;
  m[    "AS86"] =        85.9365000000;
  m[    "AS87"] =        86.9399000000;
  m[    "AS88"] =        87.9449400000;
  m[    "AS89"] =        88.9493900000;
  m[    "AS90"] =        89.9555000000;
  m[    "AS91"] =        90.9604300000;
  m[    "AS92"] =        91.9668000000;
  m[      "SE"] =        79.9165212700;
  m[    "SE65"] =        64.9646600000;
  m[    "SE66"] =        65.9552100000;
  m[    "SE67"] =        66.9500900000;
  m[    "SE68"] =        67.9417980000;
  m[    "SE69"] =        68.9395578200;
  m[    "SE70"] =        69.9333906400;
  m[    "SE71"] =        70.9322418200;
  m[    "SE72"] =        71.9271123500;
  m[    "SE73"] =        72.9267653500;
  m[    "SE74"] =        73.9224764400;
  m[    "SE75"] =        74.9225233700;
  m[    "SE76"] =        75.9192136000;
  m[    "SE77"] =        76.9199140400;
  m[    "SE78"] =        77.9173090900;
  m[    "SE79"] =        78.9184991000;
  m[    "SE80"] =        79.9165212700;
  m[    "SE81"] =        80.9179924700;
  m[    "SE82"] =        81.9166994000;
  m[    "SE83"] =        82.9191184700;
  m[    "SE84"] =        83.9184623500;
  m[    "SE85"] =        84.9222450500;
  m[    "SE86"] =        85.9242715800;
  m[    "SE87"] =        86.9285213600;
  m[    "SE88"] =        87.9314240000;
  m[    "SE89"] =        88.9364500000;
  m[    "SE90"] =        89.9399600000;
  m[    "SE91"] =        90.9459600000;
  m[    "SE92"] =        91.9499200000;
  m[    "SE93"] =        92.9562900000;
  m[    "SE94"] =        93.9604900000;
  m[      "BR"] =        78.9183370900;
  m[    "BR67"] =        66.9647900000;
  m[    "BR68"] =        67.9585160000;
  m[    "BR69"] =        68.9501060000;
  m[    "BR70"] =        69.9447920000;
  m[    "BR71"] =        70.9387400000;
  m[    "BR72"] =        71.9366445700;
  m[    "BR73"] =        72.9316915200;
  m[    "BR74"] =        73.9298910300;
  m[    "BR75"] =        74.9257762100;
  m[    "BR76"] =        75.9245414700;
  m[    "BR77"] =        76.9213790800;
  m[    "BR78"] =        77.9211457100;
  m[    "BR79"] =        78.9183370900;
  m[    "BR80"] =        79.9185293000;
  m[    "BR81"] =        80.9162905600;
  m[    "BR82"] =        81.9168041200;
  m[    "BR83"] =        82.9151804200;
  m[    "BR84"] =        83.9164789700;
  m[    "BR85"] =        84.9156084000;
  m[    "BR86"] =        85.9187975800;
  m[    "BR87"] =        86.9207113200;
  m[    "BR88"] =        87.9240659300;
  m[    "BR89"] =        88.9263853300;
  m[    "BR90"] =        89.9306277400;
  m[    "BR91"] =        90.9339681000;
  m[    "BR92"] =        91.9392587100;
  m[    "BR93"] =        92.9430500000;
  m[    "BR94"] =        93.9486800000;
  m[    "BR95"] =        94.9528700000;
  m[    "BR96"] =        95.9585300000;
  m[    "BR97"] =        96.9628000000;
  m[      "KR"] =        85.9106107300;
  m[    "KR69"] =        68.9651800000;
  m[    "KR70"] =        69.9552590000;
  m[    "KR71"] =        70.9496257400;
  m[    "KR72"] =        71.9420920400;
  m[    "KR73"] =        72.9392892000;
  m[    "KR74"] =        73.9330843700;
  m[    "KR75"] =        74.9309457500;
  m[    "KR76"] =        75.9259100800;
  m[    "KR77"] =        76.9246700000;
  m[    "KR78"] =        77.9203647800;
  m[    "KR79"] =        78.9200824300;
  m[    "KR80"] =        79.9163789700;
  m[    "KR81"] =        80.9165920200;
  m[    "KR82"] =        81.9134836000;
  m[    "KR83"] =        82.9141361000;
  m[    "KR84"] =        83.9115066900;
  m[    "KR85"] =        84.9125273300;
  m[    "KR86"] =        85.9106107300;
  m[    "KR87"] =        86.9133548600;
  m[    "KR88"] =        87.9144469700;
  m[    "KR89"] =        88.9176305800;
  m[    "KR90"] =        89.9195165600;
  m[    "KR91"] =        90.9234452200;
  m[    "KR92"] =        91.9261562100;
  m[    "KR93"] =        92.9312743600;
  m[    "KR94"] =        93.9343600000;
  m[    "KR95"] =        94.9398400000;
  m[    "KR96"] =        95.9430700000;
  m[    "KR97"] =        96.9485600000;
  m[    "KR98"] =        97.9519100000;
  m[    "KR99"] =        98.9576000000;
  m[   "KR100"] =        99.9611400000;
  m[      "RB"] =        84.9117897400;
  m[    "RB71"] =        70.9653200000;
  m[    "RB72"] =        71.9590800000;
  m[    "RB73"] =        72.9505610000;
  m[    "RB74"] =        73.9442647500;
  m[    "RB75"] =        74.9385700000;
  m[    "RB76"] =        75.9350722300;
  m[    "RB77"] =        76.9304080000;
  m[    "RB78"] =        77.9281410000;
  m[    "RB79"] =        78.9239894600;
  m[    "RB80"] =        79.9225192500;
  m[    "RB81"] =        80.9189959100;
  m[    "RB82"] =        81.9182086000;
  m[    "RB83"] =        82.9151097000;
  m[    "RB84"] =        83.9143848200;
  m[    "RB85"] =        84.9117897400;
  m[    "RB86"] =        85.9111674200;
  m[    "RB87"] =        86.9091805300;
  m[    "RB88"] =        87.9113155900;
  m[    "RB89"] =        88.9122780200;
  m[    "RB90"] =        89.9148016900;
  m[    "RB91"] =        90.9165369600;
  m[    "RB92"] =        91.9197289000;
  m[    "RB93"] =        92.9220418800;
  m[    "RB94"] =        93.9264049500;
  m[    "RB95"] =        94.9293028900;
  m[    "RB96"] =        95.9342726400;
  m[    "RB97"] =        96.9373519200;
  m[    "RB98"] =        97.9417906700;
  m[    "RB99"] =        98.9453792800;
  m[   "RB100"] =        99.9498700000;
  m[   "RB101"] =       100.9531964000;
  m[   "RB102"] =       101.9588700000;
  m[      "SR"] =        87.9056121200;
  m[    "SR73"] =        72.9659700000;
  m[    "SR74"] =        73.9563100000;
  m[    "SR75"] =        74.9499495700;
  m[    "SR76"] =        75.9417667800;
  m[    "SR77"] =        76.9379447800;
  m[    "SR78"] =        77.9321800000;
  m[    "SR79"] =        78.9297080000;
  m[    "SR80"] =        79.9245210100;
  m[    "SR81"] =        80.9232118500;
  m[    "SR82"] =        81.9184016400;
  m[    "SR83"] =        82.9175567000;
  m[    "SR84"] =        83.9134252800;
  m[    "SR85"] =        84.9129328000;
  m[    "SR86"] =        85.9092602000;
  m[    "SR87"] =        86.9088771200;
  m[    "SR88"] =        87.9056121200;
  m[    "SR89"] =        88.9074506800;
  m[    "SR90"] =        89.9077378900;
  m[    "SR91"] =        90.9102031000;
  m[    "SR92"] =        91.9110378600;
  m[    "SR93"] =        92.9140256300;
  m[    "SR94"] =        93.9153613100;
  m[    "SR95"] =        94.9193587700;
  m[    "SR96"] =        95.9216968000;
  m[    "SR97"] =        96.9261529200;
  m[    "SR98"] =        97.9284529300;
  m[    "SR99"] =        98.9332409300;
  m[   "SR100"] =        99.9353519100;
  m[   "SR101"] =       100.9405179000;
  m[   "SR102"] =       101.9430190000;
  m[   "SR103"] =       102.9489500000;
  m[   "SR104"] =       103.9523300000;
  m[   "SR105"] =       104.9585800000;
  m[       "Y"] =        88.9058483000;
  m[     "Y76"] =        75.9584500000;
  m[     "Y77"] =        76.9496450000;
  m[     "Y78"] =        77.9436100000;
  m[     "Y79"] =        78.9373516300;
  m[     "Y80"] =        79.9342800000;
  m[     "Y81"] =        80.9291274700;
  m[     "Y82"] =        81.9267924500;
  m[     "Y83"] =        82.9223542400;
  m[     "Y84"] =        83.9203882600;
  m[     "Y85"] =        84.9164330400;
  m[     "Y86"] =        85.9148855800;
  m[     "Y87"] =        86.9108757300;
  m[     "Y88"] =        87.9095011500;
  m[     "Y89"] =        88.9058483000;
  m[     "Y90"] =        89.9071518900;
  m[     "Y91"] =        90.9073047900;
  m[     "Y92"] =        91.9089491400;
  m[     "Y93"] =        92.9095827100;
  m[     "Y94"] =        93.9115952500;
  m[     "Y95"] =        94.9128206200;
  m[     "Y96"] =        95.9158913400;
  m[     "Y97"] =        96.9181340000;
  m[     "Y98"] =        97.9222030200;
  m[     "Y99"] =        98.9246362000;
  m[    "Y100"] =        99.9277565900;
  m[    "Y101"] =       100.9303139000;
  m[    "Y102"] =       101.9335557000;
  m[    "Y103"] =       102.9367300000;
  m[    "Y104"] =       103.9410500000;
  m[    "Y105"] =       104.9448700000;
  m[    "Y106"] =       105.9497900000;
  m[    "Y107"] =       106.9541400000;
  m[    "Y108"] =       107.9594800000;
  m[      "ZR"] =        89.9047044200;
  m[    "ZR78"] =        77.9552300000;
  m[    "ZR79"] =        78.9491600000;
  m[    "ZR80"] =        79.9404000000;
  m[    "ZR81"] =        80.9372100300;
  m[    "ZR82"] =        81.9310870000;
  m[    "ZR83"] =        82.9286538000;
  m[    "ZR84"] =        83.9232500000;
  m[    "ZR85"] =        84.9214711800;
  m[    "ZR86"] =        85.9164735900;
  m[    "ZR87"] =        86.9148162500;
  m[    "ZR88"] =        87.9102269000;
  m[    "ZR89"] =        88.9088895000;
  m[    "ZR90"] =        89.9047044200;
  m[    "ZR91"] =        90.9056457700;
  m[    "ZR92"] =        91.9050408500;
  m[    "ZR93"] =        92.9064760100;
  m[    "ZR94"] =        93.9063151900;
  m[    "ZR95"] =        94.9080426000;
  m[    "ZR96"] =        95.9082733900;
  m[    "ZR97"] =        96.9109531100;
  m[    "ZR98"] =        97.9127348900;
  m[    "ZR99"] =        98.9165121100;
  m[   "ZR100"] =        99.9177618900;
  m[   "ZR101"] =       100.9211404000;
  m[   "ZR102"] =       101.9229813000;
  m[   "ZR103"] =       102.9265996000;
  m[   "ZR104"] =       103.9287800000;
  m[   "ZR105"] =       104.9330500000;
  m[   "ZR106"] =       105.9359100000;
  m[   "ZR107"] =       106.9407500000;
  m[   "ZR108"] =       107.9439600000;
  m[   "ZR109"] =       108.9492400000;
  m[   "ZR110"] =       109.9528700000;
  m[      "NB"] =        92.9063780600;
  m[    "NB81"] =        80.9490300000;
  m[    "NB82"] =        81.9431300000;
  m[    "NB83"] =        82.9367053800;
  m[    "NB84"] =        83.9335700000;
  m[    "NB85"] =        84.9279124500;
  m[    "NB86"] =        85.9250383300;
  m[    "NB87"] =        86.9203611100;
  m[    "NB88"] =        87.9183321600;
  m[    "NB89"] =        88.9134182500;
  m[    "NB90"] =        89.9112648500;
  m[    "NB91"] =        90.9069962400;
  m[    "NB92"] =        91.9071938900;
  m[    "NB93"] =        92.9063780600;
  m[    "NB94"] =        93.9072838900;
  m[    "NB95"] =        94.9068357900;
  m[    "NB96"] =        95.9081006500;
  m[    "NB97"] =        96.9080985600;
  m[    "NB98"] =        97.9103284100;
  m[    "NB99"] =        98.9116183800;
  m[   "NB100"] =        99.9141816200;
  m[   "NB101"] =       100.9152520000;
  m[   "NB102"] =       101.9180376000;
  m[   "NB103"] =       102.9191438000;
  m[   "NB104"] =       103.9224647000;
  m[   "NB105"] =       104.9239365000;
  m[   "NB106"] =       105.9279700000;
  m[   "NB107"] =       106.9303100000;
  m[   "NB108"] =       107.9348400000;
  m[   "NB109"] =       108.9376300000;
  m[   "NB110"] =       109.9424400000;
  m[   "NB111"] =       110.9456500000;
  m[   "NB112"] =       111.9508300000;
  m[   "NB113"] =       112.9547000000;
  m[      "MO"] =        97.9054081700;
  m[    "MO83"] =        82.9487400000;
  m[    "MO84"] =        83.9400900000;
  m[    "MO85"] =        84.9365500000;
  m[    "MO86"] =        85.9306959000;
  m[    "MO87"] =        86.9273265000;
  m[    "MO88"] =        87.9219532400;
  m[    "MO89"] =        88.9194800100;
  m[    "MO90"] =        89.9139369000;
  m[    "MO91"] =        90.9117501900;
  m[    "MO92"] =        91.9068109900;
  m[    "MO93"] =        92.9068126100;
  m[    "MO94"] =        93.9050882700;
  m[    "MO95"] =        94.9058421300;
  m[    "MO96"] =        95.9046794800;
  m[    "MO97"] =        96.9060214700;
  m[    "MO98"] =        97.9054081700;
  m[    "MO99"] =        98.9077118700;
  m[   "MO100"] =        99.9074773400;
  m[   "MO101"] =       100.9103470000;
  m[   "MO102"] =       101.9102974000;
  m[   "MO103"] =       102.9132071000;
  m[   "MO104"] =       103.9137636000;
  m[   "MO105"] =       104.9169746000;
  m[   "MO106"] =       105.9181368000;
  m[   "MO107"] =       106.9216926000;
  m[   "MO108"] =       107.9234530000;
  m[   "MO109"] =       108.9278100000;
  m[   "MO110"] =       109.9297300000;
  m[   "MO111"] =       110.9344100000;
  m[   "MO112"] =       111.9368400000;
  m[   "MO113"] =       112.9418800000;
  m[   "MO114"] =       113.9449200000;
  m[   "MO115"] =       114.9502900000;
  m[      "TC"] =        98.9062547500;
  m[    "TC85"] =        84.9488300000;
  m[    "TC86"] =        85.9428800000;
  m[    "TC87"] =        86.9365300000;
  m[    "TC88"] =        87.9326780000;
  m[    "TC89"] =        88.9271670000;
  m[    "TC90"] =        89.9235565600;
  m[    "TC91"] =        90.9184276400;
  m[    "TC92"] =        91.9152601700;
  m[    "TC93"] =        92.9102489800;
  m[    "TC94"] =        93.9096570000;
  m[    "TC95"] =        94.9076570800;
  m[    "TC96"] =        95.9078713800;
  m[    "TC97"] =        96.9063653600;
  m[    "TC98"] =        97.9072159700;
  m[    "TC99"] =        98.9062547500;
  m[   "TC100"] =        99.9076577800;
  m[   "TC101"] =       100.9073147000;
  m[   "TC102"] =       101.9092150000;
  m[   "TC103"] =       102.9091814000;
  m[   "TC104"] =       103.9114475000;
  m[   "TC105"] =       104.9116606000;
  m[   "TC106"] =       105.9143579000;
  m[   "TC107"] =       106.9150796000;
  m[   "TC108"] =       107.9184612000;
  m[   "TC109"] =       108.9199827000;
  m[   "TC110"] =       109.9238205000;
  m[   "TC111"] =       110.9256928000;
  m[   "TC112"] =       111.9291465000;
  m[   "TC113"] =       112.9315900000;
  m[   "TC114"] =       113.9358800000;
  m[   "TC115"] =       114.9386900000;
  m[   "TC116"] =       115.9433700000;
  m[   "TC117"] =       116.9464800000;
  m[   "TC118"] =       117.9514800000;
  m[      "RU"] =       101.9043493000;
  m[    "RU87"] =        86.9491800000;
  m[    "RU88"] =        87.9402600000;
  m[    "RU89"] =        88.9361100000;
  m[    "RU90"] =        89.9298900000;
  m[    "RU91"] =        90.9262920000;
  m[    "RU92"] =        91.9201200000;
  m[    "RU93"] =        92.9170520300;
  m[    "RU94"] =        93.9113597100;
  m[    "RU95"] =        94.9104129300;
  m[    "RU96"] =        95.9075978400;
  m[    "RU97"] =        96.9075547000;
  m[    "RU98"] =        97.9052871300;
  m[    "RU99"] =        98.9059393000;
  m[   "RU100"] =        99.9042194800;
  m[   "RU101"] =       100.9055821000;
  m[   "RU102"] =       101.9043493000;
  m[   "RU103"] =       102.9063238000;
  m[   "RU104"] =       103.9054327000;
  m[   "RU105"] =       104.9077529000;
  m[   "RU106"] =       105.9073294000;
  m[   "RU107"] =       106.9099051000;
  m[   "RU108"] =       107.9101735000;
  m[   "RU109"] =       108.9132032000;
  m[   "RU110"] =       109.9141360000;
  m[   "RU111"] =       110.9176960000;
  m[   "RU112"] =       111.9189650000;
  m[   "RU113"] =       112.9224872000;
  m[   "RU114"] =       113.9242810000;
  m[   "RU115"] =       114.9286862000;
  m[   "RU116"] =       115.9308100000;
  m[   "RU117"] =       116.9355800000;
  m[   "RU118"] =       117.9378200000;
  m[   "RU119"] =       118.9428400000;
  m[   "RU120"] =       119.9453100000;
  m[      "RH"] =       102.9055043000;
  m[    "RH89"] =        88.9488370000;
  m[    "RH90"] =        89.9428700000;
  m[    "RH91"] =        90.9365500000;
  m[    "RH92"] =        91.9319800000;
  m[    "RH93"] =        92.9257400000;
  m[    "RH94"] =        93.9216980000;
  m[    "RH95"] =        94.9158987400;
  m[    "RH96"] =        95.9144606300;
  m[    "RH97"] =        96.9113368000;
  m[    "RH98"] =        97.9107081600;
  m[    "RH99"] =        98.9081321000;
  m[   "RH100"] =        99.9081215500;
  m[   "RH101"] =       100.9061636000;
  m[   "RH102"] =       101.9068432000;
  m[   "RH103"] =       102.9055043000;
  m[   "RH104"] =       103.9066555000;
  m[   "RH105"] =       104.9056938000;
  m[   "RH106"] =       105.9072871000;
  m[   "RH107"] =       106.9067484000;
  m[   "RH108"] =       107.9087280000;
  m[   "RH109"] =       108.9087373000;
  m[   "RH110"] =       109.9111364000;
  m[   "RH111"] =       110.9115859000;
  m[   "RH112"] =       111.9143942000;
  m[   "RH113"] =       112.9155306000;
  m[   "RH114"] =       113.9188060000;
  m[   "RH115"] =       114.9203340000;
  m[   "RH116"] =       115.9240620000;
  m[   "RH117"] =       116.9259800000;
  m[   "RH118"] =       117.9300700000;
  m[   "RH119"] =       118.9321100000;
  m[   "RH120"] =       119.9364100000;
  m[   "RH121"] =       120.9387200000;
  m[   "RH122"] =       121.9432100000;
  m[      "PD"] =       105.9034857000;
  m[    "PD91"] =        90.9491100000;
  m[    "PD92"] =        91.9404200000;
  m[    "PD93"] =        92.9359100000;
  m[    "PD94"] =        93.9287700000;
  m[    "PD95"] =        94.9246900000;
  m[    "PD96"] =        95.9181643600;
  m[    "PD97"] =        96.9164790700;
  m[    "PD98"] =        97.9127209000;
  m[    "PD99"] =        98.9117678300;
  m[   "PD100"] =        99.9085058900;
  m[   "PD101"] =       100.9082892000;
  m[   "PD102"] =       101.9056085000;
  m[   "PD103"] =       102.9060873000;
  m[   "PD104"] =       103.9040358000;
  m[   "PD105"] =       104.9050849000;
  m[   "PD106"] =       105.9034857000;
  m[   "PD107"] =       106.9051335000;
  m[   "PD108"] =       107.9038917000;
  m[   "PD109"] =       108.9059505000;
  m[   "PD110"] =       109.9051533000;
  m[   "PD111"] =       110.9076707000;
  m[   "PD112"] =       111.9073141000;
  m[   "PD113"] =       112.9101529000;
  m[   "PD114"] =       113.9103626000;
  m[   "PD115"] =       114.9136838000;
  m[   "PD116"] =       115.9141587000;
  m[   "PD117"] =       116.9178413000;
  m[   "PD118"] =       117.9189843000;
  m[   "PD119"] =       118.9231100000;
  m[   "PD120"] =       119.9246919000;
  m[   "PD121"] =       120.9288700000;
  m[   "PD122"] =       121.9305500000;
  m[   "PD123"] =       122.9349300000;
  m[   "PD124"] =       123.9368800000;
  m[      "AG"] =       106.9050968000;
  m[    "AG93"] =        92.9497800000;
  m[    "AG94"] =        93.9427800000;
  m[    "AG95"] =        94.9354800000;
  m[    "AG96"] =        95.9306800000;
  m[    "AG97"] =        96.9239724100;
  m[    "AG98"] =        97.9215662000;
  m[    "AG99"] =        98.9175971800;
  m[   "AG100"] =        99.9161042600;
  m[   "AG101"] =       100.9128022000;
  m[   "AG102"] =       101.9116850000;
  m[   "AG103"] =       102.9089727000;
  m[   "AG104"] =       103.9086292000;
  m[   "AG105"] =       104.9065287000;
  m[   "AG106"] =       105.9066689000;
  m[   "AG107"] =       106.9050968000;
  m[   "AG108"] =       107.9059556000;
  m[   "AG109"] =       108.9047523000;
  m[   "AG110"] =       109.9061072000;
  m[   "AG111"] =       110.9052912000;
  m[   "AG112"] =       111.9070048000;
  m[   "AG113"] =       112.9065666000;
  m[   "AG114"] =       113.9088037000;
  m[   "AG115"] =       114.9087627000;
  m[   "AG116"] =       115.9113599000;
  m[   "AG117"] =       116.9116846000;
  m[   "AG118"] =       117.9145828000;
  m[   "AG119"] =       118.9156651000;
  m[   "AG120"] =       119.9187874000;
  m[   "AG121"] =       120.9198480000;
  m[   "AG122"] =       121.9235300000;
  m[   "AG123"] =       122.9249000000;
  m[   "AG124"] =       123.9286400000;
  m[   "AG125"] =       124.9304300000;
  m[   "AG126"] =       125.9345000000;
  m[   "AG127"] =       126.9367700000;
  m[   "AG128"] =       127.9411700000;
  m[   "AG129"] =       128.9436900000;
  m[   "AG130"] =       129.9504480000;
  m[      "CD"] =       113.9033585000;
  m[    "CD95"] =        94.9498700000;
  m[    "CD96"] =        95.9397700000;
  m[    "CD97"] =        96.9349400000;
  m[    "CD98"] =        97.9273955500;
  m[    "CD99"] =        98.9250100000;
  m[   "CD100"] =        99.9202895300;
  m[   "CD101"] =       100.9186815000;
  m[   "CD102"] =       101.9144623000;
  m[   "CD103"] =       102.9134192000;
  m[   "CD104"] =       103.9098495000;
  m[   "CD105"] =       104.9094679000;
  m[   "CD106"] =       105.9064594000;
  m[   "CD107"] =       106.9066179000;
  m[   "CD108"] =       107.9041837000;
  m[   "CD109"] =       108.9049823000;
  m[   "CD110"] =       109.9030021000;
  m[   "CD111"] =       110.9041781000;
  m[   "CD112"] =       111.9027578000;
  m[   "CD113"] =       112.9044017000;
  m[   "CD114"] =       113.9033585000;
  m[   "CD115"] =       114.9054310000;
  m[   "CD116"] =       115.9047558000;
  m[   "CD117"] =       116.9072186000;
  m[   "CD118"] =       117.9069145000;
  m[   "CD119"] =       118.9099216000;
  m[   "CD120"] =       119.9098501000;
  m[   "CD121"] =       120.9129774000;
  m[   "CD122"] =       121.9133324000;
  m[   "CD123"] =       122.9170030000;
  m[   "CD124"] =       123.9176476000;
  m[   "CD125"] =       124.9212464000;
  m[   "CD126"] =       125.9223533000;
  m[   "CD127"] =       126.9264439000;
  m[   "CD128"] =       127.9277623000;
  m[   "CD129"] =       128.9321500000;
  m[   "CD130"] =       129.9339019000;
  m[   "CD131"] =       130.9406700000;
  m[   "CD132"] =       131.9455500000;
  m[      "IN"] =       114.9038785000;
  m[    "IN97"] =        96.9495400000;
  m[    "IN98"] =        97.9421400000;
  m[    "IN99"] =        98.9342200000;
  m[   "IN100"] =        99.9311108500;
  m[   "IN101"] =       100.9263400000;
  m[   "IN102"] =       101.9240902000;
  m[   "IN103"] =       102.9199142000;
  m[   "IN104"] =       103.9182962000;
  m[   "IN105"] =       104.9146735000;
  m[   "IN106"] =       105.9134654000;
  m[   "IN107"] =       106.9102951000;
  m[   "IN108"] =       107.9096982000;
  m[   "IN109"] =       108.9071505000;
  m[   "IN110"] =       109.9071653000;
  m[   "IN111"] =       110.9051033000;
  m[   "IN112"] =       111.9055323000;
  m[   "IN113"] =       112.9040578000;
  m[   "IN114"] =       113.9049139000;
  m[   "IN115"] =       114.9038785000;
  m[   "IN116"] =       115.9052597000;
  m[   "IN117"] =       116.9045136000;
  m[   "IN118"] =       117.9063544000;
  m[   "IN119"] =       118.9058454000;
  m[   "IN120"] =       119.9079596000;
  m[   "IN121"] =       120.9078458000;
  m[   "IN122"] =       121.9102760000;
  m[   "IN123"] =       122.9104383000;
  m[   "IN124"] =       123.9131752000;
  m[   "IN125"] =       124.9136006000;
  m[   "IN126"] =       125.9164639000;
  m[   "IN127"] =       126.9173531000;
  m[   "IN128"] =       127.9201723000;
  m[   "IN129"] =       128.9216970000;
  m[   "IN130"] =       129.9249700000;
  m[   "IN131"] =       130.9268518000;
  m[   "IN132"] =       131.9329903000;
  m[   "IN133"] =       132.9378100000;
  m[   "IN134"] =       133.9441500000;
  m[   "IN135"] =       134.9493300000;
  m[      "SN"] =       119.9021947000;
  m[    "SN99"] =        98.9493300000;
  m[   "SN100"] =        99.9390443400;
  m[   "SN101"] =       100.9360600000;
  m[   "SN102"] =       101.9302953000;
  m[   "SN103"] =       102.9281000000;
  m[   "SN104"] =       103.9231432000;
  m[   "SN105"] =       104.9213494000;
  m[   "SN106"] =       105.9168806000;
  m[   "SN107"] =       106.9156443000;
  m[   "SN108"] =       107.9119254000;
  m[   "SN109"] =       108.9112832000;
  m[   "SN110"] =       109.9078428000;
  m[   "SN111"] =       110.9077345000;
  m[   "SN112"] =       111.9048182000;
  m[   "SN113"] =       112.9051706000;
  m[   "SN114"] =       113.9027789000;
  m[   "SN115"] =       114.9033424000;
  m[   "SN116"] =       115.9017405000;
  m[   "SN117"] =       116.9029517000;
  m[   "SN118"] =       117.9016032000;
  m[   "SN119"] =       118.9033076000;
  m[   "SN120"] =       119.9021947000;
  m[   "SN121"] =       120.9042355000;
  m[   "SN122"] =       121.9034390000;
  m[   "SN123"] =       122.9057208000;
  m[   "SN124"] =       123.9052739000;
  m[   "SN125"] =       124.9077841000;
  m[   "SN126"] =       125.9076533000;
  m[   "SN127"] =       126.9103600000;
  m[   "SN128"] =       127.9105366000;
  m[   "SN129"] =       128.9134790000;
  m[   "SN130"] =       129.9139673000;
  m[   "SN131"] =       130.9169998000;
  m[   "SN132"] =       131.9178157000;
  m[   "SN133"] =       132.9238292000;
  m[   "SN134"] =       133.9282918000;
  m[   "SN135"] =       134.9347300000;
  m[   "SN136"] =       135.9393400000;
  m[   "SN137"] =       136.9459900000;
  m[      "SB"] =       120.9038157000;
  m[   "SB103"] =       102.9396900000;
  m[   "SB104"] =       103.9364720000;
  m[   "SB105"] =       104.9314863000;
  m[   "SB106"] =       105.9287910000;
  m[   "SB107"] =       106.9241500000;
  m[   "SB108"] =       107.9221600000;
  m[   "SB109"] =       108.9181324000;
  m[   "SB110"] =       109.9167530000;
  m[   "SB111"] =       110.9131630000;
  m[   "SB112"] =       111.9123980000;
  m[   "SB113"] =       112.9093717000;
  m[   "SB114"] =       113.9092690000;
  m[   "SB115"] =       114.9065980000;
  m[   "SB116"] =       115.9067936000;
  m[   "SB117"] =       116.9048359000;
  m[   "SB118"] =       117.9055287000;
  m[   "SB119"] =       118.9039420000;
  m[   "SB120"] =       119.9050724000;
  m[   "SB121"] =       120.9038157000;
  m[   "SB122"] =       121.9051737000;
  m[   "SB123"] =       122.9042140000;
  m[   "SB124"] =       123.9059357000;
  m[   "SB125"] =       124.9052538000;
  m[   "SB126"] =       125.9072475000;
  m[   "SB127"] =       126.9069236000;
  m[   "SB128"] =       127.9091690000;
  m[   "SB129"] =       128.9091484000;
  m[   "SB130"] =       129.9116563000;
  m[   "SB131"] =       130.9119823000;
  m[   "SB132"] =       131.9144669000;
  m[   "SB133"] =       132.9152516000;
  m[   "SB134"] =       133.9203797000;
  m[   "SB135"] =       134.9251658000;
  m[   "SB136"] =       135.9303500000;
  m[   "SB137"] =       136.9353100000;
  m[   "SB138"] =       137.9407900000;
  m[   "SB139"] =       138.9459800000;
  m[      "TE"] =       129.9062244000;
  m[   "TE105"] =       104.9436400000;
  m[   "TE106"] =       105.9375042370;
  m[   "TE107"] =       106.9350060000;
  m[   "TE108"] =       107.9294446000;
  m[   "TE109"] =       108.9274155000;
  m[   "TE110"] =       109.9224073000;
  m[   "TE111"] =       110.9211107000;
  m[   "TE112"] =       111.9170137000;
  m[   "TE113"] =       112.9158910000;
  m[   "TE114"] =       113.9120890000;
  m[   "TE115"] =       114.9119020000;
  m[   "TE116"] =       115.9084600000;
  m[   "TE117"] =       116.9086447000;
  m[   "TE118"] =       117.9058276000;
  m[   "TE119"] =       118.9064036000;
  m[   "TE120"] =       119.9040202000;
  m[   "TE121"] =       120.9049364000;
  m[   "TE122"] =       121.9030439000;
  m[   "TE123"] =       122.9042700000;
  m[   "TE124"] =       123.9028179000;
  m[   "TE125"] =       124.9044307000;
  m[   "TE126"] =       125.9033117000;
  m[   "TE127"] =       126.9052263000;
  m[   "TE128"] =       127.9044631000;
  m[   "TE129"] =       128.9065982000;
  m[   "TE130"] =       129.9062244000;
  m[   "TE131"] =       130.9085239000;
  m[   "TE132"] =       131.9085532000;
  m[   "TE133"] =       132.9109553000;
  m[   "TE134"] =       133.9113687000;
  m[   "TE135"] =       134.9164486000;
  m[   "TE136"] =       135.9201012000;
  m[   "TE137"] =       136.9253230000;
  m[   "TE138"] =       137.9292200000;
  m[   "TE139"] =       138.9347300000;
  m[   "TE140"] =       139.9388500000;
  m[   "TE141"] =       140.9446500000;
  m[   "TE142"] =       141.9490800000;
  m[       "I"] =       126.9044727000;
  m[    "I108"] =       107.9434750000;
  m[    "I109"] =       108.9381494000;
  m[    "I110"] =       109.9352420000;
  m[    "I111"] =       110.9302760000;
  m[    "I112"] =       111.9279700000;
  m[    "I113"] =       112.9236406000;
  m[    "I114"] =       113.9218500000;
  m[    "I115"] =       114.9180480000;
  m[    "I116"] =       115.9168086000;
  m[    "I117"] =       116.9136500000;
  m[    "I118"] =       117.9130740000;
  m[    "I119"] =       118.9100740000;
  m[    "I120"] =       119.9100482000;
  m[    "I121"] =       120.9073668000;
  m[    "I122"] =       121.9075893000;
  m[    "I123"] =       122.9055890000;
  m[    "I124"] =       123.9062099000;
  m[    "I125"] =       124.9046302000;
  m[    "I126"] =       125.9056242000;
  m[    "I127"] =       126.9044727000;
  m[    "I128"] =       127.9058094000;
  m[    "I129"] =       128.9049877000;
  m[    "I130"] =       129.9066742000;
  m[    "I131"] =       130.9061246000;
  m[    "I132"] =       131.9079974000;
  m[    "I133"] =       132.9077969000;
  m[    "I134"] =       133.9097445000;
  m[    "I135"] =       134.9100481000;
  m[    "I136"] =       135.9146540000;
  m[    "I137"] =       136.9178708000;
  m[    "I138"] =       137.9223496000;
  m[    "I139"] =       138.9260995000;
  m[    "I140"] =       139.9310000000;
  m[    "I141"] =       140.9350300000;
  m[    "I142"] =       141.9401800000;
  m[    "I143"] =       142.9445600000;
  m[    "I144"] =       143.9499900000;
  m[      "XE"] =       131.9041535000;
  m[   "XE110"] =       109.9442781000;
  m[   "XE111"] =       110.9416020000;
  m[   "XE112"] =       111.9356231000;
  m[   "XE113"] =       112.9333412000;
  m[   "XE114"] =       113.9279803000;
  m[   "XE115"] =       114.9262939000;
  m[   "XE116"] =       115.9215811000;
  m[   "XE117"] =       116.9203587000;
  m[   "XE118"] =       117.9161787000;
  m[   "XE119"] =       118.9154107000;
  m[   "XE120"] =       119.9117842000;
  m[   "XE121"] =       120.9114618000;
  m[   "XE122"] =       121.9083676000;
  m[   "XE123"] =       122.9084819000;
  m[   "XE124"] =       123.9058930000;
  m[   "XE125"] =       124.9063955000;
  m[   "XE126"] =       125.9042736000;
  m[   "XE127"] =       126.9051837000;
  m[   "XE128"] =       127.9035313000;
  m[   "XE129"] =       128.9047794000;
  m[   "XE130"] =       129.9035080000;
  m[   "XE131"] =       130.9050824000;
  m[   "XE132"] =       131.9041535000;
  m[   "XE133"] =       132.9059107000;
  m[   "XE134"] =       133.9053945000;
  m[   "XE135"] =       134.9072275000;
  m[   "XE136"] =       135.9072188000;
  m[   "XE137"] =       136.9115621000;
  m[   "XE138"] =       137.9139545000;
  m[   "XE139"] =       138.9187929000;
  m[   "XE140"] =       139.9216409000;
  m[   "XE141"] =       140.9266480000;
  m[   "XE142"] =       141.9297096000;
  m[   "XE143"] =       142.9351100000;
  m[   "XE144"] =       143.9385100000;
  m[   "XE145"] =       144.9440700000;
  m[   "XE146"] =       145.9477500000;
  m[   "XE147"] =       146.9535600000;
  m[      "CS"] =       132.9054519000;
  m[   "CS112"] =       111.9503010000;
  m[   "CS113"] =       112.9444933000;
  m[   "CS114"] =       113.9414500000;
  m[   "CS115"] =       114.9359100000;
  m[   "CS116"] =       115.9333670000;
  m[   "CS117"] =       116.9286707000;
  m[   "CS118"] =       117.9265595000;
  m[   "CS119"] =       118.9223773000;
  m[   "CS120"] =       119.9206773000;
  m[   "CS121"] =       120.9172292000;
  m[   "CS122"] =       121.9161134000;
  m[   "CS123"] =       122.9129960000;
  m[   "CS124"] =       123.9122578000;
  m[   "CS125"] =       124.9097283000;
  m[   "CS126"] =       125.9094520000;
  m[   "CS127"] =       126.9074175000;
  m[   "CS128"] =       127.9077489000;
  m[   "CS129"] =       128.9060644000;
  m[   "CS130"] =       129.9067086000;
  m[   "CS131"] =       130.9054639000;
  m[   "CS132"] =       131.9064343000;
  m[   "CS133"] =       132.9054519000;
  m[   "CS134"] =       133.9067185000;
  m[   "CS135"] =       134.9059770000;
  m[   "CS136"] =       135.9073116000;
  m[   "CS137"] =       136.9070895000;
  m[   "CS138"] =       137.9110167000;
  m[   "CS139"] =       138.9133640000;
  m[   "CS140"] =       139.9172824000;
  m[   "CS141"] =       140.9200458000;
  m[   "CS142"] =       141.9242989000;
  m[   "CS143"] =       142.9273518000;
  m[   "CS144"] =       143.9320769000;
  m[   "CS145"] =       144.9355262000;
  m[   "CS146"] =       145.9402894000;
  m[   "CS147"] =       146.9441550000;
  m[   "CS148"] =       147.9492182000;
  m[   "CS149"] =       148.9529300000;
  m[   "CS150"] =       149.9581700000;
  m[   "CS151"] =       150.9621900000;
  m[      "BA"] =       137.9052472000;
  m[   "BA114"] =       113.9506754000;
  m[   "BA115"] =       114.9473700000;
  m[   "BA116"] =       115.9413800000;
  m[   "BA117"] =       116.9384990000;
  m[   "BA118"] =       117.9330400000;
  m[   "BA119"] =       118.9306597000;
  m[   "BA120"] =       119.9260450000;
  m[   "BA121"] =       120.9240545000;
  m[   "BA122"] =       121.9199040000;
  m[   "BA123"] =       122.9187810000;
  m[   "BA124"] =       123.9150936000;
  m[   "BA125"] =       124.9144729000;
  m[   "BA126"] =       125.9112502000;
  m[   "BA127"] =       126.9110938000;
  m[   "BA128"] =       127.9083177000;
  m[   "BA129"] =       128.9086794000;
  m[   "BA130"] =       129.9063208000;
  m[   "BA131"] =       130.9069411000;
  m[   "BA132"] =       131.9050613000;
  m[   "BA133"] =       132.9060075000;
  m[   "BA134"] =       133.9045084000;
  m[   "BA135"] =       134.9056886000;
  m[   "BA136"] =       135.9045759000;
  m[   "BA137"] =       136.9058274000;
  m[   "BA138"] =       137.9052472000;
  m[   "BA139"] =       138.9088413000;
  m[   "BA140"] =       139.9106045000;
  m[   "BA141"] =       140.9144110000;
  m[   "BA142"] =       141.9164534000;
  m[   "BA143"] =       142.9206267000;
  m[   "BA144"] =       143.9229529000;
  m[   "BA145"] =       144.9276270000;
  m[   "BA146"] =       145.9302196000;
  m[   "BA147"] =       146.9349450000;
  m[   "BA148"] =       147.9377200000;
  m[   "BA149"] =       148.9425800000;
  m[   "BA150"] =       149.9456800000;
  m[   "BA151"] =       150.9508100000;
  m[   "BA152"] =       151.9542700000;
  m[   "BA153"] =       152.9596100000;
  m[      "LA"] =       138.9063533000;
  m[   "LA117"] =       116.9500680000;
  m[   "LA118"] =       117.9467300000;
  m[   "LA119"] =       118.9409900000;
  m[   "LA120"] =       119.9380700000;
  m[   "LA121"] =       120.9330100000;
  m[   "LA122"] =       121.9307100000;
  m[   "LA123"] =       122.9262400000;
  m[   "LA124"] =       123.9245743000;
  m[   "LA125"] =       124.9208160000;
  m[   "LA126"] =       125.9195127000;
  m[   "LA127"] =       126.9163754000;
  m[   "LA128"] =       127.9155852000;
  m[   "LA129"] =       128.9126928000;
  m[   "LA130"] =       129.9123687000;
  m[   "LA131"] =       130.9100700000;
  m[   "LA132"] =       131.9101011000;
  m[   "LA133"] =       132.9082180000;
  m[   "LA134"] =       133.9085140000;
  m[   "LA135"] =       134.9069768000;
  m[   "LA136"] =       135.9076355000;
  m[   "LA137"] =       136.9064936000;
  m[   "LA138"] =       137.9071119000;
  m[   "LA139"] =       138.9063533000;
  m[   "LA140"] =       139.9094776000;
  m[   "LA141"] =       140.9109622000;
  m[   "LA142"] =       141.9140791000;
  m[   "LA143"] =       142.9160627000;
  m[   "LA144"] =       143.9195996000;
  m[   "LA145"] =       144.9216454000;
  m[   "LA146"] =       145.9257935000;
  m[   "LA147"] =       146.9282353000;
  m[   "LA148"] =       147.9322289000;
  m[   "LA149"] =       148.9347340000;
  m[   "LA150"] =       149.9387700000;
  m[   "LA151"] =       150.9417200000;
  m[   "LA152"] =       151.9462500000;
  m[   "LA153"] =       152.9496200000;
  m[   "LA154"] =       153.9545000000;
  m[   "LA155"] =       154.9583500000;
  m[      "CE"] =       139.9054387000;
  m[   "CE119"] =       118.9527600000;
  m[   "CE120"] =       119.9466400000;
  m[   "CE121"] =       120.9434200000;
  m[   "CE122"] =       121.9379100000;
  m[   "CE123"] =       122.9354000000;
  m[   "CE124"] =       123.9304100000;
  m[   "CE125"] =       124.9284400000;
  m[   "CE126"] =       125.9239710000;
  m[   "CE127"] =       126.9227310000;
  m[   "CE128"] =       127.9189110000;
  m[   "CE129"] =       128.9181020000;
  m[   "CE130"] =       129.9147360000;
  m[   "CE131"] =       130.9144220000;
  m[   "CE132"] =       131.9114605000;
  m[   "CE133"] =       132.9115150000;
  m[   "CE134"] =       133.9089248000;
  m[   "CE135"] =       134.9091514000;
  m[   "CE136"] =       135.9071724000;
  m[   "CE137"] =       136.9078056000;
  m[   "CE138"] =       137.9059913000;
  m[   "CE139"] =       138.9066527000;
  m[   "CE140"] =       139.9054387000;
  m[   "CE141"] =       140.9082763000;
  m[   "CE142"] =       141.9092442000;
  m[   "CE143"] =       142.9123859000;
  m[   "CE144"] =       143.9136473000;
  m[   "CE145"] =       144.9172331000;
  m[   "CE146"] =       145.9187590000;
  m[   "CE147"] =       146.9226740000;
  m[   "CE148"] =       147.9244324000;
  m[   "CE149"] =       148.9283999000;
  m[   "CE150"] =       149.9304089000;
  m[   "CE151"] =       150.9339762000;
  m[   "CE152"] =       151.9365400000;
  m[   "CE153"] =       152.9405800000;
  m[   "CE154"] =       153.9434200000;
  m[   "CE155"] =       154.9480400000;
  m[   "CE156"] =       155.9512600000;
  m[   "CE157"] =       156.9563400000;
  m[      "PR"] =       140.9076528000;
  m[   "PR121"] =       120.9553640000;
  m[   "PR122"] =       121.9518100000;
  m[   "PR123"] =       122.9459600000;
  m[   "PR124"] =       123.9429600000;
  m[   "PR125"] =       124.9378300000;
  m[   "PR126"] =       125.9353100000;
  m[   "PR127"] =       126.9308300000;
  m[   "PR128"] =       127.9287910000;
  m[   "PR129"] =       128.9250950000;
  m[   "PR130"] =       129.9235900000;
  m[   "PR131"] =       130.9202590000;
  m[   "PR132"] =       131.9192550000;
  m[   "PR133"] =       132.9163305000;
  m[   "PR134"] =       133.9157117000;
  m[   "PR135"] =       134.9131117000;
  m[   "PR136"] =       135.9126916000;
  m[   "PR137"] =       136.9107055000;
  m[   "PR138"] =       137.9107546000;
  m[   "PR139"] =       138.9089384000;
  m[   "PR140"] =       139.9090759000;
  m[   "PR141"] =       140.9076528000;
  m[   "PR142"] =       141.9100448000;
  m[   "PR143"] =       142.9108169000;
  m[   "PR144"] =       143.9133052000;
  m[   "PR145"] =       144.9145117000;
  m[   "PR146"] =       145.9176443000;
  m[   "PR147"] =       146.9189960000;
  m[   "PR148"] =       147.9221350000;
  m[   "PR149"] =       148.9237177000;
  m[   "PR150"] =       149.9266730000;
  m[   "PR151"] =       150.9283186000;
  m[   "PR152"] =       151.9314992000;
  m[   "PR153"] =       152.9338389000;
  m[   "PR154"] =       153.9375182000;
  m[   "PR155"] =       154.9401200000;
  m[   "PR156"] =       155.9442700000;
  m[   "PR157"] =       156.9474300000;
  m[   "PR158"] =       157.9519800000;
  m[   "PR159"] =       158.9555000000;
  m[      "ND"] =       141.9077233000;
  m[   "ND124"] =       123.9522300000;
  m[   "ND125"] =       124.9488800000;
  m[   "ND126"] =       125.9432200000;
  m[   "ND127"] =       126.9405000000;
  m[   "ND128"] =       127.9353900000;
  m[   "ND129"] =       128.9331880000;
  m[   "ND130"] =       129.9285060000;
  m[   "ND131"] =       130.9272470000;
  m[   "ND132"] =       131.9233212000;
  m[   "ND133"] =       132.9223480000;
  m[   "ND134"] =       133.9187902000;
  m[   "ND135"] =       134.9181812000;
  m[   "ND136"] =       135.9149760000;
  m[   "ND137"] =       136.9145671000;
  m[   "ND138"] =       137.9119500000;
  m[   "ND139"] =       138.9119783000;
  m[   "ND140"] =       139.9095520000;
  m[   "ND141"] =       140.9096099000;
  m[   "ND142"] =       141.9077233000;
  m[   "ND143"] =       142.9098143000;
  m[   "ND144"] =       143.9100873000;
  m[   "ND145"] =       144.9125736000;
  m[   "ND146"] =       145.9131169000;
  m[   "ND147"] =       146.9161004000;
  m[   "ND148"] =       147.9168933000;
  m[   "ND149"] =       148.9201488000;
  m[   "ND150"] =       149.9208909000;
  m[   "ND151"] =       150.9238289000;
  m[   "ND152"] =       151.9246822000;
  m[   "ND153"] =       152.9276982000;
  m[   "ND154"] =       153.9294773000;
  m[   "ND155"] =       154.9329320000;
  m[   "ND156"] =       155.9350181000;
  m[   "ND157"] =       156.9390300000;
  m[   "ND158"] =       157.9416000000;
  m[   "ND159"] =       158.9460900000;
  m[   "ND160"] =       159.9490900000;
  m[   "ND161"] =       160.9538800000;
  m[      "PM"] =       144.9127490000;
  m[   "PM126"] =       125.9575200000;
  m[   "PM127"] =       126.9516300000;
  m[   "PM128"] =       127.9484200000;
  m[   "PM129"] =       128.9431600000;
  m[   "PM130"] =       129.9404500000;
  m[   "PM131"] =       130.9358700000;
  m[   "PM132"] =       131.9337500000;
  m[   "PM133"] =       132.9297820000;
  m[   "PM134"] =       133.9283530000;
  m[   "PM135"] =       134.9248760000;
  m[   "PM136"] =       135.9235658000;
  m[   "PM137"] =       136.9204795000;
  m[   "PM138"] =       137.9195483000;
  m[   "PM139"] =       138.9168041000;
  m[   "PM140"] =       139.9160418000;
  m[   "PM141"] =       140.9135551000;
  m[   "PM142"] =       141.9128745000;
  m[   "PM143"] =       142.9109326000;
  m[   "PM144"] =       143.9125908000;
  m[   "PM145"] =       144.9127490000;
  m[   "PM146"] =       145.9146963000;
  m[   "PM147"] =       146.9151385000;
  m[   "PM148"] =       147.9174746000;
  m[   "PM149"] =       148.9183342000;
  m[   "PM150"] =       149.9209836000;
  m[   "PM151"] =       150.9212070000;
  m[   "PM152"] =       151.9234968000;
  m[   "PM153"] =       152.9241169000;
  m[   "PM154"] =       153.9264639000;
  m[   "PM155"] =       154.9281013000;
  m[   "PM156"] =       155.9310567000;
  m[   "PM157"] =       156.9330394000;
  m[   "PM158"] =       157.9365614000;
  m[   "PM159"] =       158.9389700000;
  m[   "PM160"] =       159.9429900000;
  m[   "PM161"] =       160.9458600000;
  m[   "PM162"] =       161.9502900000;
  m[   "PM163"] =       162.9536800000;
  m[      "SM"] =       151.9197324000;
  m[   "SM128"] =       127.9580800000;
  m[   "SM129"] =       128.9546400000;
  m[   "SM130"] =       129.9489200000;
  m[   "SM131"] =       130.9461100000;
  m[   "SM132"] =       131.9406900000;
  m[   "SM133"] =       132.9386700000;
  m[   "SM134"] =       133.9339700000;
  m[   "SM135"] =       134.9325200000;
  m[   "SM136"] =       135.9282755000;
  m[   "SM137"] =       136.9269717000;
  m[   "SM138"] =       137.9232440000;
  m[   "SM139"] =       138.9222966000;
  m[   "SM140"] =       139.9189947000;
  m[   "SM141"] =       140.9184765000;
  m[   "SM142"] =       141.9151976000;
  m[   "SM143"] =       142.9146283000;
  m[   "SM144"] =       143.9119995000;
  m[   "SM145"] =       144.9134104000;
  m[   "SM146"] =       145.9130409000;
  m[   "SM147"] =       146.9148979000;
  m[   "SM148"] =       147.9148227000;
  m[   "SM149"] =       148.9171847000;
  m[   "SM150"] =       149.9172755000;
  m[   "SM151"] =       150.9199324000;
  m[   "SM152"] =       151.9197324000;
  m[   "SM153"] =       152.9220974000;
  m[   "SM154"] =       153.9222093000;
  m[   "SM155"] =       154.9246402000;
  m[   "SM156"] =       155.9255279000;
  m[   "SM157"] =       156.9283587000;
  m[   "SM158"] =       157.9299913000;
  m[   "SM159"] =       158.9332113000;
  m[   "SM160"] =       159.9351400000;
  m[   "SM161"] =       160.9388300000;
  m[   "SM162"] =       161.9412200000;
  m[   "SM163"] =       162.9453600000;
  m[   "SM164"] =       163.9482800000;
  m[   "SM165"] =       164.9529800000;
  m[      "EU"] =       152.9212303000;
  m[   "EU130"] =       129.9635690000;
  m[   "EU131"] =       130.9577530000;
  m[   "EU132"] =       131.9543700000;
  m[   "EU133"] =       132.9492400000;
  m[   "EU134"] =       133.9465100000;
  m[   "EU135"] =       134.9418200000;
  m[   "EU136"] =       135.9396000000;
  m[   "EU137"] =       136.9355700000;
  m[   "EU138"] =       137.9337090000;
  m[   "EU139"] =       138.9297923000;
  m[   "EU140"] =       139.9280876000;
  m[   "EU141"] =       140.9249307000;
  m[   "EU142"] =       141.9234349000;
  m[   "EU143"] =       142.9202975000;
  m[   "EU144"] =       143.9188168000;
  m[   "EU145"] =       144.9162652000;
  m[   "EU146"] =       145.9172058000;
  m[   "EU147"] =       146.9167461000;
  m[   "EU148"] =       147.9180859000;
  m[   "EU149"] =       148.9179312000;
  m[   "EU150"] =       149.9197018000;
  m[   "EU151"] =       150.9198502000;
  m[   "EU152"] =       151.9217445000;
  m[   "EU153"] =       152.9212303000;
  m[   "EU154"] =       153.9229792000;
  m[   "EU155"] =       154.9228933000;
  m[   "EU156"] =       155.9247522000;
  m[   "EU157"] =       156.9254236000;
  m[   "EU158"] =       157.9278453000;
  m[   "EU159"] =       158.9290889000;
  m[   "EU160"] =       159.9319710000;
  m[   "EU161"] =       160.9336800000;
  m[   "EU162"] =       161.9370400000;
  m[   "EU163"] =       162.9392100000;
  m[   "EU164"] =       163.9429900000;
  m[   "EU165"] =       164.9457200000;
  m[   "EU166"] =       165.9499700000;
  m[   "EU167"] =       166.9532100000;
  m[      "GD"] =       157.9241039000;
  m[   "GD134"] =       133.9553700000;
  m[   "GD135"] =       134.9525700000;
  m[   "GD136"] =       135.9473400000;
  m[   "GD137"] =       136.9450200000;
  m[   "GD138"] =       137.9401200000;
  m[   "GD139"] =       138.9382400000;
  m[   "GD140"] =       139.9336740000;
  m[   "GD141"] =       140.9321260000;
  m[   "GD142"] =       141.9281160000;
  m[   "GD143"] =       142.9267495000;
  m[   "GD144"] =       143.9229630000;
  m[   "GD145"] =       144.9217093000;
  m[   "GD146"] =       145.9183106000;
  m[   "GD147"] =       146.9190944000;
  m[   "GD148"] =       147.9181145000;
  m[   "GD149"] =       148.9193409000;
  m[   "GD150"] =       149.9186589000;
  m[   "GD151"] =       150.9203485000;
  m[   "GD152"] =       151.9197910000;
  m[   "GD153"] =       152.9217495000;
  m[   "GD154"] =       153.9208656000;
  m[   "GD155"] =       154.9226220000;
  m[   "GD156"] =       155.9221227000;
  m[   "GD157"] =       156.9239601000;
  m[   "GD158"] =       157.9241039000;
  m[   "GD159"] =       158.9263887000;
  m[   "GD160"] =       159.9270541000;
  m[   "GD161"] =       160.9296692000;
  m[   "GD162"] =       161.9309848000;
  m[   "GD163"] =       162.9339900000;
  m[   "GD164"] =       163.9358600000;
  m[   "GD165"] =       164.9393800000;
  m[   "GD166"] =       165.9416000000;
  m[   "GD167"] =       166.9455700000;
  m[   "GD168"] =       167.9483600000;
  m[   "GD169"] =       168.9528700000;
  m[      "TB"] =       158.9253468000;
  m[   "TB136"] =       135.9613800000;
  m[   "TB137"] =       136.9559800000;
  m[   "TB138"] =       137.9531600000;
  m[   "TB139"] =       138.9482900000;
  m[   "TB140"] =       139.9458050000;
  m[   "TB141"] =       140.9414480000;
  m[   "TB142"] =       141.9387440000;
  m[   "TB143"] =       142.9351210000;
  m[   "TB144"] =       143.9330450000;
  m[   "TB145"] =       144.9292740000;
  m[   "TB146"] =       145.9272466000;
  m[   "TB147"] =       146.9240446000;
  m[   "TB148"] =       147.9242717000;
  m[   "TB149"] =       148.9232459000;
  m[   "TB150"] =       149.9236597000;
  m[   "TB151"] =       150.9231025000;
  m[   "TB152"] =       151.9240744000;
  m[   "TB153"] =       152.9234346000;
  m[   "TB154"] =       153.9246780000;
  m[   "TB155"] =       154.9235052000;
  m[   "TB156"] =       155.9247472000;
  m[   "TB157"] =       156.9240246000;
  m[   "TB158"] =       157.9254131000;
  m[   "TB159"] =       158.9253468000;
  m[   "TB160"] =       159.9271676000;
  m[   "TB161"] =       160.9275699000;
  m[   "TB162"] =       161.9294882000;
  m[   "TB163"] =       162.9306475000;
  m[   "TB164"] =       163.9333508000;
  m[   "TB165"] =       164.9348800000;
  m[   "TB166"] =       165.9379920000;
  m[   "TB167"] =       166.9400500000;
  m[   "TB168"] =       167.9436400000;
  m[   "TB169"] =       168.9462200000;
  m[   "TB170"] =       169.9502500000;
  m[   "TB171"] =       170.9533000000;
  m[      "DY"] =       163.9291748000;
  m[   "DY138"] =       137.9624900000;
  m[   "DY139"] =       138.9595400000;
  m[   "DY140"] =       139.9540100000;
  m[   "DY141"] =       140.9513500000;
  m[   "DY142"] =       141.9463660000;
  m[   "DY143"] =       142.9438300000;
  m[   "DY144"] =       143.9392540000;
  m[   "DY145"] =       144.9374250000;
  m[   "DY146"] =       145.9328454000;
  m[   "DY147"] =       146.9310915000;
  m[   "DY148"] =       147.9271498000;
  m[   "DY149"] =       148.9273048000;
  m[   "DY150"] =       149.9255852000;
  m[   "DY151"] =       150.9261846000;
  m[   "DY152"] =       151.9247183000;
  m[   "DY153"] =       152.9257647000;
  m[   "DY154"] =       153.9244245000;
  m[   "DY155"] =       154.9257538000;
  m[   "DY156"] =       155.9242831000;
  m[   "DY157"] =       156.9254661000;
  m[   "DY158"] =       157.9244095000;
  m[   "DY159"] =       158.9257392000;
  m[   "DY160"] =       159.9251975000;
  m[   "DY161"] =       160.9269334000;
  m[   "DY162"] =       161.9267984000;
  m[   "DY163"] =       162.9287312000;
  m[   "DY164"] =       163.9291748000;
  m[   "DY165"] =       164.9317033000;
  m[   "DY166"] =       165.9328067000;
  m[   "DY167"] =       166.9356555000;
  m[   "DY168"] =       167.9371288000;
  m[   "DY169"] =       168.9403076000;
  m[   "DY170"] =       169.9423900000;
  m[   "DY171"] =       170.9462000000;
  m[   "DY172"] =       171.9487600000;
  m[   "DY173"] =       172.9530000000;
  m[      "HO"] =       164.9303221000;
  m[   "HO140"] =       139.9685390000;
  m[   "HO141"] =       140.9630980000;
  m[   "HO142"] =       141.9597700000;
  m[   "HO143"] =       142.9546100000;
  m[   "HO144"] =       143.9514800000;
  m[   "HO145"] =       144.9472000000;
  m[   "HO146"] =       145.9446400000;
  m[   "HO147"] =       146.9400560000;
  m[   "HO148"] =       147.9377180000;
  m[   "HO149"] =       148.9337748000;
  m[   "HO150"] =       149.9334962000;
  m[   "HO151"] =       150.9316881000;
  m[   "HO152"] =       151.9317137000;
  m[   "HO153"] =       152.9301988000;
  m[   "HO154"] =       153.9306016000;
  m[   "HO155"] =       154.9291035000;
  m[   "HO156"] =       155.9298390000;
  m[   "HO157"] =       156.9282562000;
  m[   "HO158"] =       157.9289410000;
  m[   "HO159"] =       158.9277120000;
  m[   "HO160"] =       159.9287295000;
  m[   "HO161"] =       160.9278548000;
  m[   "HO162"] =       161.9290955000;
  m[   "HO163"] =       162.9287339000;
  m[   "HO164"] =       163.9302335000;
  m[   "HO165"] =       164.9303221000;
  m[   "HO166"] =       165.9322842000;
  m[   "HO167"] =       166.9331326000;
  m[   "HO168"] =       167.9355157000;
  m[   "HO169"] =       168.9368723000;
  m[   "HO170"] =       169.9396189000;
  m[   "HO171"] =       170.9414652000;
  m[   "HO172"] =       171.9448200000;
  m[   "HO173"] =       172.9472900000;
  m[   "HO174"] =       173.9511500000;
  m[   "HO175"] =       174.9540500000;
  m[      "ER"] =       165.9302931000;
  m[   "ER143"] =       142.9663400000;
  m[   "ER144"] =       143.9603800000;
  m[   "ER145"] =       144.9573900000;
  m[   "ER146"] =       145.9520000000;
  m[   "ER147"] =       146.9494900000;
  m[   "ER148"] =       147.9445500000;
  m[   "ER149"] =       148.9423060000;
  m[   "ER150"] =       149.9379138000;
  m[   "ER151"] =       150.9374489000;
  m[   "ER152"] =       151.9350504000;
  m[   "ER153"] =       152.9350635000;
  m[   "ER154"] =       153.9327831000;
  m[   "ER155"] =       154.9332089000;
  m[   "ER156"] =       155.9310647000;
  m[   "ER157"] =       156.9319160000;
  m[   "ER158"] =       157.9298935000;
  m[   "ER159"] =       158.9306841000;
  m[   "ER160"] =       159.9290833000;
  m[   "ER161"] =       160.9299953000;
  m[   "ER162"] =       161.9287783000;
  m[   "ER163"] =       162.9300327000;
  m[   "ER164"] =       163.9292002000;
  m[   "ER165"] =       164.9307260000;
  m[   "ER166"] =       165.9302931000;
  m[   "ER167"] =       166.9320482000;
  m[   "ER168"] =       167.9323702000;
  m[   "ER169"] =       168.9345904000;
  m[   "ER170"] =       169.9354643000;
  m[   "ER171"] =       170.9380298000;
  m[   "ER172"] =       171.9393561000;
  m[   "ER173"] =       172.9424000000;
  m[   "ER174"] =       173.9442300000;
  m[   "ER175"] =       174.9477700000;
  m[   "ER176"] =       175.9500800000;
  m[   "ER177"] =       176.9540500000;
  m[      "TM"] =       168.9342133000;
  m[   "TM145"] =       144.9700730000;
  m[   "TM146"] =       145.9664250000;
  m[   "TM147"] =       146.9609610000;
  m[   "TM148"] =       147.9578400000;
  m[   "TM149"] =       148.9527200000;
  m[   "TM150"] =       149.9499600000;
  m[   "TM151"] =       150.9454835000;
  m[   "TM152"] =       151.9444220000;
  m[   "TM153"] =       152.9420121000;
  m[   "TM154"] =       153.9415678000;
  m[   "TM155"] =       154.9391995000;
  m[   "TM156"] =       155.9389799000;
  m[   "TM157"] =       156.9369730000;
  m[   "TM158"] =       157.9369795000;
  m[   "TM159"] =       158.9349750000;
  m[   "TM160"] =       159.9352628000;
  m[   "TM161"] =       160.9335490000;
  m[   "TM162"] =       161.9339947000;
  m[   "TM163"] =       162.9326511000;
  m[   "TM164"] =       163.9335600000;
  m[   "TM165"] =       164.9324355000;
  m[   "TM166"] =       165.9335541000;
  m[   "TM167"] =       166.9328516000;
  m[   "TM168"] =       167.9341728000;
  m[   "TM169"] =       168.9342133000;
  m[   "TM170"] =       169.9358014000;
  m[   "TM171"] =       170.9364294000;
  m[   "TM172"] =       171.9384000000;
  m[   "TM173"] =       172.9396036000;
  m[   "TM174"] =       173.9421686000;
  m[   "TM175"] =       174.9438369000;
  m[   "TM176"] =       175.9469947000;
  m[   "TM177"] =       176.9490400000;
  m[   "TM178"] =       177.9526400000;
  m[   "TM179"] =       178.9553400000;
  m[      "YB"] =       173.9388621000;
  m[   "YB148"] =       147.9674200000;
  m[   "YB149"] =       148.9640400000;
  m[   "YB150"] =       149.9584200000;
  m[   "YB151"] =       150.9554008000;
  m[   "YB152"] =       151.9502889000;
  m[   "YB153"] =       152.9494800000;
  m[   "YB154"] =       153.9463939000;
  m[   "YB155"] =       154.9457823000;
  m[   "YB156"] =       155.9428182000;
  m[   "YB157"] =       156.9426278000;
  m[   "YB158"] =       157.9398656000;
  m[   "YB159"] =       158.9400501000;
  m[   "YB160"] =       159.9375523000;
  m[   "YB161"] =       160.9379017000;
  m[   "YB162"] =       161.9357682000;
  m[   "YB163"] =       162.9363343000;
  m[   "YB164"] =       163.9344894000;
  m[   "YB165"] =       164.9352790000;
  m[   "YB166"] =       165.9338820000;
  m[   "YB167"] =       166.9349496000;
  m[   "YB168"] =       167.9338969000;
  m[   "YB169"] =       168.9351898000;
  m[   "YB170"] =       169.9347618000;
  m[   "YB171"] =       170.9363258000;
  m[   "YB172"] =       171.9363815000;
  m[   "YB173"] =       172.9382108000;
  m[   "YB174"] =       173.9388621000;
  m[   "YB175"] =       174.9412765000;
  m[   "YB176"] =       175.9425717000;
  m[   "YB177"] =       176.9452608000;
  m[   "YB178"] =       177.9466467000;
  m[   "YB179"] =       178.9501700000;
  m[   "YB180"] =       179.9523300000;
  m[   "YB181"] =       180.9561500000;
  m[      "LU"] =       174.9407718000;
  m[   "LU150"] =       149.9732280000;
  m[   "LU151"] =       150.9675770000;
  m[   "LU152"] =       151.9641200000;
  m[   "LU153"] =       152.9587673000;
  m[   "LU154"] =       153.9575220000;
  m[   "LU155"] =       154.9543162000;
  m[   "LU156"] =       155.9530325000;
  m[   "LU157"] =       156.9500983000;
  m[   "LU158"] =       157.9493133000;
  m[   "LU159"] =       158.9466288000;
  m[   "LU160"] =       159.9460330000;
  m[   "LU161"] =       160.9435720000;
  m[   "LU162"] =       161.9432773000;
  m[   "LU163"] =       162.9411790000;
  m[   "LU164"] =       163.9413390000;
  m[   "LU165"] =       164.9394067000;
  m[   "LU166"] =       165.9398590000;
  m[   "LU167"] =       166.9382700000;
  m[   "LU168"] =       167.9387391000;
  m[   "LU169"] =       168.9376514000;
  m[   "LU170"] =       169.9384750000;
  m[   "LU171"] =       170.9379131000;
  m[   "LU172"] =       171.9390857000;
  m[   "LU173"] =       172.9389306000;
  m[   "LU174"] =       173.9403375000;
  m[   "LU175"] =       174.9407718000;
  m[   "LU176"] =       175.9426863000;
  m[   "LU177"] =       176.9437581000;
  m[   "LU178"] =       177.9459546000;
  m[   "LU179"] =       178.9473274000;
  m[   "LU180"] =       179.9498812000;
  m[   "LU181"] =       180.9519700000;
  m[   "LU182"] =       181.9550400000;
  m[   "LU183"] =       182.9575700000;
  m[   "LU184"] =       183.9609100000;
  m[      "HF"] =       179.9465500000;
  m[   "HF153"] =       152.9706900000;
  m[   "HF154"] =       153.9648600000;
  m[   "HF155"] =       154.9633900000;
  m[   "HF156"] =       155.9593640000;
  m[   "HF157"] =       156.9583960000;
  m[   "HF158"] =       157.9547994000;
  m[   "HF159"] =       158.9539949000;
  m[   "HF160"] =       159.9506844000;
  m[   "HF161"] =       160.9502748000;
  m[   "HF162"] =       161.9472105000;
  m[   "HF163"] =       162.9470890000;
  m[   "HF164"] =       163.9443673000;
  m[   "HF165"] =       164.9445670000;
  m[   "HF166"] =       165.9421800000;
  m[   "HF167"] =       166.9426000000;
  m[   "HF168"] =       167.9405680000;
  m[   "HF169"] =       168.9412590000;
  m[   "HF170"] =       169.9396090000;
  m[   "HF171"] =       170.9404920000;
  m[   "HF172"] =       171.9394483000;
  m[   "HF173"] =       172.9405130000;
  m[   "HF174"] =       173.9400462000;
  m[   "HF175"] =       174.9415092000;
  m[   "HF176"] =       175.9414086000;
  m[   "HF177"] =       176.9432207000;
  m[   "HF178"] =       177.9436988000;
  m[   "HF179"] =       178.9458161000;
  m[   "HF180"] =       179.9465500000;
  m[   "HF181"] =       180.9491012000;
  m[   "HF182"] =       181.9505541000;
  m[   "HF183"] =       182.9535304000;
  m[   "HF184"] =       183.9554465000;
  m[   "HF185"] =       184.9588200000;
  m[   "HF186"] =       185.9608900000;
  m[   "HF187"] =       186.9645900000;
  m[   "HF188"] =       187.9668500000;
  m[      "TA"] =       180.9479958000;
  m[   "TA155"] =       154.9745920000;
  m[   "TA156"] =       155.9723030000;
  m[   "TA157"] =       156.9681924000;
  m[   "TA158"] =       157.9666990000;
  m[   "TA159"] =       158.9630182000;
  m[   "TA160"] =       159.9614861000;
  m[   "TA161"] =       160.9584170000;
  m[   "TA162"] =       161.9572919000;
  m[   "TA163"] =       162.9543303000;
  m[   "TA164"] =       163.9535340000;
  m[   "TA165"] =       164.9507725000;
  m[   "TA166"] =       165.9505120000;
  m[   "TA167"] =       166.9480930000;
  m[   "TA168"] =       167.9480470000;
  m[   "TA169"] =       168.9460110000;
  m[   "TA170"] =       169.9461750000;
  m[   "TA171"] =       170.9444760000;
  m[   "TA172"] =       171.9448950000;
  m[   "TA173"] =       172.9437500000;
  m[   "TA174"] =       173.9444540000;
  m[   "TA175"] =       174.9437370000;
  m[   "TA176"] =       175.9448570000;
  m[   "TA177"] =       176.9444724000;
  m[   "TA178"] =       177.9457782000;
  m[   "TA179"] =       178.9459295000;
  m[   "TA180"] =       179.9474648000;
  m[   "TA181"] =       180.9479958000;
  m[   "TA182"] =       181.9501518000;
  m[   "TA183"] =       182.9513726000;
  m[   "TA184"] =       183.9540080000;
  m[   "TA185"] =       184.9555594000;
  m[   "TA186"] =       185.9585520000;
  m[   "TA187"] =       186.9605300000;
  m[   "TA188"] =       187.9637000000;
  m[   "TA189"] =       188.9658300000;
  m[   "TA190"] =       189.9692300000;
  m[       "W"] =       183.9509312000;
  m[    "W158"] =       157.9745620000;
  m[    "W159"] =       158.9729180000;
  m[    "W160"] =       159.9684788000;
  m[    "W161"] =       160.9673570000;
  m[    "W162"] =       161.9634974000;
  m[    "W163"] =       162.9625235000;
  m[    "W164"] =       163.9589544000;
  m[    "W165"] =       164.9582799000;
  m[    "W166"] =       165.9550273000;
  m[    "W167"] =       166.9548160000;
  m[    "W168"] =       167.9518084000;
  m[    "W169"] =       168.9517788000;
  m[    "W170"] =       169.9492285000;
  m[    "W171"] =       170.9494510000;
  m[    "W172"] =       171.9472920000;
  m[    "W173"] =       172.9476890000;
  m[    "W174"] =       173.9460790000;
  m[    "W175"] =       174.9467170000;
  m[    "W176"] =       175.9456340000;
  m[    "W177"] =       176.9466430000;
  m[    "W178"] =       177.9458762000;
  m[    "W179"] =       178.9470704000;
  m[    "W180"] =       179.9467045000;
  m[    "W181"] =       180.9481972000;
  m[    "W182"] =       181.9482042000;
  m[    "W183"] =       182.9502230000;
  m[    "W184"] =       183.9509312000;
  m[    "W185"] =       184.9534193000;
  m[    "W186"] =       185.9543641000;
  m[    "W187"] =       186.9571605000;
  m[    "W188"] =       187.9584891000;
  m[    "W189"] =       188.9619129000;
  m[    "W190"] =       189.9631814000;
  m[    "W191"] =       190.9666000000;
  m[    "W192"] =       191.9681700000;
  m[      "RE"] =       186.9557531000;
  m[   "RE160"] =       159.9821150000;
  m[   "RE161"] =       160.9775891000;
  m[   "RE162"] =       161.9760020000;
  m[   "RE163"] =       162.9720805000;
  m[   "RE164"] =       163.9703230000;
  m[   "RE165"] =       164.9670886000;
  m[   "RE166"] =       165.9658080000;
  m[   "RE167"] =       166.9626010000;
  m[   "RE168"] =       167.9615726000;
  m[   "RE169"] =       168.9587911000;
  m[   "RE170"] =       169.9582201000;
  m[   "RE171"] =       170.9557160000;
  m[   "RE172"] =       171.9554230000;
  m[   "RE173"] =       172.9532430000;
  m[   "RE174"] =       173.9531150000;
  m[   "RE175"] =       174.9513810000;
  m[   "RE176"] =       175.9516230000;
  m[   "RE177"] =       176.9503280000;
  m[   "RE178"] =       177.9509890000;
  m[   "RE179"] =       178.9499876000;
  m[   "RE180"] =       179.9507891000;
  m[   "RE181"] =       180.9500679000;
  m[   "RE182"] =       181.9512101000;
  m[   "RE183"] =       182.9508198000;
  m[   "RE184"] =       183.9525208000;
  m[   "RE185"] =       184.9529550000;
  m[   "RE186"] =       185.9549861000;
  m[   "RE187"] =       186.9557531000;
  m[   "RE188"] =       187.9581144000;
  m[   "RE189"] =       188.9592290000;
  m[   "RE190"] =       189.9618180000;
  m[   "RE191"] =       190.9631252000;
  m[   "RE192"] =       191.9659600000;
  m[   "RE193"] =       192.9674700000;
  m[   "RE194"] =       193.9704200000;
  m[      "OS"] =       191.9614807000;
  m[   "OS162"] =       161.9844310000;
  m[   "OS163"] =       162.9826900000;
  m[   "OS164"] =       163.9780356000;
  m[   "OS165"] =       164.9767620000;
  m[   "OS166"] =       165.9726908000;
  m[   "OS167"] =       166.9715480000;
  m[   "OS168"] =       167.9678037000;
  m[   "OS169"] =       168.9670193000;
  m[   "OS170"] =       169.9635770000;
  m[   "OS171"] =       170.9631848000;
  m[   "OS172"] =       171.9600233000;
  m[   "OS173"] =       172.9598084000;
  m[   "OS174"] =       173.9570622000;
  m[   "OS175"] =       174.9569458000;
  m[   "OS176"] =       175.9548060000;
  m[   "OS177"] =       176.9549653000;
  m[   "OS178"] =       177.9532512000;
  m[   "OS179"] =       178.9538160000;
  m[   "OS180"] =       179.9523788000;
  m[   "OS181"] =       180.9532440000;
  m[   "OS182"] =       181.9521102000;
  m[   "OS183"] =       182.9531261000;
  m[   "OS184"] =       183.9524891000;
  m[   "OS185"] =       184.9540423000;
  m[   "OS186"] =       185.9538382000;
  m[   "OS187"] =       186.9557505000;
  m[   "OS188"] =       187.9558382000;
  m[   "OS189"] =       188.9581475000;
  m[   "OS190"] =       189.9584470000;
  m[   "OS191"] =       190.9609297000;
  m[   "OS192"] =       191.9614807000;
  m[   "OS193"] =       192.9641516000;
  m[   "OS194"] =       193.9651821000;
  m[   "OS195"] =       194.9681267000;
  m[   "OS196"] =       195.9696393000;
  m[      "IR"] =       192.9629264000;
  m[   "IR164"] =       163.9922010000;
  m[   "IR165"] =       164.9875200000;
  m[   "IR166"] =       165.9858240000;
  m[   "IR167"] =       166.9816652000;
  m[   "IR168"] =       167.9798810000;
  m[   "IR169"] =       168.9762949000;
  m[   "IR170"] =       169.9749650000;
  m[   "IR171"] =       170.9716260000;
  m[   "IR172"] =       171.9704560000;
  m[   "IR173"] =       172.9675017000;
  m[   "IR174"] =       173.9668610000;
  m[   "IR175"] =       174.9641129000;
  m[   "IR176"] =       175.9636487000;
  m[   "IR177"] =       176.9613015000;
  m[   "IR178"] =       177.9610820000;
  m[   "IR179"] =       178.9591223000;
  m[   "IR180"] =       179.9592294000;
  m[   "IR181"] =       180.9576253000;
  m[   "IR182"] =       181.9580763000;
  m[   "IR183"] =       182.9568465000;
  m[   "IR184"] =       183.9574760000;
  m[   "IR185"] =       184.9566980000;
  m[   "IR186"] =       185.9579461000;
  m[   "IR187"] =       186.9573634000;
  m[   "IR188"] =       187.9588531000;
  m[   "IR189"] =       188.9587189000;
  m[   "IR190"] =       189.9605460000;
  m[   "IR191"] =       190.9605940000;
  m[   "IR192"] =       191.9626050000;
  m[   "IR193"] =       192.9629264000;
  m[   "IR194"] =       193.9650784000;
  m[   "IR195"] =       194.9659796000;
  m[   "IR196"] =       195.9683965000;
  m[   "IR197"] =       196.9696533000;
  m[   "IR198"] =       197.9722800000;
  m[   "IR199"] =       198.9738046000;
  m[      "PT"] =       194.9647911000;
  m[   "PT166"] =       165.9948550000;
  m[   "PT167"] =       166.9929790000;
  m[   "PT168"] =       167.9881507000;
  m[   "PT169"] =       168.9867150000;
  m[   "PT170"] =       169.9824953000;
  m[   "PT171"] =       170.9812445000;
  m[   "PT172"] =       171.9773471000;
  m[   "PT173"] =       172.9764448000;
  m[   "PT174"] =       173.9728188000;
  m[   "PT175"] =       174.9724206000;
  m[   "PT176"] =       175.9689446000;
  m[   "PT177"] =       176.9684695000;
  m[   "PT178"] =       177.9656487000;
  m[   "PT179"] =       178.9653634000;
  m[   "PT180"] =       179.9630315000;
  m[   "PT181"] =       180.9630973000;
  m[   "PT182"] =       181.9611707000;
  m[   "PT183"] =       182.9615967000;
  m[   "PT184"] =       183.9599223000;
  m[   "PT185"] =       184.9606190000;
  m[   "PT186"] =       185.9593508000;
  m[   "PT187"] =       186.9605870000;
  m[   "PT188"] =       187.9593954000;
  m[   "PT189"] =       188.9608337000;
  m[   "PT190"] =       189.9599317000;
  m[   "PT191"] =       190.9616767000;
  m[   "PT192"] =       191.9610380000;
  m[   "PT193"] =       192.9629874000;
  m[   "PT194"] =       193.9626803000;
  m[   "PT195"] =       194.9647911000;
  m[   "PT196"] =       195.9649515000;
  m[   "PT197"] =       196.9673402000;
  m[   "PT198"] =       197.9678928000;
  m[   "PT199"] =       198.9705931000;
  m[   "PT200"] =       199.9714407000;
  m[   "PT201"] =       200.9745129000;
  m[   "PT202"] =       201.9757400000;
  m[      "AU"] =       196.9665687000;
  m[   "AU169"] =       168.9980800000;
  m[   "AU170"] =       169.9961220000;
  m[   "AU171"] =       170.9918789000;
  m[   "AU172"] =       171.9900350000;
  m[   "AU173"] =       172.9862374000;
  m[   "AU174"] =       173.9847610000;
  m[   "AU175"] =       174.9812741000;
  m[   "AU176"] =       175.9800990000;
  m[   "AU177"] =       176.9768649000;
  m[   "AU178"] =       177.9760319000;
  m[   "AU179"] =       178.9732128000;
  m[   "AU180"] =       179.9725211000;
  m[   "AU181"] =       180.9700790000;
  m[   "AU182"] =       181.9696179000;
  m[   "AU183"] =       182.9675930000;
  m[   "AU184"] =       183.9674515000;
  m[   "AU185"] =       184.9657894000;
  m[   "AU186"] =       185.9659527000;
  m[   "AU187"] =       186.9645675000;
  m[   "AU188"] =       187.9653237000;
  m[   "AU189"] =       188.9639483000;
  m[   "AU190"] =       189.9647003000;
  m[   "AU191"] =       190.9637042000;
  m[   "AU192"] =       191.9648130000;
  m[   "AU193"] =       192.9641497000;
  m[   "AU194"] =       193.9653653000;
  m[   "AU195"] =       194.9650346000;
  m[   "AU196"] =       195.9665698000;
  m[   "AU197"] =       196.9665687000;
  m[   "AU198"] =       197.9682423000;
  m[   "AU199"] =       198.9687652000;
  m[   "AU200"] =       199.9707256000;
  m[   "AU201"] =       200.9716572000;
  m[   "AU202"] =       201.9738058000;
  m[   "AU203"] =       202.9751545000;
  m[   "AU204"] =       203.9777240000;
  m[   "AU205"] =       204.9798700000;
  m[      "HG"] =       201.9706430000;
  m[   "HG171"] =       171.0037600000;
  m[   "HG172"] =       171.9988327000;
  m[   "HG173"] =       172.9972420000;
  m[   "HG174"] =       173.9928637000;
  m[   "HG175"] =       174.9914233000;
  m[   "HG176"] =       175.9873546000;
  m[   "HG177"] =       176.9862792000;
  m[   "HG178"] =       177.9824831000;
  m[   "HG179"] =       178.9818339000;
  m[   "HG180"] =       179.9782664000;
  m[   "HG181"] =       180.9778193000;
  m[   "HG182"] =       181.9746900000;
  m[   "HG183"] =       182.9744498000;
  m[   "HG184"] =       183.9717131000;
  m[   "HG185"] =       184.9718991000;
  m[   "HG186"] =       185.9693618000;
  m[   "HG187"] =       186.9698142000;
  m[   "HG188"] =       187.9675770000;
  m[   "HG189"] =       188.9681900000;
  m[   "HG190"] =       189.9663224000;
  m[   "HG191"] =       190.9671571000;
  m[   "HG192"] =       191.9656343000;
  m[   "HG193"] =       192.9666654000;
  m[   "HG194"] =       193.9654394000;
  m[   "HG195"] =       194.9667201000;
  m[   "HG196"] =       195.9658326000;
  m[   "HG197"] =       196.9672129000;
  m[   "HG198"] =       197.9667690000;
  m[   "HG199"] =       198.9682799000;
  m[   "HG200"] =       199.9683260000;
  m[   "HG201"] =       200.9703023000;
  m[   "HG202"] =       201.9706430000;
  m[   "HG203"] =       202.9728725000;
  m[   "HG204"] =       203.9734939000;
  m[   "HG205"] =       204.9760734000;
  m[   "HG206"] =       205.9775141000;
  m[   "HG207"] =       206.9825885000;
  m[   "HG208"] =       207.9859400000;
  m[   "HG209"] =       208.9910400000;
  m[   "HG210"] =       209.9945100000;
  m[      "TL"] =       204.9744275000;
  m[   "TL176"] =       176.0005900000;
  m[   "TL177"] =       176.9964273000;
  m[   "TL178"] =       177.9948970000;
  m[   "TL179"] =       178.9910891000;
  m[   "TL180"] =       179.9899060000;
  m[   "TL181"] =       180.9862574000;
  m[   "TL182"] =       181.9856671000;
  m[   "TL183"] =       182.9821928000;
  m[   "TL184"] =       183.9818731000;
  m[   "TL185"] =       184.9787913000;
  m[   "TL186"] =       185.9783250000;
  m[   "TL187"] =       186.9759059000;
  m[   "TL188"] =       187.9760098000;
  m[   "TL189"] =       188.9735884000;
  m[   "TL190"] =       189.9738771000;
  m[   "TL191"] =       190.9717862000;
  m[   "TL192"] =       191.9722250000;
  m[   "TL193"] =       192.9706720000;
  m[   "TL194"] =       193.9712000000;
  m[   "TL195"] =       194.9697743000;
  m[   "TL196"] =       195.9704812000;
  m[   "TL197"] =       196.9695745000;
  m[   "TL198"] =       197.9704835000;
  m[   "TL199"] =       198.9698770000;
  m[   "TL200"] =       199.9709627000;
  m[   "TL201"] =       200.9708189000;
  m[   "TL202"] =       201.9721058000;
  m[   "TL203"] =       202.9723442000;
  m[   "TL204"] =       203.9738635000;
  m[   "TL205"] =       204.9744275000;
  m[   "TL206"] =       205.9761103000;
  m[   "TL207"] =       206.9774194000;
  m[   "TL208"] =       207.9820187000;
  m[   "TL209"] =       208.9853590000;
  m[   "TL210"] =       209.9900737000;
  m[   "TL211"] =       210.9934770000;
  m[   "TL212"] =       211.9982280000;
  m[      "PB"] =       207.9766521000;
  m[   "PB178"] =       178.0038302000;
  m[   "PB179"] =       179.0021500000;
  m[   "PB180"] =       179.9979182000;
  m[   "PB181"] =       180.9966240000;
  m[   "PB182"] =       181.9926718000;
  m[   "PB183"] =       182.9918746000;
  m[   "PB184"] =       183.9881423000;
  m[   "PB185"] =       184.9876099000;
  m[   "PB186"] =       185.9842389000;
  m[   "PB187"] =       186.9839184000;
  m[   "PB188"] =       187.9808743000;
  m[   "PB189"] =       188.9808070000;
  m[   "PB190"] =       189.9780815000;
  m[   "PB191"] =       190.9782650000;
  m[   "PB192"] =       191.9757852000;
  m[   "PB193"] =       192.9761732000;
  m[   "PB194"] =       193.9740121000;
  m[   "PB195"] =       194.9745421000;
  m[   "PB196"] =       195.9727741000;
  m[   "PB197"] =       196.9734311000;
  m[   "PB198"] =       197.9720340000;
  m[   "PB199"] =       198.9729167000;
  m[   "PB200"] =       199.9718267000;
  m[   "PB201"] =       200.9728845000;
  m[   "PB202"] =       201.9721591000;
  m[   "PB203"] =       202.9733905000;
  m[   "PB204"] =       203.9730436000;
  m[   "PB205"] =       204.9744818000;
  m[   "PB206"] =       205.9744653000;
  m[   "PB207"] =       206.9758969000;
  m[   "PB208"] =       207.9766521000;
  m[   "PB209"] =       208.9810901000;
  m[   "PB210"] =       209.9841885000;
  m[   "PB211"] =       210.9887370000;
  m[   "PB212"] =       211.9918975000;
  m[   "PB213"] =       212.9965815000;
  m[   "PB214"] =       213.9998054000;
  m[   "PB215"] =       215.0048070000;
  m[      "BI"] =       208.9803987000;
  m[   "BI184"] =       184.0011240000;
  m[   "BI185"] =       184.9976250000;
  m[   "BI186"] =       185.9965976000;
  m[   "BI187"] =       186.9931578000;
  m[   "BI188"] =       187.9922652000;
  m[   "BI189"] =       188.9891990000;
  m[   "BI190"] =       189.9882951000;
  m[   "BI191"] =       190.9857861000;
  m[   "BI192"] =       191.9854580000;
  m[   "BI193"] =       192.9829598000;
  m[   "BI194"] =       193.9828340000;
  m[   "BI195"] =       194.9806507000;
  m[   "BI196"] =       195.9806665000;
  m[   "BI197"] =       196.9788645000;
  m[   "BI198"] =       197.9792060000;
  m[   "BI199"] =       198.9776720000;
  m[   "BI200"] =       199.9781318000;
  m[   "BI201"] =       200.9770090000;
  m[   "BI202"] =       201.9777423000;
  m[   "BI203"] =       202.9768760000;
  m[   "BI204"] =       203.9778127000;
  m[   "BI205"] =       204.9773894000;
  m[   "BI206"] =       205.9784991000;
  m[   "BI207"] =       206.9784707000;
  m[   "BI208"] =       207.9797422000;
  m[   "BI209"] =       208.9803987000;
  m[   "BI210"] =       209.9841204000;
  m[   "BI211"] =       210.9872695000;
  m[   "BI212"] =       211.9912857000;
  m[   "BI213"] =       212.9943847000;
  m[   "BI214"] =       213.9987115000;
  m[   "BI215"] =       215.0017698000;
  m[   "BI216"] =       216.0063059000;
  m[   "BI217"] =       217.0094700000;
  m[   "BI218"] =       218.0143160000;
  m[      "PO"] =       208.9824304000;
  m[   "PO188"] =       187.9994220000;
  m[   "PO189"] =       188.9984806000;
  m[   "PO190"] =       189.9951012000;
  m[   "PO191"] =       190.9945745000;
  m[   "PO192"] =       191.9913351000;
  m[   "PO193"] =       192.9910253000;
  m[   "PO194"] =       193.9881856000;
  m[   "PO195"] =       194.9881107000;
  m[   "PO196"] =       195.9855346000;
  m[   "PO197"] =       196.9856596000;
  m[   "PO198"] =       197.9833886000;
  m[   "PO199"] =       198.9836661000;
  m[   "PO200"] =       199.9817986000;
  m[   "PO201"] =       200.9822598000;
  m[   "PO202"] =       201.9807575000;
  m[   "PO203"] =       202.9814201000;
  m[   "PO204"] =       203.9803181000;
  m[   "PO205"] =       204.9812033000;
  m[   "PO206"] =       205.9804811000;
  m[   "PO207"] =       206.9815932000;
  m[   "PO208"] =       207.9812457000;
  m[   "PO209"] =       208.9824304000;
  m[   "PO210"] =       209.9828737000;
  m[   "PO211"] =       210.9866532000;
  m[   "PO212"] =       211.9888680000;
  m[   "PO213"] =       212.9928573000;
  m[   "PO214"] =       213.9952014000;
  m[   "PO215"] =       214.9994200000;
  m[   "PO216"] =       216.0019150000;
  m[   "PO217"] =       217.0063348000;
  m[   "PO218"] =       218.0089730000;
  m[   "PO219"] =       219.0137440000;
  m[   "PO220"] =       220.0166020000;
  m[      "AT"] =       210.9874963000;
  m[   "AT193"] =       192.9998431000;
  m[   "AT194"] =       193.9987251000;
  m[   "AT195"] =       194.9962681000;
  m[   "AT196"] =       195.9957881000;
  m[   "AT197"] =       196.9931892000;
  m[   "AT198"] =       197.9928372000;
  m[   "AT199"] =       198.9905323000;
  m[   "AT200"] =       199.9903513000;
  m[   "AT201"] =       200.9884170000;
  m[   "AT202"] =       201.9886302000;
  m[   "AT203"] =       202.9869420000;
  m[   "AT204"] =       203.9872513000;
  m[   "AT205"] =       204.9860745000;
  m[   "AT206"] =       205.9866670000;
  m[   "AT207"] =       206.9857835000;
  m[   "AT208"] =       207.9865900000;
  m[   "AT209"] =       208.9861731000;
  m[   "AT210"] =       209.9871477000;
  m[   "AT211"] =       210.9874963000;
  m[   "AT212"] =       211.9907448000;
  m[   "AT213"] =       212.9929366000;
  m[   "AT214"] =       213.9963717000;
  m[   "AT215"] =       214.9986526000;
  m[   "AT216"] =       216.0024233000;
  m[   "AT217"] =       217.0047188000;
  m[   "AT218"] =       218.0086943000;
  m[   "AT219"] =       219.0111617000;
  m[   "AT220"] =       220.0154077000;
  m[   "AT221"] =       221.0180500000;
  m[   "AT222"] =       222.0223300000;
  m[   "AT223"] =       223.0251900000;
  m[      "RN"] =       222.0175777000;
  m[   "RN195"] =       195.0054377000;
  m[   "RN196"] =       196.0021152000;
  m[   "RN197"] =       197.0015844000;
  m[   "RN198"] =       197.9986787000;
  m[   "RN199"] =       198.9983703000;
  m[   "RN200"] =       199.9956993000;
  m[   "RN201"] =       200.9956283000;
  m[   "RN202"] =       201.9932635000;
  m[   "RN203"] =       202.9933867000;
  m[   "RN204"] =       203.9914287000;
  m[   "RN205"] =       204.9917188000;
  m[   "RN206"] =       205.9902141000;
  m[   "RN207"] =       206.9907342000;
  m[   "RN208"] =       207.9896425000;
  m[   "RN209"] =       208.9904147000;
  m[   "RN210"] =       209.9896962000;
  m[   "RN211"] =       210.9906005000;
  m[   "RN212"] =       211.9907035000;
  m[   "RN213"] =       212.9938827000;
  m[   "RN214"] =       213.9953626000;
  m[   "RN215"] =       214.9987455000;
  m[   "RN216"] =       216.0002744000;
  m[   "RN217"] =       217.0039277000;
  m[   "RN218"] =       218.0056013000;
  m[   "RN219"] =       219.0094802000;
  m[   "RN220"] =       220.0113940000;
  m[   "RN221"] =       221.0155368000;
  m[   "RN222"] =       222.0175777000;
  m[   "RN223"] =       223.0217900000;
  m[   "RN224"] =       224.0240900000;
  m[   "RN225"] =       225.0284400000;
  m[   "RN226"] =       226.0308900000;
  m[   "RN227"] =       227.0354070000;
  m[   "RN228"] =       228.0379860000;
  m[      "FR"] =       222.0175517000;
  m[   "FR199"] =       199.0072581000;
  m[   "FR200"] =       200.0065725000;
  m[   "FR201"] =       201.0038609000;
  m[   "FR202"] =       202.0033728000;
  m[   "FR203"] =       203.0009246000;
  m[   "FR204"] =       204.0006532000;
  m[   "FR205"] =       204.9985940000;
  m[   "FR206"] =       205.9986661000;
  m[   "FR207"] =       206.9969494000;
  m[   "FR208"] =       207.9971388000;
  m[   "FR209"] =       208.9959536000;
  m[   "FR210"] =       209.9964077000;
  m[   "FR211"] =       210.9955365000;
  m[   "FR212"] =       211.9962022000;
  m[   "FR213"] =       212.9961891000;
  m[   "FR214"] =       213.9989711000;
  m[   "FR215"] =       215.0003415000;
  m[   "FR216"] =       216.0031980000;
  m[   "FR217"] =       217.0046320000;
  m[   "FR218"] =       218.0075783000;
  m[   "FR219"] =       219.0092521000;
  m[   "FR220"] =       220.0123274000;
  m[   "FR221"] =       221.0142548000;
  m[   "FR222"] =       222.0175517000;
  m[   "FR223"] =       223.0197359000;
  m[   "FR224"] =       224.0232500000;
  m[   "FR225"] =       225.0255654000;
  m[   "FR226"] =       226.0293862000;
  m[   "FR227"] =       227.0318359000;
  m[   "FR228"] =       228.0357290000;
  m[   "FR229"] =       229.0384502000;
  m[   "FR230"] =       230.0425100000;
  m[   "FR231"] =       231.0454400000;
  m[   "FR232"] =       232.0497720000;
  m[      "RA"] =       228.0310703000;
  m[   "RA202"] =       202.0098907000;
  m[   "RA203"] =       203.0092716000;
  m[   "RA204"] =       204.0064997000;
  m[   "RA205"] =       205.0062686000;
  m[   "RA206"] =       206.0038273000;
  m[   "RA207"] =       207.0037981000;
  m[   "RA208"] =       208.0018399000;
  m[   "RA209"] =       209.0019914000;
  m[   "RA210"] =       210.0004950000;
  m[   "RA211"] =       211.0008980000;
  m[   "RA212"] =       211.9997945000;
  m[   "RA213"] =       213.0003840000;
  m[   "RA214"] =       214.0001079000;
  m[   "RA215"] =       215.0027198000;
  m[   "RA216"] =       216.0035330000;
  m[   "RA217"] =       217.0063203000;
  m[   "RA218"] =       218.0071402000;
  m[   "RA219"] =       219.0100851000;
  m[   "RA220"] =       220.0110284000;
  m[   "RA221"] =       221.0139173000;
  m[   "RA222"] =       222.0153745000;
  m[   "RA223"] =       223.0185022000;
  m[   "RA224"] =       224.0202118000;
  m[   "RA225"] =       225.0236116000;
  m[   "RA226"] =       226.0254098000;
  m[   "RA227"] =       227.0291778000;
  m[   "RA228"] =       228.0310703000;
  m[   "RA229"] =       229.0349576000;
  m[   "RA230"] =       230.0370564000;
  m[   "RA231"] =       231.0412200000;
  m[   "RA232"] =       232.0436380000;
  m[   "RA233"] =       233.0480600000;
  m[   "RA234"] =       234.0507040000;
  m[      "AC"] =       227.0277521000;
  m[   "AC206"] =       206.0145050000;
  m[   "AC207"] =       207.0119497000;
  m[   "AC208"] =       208.0115516000;
  m[   "AC209"] =       209.0094949000;
  m[   "AC210"] =       210.0094360000;
  m[   "AC211"] =       211.0077348000;
  m[   "AC212"] =       212.0078138000;
  m[   "AC213"] =       213.0066076000;
  m[   "AC214"] =       214.0069018000;
  m[   "AC215"] =       215.0064536000;
  m[   "AC216"] =       216.0087201000;
  m[   "AC217"] =       217.0093469000;
  m[   "AC218"] =       218.0116415000;
  m[   "AC219"] =       219.0124204000;
  m[   "AC220"] =       220.0147630000;
  m[   "AC221"] =       221.0155912000;
  m[   "AC222"] =       222.0178439000;
  m[   "AC223"] =       223.0191375000;
  m[   "AC224"] =       224.0217229000;
  m[   "AC225"] =       225.0232296000;
  m[   "AC226"] =       226.0260981000;
  m[   "AC227"] =       227.0277521000;
  m[   "AC228"] =       228.0310211000;
  m[   "AC229"] =       229.0330152000;
  m[   "AC230"] =       230.0362942000;
  m[   "AC231"] =       231.0385588000;
  m[   "AC232"] =       232.0420274000;
  m[   "AC233"] =       233.0445500000;
  m[   "AC234"] =       234.0484200000;
  m[   "AC235"] =       235.0512320000;
  m[   "AC236"] =       236.0552960000;
  m[      "TH"] =       232.0380553000;
  m[   "TH209"] =       209.0177157000;
  m[   "TH210"] =       210.0150753000;
  m[   "TH211"] =       211.0149284000;
  m[   "TH212"] =       212.0129803000;
  m[   "TH213"] =       213.0130101000;
  m[   "TH214"] =       214.0114998000;
  m[   "TH215"] =       215.0117303000;
  m[   "TH216"] =       216.0110621000;
  m[   "TH217"] =       217.0131143000;
  m[   "TH218"] =       218.0132845000;
  m[   "TH219"] =       219.0155369000;
  m[   "TH220"] =       220.0157478000;
  m[   "TH221"] =       221.0181837000;
  m[   "TH222"] =       222.0184681000;
  m[   "TH223"] =       223.0208114000;
  m[   "TH224"] =       224.0214669000;
  m[   "TH225"] =       225.0239510000;
  m[   "TH226"] =       226.0249031000;
  m[   "TH227"] =       227.0277041000;
  m[   "TH228"] =       228.0287411000;
  m[   "TH229"] =       229.0317624000;
  m[   "TH230"] =       230.0331338000;
  m[   "TH231"] =       231.0363043000;
  m[   "TH232"] =       232.0380553000;
  m[   "TH233"] =       233.0415818000;
  m[   "TH234"] =       234.0436012000;
  m[   "TH235"] =       235.0475101000;
  m[   "TH236"] =       236.0498700000;
  m[   "TH237"] =       237.0538940000;
  m[   "TH238"] =       238.0564960000;
  m[      "PA"] =       231.0358840000;
  m[   "PA212"] =       212.0232041000;
  m[   "PA213"] =       213.0211093000;
  m[   "PA214"] =       214.0209184000;
  m[   "PA215"] =       215.0191859000;
  m[   "PA216"] =       216.0191096000;
  m[   "PA217"] =       217.0183240000;
  m[   "PA218"] =       218.0200419000;
  m[   "PA219"] =       219.0198831000;
  m[   "PA220"] =       220.0218753000;
  m[   "PA221"] =       221.0218780000;
  m[   "PA222"] =       222.0237420000;
  m[   "PA223"] =       223.0239623000;
  m[   "PA224"] =       224.0256257000;
  m[   "PA225"] =       225.0261307000;
  m[   "PA226"] =       226.0279478000;
  m[   "PA227"] =       227.0288051000;
  m[   "PA228"] =       228.0310514000;
  m[   "PA229"] =       229.0320968000;
  m[   "PA230"] =       230.0345408000;
  m[   "PA231"] =       231.0358840000;
  m[   "PA232"] =       232.0385916000;
  m[   "PA233"] =       233.0402473000;
  m[   "PA234"] =       234.0433081000;
  m[   "PA235"] =       235.0454436000;
  m[   "PA236"] =       236.0486813000;
  m[   "PA237"] =       237.0511457000;
  m[   "PA238"] =       238.0545027000;
  m[   "PA239"] =       239.0572600000;
  m[   "PA240"] =       240.0609800000;
  m[       "U"] =       238.0507882000;
  m[    "U217"] =       217.0243688000;
  m[    "U218"] =       218.0235357000;
  m[    "U219"] =       219.0249192000;
  m[    "U220"] =       220.0247230000;
  m[    "U221"] =       221.0263990000;
  m[    "U222"] =       222.0260860000;
  m[    "U223"] =       223.0277386000;
  m[    "U224"] =       224.0276048000;
  m[    "U225"] =       225.0293907000;
  m[    "U226"] =       226.0293387000;
  m[    "U227"] =       227.0311564000;
  m[    "U228"] =       228.0313740000;
  m[    "U229"] =       229.0335059000;
  m[    "U230"] =       230.0339398000;
  m[    "U231"] =       231.0362937000;
  m[    "U232"] =       232.0371562000;
  m[    "U233"] =       233.0396352000;
  m[    "U234"] =       234.0409521000;
  m[    "U235"] =       235.0439299000;
  m[    "U236"] =       236.0455680000;
  m[    "U237"] =       237.0487302000;
  m[    "U238"] =       238.0507882000;
  m[    "U239"] =       239.0542933000;
  m[    "U240"] =       240.0565920000;
  m[    "U241"] =       241.0603300000;
  m[    "U242"] =       242.0629310000;
  m[      "NP"] =       237.0481734000;
  m[   "NP225"] =       225.0339139000;
  m[   "NP226"] =       226.0351450000;
  m[   "NP227"] =       227.0349568000;
  m[   "NP228"] =       228.0361800000;
  m[   "NP229"] =       229.0362638000;
  m[   "NP230"] =       230.0378276000;
  m[   "NP231"] =       231.0382451000;
  m[   "NP232"] =       232.0401080000;
  m[   "NP233"] =       233.0407405000;
  m[   "NP234"] =       234.0428950000;
  m[   "NP235"] =       235.0440633000;
  m[   "NP236"] =       236.0465696000;
  m[   "NP237"] =       237.0481734000;
  m[   "NP238"] =       238.0509464000;
  m[   "NP239"] =       239.0529390000;
  m[   "NP240"] =       240.0561622000;
  m[   "NP241"] =       241.0582524000;
  m[   "NP242"] =       242.0616412000;
  m[   "NP243"] =       243.0642790000;
  m[   "NP244"] =       244.0678500000;
  m[      "PU"] =       242.0587426000;
  m[   "PU228"] =       228.0387423000;
  m[   "PU229"] =       229.0401502000;
  m[   "PU230"] =       230.0396499000;
  m[   "PU231"] =       231.0411011000;
  m[   "PU232"] =       232.0411871000;
  m[   "PU233"] =       233.0429974000;
  m[   "PU234"] =       234.0433171000;
  m[   "PU235"] =       235.0452861000;
  m[   "PU236"] =       236.0460580000;
  m[   "PU237"] =       237.0484097000;
  m[   "PU238"] =       238.0495599000;
  m[   "PU239"] =       239.0521634000;
  m[   "PU240"] =       240.0538135000;
  m[   "PU241"] =       241.0568515000;
  m[   "PU242"] =       242.0587426000;
  m[   "PU243"] =       243.0620031000;
  m[   "PU244"] =       244.0642039000;
  m[   "PU245"] =       245.0677472000;
  m[   "PU246"] =       246.0702046000;
  m[   "PU247"] =       247.0740700000;
  m[      "AM"] =       243.0613811000;
  m[   "AM231"] =       231.0455600000;
  m[   "AM232"] =       232.0465900000;
  m[   "AM233"] =       233.0463480000;
  m[   "AM234"] =       234.0478090000;
  m[   "AM235"] =       235.0479460000;
  m[   "AM236"] =       236.0495790000;
  m[   "AM237"] =       237.0499960000;
  m[   "AM238"] =       238.0519843000;
  m[   "AM239"] =       239.0530245000;
  m[   "AM240"] =       240.0553002000;
  m[   "AM241"] =       241.0568291000;
  m[   "AM242"] =       242.0595492000;
  m[   "AM243"] =       243.0613811000;
  m[   "AM244"] =       244.0642848000;
  m[   "AM245"] =       245.0664521000;
  m[   "AM246"] =       246.0697746000;
  m[   "AM247"] =       247.0720930000;
  m[   "AM248"] =       248.0757520000;
  m[   "AM249"] =       249.0784800000;
  m[      "CM"] =       247.0703535000;
  m[   "CM233"] =       233.0507712000;
  m[   "CM234"] =       234.0501598000;
  m[   "CM235"] =       235.0514340000;
  m[   "CM236"] =       236.0514130000;
  m[   "CM237"] =       237.0529010000;
  m[   "CM238"] =       238.0530287000;
  m[   "CM239"] =       239.0549570000;
  m[   "CM240"] =       240.0555295000;
  m[   "CM241"] =       241.0576530000;
  m[   "CM242"] =       242.0588358000;
  m[   "CM243"] =       243.0613891000;
  m[   "CM244"] =       244.0627526000;
  m[   "CM245"] =       245.0654912000;
  m[   "CM246"] =       246.0672237000;
  m[   "CM247"] =       247.0703535000;
  m[   "CM248"] =       248.0723485000;
  m[   "CM249"] =       249.0759534000;
  m[   "CM250"] =       250.0783570000;
  m[   "CM251"] =       251.0822846000;
  m[   "CM252"] =       252.0848700000;
  m[      "BK"] =       247.0703071000;
  m[   "BK235"] =       235.0565800000;
  m[   "BK236"] =       236.0573300000;
  m[   "BK237"] =       237.0570030000;
  m[   "BK238"] =       238.0582810000;
  m[   "BK239"] =       239.0582790000;
  m[   "BK240"] =       240.0597590000;
  m[   "BK241"] =       241.0602300000;
  m[   "BK242"] =       242.0619810000;
  m[   "BK243"] =       243.0630076000;
  m[   "BK244"] =       244.0651808000;
  m[   "BK245"] =       245.0663616000;
  m[   "BK246"] =       246.0686729000;
  m[   "BK247"] =       247.0703071000;
  m[   "BK248"] =       248.0730860000;
  m[   "BK249"] =       249.0749867000;
  m[   "BK250"] =       250.0783165000;
  m[   "BK251"] =       251.0807602000;
  m[   "BK252"] =       252.0843100000;
  m[   "BK253"] =       253.0868800000;
  m[   "BK254"] =       254.0906000000;
  m[      "CF"] =       251.0795868000;
  m[   "CF237"] =       237.0620700000;
  m[   "CF238"] =       238.0614100000;
  m[   "CF239"] =       239.0624220000;
  m[   "CF240"] =       240.0623020000;
  m[   "CF241"] =       241.0637260000;
  m[   "CF242"] =       242.0637016000;
  m[   "CF243"] =       243.0654270000;
  m[   "CF244"] =       244.0660007000;
  m[   "CF245"] =       245.0680486000;
  m[   "CF246"] =       246.0688053000;
  m[   "CF247"] =       247.0710006000;
  m[   "CF248"] =       248.0721849000;
  m[   "CF249"] =       249.0748535000;
  m[   "CF250"] =       250.0764061000;
  m[   "CF251"] =       251.0795868000;
  m[   "CF252"] =       252.0816258000;
  m[   "CF253"] =       253.0851331000;
  m[   "CF254"] =       254.0873229000;
  m[   "CF255"] =       255.0910460000;
  m[   "CF256"] =       256.0934400000;
  m[      "ES"] =       252.0829785000;
  m[   "ES240"] =       240.0689200000;
  m[   "ES241"] =       241.0685380000;
  m[   "ES242"] =       242.0697450000;
  m[   "ES243"] =       243.0695480000;
  m[   "ES244"] =       244.0708830000;
  m[   "ES245"] =       245.0713240000;
  m[   "ES246"] =       246.0728960000;
  m[   "ES247"] =       247.0736560000;
  m[   "ES248"] =       248.0754710000;
  m[   "ES249"] =       249.0764110000;
  m[   "ES250"] =       250.0786120000;
  m[   "ES251"] =       251.0799921000;
  m[   "ES252"] =       252.0829785000;
  m[   "ES253"] =       253.0848247000;
  m[   "ES254"] =       254.0880220000;
  m[   "ES255"] =       255.0902731000;
  m[   "ES256"] =       256.0935980000;
  m[   "ES257"] =       257.0959790000;
  m[   "ES258"] =       258.0995200000;
  m[      "FM"] =       257.0951047000;
  m[   "FM242"] =       242.0734300000;
  m[   "FM243"] =       243.0743530000;
  m[   "FM244"] =       244.0740840000;
  m[   "FM245"] =       245.0753850000;
  m[   "FM246"] =       246.0752990000;
  m[   "FM247"] =       247.0768470000;
  m[   "FM248"] =       248.0771947000;
  m[   "FM249"] =       249.0790340000;
  m[   "FM250"] =       250.0795213000;
  m[   "FM251"] =       251.0815750000;
  m[   "FM252"] =       252.0824669000;
  m[   "FM253"] =       253.0851852000;
  m[   "FM254"] =       254.0868542000;
  m[   "FM255"] =       255.0899622000;
  m[   "FM256"] =       256.0917731000;
  m[   "FM257"] =       257.0951047000;
  m[   "FM258"] =       258.0970760000;
  m[   "FM259"] =       259.1005950000;
  m[   "FM260"] =       260.1026780000;
  m[      "MD"] =       258.0984313000;
  m[   "MD245"] =       245.0808290000;
  m[   "MD246"] =       246.0818860000;
  m[   "MD247"] =       247.0816350000;
  m[   "MD248"] =       248.0828230000;
  m[   "MD249"] =       249.0830130000;
  m[   "MD250"] =       250.0844200000;
  m[   "MD251"] =       251.0848390000;
  m[   "MD252"] =       252.0865600000;
  m[   "MD253"] =       253.0872800000;
  m[   "MD254"] =       254.0896560000;
  m[   "MD255"] =       255.0910827000;
  m[   "MD256"] =       256.0940590000;
  m[   "MD257"] =       257.0955414000;
  m[   "MD258"] =       258.0984313000;
  m[   "MD259"] =       259.1005090000;
  m[   "MD260"] =       260.1036520000;
  m[   "MD261"] =       261.1057210000;
  m[   "MD262"] =       262.1088650000;
  m[      "NO"] =       255.0932411000;
  m[   "NO248"] =       248.0865960000;
  m[   "NO249"] =       249.0878330000;
  m[   "NO250"] =       250.0875100000;
  m[   "NO251"] =       251.0890120000;
  m[   "NO252"] =       252.0889765000;
  m[   "NO253"] =       253.0906780000;
  m[   "NO254"] =       254.0909553000;
  m[   "NO255"] =       255.0932411000;
  m[   "NO256"] =       256.0942827000;
  m[   "NO257"] =       257.0968772000;
  m[   "NO258"] =       258.0982070000;
  m[   "NO259"] =       259.1010310000;
  m[   "NO260"] =       260.1026430000;
  m[   "NO261"] =       261.1057490000;
  m[   "NO262"] =       262.1073010000;
  m[   "NO263"] =       263.1105520000;
  m[   "NO264"] =       264.1123450000;
  m[      "LR"] =       260.1055040000;
  m[   "LR251"] =       251.0943600000;
  m[   "LR252"] =       252.0953710000;
  m[   "LR253"] =       253.0952100000;
  m[   "LR254"] =       254.0964540000;
  m[   "LR255"] =       255.0966810000;
  m[   "LR256"] =       256.0986290000;
  m[   "LR257"] =       257.0995550000;
  m[   "LR258"] =       258.1018140000;
  m[   "LR259"] =       259.1029010000;
  m[   "LR260"] =       260.1055040000;
  m[   "LR261"] =       261.1068830000;
  m[   "LR262"] =       262.1096340000;
  m[   "LR263"] =       263.1112930000;
  m[   "LR264"] =       264.1140380000;
  m[   "LR265"] =       265.1158390000;
  m[   "LR266"] =       266.1193050000;
  m[      "RF"] =       263.1125470000;
  m[   "RF253"] =       253.1006890000;
  m[   "RF254"] =       254.1001840000;
  m[   "RF255"] =       255.1013400000;
  m[   "RF256"] =       256.1011662000;
  m[   "RF257"] =       257.1029900000;
  m[   "RF258"] =       258.1034890000;
  m[   "RF259"] =       259.1056370000;
  m[   "RF260"] =       260.1064400000;
  m[   "RF261"] =       261.1087666000;
  m[   "RF262"] =       262.1099250000;
  m[   "RF263"] =       263.1125470000;
  m[   "RF264"] =       264.1139850000;
  m[   "RF265"] =       265.1167040000;
  m[   "RF266"] =       266.1179560000;
  m[   "RF267"] =       267.1215290000;
  m[   "RF268"] =       268.1236440000;
  m[      "DB"] =       255.1073980000;
  m[   "DB255"] =       255.1073980000;
  m[   "DB256"] =       256.1081270000;
  m[   "DB257"] =       257.1077220000;
  m[   "DB258"] =       258.1092310000;
  m[   "DB259"] =       259.1096100000;
  m[   "DB260"] =       260.1113000000;
  m[   "DB261"] =       261.1120560000;
  m[   "DB262"] =       262.1140840000;
  m[   "DB263"] =       263.1149880000;
  m[   "DB264"] =       264.1174040000;
  m[   "DB265"] =       265.1186010000;
  m[   "DB266"] =       266.1210290000;
  m[   "DB267"] =       267.1223770000;
  m[   "DB268"] =       268.1254450000;
  m[   "DB269"] =       269.1274600000;
  m[   "DB270"] =       270.1307120000;
  m[      "SG"] =       259.1145000000;
  m[   "SG258"] =       258.1131680000;
  m[   "SG259"] =       259.1145000000;
  m[   "SG260"] =       260.1144221000;
  m[   "SG261"] =       261.1161170000;
  m[   "SG262"] =       262.1163980000;
  m[   "SG263"] =       263.1183220000;
  m[   "SG264"] =       264.1189310000;
  m[   "SG265"] =       265.1211147000;
  m[   "SG266"] =       266.1220650000;
  m[   "SG267"] =       267.1244250000;
  m[   "SG268"] =       268.1256060000;
  m[   "SG269"] =       269.1287550000;
  m[   "SG270"] =       270.1303290000;
  m[   "SG271"] =       271.1334720000;
  m[   "SG272"] =       272.1351580000;
  m[   "SG273"] =       273.1382200000;
  m[      "BH"] =       262.1228920000;
  m[   "BH260"] =       260.1219700000;
  m[   "BH261"] =       261.1216640000;
  m[   "BH262"] =       262.1228920000;
  m[   "BH263"] =       263.1230350000;
  m[   "BH264"] =       264.1246040000;
  m[   "BH265"] =       265.1251470000;
  m[   "BH266"] =       266.1269420000;
  m[   "BH267"] =       267.1276500000;
  m[   "BH268"] =       268.1297550000;
  m[   "BH269"] =       269.1306940000;
  m[   "BH270"] =       270.1336160000;
  m[   "BH271"] =       271.1351790000;
  m[   "BH272"] =       272.1380320000;
  m[   "BH273"] =       273.1396180000;
  m[   "BH274"] =       274.1424400000;
  m[   "BH275"] =       275.1442500000;
  m[      "HS"] =       263.1285580000;
  m[   "HS263"] =       263.1285580000;
  m[   "HS264"] =       264.1283949000;
  m[   "HS265"] =       265.1300850000;
  m[   "HS266"] =       266.1300970000;
  m[   "HS267"] =       267.1317890000;
  m[   "HS268"] =       268.1321620000;
  m[   "HS269"] =       269.1340560000;
  m[   "HS270"] =       270.1346500000;
  m[   "HS271"] =       271.1376570000;
  m[   "HS272"] =       272.1390520000;
  m[   "HS273"] =       273.1419860000;
  m[   "HS274"] =       274.1431310000;
  m[   "HS275"] =       275.1459520000;
  m[   "HS276"] =       276.1472080000;
  m[   "HS277"] =       277.1498410000;
  m[      "MT"] =       265.1361510000;
  m[   "MT265"] =       265.1361510000;
  m[   "MT266"] =       266.1372990000;
  m[   "MT267"] =       267.1373070000;
  m[   "MT268"] =       268.1387280000;
  m[   "MT269"] =       269.1390550000;
  m[   "MT270"] =       270.1406570000;
  m[   "MT271"] =       271.1411390000;
  m[   "MT272"] =       272.1437380000;
  m[   "MT273"] =       273.1449130000;
  m[   "MT274"] =       274.1474920000;
  m[   "MT275"] =       275.1486470000;
  m[   "MT276"] =       276.1511560000;
  m[   "MT277"] =       277.1524200000;
  m[   "MT278"] =       278.1548120000;
  m[   "MT279"] =       279.1561930000;
  m[      "DS"] =       281.1620610000;
  m[   "DS267"] =       267.1443410000;
  m[   "DS268"] =       268.1437950000;
  m[   "DS269"] =       269.1451240000;
  m[   "DS270"] =       270.1447200000;
  m[   "DS271"] =       271.1460620000;
  m[   "DS272"] =       272.1463170000;
  m[   "DS273"] =       273.1488630000;
  m[   "DS274"] =       274.1494920000;
  m[   "DS275"] =       275.1521760000;
  m[   "DS276"] =       276.1530340000;
  m[   "DS277"] =       277.1556470000;
  m[   "DS278"] =       278.1564690000;
  m[   "DS279"] =       279.1588610000;
  m[   "DS280"] =       280.1597950000;
  m[   "DS281"] =       281.1620610000;
  m[      "RG"] =       272.1536150000;
  m[   "RG272"] =       272.1536150000;
  m[   "RG273"] =       273.1536820000;
  m[   "RG274"] =       274.1557130000;
  m[   "RG275"] =       275.1561420000;
  m[   "RG276"] =       276.1584930000;
  m[   "RG277"] =       277.1595190000;
  m[   "RG278"] =       278.1616040000;
  m[   "RG279"] =       279.1624680000;
  m[   "RG280"] =       280.1644730000;
  m[   "RG281"] =       281.1653720000;
  m[   "RG282"] =       282.1674860000;
  m[   "RG283"] =       283.1684150000;
  m[     "UUB"] =       283.1717920000;
  m[  "UUB277"] =       277.1639430000;
  m[  "UUB278"] =       278.1643120000;
  m[  "UUB279"] =       279.1665460000;
  m[  "UUB280"] =       280.1670390000;
  m[  "UUB281"] =       281.1692860000;
  m[  "UUB282"] =       282.1697650000;
  m[  "UUB283"] =       283.1717920000;
  m[  "UUB284"] =       284.1723840000;
  m[  "UUB285"] =       285.1741050000;
  m[     "UUT"] =       283.1764510000;
  m[  "UUT283"] =       283.1764510000;
  m[  "UUT284"] =       284.1780800000;
  m[  "UUT285"] =       285.1787320000;
  m[  "UUT286"] =       286.1804810000;
  m[  "UUT287"] =       287.1810450000;
  m[     "UUQ"] =       285.1836980000;
  m[  "UUQ285"] =       285.1836980000;
  m[  "UUQ286"] =       286.1838550000;
  m[  "UUQ287"] =       287.1855990000;
  m[  "UUQ288"] =       288.1856890000;
  m[  "UUQ289"] =       289.1872790000;
  m[     "UUP"] =       287.1911860000;
  m[  "UUP287"] =       287.1911860000;
  m[  "UUP288"] =       288.1924920000;
  m[  "UUP289"] =       289.1927150000;
  m[  "UUP290"] =       290.1941410000;
  m[  "UUP291"] =       291.1943840000;
  m[     "UUH"] =       292.1997860000;
  m[  "UUH289"] =       289.1988620000;
  m[  "UUH290"] =       290.1985900000;
  m[  "UUH291"] =       291.2000110000;
  m[  "UUH292"] =       292.1997860000;
  m[     "UUS"] =       291.2065640000;
  m[  "UUS291"] =       291.2065640000;
  m[  "UUS292"] =       292.2075490000;
  m[     "UUO"] =       293.2146700000;
  m[  "UUO293"] =       293.2146700000;
  return m;
}

// Create static map for isotope symbol->mass
static map<string,double> isotope_to_mass_map = create_isotope_to_mass_map();

// Define function to access map ; could be eliminated later
double Isotope_to_mass(std::string lbl) {
  return opt::isotope_to_mass_map[lbl];
}
*/

}
