use ui;

/**
 * View hints, as described in the comment for Hints.
 */
class ViewHints on Render {
	// Should we pause on function entry?
	Bool pauseOnEntry() { true; }

	// Create an element for objects found during traversal. You don't need to wrap these in
	// frames. Return 'null' to leave handling to the default implementation.
	Data? createData(Workspace ws, TypeInfo info, unsafe:RawPtr base, Nat offset) : abstract;

	// Get a title for the type.
	Str? typeTitle(Workspace ws, TypeInfo info) : abstract;
}


/**
 * Hints used when calling "traverse" to inspect and impact object creation.
 */
class CreateHints on Render {
	// Get an opportunity to decorate a newly created object.
	Data decorate(Data data) : abstract;
}


/**
 * Default view hints.
 */
class DefaultViewHints extends ViewHints {
	// Default implementation. Handles most of Basic Storm.
	Data? createData(Workspace ws, TypeInfo info, unsafe:RawPtr object, Nat offset) : override {
		if (info.isRef | !info.isValue) {
			// It is a ref to a value. We don't generally support these, so just show them as a pointer.
			return Pointer();
		} else if (c = convertPrimitive(info.type, object, offset)) {
			return Primitive(info.type, c);
		} else {
			return createComposite(ws, info.type, object, offset);
		}
	}

	// Composite object.
	private Data createComposite(Workspace ws, core:lang:Type type, unsafe:RawPtr object, Nat offset) {
		if (type as core:lang:ArrayType) {
			return createArray(type, object, offset);
		}

		var members = findMembers(type);
		if (members.empty)
			return UnknownData();

		Composite result(type);
		for (m in members) {
			result.add(m.name, m.offset, ws.createData(m.type, object, offset + m.offset), m.first);
		}

		result;
	}

	// Create an array.
	private Data createArray(core:lang:ArrayType type, unsafe:RawPtr object, Nat offset) {
		if (t = type.param.type) {
			ArrayData(t);
		} else {
			UnknownData();
		}
	}

	// Type names.
	Str? typeTitle(Workspace ws, TypeInfo info) : override {
		if (type = info.type as core:lang:ArrayType) {
			if (t = type.param.type) {
				TypeInfo info(t, true, false);
				return ws.typeTitle(info) + "[]";
			}
		}

		info.type.name;
	}
}
