# Metadata Providers

Authors: [Moritz Lang](https://github.com/slashmo), [Konrad 'ktoso' Malawski](https://github.com/ktoso)

## Introduction

While global metadata attributes may be manually set on a `LogHandler` level, there's currently no way of reliably providing contextual, automatically propagated, metadata when logging with swift-log.

## Motivation

To benefit from tools such as [Distributed Tracing](https://github.com/apple/swift-distributed-tracing) it is necessary for libraries to make use of the trace information.

Most notably, loggers should participate in tracing by including some trace metadata (such as e.g. a `trace-id`) when logging, as it transparently enables developers to benefit from log correlation using those IDs.

Today, the only supported way of providing such metadata is to pass them along to each log call explicitly:

```swift
logger.info("first this ...", metadata: ["trace-id": MyTracingLibrary.currentTraceID])
logger.info("... now this", metadata: ["trace-id": MyTracingLibrary.currentTraceID])
```

This comes with a couple of downsides:

### Error-prone and repetitive

It's easy to forget passing this metadata to _all_ log statements, resulting in an inconsistent debugging experience as these log statements cannot be found using correlation IDs.

The repetitiveness and verboseness of logging multiple metadata in-line quickly becomes annoying and vastly decreases the signal-to-noise ratio of Swift code trying to be a good citizen and making use of log correlation techniques such as distributed tracing.

### Impossible to implement for libraries

A large portion of logs are not generated by the end-user but by libraries such as AsyncHTTPClient, Vapor etc.
These libraries, by design, don't know about the specific metadata keys that should be included in the logs.
Those keys are after all runtime dependent, and may change depending on what tracing system is configured by the end user.

For example, a specific `Tracer` implementation would use a type representing a trace ID, and has a way of extracting
that trace ID from a [Swift Distributed Tracing Baggage](https://github.com/apple/swift-distributed-tracing-baggage). But other libraries can't know about this _specific_ trace ID and therefore would not be
able to pass such values along to their log statements.

## Proposed solution

To support this kind of runtime-generated metadata in `swift-log`, we need to extend the logging APIs in an open-ended way, to allow any kinds of metadata to be provided from the asynchronous context (i.e. task local values).

We also have a desire to keep `swift-log` a "zero dependencies" library, as it intends only to focus on describing a logging API, and not incur any additional dependencies, so it can be used in the most minimal of projects.

To solve this, we propose the extension of swift-log APIs with a new concept: _metadata providers_.

`MetadataProvider` is a struct nested in the `Logger` type, sitting alongside `MetadataValue` and the `Metadata`. 
Its purpose is to _provide_ `Logger.Metadata` from the asynchronous context, by e.g. looking up various task-local values,
and converting them into `Logger.Metadata`. This is performed by calling the `get()` function, like so:

```swift
extension Logger {
    public struct MetadataProvider: Sendable {
        // ... 
        public init(_ provideMetadata: @escaping @Sendable () -> Metadata)
    
        public get() -> Metadata
    }
}
```

### Defining a `MetadataProvider`

While `MetadataProvider`s can be created in an ad-hoc fashion, the struct may be used as a namespace
to define providers in. 

For example, a `MyTracer` implementation of swift-distributed-tracing would be expected to provide a metadata provider
that is aware of its own `Baggage` and specific keys it uses to carry the trace and span identifiers, like this:

```swift
import Tracing // swift-distributed-tracing

extension Logger.MetadataProvider {
    static let myTracer = Logger.MetadataProvider {
        guard let baggage = Baggage.current else {
            return [:]
        }
        guard let spanContext = baggage.spanContext else { 
            return [:]
        }
        
        return [
          "traceID": "\(spanContext.traceID)",
          "spanID": "\(spanContext.spanID)",
        ]
    }
}
```

### Using a `MetadataProvider`

A `MetadataProvider` can be set up either globally, on a boot-strapped logging system:

```swift
LoggingSystem.bootstrap(
    metadataProvider: .myTracer,
    StreamLogHandler.standardOutput
)
```

or, using the metadata provider specific bootstrap method:

```swift
LoggingSystem.bootstrapMetadataProvider(.myTracer)
```

It is also possible to configure a metadata provider on a per-logger basis, like this:

```swift
let logger = Logger(label: "example", metadataProvider: Logger.MetadataProvider { 
    guard let baggage = Baggage.current else {
        return [:]
    }
    guard let operationID = baggage.operationID else { 
        return nil
    }
    return ["extra/opID": "\(opID)"]
})
```

which _overrides_ the default bootstrapped metadata provider.

> NOTE: Setting the metadata provider on the logger directly means the `LoggingSystem` metadata provider
> is skipped (if defined), following how an explicitly passed handler `factory`
> overrides the `LoggingSystem`s `factory`.

Once a metadata provider was set up, when a log statement is about to be emitted, the log handler implementation shall
invoke it whenever it is about to emit a log statement. This must be done from the same asynchronous context as the log statement
was made; I.e. if a log handler were to asynchronously–in a _detached_ task–create the actual log message, the invocation of
the metadata provider still _must_ be performed before passing the data over to the other detached task.

Usually, metadata providers will reach for the task's Swift Distributed Tracing [`Baggage`](https://github.com/apple/swift-distributed-tracing-baggage) which is the mechanism that swift-distributed-tracing 
and instrumentation libraries use to propagate metadata across asynchronous, as well as process, boundaries. Handlers 
may also inspect other task-local values, however they should not expect other libraries to be able to propagage those
as well as they will propagate `Baggage` values.

Those metadata will then be included in the log statement, e.g. like this:

```swift
var baggage = Baggage.topLevel
baggage.spanContext = SpanContext()
Baggage.withValue(baggage) {
    test()
}

func test() {
    log.info("Test", metadata: ["oneOff": "42"])
    // info [traceID: abc, spanID: 123, onOff: 42] Test
}
```

### Multiple `MetadataProvider`s using `MetadataProvider.multiplex(_:)`

Borrowing the concept from log handlers, metadata providers also have a multiplexing implementation.
It is defined as an extension on `MetadataProvider` and is useful in cases where users want to utilize
more than one metadata provider at the same time:

```swift
extension Logger.MetadataProvider {
    public static func multiplex(_ providers: [Logger.MetadataProvider]) -> Logger.MetadataProvider {
        precondition(!providers.isEmpty, "providers MUST NOT be empty")
        return Logger.MetadataProvider { baggage in
            providers.reduce(into: nil) { metadata, provider in
                if let providedMetadata = provider.metadata(baggage) {
                    if metadata != nil {
                        metadata!.merge(providedMetadata, uniquingKeysWith: { _, rhs in rhs })
                    } else {
                        metadata = providedMetadata
                    }
                }
            }
        }
    }
}
```

Metadata keys are unique, so in case multiple metadata providers return the same key,
the last provider in the array _wins_ and provides the value.

Note that it is possible to query the `LoggingSystem` for the configured, system-wide, metadata provider,
and combine it using a `multiplex` provider if the system-wide provider should not be replaced, but augmented
by additional providers.

### Understanding Swift Distributed Tracing `Baggage`

The `Baggage` type is more than just a fancy type-safe container for your values which are meant to be carried across
using a single, well-known, [task-local value](https://developer.apple.com/documentation/swift/tasklocal).

Values stored in `Baggage` are intended to be carried _across process boundaries_, and e.g. libraries such as HTTP clients,
or other RPC mechanisms, or even messaging systems implement distributed tracing [instruments](https://github.com/apple/swift-distributed-tracing/blob/main/Sources/Instrumentation/Instrument.swift) 
which `inject` and `extract` baggage values into their respective carrier objects, e.g. an `HTTPRequest` in the case of an HTTP client.

In other words, whenever intending to propagate information _across processes_ utilize the `Baggage` type to carry it, 
and ensure to provide an `Instrument` that is able to inject/extract the values of interest into the carrier types you are interested in.
To learn more about baggage and instrumentation, refer to the [Swift Distributed Tracing](https://github.com/apple/swift-distributed-tracing/) library documentation.

#### When to use `Baggage` vs. `Logger[metadataKey:]`

While `Baggage` is context-dependent and changes depending on where the log methods are being called from, the metadata set on a `Logger` is static and not context-dependent. E.g, if you wanted to add things like an instance ID or a subsystem name to a `Logger`, that could be seen as static information and set via `Logger[metadataKey:]`.

```swift
var logger = Logger(label: "org.swift.my-service")
logger[metadataKey: "instanceId"] = "123"

logger.info("Service started.")
// [instanceId: 123] Service started.
```

On the other hand, things like a trace ID are dynamic and context-dependent, therefore would be obtained via `Baggage`:

```swift
logger.info("Product fetched.")
// [traceId: 42] Product fetched.
```

Inline-metadata is suitable for one-offs such as a `productId` or a `paymentMethod` in an online store service, but are not enough to corralate the following log statements, i.e. tying them both to the same request:

```swift
logger.info("Product fetched.", metadata: ["productId": "42"])
logger.info("Product purchased.", metadata: ["paymentMethod": "apple-pay"])

// [productId: 42] Product fetched.
// [paymentMethod: apple-pay] Product fetched.
```

If there was a `Baggage` value with a trace ID surrounding these log statements, they would be automatically correlatable:

```swift
var baggage = Baggage.topLevel
baggage.traceID = 42
Baggage.$current.withValue(baggage) {
    logger.info("Product fetched.", metadata: ["productId": "42"])
    logger.info("Product purchased.", metadata: ["paymentMethod": "apple-pay"])
}

// [trace-id: 42, productId: 42] Product fetched.
// [trace-id: 42, paymentMethod: apple-pay] Product fetched.
```

## Alternatives considered

### MetadataProviders as a function of `Baggage -> Metadata`

This was considered and fully developed, however it would cause swift-log to have a dependency on the instrumentation type `Baggage`, 
which was seen as in-conflict-with the interest of swift-log remaining a zero-dependency library.

`Baggage` _is_ the well known type that is intended to be used for any kind of distributed systems instrumentation and tracing,
however it adds additional complexity and confusion for users who are not interested in this domain. For example, developers
may be confused about why `Baggage` and `Logger.Metadata` look somewhat similar, but behave very differently. This complexity
is inherent to the two types actually being _very_ different, however we do not want to overwhelm newcomers or developers
who are only intending to use swift-log within process. Such developers do not need to care about the added complexities 
of distributed systems.

The tradeoff we take here is that every metadata provider will have to perform its own task-local (and therefore also thread-local),
access in order to obtain the `Baggage` value, rather than the lookup being able to be performed _once_ and shared between 
multiple providers when a multiplex provider was configured in the system. We view this tradeoff as worth taking, as the cost
of actually formatting the metadata usually strongly dominates the cost of the task-local lookup. 

### Removing `LogHandler.Metadata` in favor of `Baggage`

Removing logger metadata is not a good option because it serves a slightly different style of metadata than the baggage.

Baggage is intended for contextual, task-local, metadata which is _carried across multiple libraries_. The values stored in baggage are well-typed, and must declare keys for accessing values in a baggage, this works well for multiple pieces of an application needing to reach for specific baggage items: everyone aware of the `traceID` key, is able to query the baggage for this key (`baggage.traceID`) and obtain a well-typed value. This comes with a higher cost on declaring keys though, as well as a global namespace for those - which is desirable for such kinds of metadata.

This is not the same usage pattern as emitting a plain structured log where we'd like to include the name of an item we just queried:

```swift
log.info("Obtained item! Hooray!", metadata: ["item": "\(item)"])
```

In this example, the key/value pair for `"item"` is pretty ad-hoc, and we never need to refer to it elsewhere in the program. It never is queried by other pieces of the application, nor would it be useful to set it in baggage metadata, as the only purpose of the `item` key here is to log, and forget about it.

### Explicitly passing `Baggage` always

Baggage is designed for use cases like distributed tracing, or similar instrumentation patterns where "all" participating code may need to reach for it.

Specifically in logging, this means that _every_ call site for _every_ log statement would have to pass it explicitly resulting in annoying noisy code:

```swift
class StoresRepository {
    func store(byID id: String, eventLoop: EventLoop, logger: Logger, baggage: Baggage) async throws -> Store {
        InstrumentationSystem.tracer.withSpan("Fetch Store") { span in
            logger.info("Fetched store.", baggage: span.baggage)
        }
    }
}

try await storesRepository.store(
    byID: storeID,
    eventLoop: eventLoop,
    logger: logger,
    baggage: baggage
)
```

### Explicitly passing `Logger` always

Imagine we don't have metadata providers, we'd have to manually set trace IDs on loggers which doesn't really work as all libraries involved would need to know about the same specific trace ID. Even if we inverted the dependency to have `Tracing` depend on `Logging` so that we'd be able to define something like "Tracing, please populate this logger with metadata", we'd have to make sure this thing is called in all places to avoid dropping contextual metadata.

```swift
import Tracing
import Logging

let contextualLogger = InstrumentationSystem.tracer.populateTraceMetadata(logger)
contextualLogger.info("Request received.")
```

### Tracing providing extensions on Logger

Instead of having `swift-log` depend on `swift-distributed-tracing-baggage` we could also create extensions for `Logger` inside `swift-distributed-tracing` and have users call these new overloaded methods instead:

```swift
extension Logger {
    func tinfo(_ message: ..., baggage: Baggage?) {
        // ...
    }
}
```

Such extensions could work like the currently-proposed APIs, but the risk of users calling the wrong methods is incredibly high and we cannot overload the existing methods' signatures because of ambiguity of call-sides without explicit Baggage being passed:

```swift
logger.info("Hello")
// we want this to pick up Baggage, but the signature would be ambiguous
```

Also, this extension-based contextual metadata would require basically everyone in Server-side Swift to adapt their usage of `Logging` to use these extensions instead. With the proposed APIs, we'd only need to modify `Logging` and any NIO-based libraries such as `AsyncHTTPClient`, `Vapor`, etc. and not every single log statement in every application.
