// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (c) 2000-2001,2005 Silicon Graphics, Inc.
 * All Rights Reserved.
 */

#include "libxfs.h"
#include "command.h"
#include "type.h"
#include "faddr.h"
#include "fprint.h"
#include "field.h"
#include "io.h"
#include "bit.h"
#include "output.h"
#include "init.h"
#include "agf.h"

static int agf_f(int argc, char **argv);
static void agf_help(void);

static const cmdinfo_t agf_cmd =
	{ "agf", NULL, agf_f, 0, 1, 1, N_("[agno]"),
	  N_("set address to agf header"), agf_help };

const field_t	agf_hfld[] = {
	{ "", FLDT_AGF, OI(0), C1, 0, TYP_NONE },
	{ NULL }
};

#define	OFF(f)	bitize(offsetof(xfs_agf_t, agf_ ## f))
#define	SZ(f)	bitszof(xfs_agf_t, agf_ ## f)
const field_t	agf_flds[] = {
	{ "magicnum", FLDT_UINT32X, OI(OFF(magicnum)), C1, 0, TYP_NONE },
	{ "versionnum", FLDT_UINT32D, OI(OFF(versionnum)), C1, 0, TYP_NONE },
	{ "seqno", FLDT_AGNUMBER, OI(OFF(seqno)), C1, 0, TYP_NONE },
	{ "length", FLDT_AGBLOCK, OI(OFF(length)), C1, 0, TYP_NONE },
	{ "bnoroot", FLDT_AGBLOCK, OI(OFF(bno_root)), C1, 0, TYP_BNOBT },
	{ "cntroot", FLDT_AGBLOCK, OI(OFF(cnt_root)), C1, 0, TYP_CNTBT },
	{ "rmaproot", FLDT_AGBLOCKNZ, OI(OFF(rmap_root)), C1, 0, TYP_RMAPBT },
	{ "refcntroot", FLDT_AGBLOCKNZ,
	  OI(OFF(refcount_root)), C1, 0,
	  TYP_REFCBT },
	{ "bnolevel", FLDT_UINT32D, OI(OFF(bno_level)), C1, 0, TYP_NONE },
	{ "cntlevel", FLDT_UINT32D, OI(OFF(cnt_level)), C1, 0, TYP_NONE },
	{ "rmaplevel", FLDT_UINT32D, OI(OFF(rmap_level)), C1, 0, TYP_NONE },
	{ "refcntlevel", FLDT_UINT32D,
	  OI(OFF(refcount_level)), C1, 0,
	  TYP_NONE },
	{ "rmapblocks", FLDT_UINT32D,
	  OI(OFF(rmap_blocks)), C1, 0,
	  TYP_NONE },
	{ "refcntblocks", FLDT_UINT32D,
	  OI(OFF(refcount_blocks)), C1, 0,
	  TYP_NONE },
	{ "flfirst", FLDT_UINT32D, OI(OFF(flfirst)), C1, 0, TYP_NONE },
	{ "fllast", FLDT_UINT32D, OI(OFF(fllast)), C1, 0, TYP_NONE },
	{ "flcount", FLDT_UINT32D, OI(OFF(flcount)), C1, 0, TYP_NONE },
	{ "freeblks", FLDT_EXTLEN, OI(OFF(freeblks)), C1, 0, TYP_NONE },
	{ "longest", FLDT_EXTLEN, OI(OFF(longest)), C1, 0, TYP_NONE },
	{ "btreeblks", FLDT_UINT32D, OI(OFF(btreeblks)), C1, 0, TYP_NONE },
	{ "uuid", FLDT_UUID, OI(OFF(uuid)), C1, 0, TYP_NONE },
	{ "lsn", FLDT_UINT64X, OI(OFF(lsn)), C1, 0, TYP_NONE },
	{ "crc", FLDT_CRC, OI(OFF(crc)), C1, 0, TYP_NONE },
	{ NULL }
};

static void
agf_help(void)
{
	dbprintf(_(
"\n"
" set allocation group free block list\n"
"\n"
" Example:\n"
"\n"
" agf 2 - move location to AGF in 2nd filesystem allocation group\n"
"\n"
" Located in the second sector of each allocation group, the AGF\n"
" contains the root of two different freespace btrees:\n"
" The 'cnt' btree keeps track freespace indexed on section size.\n"
" The 'bno' btree tracks sections of freespace indexed on block number.\n"
));
}

static int
agf_f(
	int		argc,
	char		**argv)
{
	xfs_agnumber_t	agno;
	char		*p;

	if (argc > 1) {
		agno = (xfs_agnumber_t)strtoul(argv[1], &p, 0);
		if (*p != '\0' || agno >= mp->m_sb.sb_agcount) {
			dbprintf(_("bad allocation group number %s\n"), argv[1]);
			return 0;
		}
		cur_agno = agno;
	} else if (cur_agno == NULLAGNUMBER)
		cur_agno = 0;
	ASSERT(typtab[TYP_AGF].typnm == TYP_AGF);
	set_cur(&typtab[TYP_AGF],
		XFS_AG_DADDR(mp, cur_agno, XFS_AGF_DADDR(mp)),
		XFS_FSS_TO_BB(mp, 1), DB_RING_ADD, NULL);
	return 0;
}

void
agf_init(void)
{
	add_command(&agf_cmd);
}

int
agf_size(
	void	*obj,
	int	startoff,
	int	idx)
{
	return bitize(mp->m_sb.sb_sectsize);
}
