//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Sim/Scan/IBeamScan.h
//! @brief     Declares interface IBeamScan.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_SIM_SCAN_IBEAMSCAN_H
#define BORNAGAIN_SIM_SCAN_IBEAMSCAN_H

#include "Base/Types/ICloneable.h"
#include "Param/Node/INode.h"
#include <cstddef>
#include <heinz/Vectors3D.h>
#include <memory>
#include <vector>

class CoordSystem1D;
class Scale;
class IFootprint;
class PolFilter;
class SpecularElement;
class SpinMatrix;

//! Abstract base class for all types of specular scans.

class IBeamScan : public ICloneable, public INode {
public:
    IBeamScan(Scale* axis, double lambda0);
    ~IBeamScan() override;

    IBeamScan* clone() const override = 0;
    std::vector<const INode*> nodeChildren() const override;

    void setWavelength(double lambda0);

    //! Sets the intensity fudge factor
    void setIntensity(double intensity);

    //! Sets the polarization density matrix according to the given Bloch vector
    void setPolarization(R3 bloch_vector);

    //! Sets the polarization analyzer characteristics of the detector
    void setAnalyzer(R3 Bloch_vector = {}, double mean_transmission = 0.5);
    void setAnalyzer(R3 direction, double efficiency, double transmission); // OBSOLETE since v21

    void setFootprint(const IFootprint* footprint);

#ifndef SWIG
    const PolFilter* analyzer() const
    {
        return m_polAnalyzer.get();
    }

    double wavelength() const
    {
        return m_lambda0;
    }

    double intensity() const
    {
        return m_intensity;
    }

    SpinMatrix polarizerMatrix() const;
    SpinMatrix analyzerMatrix() const;

    //! Generates simulation elements for specular simulations
    virtual std::vector<SpecularElement> generateElements() const = 0;

    //! Returns coordinate axis assigned to the data holder
    const Scale* coordinateAxis() const
    {
        return m_axis.get();
    }

    //! Returns the number of scan points
    size_t nScan() const;

    //! Returns the number of distribution samples
    virtual size_t nDistributionSamples() const = 0;

    virtual CoordSystem1D* scanCoordSystem() const = 0;

    bool polarized() const;

    const IFootprint* footprint() const
    {
        return m_footprint.get();
    }

protected:
    const std::unique_ptr<const Scale> m_axis;
    double m_lambda0;
    double m_intensity = 1; //!< Fudge factor to adjust for imperfect normalization of exp. data
    std::unique_ptr<R3> m_beamPolarization; //!< Bloch vector encoding the beam's polarization
    std::unique_ptr<PolFilter> m_polAnalyzer;
    std::unique_ptr<IFootprint> m_footprint; //!< footprint correction handler
#endif                                       // SWIG
};

#endif // BORNAGAIN_SIM_SCAN_IBEAMSCAN_H
