/* $Id: main-toolbar.c 3767 2011-12-08 19:55:28Z ensonic $
 *
 * Buzztard
 * Copyright (C) 2006 Buzztard team <buzztard-devel@lists.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/**
 * SECTION:btmaintoolbar
 * @short_description: class for the editor main toolbar
 *
 * Contains typical applications buttons for file i/o and playback, plus volume
 * meters and volume control.
 */

/* @todo should we separate the toolbars?
 * - common - load, save, ...
 * - volume - gain, levels
 */
#define BT_EDIT
#define BT_MAIN_TOOLBAR_C

#include "bt-edit.h"
#include "gtkvumeter.h"

/* lets keep multichannel audio for later :) */
//#define MAX_VUMETER 4
#define MAX_VUMETER 2
#define DEF_VUMETER 2
#define LOW_VUMETER_VAL -90.0

struct _BtMainToolbarPrivate {
  /* used to validate if dispose has run */
  gboolean dispose_has_run;

  /* the application */
  BtEditApplication *app;

  /* the level meters */
  GtkVUMeter *vumeter[MAX_VUMETER];
  G_POINTER_ALIAS(GstElement *,level);
  GstClock *clock;

  /* the volume gain */
  GtkScale *volume;
  G_POINTER_ALIAS(GstElement *,gain);
  G_POINTER_ALIAS(BtMachine *,master);

  /* action buttons */
  GtkWidget *save_button;
  GtkWidget *play_button;
  GtkWidget *stop_button;
  GtkWidget *loop_button;

  /* signal handler ids */
  guint playback_update_id;
  guint playback_rate_id;

  /* playback state */
  gboolean is_playing;
  gboolean has_error;
  gdouble playback_rate;

  /* lock for multithreaded access */
  GMutex        *lock;
};

static GQuark bus_msg_level_quark=0;
static GQuark bus_msg_level_caps_changed_quark=0;

static void on_toolbar_play_clicked(GtkButton *button, gpointer user_data);
static void reset_playback_rate(BtMainToolbar *self);
static void on_song_volume_changed(GstElement *gain,GParamSpec *arg,gpointer user_data);

//-- the class

G_DEFINE_TYPE (BtMainToolbar, bt_main_toolbar, GTK_TYPE_TOOLBAR);


//-- helper

static gint gst_caps_get_channels(GstCaps *caps) {
  GstStructure *structure;
  gint channels=0;

  g_assert(caps);

  if(GST_CAPS_IS_SIMPLE(caps)) {
    if((structure=gst_caps_get_structure(caps,0))) {
      gst_structure_get_int(structure,"channels",&channels);
      GST_DEBUG("---    simple caps with channels=%d",channels);
    }
  }
  else {
    gint size,i;
    size=gst_caps_get_size(caps);
    for(i=0;i<size;i++) {
      if((structure=gst_caps_get_structure(caps,i))) {
        gst_structure_get_int(structure,"channels",&channels);
        GST_DEBUG("---    caps %d with channels=%d",i,channels);
      }
    }
  }
  return(channels);
}

//-- event handler

static gboolean on_song_playback_update(gpointer user_data) {
  BtSong *self=BT_SONG(user_data);

  return(bt_song_update_playback_position(self));
}

static void on_song_is_playing_notify(const BtSong *song,GParamSpec *arg,gpointer user_data) {
  BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);

  g_object_get((gpointer)song,"is-playing",&self->priv->is_playing,NULL);
  if(!self->priv->is_playing) {
    gint i;

    GST_INFO("song stop event occurred: %p",g_thread_self());
    // stop update timer and reset trick playback
    if(self->priv->playback_update_id) {
      g_source_remove(self->priv->playback_update_id);
      self->priv->playback_update_id=0;
    }
    bt_song_update_playback_position(song);
    reset_playback_rate(self);
    // disable stop button
    gtk_widget_set_sensitive(GTK_WIDGET(self->priv->stop_button),FALSE);
    // switch off play button
    gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(self->priv->play_button),FALSE);
    // enable play button
    gtk_widget_set_sensitive(GTK_WIDGET(self->priv->play_button),TRUE);
    // reset level meters
    for(i=0;i<MAX_VUMETER;i++) {
      gtk_vumeter_set_levels(self->priv->vumeter[i], LOW_VUMETER_VAL, LOW_VUMETER_VAL);
    }

    self->priv->has_error = FALSE;

    GST_INFO("song stop event handled");
  }
  else {
    // update playback position 10 times a second
    //self->priv->playback_update_id=g_timeout_add(100,on_song_playback_update,(gpointer)song);
    self->priv->playback_update_id=g_timeout_add_full(G_PRIORITY_HIGH,100,on_song_playback_update,(gpointer)song,NULL);
    bt_song_update_playback_position(song);

    // if we started playback remotely activate playbutton
    if(!gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON(self->priv->play_button))) {
      g_signal_handlers_block_matched(self->priv->play_button,G_SIGNAL_MATCH_FUNC|G_SIGNAL_MATCH_DATA,0,0,NULL,on_toolbar_play_clicked,(gpointer)self);
      gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(self->priv->play_button),TRUE);
      g_signal_handlers_unblock_matched(self->priv->play_button,G_SIGNAL_MATCH_FUNC|G_SIGNAL_MATCH_DATA,0,0,NULL,on_toolbar_play_clicked,(gpointer)self);
    }
    // disable play button
    gtk_widget_set_sensitive(GTK_WIDGET(self->priv->play_button),FALSE);
    // enable stop button
    gtk_widget_set_sensitive(GTK_WIDGET(self->priv->stop_button),TRUE);
  }
}

static void on_toolbar_new_clicked(GtkButton *button, gpointer user_data) {
  BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);
  BtMainWindow *main_window;

  GST_INFO("toolbar new event occurred");
  g_object_get(self->priv->app,"main-window",&main_window,NULL);
  bt_main_window_new_song(main_window);
  g_object_unref(main_window);
}

static void on_toolbar_open_clicked(GtkButton *button, gpointer user_data) {
  BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);
  BtMainWindow *main_window;

  GST_INFO("toolbar open event occurred");
  g_object_get(self->priv->app,"main-window",&main_window,NULL);
  bt_main_window_open_song(main_window);
  g_object_unref(main_window);
}

static void on_toolbar_save_clicked(GtkButton *button, gpointer user_data) {
  BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);
  BtMainWindow *main_window;

  GST_INFO("toolbar open event occurred");
  g_object_get(self->priv->app,"main-window",&main_window,NULL);
  bt_main_window_save_song(main_window);
  g_object_unref(main_window);
}

static void on_toolbar_play_clicked(GtkButton *button, gpointer user_data) {
  BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);

  if(gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON(button))) {
    BtSong *song;

    GST_INFO("toolbar play activated");

    // get song from app and start playback
    g_object_get(self->priv->app,"song",&song,NULL);
    if(!bt_song_play(song)) {
      // switch off play button
      gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(button),FALSE);
    }

    // release the reference
    g_object_unref(song);
  }
}

static void on_toolbar_stop_clicked(GtkButton *button, gpointer user_data) {
  BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);
  BtSong *song;

  GST_INFO("toolbar stop event occurred");
  // get song from app
  g_object_get(self->priv->app,"song",&song,NULL);
  bt_song_stop(song);
  reset_playback_rate(self);

  GST_INFO("  song stopped");
  // release the reference
  g_object_unref(song);
}

static void on_toolbar_loop_toggled(GtkButton *button, gpointer user_data) {
  BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);
  BtSong *song;
  BtSequence *sequence;
  gboolean loop;

  loop=gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON(button));
  GST_INFO("toolbar loop toggle event occurred, new-state=%d",loop);
  // get song from app
  g_object_get(self->priv->app,"song",&song,NULL);
  g_object_get(song,"sequence",&sequence,NULL);
  g_object_set(sequence,"loop",loop,NULL);
  bt_edit_application_set_song_unsaved(self->priv->app);
  // release the references
  g_object_unref(sequence);
  g_object_unref(song);
}

static void set_new_playback_rate(BtMainToolbar *self,gdouble playback_rate) {
  BtSong *song;

  self->priv->playback_rate=playback_rate;
  g_object_get(self->priv->app,"song",&song,NULL);
  g_object_set(song,"play-rate",self->priv->playback_rate,NULL);
  g_object_unref(song);
}

static void reset_playback_rate(BtMainToolbar *self) {
  set_new_playback_rate(self,1.0);

  if(self->priv->playback_rate_id) {
    g_source_remove(self->priv->playback_rate_id);
    self->priv->playback_rate_id=0;
  }
}

#define SEEK_FACTOR 1.2
#define SEEK_TIMEOUT 2000    
#define SEEK_MAX_RATE 4.0

static gboolean on_song_playback_rate_rewind(gpointer user_data) {
  BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);
  gdouble playback_rate = self->priv->playback_rate * SEEK_FACTOR;

  GST_DEBUG(" << speedup");

  if (playback_rate > -SEEK_MAX_RATE) {
    set_new_playback_rate(self, playback_rate);
    return(TRUE);
  } else {
    self->priv->playback_rate_id=0;
    return(FALSE);
  }
}

static gboolean on_toolbar_rewind_pressed(GtkWidget *widget,GdkEventButton *event,gpointer user_data) {
  BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);

  if(event->button!=1)
    return(FALSE);

  GST_DEBUG(" << pressed");

  set_new_playback_rate(self,-1.0);
  self->priv->playback_rate_id=g_timeout_add(SEEK_TIMEOUT,on_song_playback_rate_rewind,(gpointer)self);

  GST_DEBUG(" << <<");

  return(FALSE);
}

static gboolean on_toolbar_rewind_released(GtkWidget *widget,GdkEventButton *event,gpointer user_data) {
  BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);

  if(event->button!=1)
    return(FALSE);

  GST_DEBUG(" << released");
  reset_playback_rate(self);

  return(FALSE);
}

static gboolean on_song_playback_rate_forward(gpointer user_data) {
  BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);
  gdouble playback_rate = self->priv->playback_rate * SEEK_FACTOR;

  GST_WARNING(" >> speedup");

  if (playback_rate < SEEK_MAX_RATE) {
    set_new_playback_rate(self,playback_rate);
    return(TRUE);
  } else {
    self->priv->playback_rate_id=0;
    return(FALSE);
  }
}

static gboolean on_toolbar_forward_pressed(GtkWidget *widget,GdkEventButton *event,gpointer user_data) {
  BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);

  if(event->button!=1)
    return(FALSE);

  GST_DEBUG(" >> pressed");

  set_new_playback_rate(self,SEEK_FACTOR);

  self->priv->playback_rate_id=g_timeout_add(SEEK_TIMEOUT,on_song_playback_rate_forward,(gpointer)self);

  GST_DEBUG(" >> >>");
  return(FALSE);
}

static gboolean on_toolbar_forward_released(GtkWidget *widget,GdkEventButton *event,gpointer user_data) {
  BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);

  if(event->button!=1)
    return(FALSE);

  GST_DEBUG(" >> released");
  reset_playback_rate(self);

  return(FALSE);
}

static void on_song_error(const GstBus * const bus, GstMessage *message, gconstpointer user_data) {
  const BtMainToolbar * const self=BT_MAIN_TOOLBAR(user_data);
  GError *err = NULL;
  gchar *dbg = NULL;

  GST_INFO("received %s bus message from %s",
    GST_MESSAGE_TYPE_NAME(message), GST_OBJECT_NAME(GST_MESSAGE_SRC(message)));

  // @todo: check domain and code
  gst_message_parse_error(message, &err, &dbg);
  GST_WARNING_OBJECT(GST_MESSAGE_SRC(message),"ERROR: %s (%s)", err->message, (dbg ? dbg : "no details"));

  if(!self->priv->has_error) {
    BtSong *song;
    BtMainWindow *main_window;

    // get song from app
    g_object_get(self->priv->app,"song",&song,"main-window",&main_window,NULL);
    // debug the state
    bt_song_write_to_lowlevel_dot_file(song);
    bt_song_stop(song);

    bt_dialog_message(main_window,_("Error"),_("An error occurred"),err->message);

    // release the reference
    g_object_unref(song);
    g_object_unref(main_window);
  }
  g_error_free(err);
  g_free(dbg);

  self->priv->has_error = TRUE;
}

static void on_song_warning(const GstBus * const bus, GstMessage *message, gconstpointer user_data) {
  const BtMainToolbar * const self=BT_MAIN_TOOLBAR(user_data);
  GError *err = NULL;
  gchar *dbg = NULL;

  GST_INFO("received %s bus message from %s",
    GST_MESSAGE_TYPE_NAME(message), GST_OBJECT_NAME(GST_MESSAGE_SRC(message)));

  // @todo: check domain and code
  gst_message_parse_warning(message, &err, &dbg);
  GST_WARNING_OBJECT(GST_MESSAGE_SRC(message),"WARNING: %s (%s)", err->message, (dbg ? dbg : "no details"));

  if(!self->priv->has_error) {
    BtSong *song;
    BtMainWindow *main_window;

    // get song from app
    g_object_get(self->priv->app,"song",&song,"main-window",&main_window,NULL);
    //bt_song_stop(song);

    bt_dialog_message(main_window,_("Warning"),_("A problem occurred"),err->message);

    // release the reference
    g_object_unref(song);
    g_object_unref(main_window);
  }
  g_error_free(err);
  g_free(dbg);
}

static gboolean on_delayed_idle_song_level_change(gpointer user_data) {
  gconstpointer * const params=(gconstpointer *)user_data;
  BtMainToolbar *self=(BtMainToolbar *)params[0];
  GstMessage *message=(GstMessage *)params[1];

  if(self) {
    const GstStructure *structure=gst_message_get_structure(message);
    const GValue *l_cur,*l_peak;
    gdouble cur, peak;
    guint i;

    g_mutex_lock(self->priv->lock);
    g_object_remove_weak_pointer((gpointer)self,(gpointer *)&params[0]);
    g_mutex_unlock(self->priv->lock);

    if(!self->priv->is_playing)
      goto done;

    l_cur=(GValue *)gst_structure_get_value(structure, "decay");
    l_peak=(GValue *)gst_structure_get_value(structure, "peak");

    for(i=0;i<gst_value_list_get_size(l_cur);i++) {
      cur=g_value_get_double(gst_value_list_get_value(l_cur,i));
      peak=g_value_get_double(gst_value_list_get_value(l_peak,i));
      if(isinf(cur) || isnan(cur)) cur=LOW_VUMETER_VAL;
      if(isinf(peak) || isnan(peak)) peak=LOW_VUMETER_VAL;
      //GST_INFO("level.%d  %.3f %.3f", i, peak, cur);
      //gtk_vumeter_set_levels(self->priv->vumeter[i], (gint)cur, (gint)peak);
      gtk_vumeter_set_levels(self->priv->vumeter[i], (gint)peak, (gint)cur);
    }
  }
done:
  gst_message_unref(message);
  g_slice_free1(2*sizeof(gconstpointer),params);
  return(FALSE);
}

static gboolean on_delayed_song_level_change(GstClock *clock,GstClockTime time,GstClockID id,gpointer user_data) {
  // the callback is called from a clock thread
  if(GST_CLOCK_TIME_IS_VALID(time))
    g_idle_add(on_delayed_idle_song_level_change,user_data);
  else {
    gconstpointer * const params=(gconstpointer *)user_data;
    GstMessage *message=(GstMessage *)params[1];
    gst_message_unref(message);
    g_slice_free1(2*sizeof(gconstpointer),user_data);
  }
  return(TRUE);
}


static void on_song_level_change(GstBus * bus, GstMessage * message, gpointer user_data) {
  const GstStructure *structure=gst_message_get_structure(message);
  const GQuark name_id=gst_structure_get_name_id(structure);

  if(name_id==bus_msg_level_quark) {
    BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);
    GstElement *level=GST_ELEMENT(GST_MESSAGE_SRC(message));

    // check if its our element (we can have multiple level meters)
    if(level==self->priv->level) {
      GstClockTime timestamp, duration;
      GstClockTime waittime=GST_CLOCK_TIME_NONE;

      if(gst_structure_get_clock_time (structure, "running-time", &timestamp) &&
        gst_structure_get_clock_time (structure, "duration", &duration)) {
        /* wait for middle of buffer */
        waittime=timestamp+duration/2;
      }
      else if(gst_structure_get_clock_time (structure, "endtime", &timestamp)) {
        /* level send endtime as stream_time and not as running_time */
        waittime=gst_segment_to_running_time(&GST_BASE_TRANSFORM(level)->segment, GST_FORMAT_TIME, timestamp);
      }
      if(GST_CLOCK_TIME_IS_VALID(waittime)) {
        gconstpointer *params=(gconstpointer *)g_slice_alloc(2*sizeof(gconstpointer));
        GstClockID clock_id;
        GstClockTime basetime=gst_element_get_base_time(level);

        //GST_WARNING("target %"GST_TIME_FORMAT" %"GST_TIME_FORMAT,
        //  GST_TIME_ARGS(endtime),GST_TIME_ARGS(waittime));

        params[0]=(gpointer)self;
        params[1]=(gpointer)gst_message_ref(message);
        g_mutex_lock(self->priv->lock);
        g_object_add_weak_pointer((gpointer)self,(gpointer *)&params[0]);
        g_mutex_unlock(self->priv->lock);
        clock_id=gst_clock_new_single_shot_id(self->priv->clock,waittime+basetime);
        if(gst_clock_id_wait_async(clock_id,on_delayed_song_level_change,(gpointer)params)!=GST_CLOCK_OK) {
          gst_message_unref(message);
          g_slice_free1(2*sizeof(gconstpointer),params);
        }
        gst_clock_id_unref(clock_id);
      }
    }
  }
}

static void on_song_level_negotiated(GstBus * bus, GstMessage * message, gpointer user_data) {
  const GstStructure *structure=gst_message_get_structure(message);
  const GQuark name_id=gst_structure_get_name_id(structure);

  // receive message from on_channels_negotiated()
  if(name_id==bus_msg_level_caps_changed_quark) {
    BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);
    gint i,channels;

    gst_structure_get_int(structure,"channels",&channels);
    GST_INFO("received application bus message: channel=%d",channels);

    for(i=0;i<channels;i++) {
      gtk_widget_show(GTK_WIDGET(self->priv->vumeter[i]));
    }
    for(i=channels;i<MAX_VUMETER;i++) {
      gtk_widget_hide(GTK_WIDGET(self->priv->vumeter[i]));
    }
  }
}


static gdouble volume_slider2real(gdouble lin)
{
  if (lin <= 0)
    return 0.0;
  return pow(10000.0, lin - 1.0);
}

static gdouble volume_real2slider(gdouble logv)
{
  if (logv <= (1.0 / 10000.0))
    return 0.0;
  return log(logv) / log(10000.0) + 1.0;
}

static void on_song_volume_slider_change(GtkRange *range,gpointer user_data) {
  BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);
  gdouble nvalue,ovalue;

  g_assert(self->priv->gain);
  g_assert(self->priv->volume);

  // get value from HScale and change volume
  nvalue=volume_slider2real(gtk_range_get_value(GTK_RANGE(self->priv->volume)));
  g_object_get(self->priv->gain,"master-volume",&ovalue,NULL);
  if(fabs(nvalue-ovalue)>0.000001) {
    GST_DEBUG("volume-slider has changed : %f->%f",ovalue,nvalue);
    g_signal_handlers_block_matched(self->priv->volume,G_SIGNAL_MATCH_FUNC|G_SIGNAL_MATCH_DATA,0,0,NULL,on_song_volume_changed,(gpointer)self);
    g_object_set(self->priv->gain,"master-volume",nvalue,NULL);
    g_signal_handlers_unblock_matched(self->priv->volume,G_SIGNAL_MATCH_FUNC|G_SIGNAL_MATCH_DATA,0,0,NULL,on_song_volume_changed,(gpointer)self);
    bt_edit_application_set_song_unsaved(self->priv->app);
  }
  /*else {
    GST_WARNING("IGN volume-slider has changed : %f->%f",ovalue,nvalue);
  }*/
}

static gboolean on_song_volume_slider_press_event(GtkWidget *widget, GdkEventButton *event, gpointer user_data) {
  if(event->type == GDK_BUTTON_PRESS) {
    if(event->button == 1) {
      BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);
      GstElement *machine;
      GstController *ctrl;

      g_object_get(self->priv->master,"machine",&machine,NULL);
      if((ctrl=gst_object_get_controller(G_OBJECT(machine)))) {
        gst_controller_set_property_disabled(ctrl,"master-volume",TRUE);
      }
      g_object_unref(machine);
    }
  }
  return(FALSE);
}

static gboolean on_song_volume_slider_release_event(GtkWidget *widget, GdkEventButton *event, gpointer user_data) {
  if(event->button == 1 && event->type == GDK_BUTTON_RELEASE) {
    BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);
    GstElement *machine;
    GstController *ctrl;

    g_object_get(self->priv->master,"machine",&machine,NULL);
    if((ctrl=gst_object_get_controller(G_OBJECT(self->priv->master)))) {
      // update the default value at ts=0
      bt_machine_set_global_param_default(self->priv->master,
        bt_machine_get_global_param_index(self->priv->master,"master-volume",NULL));

      /*
       * @todo: it should actualy postpone the disable to the next timestamp
       * (not possible right now).
       *
       * @idea: can we have a livecontrolsource that subclasses interpolationcs
       * - when enabling, if would need to delay the enabled to the next control-point
       * - it would need to peek at the control-point list :/
       */
      gst_controller_set_property_disabled(ctrl,"master-volume",FALSE);
    }
    g_object_unref(machine);
  }
  return(FALSE);
}

static void on_song_volume_changed(GstElement *gain,GParamSpec *arg,gpointer user_data) {
  BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);
  gdouble nvalue,ovalue;

  g_assert(self->priv->gain);
  g_assert(self->priv->volume);

  // get value from Element and change HScale
  g_object_get(self->priv->gain,"master-volume",&nvalue,NULL);
  nvalue=volume_real2slider(nvalue);
  ovalue=gtk_range_get_value(GTK_RANGE(self->priv->volume));
  if(fabs(nvalue-ovalue)>0.000001) {
    GST_DEBUG("volume-slider notify : %f",nvalue);

    g_signal_handlers_block_matched(self->priv->gain,G_SIGNAL_MATCH_FUNC|G_SIGNAL_MATCH_DATA,0,0,NULL,on_song_volume_slider_change,(gpointer)self);
    gtk_range_set_value(GTK_RANGE(self->priv->volume),nvalue);
    g_signal_handlers_unblock_matched(self->priv->gain,G_SIGNAL_MATCH_FUNC|G_SIGNAL_MATCH_DATA,0,0,NULL,on_song_volume_slider_change,(gpointer)self);
  }
  /*else {
    GST_WARNING("IGN volume-slider notify : %f",nvalue);
  }*/
}


static void on_channels_negotiated(GstPad *pad,GParamSpec *arg,gpointer user_data) {
  GstCaps *caps;

  if((caps=(GstCaps *)gst_pad_get_negotiated_caps(pad))) {
    BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);
    gint channels;
    GstStructure *structure;
    GstMessage *message;
    GstElement *bin;

    channels=gst_caps_get_channels(caps);
    gst_caps_unref(caps);
    GST_INFO("!!!  input level src has %d output channels",channels);

    // post a message to the bus (we can't do gtk+ stuff here)
    structure = gst_structure_new ("level-caps-changed",
        "channels", G_TYPE_INT, channels, NULL);
    message = gst_message_new_application (NULL, structure);

    g_object_get(self->priv->app,"bin",&bin,NULL);
    gst_element_post_message(bin,message);
    gst_object_unref(bin);
  }
}

static void on_song_unsaved_changed(const GObject *object,GParamSpec *arg,gpointer user_data) {
  BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);
  gboolean unsaved=bt_edit_application_is_song_unsaved(self->priv->app);

  gtk_widget_set_sensitive(self->priv->save_button,unsaved);
}

static void on_sequence_loop_notify(const BtSequence *sequence,GParamSpec *arg,gpointer user_data) {
  BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);
  gboolean loop;

  g_object_get((gpointer)sequence,"loop",&loop,NULL);
  g_signal_handlers_block_matched(self->priv->loop_button,G_SIGNAL_MATCH_FUNC|G_SIGNAL_MATCH_DATA,0,0,NULL,on_toolbar_loop_toggled,(gpointer)self);
  gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(self->priv->loop_button),loop);
  g_signal_handlers_unblock_matched(self->priv->loop_button,G_SIGNAL_MATCH_FUNC|G_SIGNAL_MATCH_DATA,0,0,NULL,on_toolbar_loop_toggled,(gpointer)self);
}

static void on_song_changed(const BtEditApplication *app,GParamSpec *arg,gpointer user_data) {
  BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);
  BtSong *song;
  BtSequence *sequence;
  GstBin *bin;
  //gboolean loop;

  GST_INFO("song has changed : app=%p, toolbar=%p",app,user_data);

  g_object_get(self->priv->app,"song",&song,NULL);
  if(!song) return;

  // get the audio_sink (song->master is a bt_sink_machine) if there is one already
  g_object_try_weak_unref(self->priv->master);
  g_object_get(song,"master",&self->priv->master,"sequence",&sequence,"bin",&bin,NULL);

  if(self->priv->master) {
    GstPad *pad;
    GstBus *bus;

    GST_INFO("connect to input-level : song=%p,  master=%p (ref_ct=%d)",song,self->priv->master,G_OBJECT_REF_COUNT(self->priv->master));
    g_object_try_weak_ref(self->priv->master);

    // get the input_level and input_gain properties from audio_sink
    g_object_try_weak_unref(self->priv->gain);
    g_object_try_weak_unref(self->priv->level);
    g_object_get(self->priv->master,"input-post-level",&self->priv->level,"machine",&self->priv->gain,NULL);
    g_object_try_weak_ref(self->priv->gain);
    g_object_try_weak_ref(self->priv->level);

    // connect bus signals
    bus=gst_element_get_bus(GST_ELEMENT(bin));
    g_signal_connect(bus, "message::error", G_CALLBACK(on_song_error), (gpointer)self);
    g_signal_connect(bus, "message::warning", G_CALLBACK(on_song_warning), (gpointer)self);
    g_signal_connect(bus, "message::element", G_CALLBACK(on_song_level_change), (gpointer)self);
    g_signal_connect(bus, "message::application", G_CALLBACK(on_song_level_negotiated), (gpointer)self);
    gst_object_unref(bus);

    if(self->priv->clock) gst_object_unref(self->priv->clock);
    self->priv->clock=gst_pipeline_get_clock (GST_PIPELINE(bin));

    // get the pad from the input-level and listen there for channel negotiation
    g_assert(GST_IS_ELEMENT(self->priv->level));
    if((pad=gst_element_get_static_pad(self->priv->level,"src"))) {
      g_signal_connect(pad,"notify::caps",G_CALLBACK(on_channels_negotiated),(gpointer)self);
      gst_object_unref(pad);
    }

    g_assert(GST_IS_ELEMENT(self->priv->gain));
    // get the current input_gain and adjust volume widget
    on_song_volume_changed(self->priv->gain,NULL,(gpointer)self);

    // connect slider changed and volume changed events
    g_signal_connect(self->priv->volume,"value_changed",G_CALLBACK(on_song_volume_slider_change),(gpointer)self);
    g_signal_connect(self->priv->volume,"button-press-event",G_CALLBACK(on_song_volume_slider_press_event),(gpointer)self);
    g_signal_connect(self->priv->volume,"button-release-event",G_CALLBACK(on_song_volume_slider_release_event),(gpointer)self);
    //g_signal_connect(self->priv->gain ,"notify::volume",G_CALLBACK(on_song_volume_changed),(gpointer)self);
    g_signal_connect(self->priv->gain ,"notify::master-volume",G_CALLBACK(on_song_volume_changed),(gpointer)self);

    gst_object_unref(self->priv->gain);
    gst_object_unref(self->priv->level);
    g_object_unref(self->priv->master);
  }
  else {
    GST_WARNING("failed to get the master element of the song");
  }
  g_signal_connect(song,"notify::is-playing",G_CALLBACK(on_song_is_playing_notify),(gpointer)self);
  on_sequence_loop_notify(sequence,NULL,(gpointer)self);
  g_signal_connect(sequence,"notify::loop",G_CALLBACK(on_sequence_loop_notify),(gpointer)self);
  //-- release the references
  gst_object_unref(bin);
  g_object_unref(sequence);
  g_object_unref(song);
}

static void on_toolbar_style_changed(const BtSettings *settings,GParamSpec *arg,gpointer user_data) {
  BtMainToolbar *self=BT_MAIN_TOOLBAR(user_data);
  gchar *toolbar_style;

  g_object_get((gpointer)settings,"toolbar-style",&toolbar_style,NULL);
  if(!BT_IS_STRING(toolbar_style)) return;

  GST_INFO("!!!  toolbar style has changed '%s'", toolbar_style);
  gtk_toolbar_set_style(GTK_TOOLBAR(self),gtk_toolbar_get_style_from_string(toolbar_style));
  g_free(toolbar_style);
}

//-- helper methods

static void bt_main_toolbar_init_ui(const BtMainToolbar *self) {
  BtSettings *settings;
  GtkWidget *tool_item;
  GtkWidget *box, *child;
  gulong i;
  BtChangeLog *change_log;

  gtk_widget_set_name(GTK_WIDGET(self),"main toolbar");

  //-- file controls

  tool_item=GTK_WIDGET(gtk_tool_button_new_from_stock(GTK_STOCK_NEW));
  gtk_widget_set_name(tool_item,"New");
  gtk_tool_item_set_tooltip_text (GTK_TOOL_ITEM(tool_item),_("Prepare a new empty song"));
  gtk_toolbar_insert(GTK_TOOLBAR(self),GTK_TOOL_ITEM(tool_item),-1);
  g_signal_connect(tool_item,"clicked",G_CALLBACK(on_toolbar_new_clicked),(gpointer)self);

  tool_item=GTK_WIDGET(gtk_tool_button_new_from_stock(GTK_STOCK_OPEN));
  gtk_widget_set_name(tool_item,"Open");
  gtk_tool_item_set_tooltip_text (GTK_TOOL_ITEM(tool_item),_("Load a new song"));

  gtk_toolbar_insert(GTK_TOOLBAR(self),GTK_TOOL_ITEM(tool_item),-1);
  g_signal_connect(tool_item,"clicked",G_CALLBACK(on_toolbar_open_clicked),(gpointer)self);

  tool_item=GTK_WIDGET(gtk_tool_button_new_from_stock(GTK_STOCK_SAVE));
  gtk_widget_set_name(tool_item,"Save");
  gtk_tool_item_set_tooltip_text (GTK_TOOL_ITEM(tool_item),_("Save this song"));
  gtk_toolbar_insert(GTK_TOOLBAR(self),GTK_TOOL_ITEM(tool_item),-1);
  g_signal_connect(tool_item,"clicked",G_CALLBACK(on_toolbar_save_clicked),(gpointer)self);
  self->priv->save_button=tool_item;

  gtk_toolbar_insert(GTK_TOOLBAR(self),gtk_separator_tool_item_new(),-1);

  //-- media controls

  tool_item=GTK_WIDGET(gtk_tool_button_new_from_stock(GTK_STOCK_MEDIA_REWIND));
  gtk_tool_item_set_tooltip_text(GTK_TOOL_ITEM(tool_item),_("Rewind playback position of this song"));
  gtk_toolbar_insert(GTK_TOOLBAR(self),GTK_TOOL_ITEM(tool_item),-1);
  child=gtk_bin_get_child(GTK_BIN(tool_item));
  gtk_widget_add_events(child,GDK_BUTTON_PRESS_MASK|GDK_BUTTON_RELEASE_MASK);
  g_signal_connect(child,"button-press-event",G_CALLBACK(on_toolbar_rewind_pressed),(gpointer)self);
  g_signal_connect(child,"button-release-event",G_CALLBACK(on_toolbar_rewind_released),(gpointer)self);

  tool_item=GTK_WIDGET(gtk_toggle_tool_button_new_from_stock(GTK_STOCK_MEDIA_PLAY));
  gtk_widget_set_name(tool_item,"Play");
  gtk_tool_item_set_tooltip_text(GTK_TOOL_ITEM(tool_item),_("Play this song"));
  gtk_toolbar_insert(GTK_TOOLBAR(self),GTK_TOOL_ITEM(tool_item),-1);
  g_signal_connect(tool_item,"clicked",G_CALLBACK(on_toolbar_play_clicked),(gpointer)self);
  self->priv->play_button=tool_item;

  tool_item=GTK_WIDGET(gtk_tool_button_new_from_stock(GTK_STOCK_MEDIA_FORWARD));
  gtk_tool_item_set_tooltip_text(GTK_TOOL_ITEM(tool_item),_("Forward playback position of this song"));
  gtk_toolbar_insert(GTK_TOOLBAR(self),GTK_TOOL_ITEM(tool_item),-1);
  child=gtk_bin_get_child(GTK_BIN(tool_item));
  gtk_widget_add_events(child,GDK_BUTTON_PRESS_MASK|GDK_BUTTON_RELEASE_MASK);
  g_signal_connect(child,"button-press-event",G_CALLBACK(on_toolbar_forward_pressed),(gpointer)self);
  g_signal_connect(child,"button-release-event",G_CALLBACK(on_toolbar_forward_released),(gpointer)self);

  tool_item=GTK_WIDGET(gtk_tool_button_new_from_stock(GTK_STOCK_MEDIA_STOP));
  gtk_widget_set_name(tool_item,"Stop");
  gtk_tool_item_set_tooltip_text(GTK_TOOL_ITEM(tool_item),_("Stop playback of this song"));
  gtk_toolbar_insert(GTK_TOOLBAR(self),GTK_TOOL_ITEM(tool_item),-1);
  g_signal_connect(tool_item,"clicked",G_CALLBACK(on_toolbar_stop_clicked),(gpointer)self);
  gtk_widget_set_sensitive(tool_item,FALSE);
  self->priv->stop_button=tool_item;

  tool_item=GTK_WIDGET(gtk_toggle_tool_button_new());
  gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(tool_item),gtk_image_new_from_filename("stock_repeat.png"));
  gtk_tool_button_set_label(GTK_TOOL_BUTTON(tool_item),_("Loop"));
  gtk_widget_set_name(tool_item,"Loop");
  gtk_tool_item_set_tooltip_text(GTK_TOOL_ITEM(tool_item),_("Toggle looping of playback"));
  gtk_toolbar_insert(GTK_TOOLBAR(self),GTK_TOOL_ITEM(tool_item),-1);
  g_signal_connect(tool_item,"toggled",G_CALLBACK(on_toolbar_loop_toggled),(gpointer)self);
  self->priv->loop_button=tool_item;

  gtk_toolbar_insert(GTK_TOOLBAR(self),gtk_separator_tool_item_new(),-1);

  //-- volume level and control

  box=gtk_vbox_new(FALSE,0);
  gtk_container_set_border_width(GTK_CONTAINER(box),2);
#ifndef USE_HILDON
  gtk_widget_set_size_request(GTK_WIDGET(box),250,-1);
#else
  gtk_widget_set_size_request(GTK_WIDGET(box),200,-1);
#endif
  // add gtk_vumeter widgets and update from level_callback
  for(i=0;i<MAX_VUMETER;i++) {
    self->priv->vumeter[i]=GTK_VUMETER(gtk_vumeter_new(FALSE));
    // @idea have distinct tooltips with channel names
    gtk_widget_set_tooltip_text(GTK_WIDGET(self->priv->vumeter[i]),_("playback volume"));
    gtk_vumeter_set_min_max(self->priv->vumeter[i], LOW_VUMETER_VAL, 0);
    gtk_vumeter_set_levels(self->priv->vumeter[i], LOW_VUMETER_VAL, LOW_VUMETER_VAL);
    // no falloff in widget, we have falloff in GstLevel
    //gtk_vumeter_set_peaks_falloff(self->priv->vumeter[i], GTK_VUMETER_PEAKS_FALLOFF_MEDIUM);
    gtk_vumeter_set_scale(self->priv->vumeter[i], GTK_VUMETER_SCALE_LINEAR);
    gtk_widget_set_no_show_all(GTK_WIDGET(self->priv->vumeter[i]),TRUE);
    if(i<DEF_VUMETER) {
      gtk_widget_show(GTK_WIDGET(self->priv->vumeter[i]));
    }
    else {
      gtk_widget_hide(GTK_WIDGET(self->priv->vumeter[i]));
    }
    gtk_box_pack_start(GTK_BOX(box),GTK_WIDGET(self->priv->vumeter[i]),TRUE,TRUE,0);
  }

  // add gain-control
  self->priv->volume=GTK_SCALE(gtk_hscale_new_with_range(/*min=*/0.0,/*max=*/1.0,/*step=*/0.01));
  gtk_widget_set_tooltip_text(GTK_WIDGET(self->priv->volume),_("Change playback volume"));
  gtk_scale_set_draw_value(self->priv->volume,FALSE);
  //gtk_range_set_update_policy(GTK_RANGE(self->priv->volume),GTK_UPDATE_DELAYED);
#ifndef USE_HILDON
  gtk_box_pack_start(GTK_BOX(box),GTK_WIDGET(self->priv->volume),TRUE,TRUE,0);
#else
  gtk_widget_set_size_request(GTK_WIDGET(self->priv->volume),200,-1);
  tool_item=GTK_WIDGET(gtk_tool_item_new());
  gtk_widget_set_name(tool_item,"volume-control");
  gtk_container_add(GTK_CONTAINER(tool_item),self->priv->volume);
  gtk_toolbar_insert(GTK_TOOLBAR(self),GTK_TOOL_ITEM(tool_item),-1);
#endif
  gtk_widget_show_all(GTK_WIDGET(box));

  tool_item=GTK_WIDGET(gtk_tool_item_new());
  gtk_widget_set_name(tool_item,"volume");
  gtk_container_add(GTK_CONTAINER(tool_item),box);
  gtk_toolbar_insert(GTK_TOOLBAR(self),GTK_TOOL_ITEM(tool_item),-1);

  // register event handlers
  g_signal_connect(self->priv->app, "notify::song", G_CALLBACK(on_song_changed), (gpointer)self);
  g_signal_connect(self->priv->app, "notify::unsaved", G_CALLBACK(on_song_unsaved_changed), (gpointer)self);

  change_log=bt_change_log_new();
  g_signal_connect(change_log, "notify::can-undo", G_CALLBACK(on_song_unsaved_changed), (gpointer)self);
  g_object_unref(change_log);
  
  // let settings control toolbar style
  g_object_get(self->priv->app,"settings",&settings,NULL);
  on_toolbar_style_changed(settings,NULL,(gpointer)self);
  g_signal_connect(settings, "notify::toolbar-style", G_CALLBACK(on_toolbar_style_changed), (gpointer)self);
  g_object_unref(settings);
}

//-- constructor methods

/**
 * bt_main_toolbar_new:
 *
 * Create a new instance
 *
 * Returns: the new instance
 */
BtMainToolbar *bt_main_toolbar_new(void) {
  BtMainToolbar *self;

  self=BT_MAIN_TOOLBAR(g_object_new(BT_TYPE_MAIN_TOOLBAR,NULL));
  bt_main_toolbar_init_ui(self);
  return(self);
}

//-- methods


//-- class internals

static void bt_main_toolbar_dispose(GObject *object) {
  BtMainToolbar *self = BT_MAIN_TOOLBAR(object);
	BtSong *song;

  return_if_disposed();
  self->priv->dispose_has_run = TRUE;

  GST_DEBUG("!!!! self=%p",self);

  g_object_get(self->priv->app,"song",&song,NULL);
  if(song) {
    GstBin *bin;
    GstBus *bus;

    GST_DEBUG("disconnect handlers from song=%p",song);

    g_signal_handlers_disconnect_matched(song,G_SIGNAL_MATCH_FUNC,0,0,NULL,on_song_is_playing_notify,NULL);

    g_object_get(song,"bin", &bin, NULL);
    bus=gst_element_get_bus(GST_ELEMENT(bin));
    g_signal_handlers_disconnect_matched(bus,G_SIGNAL_MATCH_FUNC,0,0,NULL,on_song_error,NULL);
    g_signal_handlers_disconnect_matched(bus,G_SIGNAL_MATCH_FUNC,0,0,NULL,on_song_warning,NULL);
    g_signal_handlers_disconnect_matched(bus,G_SIGNAL_MATCH_FUNC,0,0,NULL,on_song_level_change,NULL);
    g_signal_handlers_disconnect_matched(bus,G_SIGNAL_MATCH_FUNC,0,0,NULL,on_song_level_negotiated,NULL);
    gst_object_unref(bus);
    gst_object_unref(bin);
    g_object_unref(song);
  }

  g_object_try_weak_unref(self->priv->master);
  g_object_try_weak_unref(self->priv->gain);
  g_object_try_weak_unref(self->priv->level);

  if(self->priv->clock) gst_object_unref(self->priv->clock);
  
  g_object_unref(self->priv->app);

  G_OBJECT_CLASS(bt_main_toolbar_parent_class)->dispose(object);
}

static void bt_main_toolbar_finalize(GObject *object) {
  BtMainToolbar *self = BT_MAIN_TOOLBAR(object);

  GST_DEBUG("!!!! self=%p",self);
  g_mutex_free(self->priv->lock);

  G_OBJECT_CLASS(bt_main_toolbar_parent_class)->finalize(object);
}

static void bt_main_toolbar_init(BtMainToolbar *self) {
  self->priv = G_TYPE_INSTANCE_GET_PRIVATE(self, BT_TYPE_MAIN_TOOLBAR, BtMainToolbarPrivate);
  GST_DEBUG("!!!! self=%p",self);
  self->priv->app = bt_edit_application_new();
  self->priv->lock=g_mutex_new();
  self->priv->playback_rate=1.0;
}

static void bt_main_toolbar_class_init(BtMainToolbarClass *klass) {
  GObjectClass *gobject_class = G_OBJECT_CLASS(klass);

  bus_msg_level_quark=g_quark_from_static_string("level");
  bus_msg_level_caps_changed_quark=g_quark_from_static_string("level-caps-changed");

  g_type_class_add_private(klass,sizeof(BtMainToolbarPrivate));

  gobject_class->dispose      = bt_main_toolbar_dispose;
  gobject_class->finalize     = bt_main_toolbar_finalize;

}

