(*:Name: LabelContour *)

(*:Title: LabelContour *)

(*:Author: Tom Wickham-Jones*)

(*:Package Version: 1.0 *)

(*:Mathematica Version: 2.2 *)

(*:Summary:
	This package provides functions to label contour plots.
*)


(*:History:
	Created summer 1993 by Tom Wickham-Jones.

	This package is described in the book
	Mathematica Graphics: Techniques and Applications.
	Tom Wickham-Jones, TELOS/Springer-Verlag 1994.

*)

(*:Warnings:
*)


BeginPackage[ 
		"ExtendGraphics`LabelContour`", 
		"Graphics`Legend`","Utilities`FilterOptions`"]


LabelContourLines::usage = 
	"LabelContourLines[ cont, opts] displays the contour plot
	 cont with a label on each contour line giving its height."

LabelPlacement::usage =
	"LabelPlacement is an option of LabelContourLines that
	 determines how the labels are placed.  A value
	 of Center will use the mid-point of each contour line.
	 A value of Automatic will attempt to find labels that
	 are spaced away from each other."

PointFactor::usage =
	"PointFactor is an option of LabelContourLines that is used
	 when the LabelPlacement is set to Automatic.  It takes a
	 positive number that determines the weighting function
	 that is used to find the points at which labels are placed.
	 A value of 1 means that the same weight is given in the x
	 and y directions.  A value of less than 1 spreads labels
	 in the x direction while a value of greater than 1 spreads
	 them out in the y direction."

PointSkip::usage =
	"PointSkip is an option of LabelContourLines that is used
	 when the LabelPlacement is not set to be Automatic.  It
	 determines how many points are examined as candidates
	 for placing labels.   A setting of 4 will ensure that
	 every fourth point along each line is examined."


LabelContourLegend::usage = 
	"LabelContourLegend[ cont, opts] displays the contour plot cont
	 with a legend containing the values of the heights of the contours."

HeightName::usage =
	"HeightName is an option of LabelContourLegend that determines
	 the text label describing the name of the height."

LegendSize::usage =
	"LegendSize is an option of various graphics functions that determines
	 the size of the legend."

LegendPosition::usage =
	"LegendPosition is an option of various graphics functions that determines
	 the position of the legend."

LabelFont::usage =
	"LabelFont is an option of LabelContourLines and LabelContourLegend
	 that describes the font to be used for labels."


ContourDisplayFunction::usage =
	"ContourDisplayFunction is a display function for contour plots
	 that causes automatic label placement."
	

Begin[ "`Private`"]

ContourDisplayFunction[ g:ContourGraphics[ data_, opts___]] :=
  Block[{shade, topts, copts, fun, lopts},
    shade = ContourShading /. 
                Flatten[ {opts}] /. 
                Options[ ContourGraphics] ;
	topts = Flatten[ {opts}] ;
	copts = FilterOptions[ ContourGraphics, Apply[ Sequence, topts]] ;
	fun = If[ shade === True, LabelContourLegend, LabelContourLines] ;
	lopts = FilterOptions[ fun, Apply[ Sequence, topts]] ;
	fun[ ContourGraphics[ data, copts], lopts] ;
    g]




If[ ($VersionNumber > 2.2 && $ReleaseNumber > 0) ||
     $VersionNumber > 3.0,
	textFun[ Text[ t_, {x_, y_, ang_}], {font_,size_}, form_] :=
		If[ PretendToBeOld === True,
			Text[ FontForm[ t, font], {x,y}, 1. {Sin[ang],Cos[ang]}],
			Text[ t, {x,y}, 
				TextStyle-> {FontName->font, FontSize->size},
				FormatType -> form,
				Background -> Automatic]],
	textFun[ Text[ t_, {x_, y_, ang_}], font_, form_] :=
		Text[ FontForm[ t, font], {x,y}, 1. {Sin[ang],Cos[ang]}]
		]
	


Options[ LabelContourLines] =
	{
	LabelFont -> {"Times-Roman", 8},
	(* FormatType -> OutputForm, *)
	LabelPlacement -> Center,
	PointFactor -> 1,
	PointSkip -> 4
	}	

GetFont[ {name_String, size_}, def_] :=
	{ name, CheckSize[ size]}

GetFont[ name_String, def_] :=
	If[ Length[ def] === 2,
			{name, CheckSize[ Last[ def]]},
			{name, 6}]

GetFont[ size_, def_] :=
	If[ Length[ def] === 2,
			{First[ def], CheckSize[size]},
			{"Courier", CheckSize[ size]}]
	
CheckSize[ x_] := If[ Positive[ x], x, 6, 6]

CheckPlace[ x_] := Switch[ x, Center, Center, Automatic, Automatic, _, Center]

CheckFactor[ x_] := If[ NumberQ[x] && x > 0, x, 1]

If[ Positive[ x], x, Automatic, Automatic]		

LabelContourLines[ g:ContourGraphics[ data_, opts___], lopts___] :=
    Block[{conts, style, shade, factor, form,
    	   c, prims, opt, levs, text, font,
	   	   ostyle, place, skip, fact, max},

		opt = Join[ {lopts}, Options[ LabelContourLines]] ;
	    font = GetFont[ LabelFont /. opt, 
							LabelFont /. Options[ LabelContourLines]] ;
	    form = (* FormatType /. opt ;*) OutputForm ;
		place = CheckPlace[ LabelPlacement /. opt] ;
		skip = PointSkip /. opt ;
		fact = CheckFactor[ PointFactor /. opt] ;
		opt = Join[ Flatten[{opts}], Options[ ContourGraphics]] ;
		ostyle = ContourStyle /. opt ;
		shade = ContourShading /. opt ;
		conts = Contours /. opt ;
		conts = FindContours[ conts, 
					Last[ FullOptions[ g, PlotRange]]] ;
    	max = Max[ Abs[ conts]] ;
		style = Table[ {AbsoluteThickness[ i]}, {i,Length[ conts]}] ;

		c = Show[ g,
			Contours -> conts,
			ContourStyle -> style, 
			DisplayFunction -> Identity] ;
		c = Graphics[ c] ;
		prims = First[ c] ;
		opt = List @@ Rest[ c] ;
		levs = If[ shade,
		    levs = Drop[ prims, 1] ;
	 	    Map[ (# /. 
		         {__, {___, AbsoluteThickness[ t_], ___, Line[ pts_]}} -> 
			 		{t, pts})&,
			  levs]
		(* else *) ,
	 	    Map[ (# /. 
		         {___, AbsoluteThickness[ t_], ___, Line[ pts_]} -> 
			 		{t, pts})&,
			  prims]] ;
		ostyle = FixStyle[ ostyle, Length[ conts]] ;
		prims = Map[ (# /. 
		    AbsoluteThickness[ t_] :> 
		    Sequence @@ Part[ ostyle, t])&, prims] ; 
		
		If[ place === Center,
				levs = Map[ { Part[conts, First[#]], Last[#]}&, levs] ;
				text = Map[ MidTextCalc[#,max]&, levs]
			
			, (* else *)

			levs = Map[ FixLevs[ #, skip]&, levs] ;
			text = Table[ TextCalc[ conts, levs, i, fact, max], 
										{i, Length[ levs]}]
			] ;

		text = Map[ textFun[#, font, form]&, text] ;
		
		opt = opt /. (DisplayFunction -> Identity) -> 
								Options[ Graphics, DisplayFunction] ;
		Show[ 
	    	Graphics[ {prims, text}, 
				FilterOptions[ Graphics, lopts], opt]]
		]

(*
MidTextCalc[ {cont_, line_}, max_] :=
	Text[ NumberFunction[ cont, max], GetLinePosition[ line]]

GetLinePosition[ line_] :=
		If[ Length[ line] === 2,
			Apply[ Plus, line]/2,
			Part[ line, Ceiling[ Length[ line]/2.]]]
*)

MidTextCalc[ {cont_, line_}, max_] :=
	Text[ NumberFunction[ cont, max], GetLinePosition[ line]]

GetLinePosition[ line_] :=
	Block[{pos},
		If[ Length[ line] === 2,
			Append[ Apply[ Plus, line]/2, Apply[ AngCal, line]],
			pos = Ceiling[ Length[ line]/2.] ;
			Append[ Part[ line, pos], AngCal[ line[[pos-1]], line[[pos+1]]]]
			]
		]

AngCal[ p1_, p2_] :=
	If[ Chop[ p1- p2] =!= {0, 0},
		N[ 2Pi - Apply[ ArcTan, p2-p1]],
		0.0]


FixLevs[ {num_, lev_}, fact_] :=
    Block[{len, nfact},
    	nfact = fact ;
    	len = Length[ lev] ;
	If[ len < nfact*4,
		nfact = Floor[ len/4]] ;
	If[ Head[ nfact] =!= Integer || nfact < 1,
		nfact = 1] ;
 	If[ len < 4,
		{num, lev},
		{num, Map[ First, Partition[ Take[ lev, {2, -2}], nfact]]}]
	]

FindMin[ pt_, test_, wght_] :=
	Min[ Map[ (tmp = (#-pt)*{1,wght}; tmp.tmp)&, test]]


FixDists[ inp_List] :=
    Block[{},
    	{coord, dists} = Transpose[ inp] ;
	If[ True === SameQ @@ coord, , Print[ "Mess Up in FixDists"]] ;
	{Part[ coord, 1], Times @@ dists}
	]


TextCalc[ vals_, levs_, num_, wght_, cmax_] :=
    Block[{test, cnum, dists, max, i, pos, cont, ang, rpos, lpos}, 
    	cont = Part[ levs, num] ;
		cnum = Part[ cont, 1] ;
		test = Select[ levs, (Abs[ Part[ #, 1]-cnum] === 1.)&] ;
		test = Map[ Part[ #, 2]&, test] ;
		cont = Part[ cont, 2] ;
		dists = 
	    	Map[ 
	        	Table[ 
		    	{
		    	i,
		    	FindMin[ Part[ cont, i], #, wght]
		    	}, {i, Length[ cont]}]&, test] ;
		dists = Transpose[ dists] ;
		dists = Map[ FixDists, dists] ;
		dists = dists * 
			Join[ {{1, 0.5}}, 
				Table[ {1, 1}, {Length[dists]-2}], {{1, 0.5}}] ;
		max = Part[ dists, 1] ;
		Do[ If[ Part[ max, 2] < Part[ dists, i, 2], max = Part[ dists,i]],
			{i, 2, Length[ dists]}] ;
		pos = Part[ max, 1] ;
		If[ Length[ cont] < 3,
			ang = Apply[ AngCal, cont],
			lpos = If[ pos == 1, 1, pos-1] ;
			rpos = If[ pos == Length[ cont], pos, pos+1] ;
			ang = AngCal[ cont[[lpos]], cont[[rpos]]]] ;
		pos = Append[ Part[ cont, pos], ang] ;
		Text[ NumberFunction[ Part[ vals, cnum], cmax], pos]
	]
	
FixStyle[ Automatic, num_] := FixStyle[ Thickness[ 0.001], num]

FixStyle[ style_, num_] := FixStyle[ {style}, num]

FixStyle[ style_List, num_] :=
	If[ ListQ[ style] && Length[ style] > 0 &&
	    ListQ[ First[ style]],
		Table[ First[ RotateLeft[ style, i-1]], {i,num}],
		Table[ style, {num}]]


FindContours[ x_List, {z1_, z2_}] := x

FindContours[ n_Integer /; n > 0, {z1_, z2_}] :=
    Block[{zinc},
	zinc = (z2 - z1)/ (n + 1) ;
	Table[ i, {i, z1 + zinc, z2 - zinc, zinc}]
	]


NumberFunction[ x_, max_] :=
    Block[{t, digs},
    	digs = Floor[ Log[ 10, max]] ;
	digs = N[ 10^digs] ;
	t = x/digs ;
	t = Floor[ t 100] ;
	t = t*digs/100 ;
	ToString[ NumberForm[ t, 3]]
	]


Options[ LabelContourLegend] =
	{
	LabelFont -> {"Times-Roman", 6},
	HeightName -> "z",
	LegendSize -> Automatic,
	LegendPosition -> Automatic
	}


LabelContourLegend[ g:ContourGraphics[ data_, opts___], lopts___] :=
    Block[{z1, z2, conts, cfun, shade, name, lsize, lpos,
    	   prims, text, font, size, opt},
    	opt = Join[ {lopts}, Options[ LabelContourLegend]] ;
	    font = GetFont[ LabelFont /. opt, 
					LabelFont /. Options[ LabelContourLegend]] ;
		size = Last[ font] ;
      	name = HeightName /. opt ;
      	lsize = LegendSize /. opt ;
      	lpos = LegendPosition /. opt ;
		If[ !StringQ[ name], name = ToString[ name]] ;
		{z1,z2} = Last[ FullOptions[ g, PlotRange]] ;
    	opt = Join[ Flatten[ {opts}], Options[ ContourGraphics]] ;
		conts = Contours /. opt ;
		cfun = ColorFunction /. opt ;

		If[ cfun === Automatic, cfun = GrayLevel] ;
		conts = FindContours[ conts, {z1, z2}] ;
        shade = ColorFunctionValues[ conts, z1, z2] ;
		shade = Map[ cfun, shade] ;
		max = Max[ Abs[ conts]] ;
		conts = Map[ NumberFunction[ #, max]&, conts] ;

		text = Table[ 
	            FontForm[ StringJoin[ "  < ", conts[[i]]], 
		    	      font],
		    {i, Length[ conts]}] ;
        text = Append[ text, 
			FontForm[ StringJoin[ "  > ", Last[ conts]], 
			          font]] ;
		prims = Graphics[ 
			{{Rectangle[ {0,0}, {1,1}]},
		  		Line[ {{0,0},{0,1},{1,1},{1,0},{0,0}}]}] ;
		prims = Table[ 
	         			{
		 				Insert[ prims, shade[[i]], {1,1,1}],
		 				text[[i]]
		 				}, {i, Length[ shade]}] ;
		prims = Reverse[ prims] ;

		lsize = FixLSize[ lsize, Length[ conts]+1, size] ;
		lpos = FixLPos[ lpos, lsize] ;
		ShowLegend[g, 
	           		{
		   			prims, 
					LegendLabel -> FontForm[ name, font],
	           		LegendPosition -> lpos,
                   	LegendSize -> lsize,
		   			LegendShadow -> {0,0},
					LegendBorder -> {Thickness[0.0001],GrayLevel[0]},
		   			LegendSpacing -> Automatic,
		   			LegendTextSpacing -> Automatic,
		   			FilterOptions[ Graphics, lopts]
		   			}]
	]


FixLPos[ {x_ /; NumberQ[x], y_ /; NumberQ[y]}, _] := {x, y}

FixLPos[ _, {x_, y_}] := {1.1, -y/2.}


FixLSize[ {x_ /; Positive[x], y_ /; Positive[ y]}, len_, size_] := {x, y}

If[ BookPrint === True,
	FixLSize[ _, clen_, size_] := {0.11, 0.027 clen} size,
	FixLSize[ _, clen_, size_] := {0.06, 0.02 clen} size]
	

ColorFunctionValues[ vals_List, z1_, z2_] :=
    Block[{p1, p2, pf, res},
    	p1 = (z1 + First[ vals])/2. ;
    	p2 = (z2 + Last[ vals])/2. ;
	pf = p2 - p1 ;
	res =
	  Table[ 
	      ((Part[ vals, i] + Part[ vals, i+1])/2. - p1)/pf,
              {i,Length[ vals] - 1}] ;
	Join[ {0}, res, {1}]
	]      


End[]

EndPackage[]


(* Examples:


For unshaded contours...

Get[ "ExtendGraphics`LabelContour`"]

c = ContourPlot[ x y, {x,-2,2}, {y,-2,2},
			ContourShading -> False]

LabelContourLines[ c]

c = ContourPlot[ x^2 -y^2, {x,-2,2},{y,-2,2},
			ContourShading -> False]

LabelContourLines[ c]

LabelContourLines[ c, LabelPlacement -> 1]


For shaded contours...

c = ContourPlot[ x^2 y + x y^2, {x, -2,2}, {y,-2,2}]

LabelContourLegend[ c]


*)







