/*
  This file is part of CDO. CDO is a collection of Operators to manipulate and analyse Climate model Data.

  Author: Uwe Schulzweida

*/

#include "cdo_task.h"

#include <cstdio>

namespace cdo
{

static void
task(cdo::Task *taskInfo)
{
  // cond.wait mutex must be locked before we can wait
  std::unique_lock<std::mutex> workLock(taskInfo->workMutex);

  // ensure boss is waiting
  taskInfo->bossMutex.lock();

  // signal to boss that setup is complete
  taskInfo->state = TaskState::IDLE;

  // wake-up signal
  taskInfo->bossCond.notify_one();
  taskInfo->bossMutex.unlock();

  while (1)
    {
      taskInfo->workCond.wait(workLock);

      if (TaskState::DIE == taskInfo->state) break;  // kill thread

      if (TaskState::IDLE == taskInfo->state) continue;  // accidental wake-up

      // do blocking task
      // printf("<worker> JOB start\n");
      taskInfo->result = taskInfo->routine(taskInfo->arg);
      // printf("<worker> JOB end\n");

      // ensure boss is waiting
      taskInfo->bossMutex.lock();

      // indicate that job is done
      taskInfo->state = TaskState::IDLE;

      // wake-up signal
      taskInfo->bossCond.notify_one();
      taskInfo->bossMutex.unlock();
    }
}

void
Task::start(void *(*taskRoutine)(void *), void *taskArg)
{
  // ensure worker is waiting
  workMutex.lock();

  // set job information & state
  this->routine = taskRoutine;
  this->arg = taskArg;
  this->state = TaskState::JOB;

  // wake-up signal
  workCond.notify_one();
  workMutex.unlock();
}

void *
Task::wait()
{
  while (1)
    {
      if (TaskState::IDLE == this->state) break;

      bossCond.wait(bossMutex);
    }

  return this->result;
}

Task::Task()
{
  bossMutex.lock();

  this->thread = std::thread(task, this);

  this->wait();
}

Task::~Task()
{
  // ensure the worker is waiting
  workMutex.lock();

  // printf("Task::delete: send DIE to <worker>\n");
  this->state = TaskState::DIE;

  // wake-up signal
  workCond.notify_one();
  workMutex.unlock();

  // wait for thread to exit
  this->thread.join();

  bossMutex.unlock();
}

}  // namespace cdo

#ifdef TEST_CDO_TASK
// g++ -g -Wall -O2 -DTEST_CDO_TASK cdo_task.cc

void *
mytask(void *arg)
{
  printf("run mytask\n");
  return nullptr;
}

int
main(int argc, char **argv)
{
  cdo::Task task;

  void *myarg = nullptr;
  void *myresult;

  task.start(mytask, myarg);

  myresult = task.wait();

  task.start(mytask, myarg);

  myresult = task.wait();

  return 0;
}
#endif
