/*
 *  Copyright 2004 The Apache Software Foundation
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.apache.commons.collections15.bidimap;

import junit.framework.Test;
import junit.textui.TestRunner;
import org.apache.commons.collections15.BidiMap;
import org.apache.commons.collections15.BulkTest;
import org.apache.commons.collections15.SortedBidiMap;
import org.apache.commons.collections15.comparators.ComparableComparator;
import org.apache.commons.collections15.comparators.ReverseComparator;

import java.io.*;
import java.util.*;

/**
 * JUnit tests.
 *
 * @author Matthew Hawthorne
 * @author Stephen Colebourne
 * @author Matt Hall, John Watkinson, Jonas Van Poucke
 * @version $Revision: 1.1 $ $Date: 2005/10/11 17:05:44 $
 */
public class TestDualTreeBidiMap2 extends AbstractTestSortedBidiMap {

    public static void main(String[] args) {
        TestRunner.run(suite());
    }

    public static Test suite() {
        return BulkTest.makeSuite(TestDualTreeBidiMap2.class);
    }

    public TestDualTreeBidiMap2(String testName) {
        super(testName);
    }

    public BidiMap makeEmptyBidiMap() {
        return DualTreeBidiMap.createTwoWayBidiMap(new ReverseComparator(ComparableComparator.getInstance()));
    }

    public Map makeConfirmedMap() {
        return new TreeMap(new ReverseComparator(ComparableComparator.getInstance()));
    }

    public void testComparator() {
        resetEmpty();
        SortedBidiMap bidi = (SortedBidiMap) map;
        assertNotNull(bidi.comparator());
        assertTrue(bidi.comparator() instanceof ReverseComparator);
    }

    public void testSerializeDeserializeCheckComparator() throws Exception {
        SortedBidiMap obj = (SortedBidiMap) makeObject();
        if (obj instanceof Serializable && isTestSerialization()) {
            ByteArrayOutputStream buffer = new ByteArrayOutputStream();
            ObjectOutputStream out = new ObjectOutputStream(buffer);
            out.writeObject(obj);
            out.close();

            ObjectInputStream in = new ObjectInputStream(new ByteArrayInputStream(buffer.toByteArray()));
            Object dest = in.readObject();
            in.close();

            SortedBidiMap bidi = (SortedBidiMap) dest;
            assertNotNull(obj.comparator());
            assertNotNull(bidi.comparator());
            assertTrue(bidi.comparator() instanceof ReverseComparator);
        }
    }

    public void testSortOrder() throws Exception {
        SortedBidiMap sm = (SortedBidiMap) makeFullMap();

        // Sort by the comparator used in the makeEmptyBidiMap() method
        List newSortedKeys = Arrays.asList(getSampleKeys());
        Collections.sort(newSortedKeys, new ReverseComparator(ComparableComparator.getInstance()));
        newSortedKeys = Collections.unmodifiableList(newSortedKeys);

        Iterator mapIter = sm.keySet().iterator();
        Iterator expectedIter = newSortedKeys.iterator();
        while (expectedIter.hasNext()) {
            Object expectedKey = expectedIter.next();
            Object mapKey = mapIter.next();
            assertNotNull("key in sorted list may not be null", expectedKey);
            assertNotNull("key in map may not be null", mapKey);
            assertEquals("key from sorted list and map must be equal", expectedKey, mapKey);
        }
    }

    public String getCompatibilityVersion() {
        return "3.Test2";
    }

    /**
     * Override to prevent infinite recursion of tests.
     */
    public String[] ignoredTests() {
        return new String[]{"TestDualTreeBidiMap2.bulkTestInverseMap.bulkTestInverseMap"};
    }

    //    public void testCreate() throws Exception {
    //        resetEmpty();
    //        writeExternalFormToDisk((java.io.Serializable) map, "D:/dev/collections15/data/test/DualTreeBidiMap.emptyCollection.version3.Test2.obj");
    //        resetFull();
    //        writeExternalFormToDisk((java.io.Serializable) map, "D:/dev/collections15/data/test/DualTreeBidiMap.fullCollection.version3.Test2.obj");
    //    }
}
