/*
------------------------------------------------------------------------------
denef - Decode NEF image files
Copyright (C) 2000 Daniel Stephens (daniel@cheeseplant.org)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
------------------------------------------------------------------------------
*/

#include <iostream.h>
#include <errno.h>
#include <string.h>
#include <netinet/in.h>

#include "metacam.h"

static const char *rcsid="$Id: tifffile.cc,v 1.3 2000/09/18 07:51:43 daniel Exp $";

tiffFile::tiffFile(istream &i, unsigned long ofs)
    : is(i), global_offset(ofs)
{
    Seek(0);
    unsigned char header[2];
    Get_Data(header,2);

    if ((header[0] == 'I') && (header[1] == 'I')) {
	bigendian = false;
    } else if ((header[0] == 'M') && (header[1] == 'M')) {
	bigendian = true;
    } else {
	cerr << "Unknown byte ordering - aborting" << endl;
	exit(2);
    }

    unsigned short checknum = Get_UShort();

    if (checknum != 42) {
	cerr << "Bad checknum: " << checknum << endl;
	exit(2);
    }
}

   
void
tiffFile::Seek(unsigned long ofs)
{
    is.seekg(ofs+global_offset, ios::beg);
}

void
tiffFile::Get_Data(unsigned char *buf, size_t bytes)
{
    is.read(buf, bytes);
    if (is.gcount() != bytes) {
	int err = errno;
	cerr << "Read failed: " << strerror(err) << endl;
	exit(2);
    }
}

unsigned char
tiffFile::Get_UByte()
{
    unsigned char b;
    Get_Data((unsigned char *)&b, 1);
    return b;
}

unsigned short
tiffFile::Get_UShort()
{
    unsigned short s;
    Get_Data((unsigned char *)&s, 2);
    if (bigendian) {return ntohs(s);}
    return s;
}

unsigned long
tiffFile::Get_ULong()
{
    unsigned long l;
    Get_Data((unsigned char *)&l, 4);
    if (bigendian) {return ntohl(l);}
    return l;
}

signed char
tiffFile::Get_SByte()
{
    signed char b;
    Get_Data((unsigned char *)&b, 1);
    return b;
}

signed short
tiffFile::Get_SShort()
{
    signed short s;
    Get_Data((unsigned char *)&s, 2);
    if (bigendian) {return ntohs(s);}
    return s;
}

signed long
tiffFile::Get_SLong()
{
    signed long l;
    Get_Data((unsigned char *)&l, 4);
    if (bigendian) {return ntohl(l);}
    return l;
}

IFD
tiffFile::Get_IFD(unsigned long ofs, unsigned long tagofs)
{
    return IFD(*this, ofs, tagofs);
}


IFD
tiffFile::Get_IFD()
{
    return IFD(*this, First_IFD());
}

