/****************************************************************************
** Dooble - The Secure Internet Web Browser
**
** Copyright (c) 2008, 2009 Alexis Megas, 
** Gunther van Dooble, and the Dooble Team. 
** All rights reserved.
**
**
** License: GPL2 only: 
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; version 2 of the License only.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
** or see here: http://www.gnu.org/licenses/gpl.html
**
**
** For the WebKit library, please see: http://webkit.org.
**
** THE CODE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY
** EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
** PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
** ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
** IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
** ARISING IN ANY WAY OUT OF THE USE OF THIS APPLICATION, EVEN IF ADVISED
** OF THE POSSIBILITY OF SUCH DAMAGE.
**
** Please report all praise, requests, bugs, and problems to the project
** team and administrators: http://sf.net/projects/dooble.
**
** You can find us listed at our project page. New team members are welcome.
** The name of the authors should not be used to endorse or promote products
** derived from Dooble without specific prior written permission.
** If you use this code for other projects, please let us know.
**
** Web sites:
**   http://dooble.net/projects/dooble
**   http://dooble.sf.net
**   http://www.getadooble.com
****************************************************************************/


#include <QFtp>
#include <QFile>
#include <QHttp>
#include <QtDebug>
#include <QFileInfo>
#include <QByteArray>

#include "downloadwindowitem.h"

downloadwindowitem::downloadwindowitem(void):QWidget()
{
  ui.setupUi(this);
}

void downloadwindowitem::download(const QString &srcFileName,
				  const QString &dstFileName)
{
  ui.sourceLabel->setText(srcFileName);
  ui.destinationLabel->setText(dstFileName);
  ui.abortToolButton->setVisible(false);
  ui.progressBar->setVisible(false);

  /*
  ** Copy the local file.
  */

  QFile file(srcFileName);

  if(file.open(QIODevice::ReadOnly))
    {
      QFile::remove(dstFileName);
      file.copy(dstFileName);
      file.close();
      downloadFinished(file.size());
    }
}

void downloadwindowitem::download(const QUrl &url, const QString &fileName_)
{
  requestId = 0;
  fileName = fileName_;
  ui.sourceLabel->setText(url.toString());
  ui.destinationLabel->setText(fileName);
  ui.abortToolButton->setIcon(QIcon("Icons/nuvola/16x16/cancel.png"));
  ui.abortToolButton->setEnabled(false);
  connect(ui.abortToolButton, SIGNAL(clicked(void)),
	  this, SLOT(slotAbortDownload(void)));

  QFile *file = new QFile(fileName);

  if(file->open(QIODevice::WriteOnly))
    {
      if(url.scheme().toLower() == "ftp")
	{
	  QFtp *ftp = new QFtp(this);

	  connect(ftp, SIGNAL(commandStarted(int)),
		  this, SLOT(slotRequestStarted(int)));
	  connect(ftp, SIGNAL(commandFinished(int, bool)),
		  this, SLOT(slotDownloadFinished(int, bool)));
	  connect(ftp, SIGNAL(dataTransferProgress(qint64, qint64)),
		  this, SLOT(slotDataTransferProgress(qint64, qint64)));
	  connect(this, SIGNAL(downloadAbortedSignal(int, bool)),
		  this, SLOT(slotDownloadFinished(int, bool)));

	  QByteArray path = QUrl::toPercentEncoding
	    (url.path(), "!$&'()*+,;=:@/");

	  ftp->connectToHost(url.host());
	  ftp->login();
	  requestId = ftp->get(path, file);
	}
      else if(url.scheme().toLower() == "http" ||
	      url.scheme().toLower() == "https")
	{
	  QHttp *http = new QHttp(this);

	  connect(http, SIGNAL(requestStarted(int)),
		  this, SLOT(slotRequestStarted(int)));
	  connect(http, SIGNAL(requestFinished(int, bool)),
		  this, SLOT(slotDownloadFinished(int, bool)));
	  connect(http, SIGNAL(dataReadProgress(int, int)),
		  this, SLOT(slotDataReadProgress(int, int)));
	  http->setHost(url.host());

	  QByteArray path = QUrl::toPercentEncoding
	    (url.path(), "!$&'()*+,;=:@/");

	  requestId = http->get(path, file);
	}
      else
	file->deleteLater();
    }
  else
    file->deleteLater();
}

downloadwindowitem::~downloadwindowitem()
{
}

void downloadwindowitem::slotDownloadFinished(int id, bool error)
{
  if(id != requestId)
    return;

  QIODevice *file = 0;

  Q_UNUSED(id);
  Q_UNUSED(error);

  QFtp *ftp = findChild<QFtp *> ();

  if(ftp)
    file = ftp->currentDevice();

  QHttp *http = findChild<QHttp *> ();

  if(http)
    file = http->currentDestinationDevice();

  if(file)
    {
      file->close();
      downloadFinished(file->size());
      ui.progressBar->setVisible(false);
      ui.abortToolButton->setVisible(false);
      file->deleteLater();
    }

  if(ftp)
    ftp->deleteLater();

  if(http)
    http->deleteLater();

  requestId = 0;
}

void downloadwindowitem::slotDataReadProgress(int done, int total)
{
  updateProgress(static_cast<qint64> (done), static_cast<qint64> (total));
}

void downloadwindowitem::slotDataTransferProgress(qint64 done, qint64 total)
{
  updateProgress(done, total);
}

void downloadwindowitem::updateProgress(const qint64 done, const qint64 total)
{
  ui.progressBar->setMaximum(total);
  ui.progressBar->setValue(done);

  int secs = 0;
  QTime timeNow = QTime::currentTime();

  if((secs = startTime.secsTo(timeNow)) >= 1)
    {
      QFileInfo fileInfo(fileName);

      if(fileInfo.exists())
	{
	  int kB = fileInfo.size() / 1024;
	  int rate = kB / secs;

	  ui.downloadInformationLabel->setText
	    (QString(tr("%1 kB/s (%2)")).arg(rate).
	     arg(formattedSize(fileInfo.size())));
	}
      else
	slotAbortDownload();
    }
}

void downloadwindowitem::slotRequestStarted(int id)
{
  Q_UNUSED(id);

  startTime = QTime::currentTime();
  ui.abortToolButton->setEnabled(true);
}

void downloadwindowitem::slotAbortDownload(void)
{
  QFtp *ftp = findChild<QFtp *> ();

  if(ftp)
    {
      ftp->abort();
      emit downloadAbortedSignal(-1, true); /*
					    ** The commandFinished() signal
					    ** may not be emitted. Therefore,
					    ** the slotDownloadFinished() will
					    ** not be issued. Hence the new
					    ** signal.
					    */
    }

  QHttp *http = findChild<QHttp *> ();

  if(http)
    http->abort();

  ui.progressBar->setVisible(false);
  ui.abortToolButton->setVisible(false);
  QFile::remove(fileName);
  ui.downloadInformationLabel->setText(tr("Download Aborted"));
  requestId = 0;
}

bool downloadwindowitem::isDownloading(void)
{
  /*
  ** What should we do for local copies of large files?
  */

  return ui.abortToolButton->isVisible();
}

void downloadwindowitem::downloadFinished(const int fileSize)
{
  ui.downloadInformationLabel->setText(formattedSize(fileSize));
  emit downloadFinishedSignal();
}

QString downloadwindowitem::formattedSize(const int fileSize)
{
  QString str("");

  if(fileSize < 1024)
    str = QString(tr("%1 Byte(s)")).arg(fileSize);
  else if(fileSize < 1048576)
    str = QString(tr("%1 kB")).arg
      (QString::number((fileSize * 1.0) / 1024, 'f', 2));
  else
    str = QString(tr("%1 MB")).arg
      (QString::number((fileSize * 1.0) / 1048576, 'f', 2));

  return str;
}
