/* PROGRAM:	basket
 * FILE:	$Header: /home/egg/src/RCS/basket.c,v 1.8 1999/02/28 19:58:49 ghn Exp $
 * PURPOSE:	Main program for basket software
 * AUTHOR:	Greg Nelson
 * DATE:	98-06-28
 *
 * REVISED:
 * $Log: basket.c,v $
 * Revision 1.8  1999/02/28 19:58:49  ghn
 * Version 5.1: Support for ip address masks to allow us to limit egg
 * spoofing to users on a limited subnet (cf. EGG_DYNAMIC).  Added
 * flexible interface/port cmdline arguments and config file arguments.
 * Report the eggs we are accepting.
 *
 * Revision 1.7  1999/01/01 23:52:31  ghn
 * Allow basket to start with no net connection and handle connection
 * going up and down.
 *
 * Revision 1.6  1998/12/31 22:07:56  ghn
 * Rev 5 code: includes multi-reg support, HTML, etc.
 *
 * Revision 1.5  1998/08/03 20:40:13  kelvin
 * PACKETDUMP, SIGHUP reset when signal caught.
 * 
 * Revision 1.4  1998/08/01  18:52:49  ghn
 * Added John's byte-order-independence changes.
 *
 * Revision 1.3  1998/08/01 17:02:05  ghn
 * Incorporate John's Solaris fixes and terminate STAT lines properly.
 *
 * Revision 1.2  1998/07/21 11:44:04  ghn
 * Added headers.
 *
 * Copyright 1998 - Greg Nelson
 */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include "global.h"
#include "genlib.h"
#include "storage.h"
#include "network.h"
#include "errnos.h"
#include "version.h"

#define EGGSTATS        "egg.status"

static int32 BasketReceiveDataPacket(char *pktbuf, int16 isegg, int16 thisegg,
			      struct sockaddr_in *rhost);
static void MakeRequest(ReqPacket *pkt, uint16 eggid, uint32 whence);
static void MakeSettings(SettingsPacket *pkt, uint16 eggid);
static void LoadRCFile(void);
static void LoadEggStats(void);
static void SaveEggStats(void);
static void handle_sighup(int arg);

EggEntry eggtable[MAX_EGGS];
BasketEntry baskettable[MAX_BASKETS];
EggHeader protocol;

short numeggs, numbaskets;
char	*pgmname;		      /* Program name from argv[0] */
char	*myaddr;		      /* Interface to bind */
int16	myport;			      /* Service port to bind */

static int htmlInterval = 0;	      /* HTML status file update interval in seconds */
static char htmlFile[256] = "";       /* HTML status file name */
static uint32 upsince = 0;	      /* Basket start time */
#define htmlReport (htmlFile[0] != 0) /* Make HTML report ? */

struct {
    uint32 firstPacket;               /* Time of "first contact" with egg */
    uint32 trials;		      /* Number of trials received */
    uint32 missing;		      /* Number of missing samples */
    double sum; 		      /* Sum of trials to date */
} eggStatistics[MAX_EGGS];

/*  updateHTML	--  Update HTML status file if a decent interval
		    has passed since the last update.  */

static void updateHTML(void)
{
    if (htmlReport) {
	int i;
	static uint32 lastHtml = 0;
	uint32 now = getzulutime(NULL);
	char udate[256], ustime[256];
        static char timeFormat[] = "%Y-%m-%d %T";

	/* When first called, set the last HTML update interval to
	   the present moment.	This defers generation of the first
	   HTML report until htmlInterval elapses.  This prevents
	   issuing a potentially-misleading HTML report based on
	   incomplete information. */

	if (lastHtml == 0) {
	    lastHtml = now;
	}

	if ((now - lastHtml) >= htmlInterval) {
	    FILE *hf;

	    lastHtml = now;
            hf = fopen(htmlFile, "w");
	    if (hf == NULL) {
                fprintf(stderr, "Cannot open HTML status file %s.  Updates discarded.\n", htmlFile);
		htmlFile[0] = 0;      /* Suppress further updates */
		return;
	    }

	    strftime(udate, sizeof udate, timeFormat, gmtime((time_t *) &now));
	    strftime(ustime, sizeof ustime, timeFormat, gmtime((time_t *) &upsince));

            fprintf(hf, "\
<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2 Final//EN\">\n\
<html version=\"-//W3C//DTD HTML 3.2 Final//EN\">\n\
<head>\n\
<title>Basket %s Status Report</title>\n\
<meta http-equiv=\"Refresh\" content=\"%d\">\n\
</head>\n\
<body>\n\
<center>\n\
<h1>Basket %s Status Report</h1>\n\
<h3>Last update: %s UTC</h3>\n\
<h4>Basket started at %s UTC<br>\n\
%s</h4>\n\
</center>\n\
<center>\n\
<h2>Egg Status</h2>\n\
<table border cellpadding=4>\n\
<tr><th>Name<th>Number<th>Up Since<th>Last Packet<th>Active\n\
<th>Trials<th>Missed<th>Coverage<th>Mean\n\
",
	    baskettable[0].name, htmlInterval, baskettable[0].name, udate, ustime, Version);

	    for (i = 0; i < numeggs; i++) {
	      char url1[256], url2[256];

	      if ((eggtable[i].lastupd == 0) || (eggStatistics[i].firstPacket == 0)) {
                  strcpy(udate, "<td colspan=2 align=center><em>Never contacted</em>");
                  strcpy(ustime, "");
	      } else {
                  strftime(udate, sizeof udate, "<td align=center>%Y-%m-%d %T", gmtime((time_t *) &eggtable[i].lastupd));
                  strftime(ustime, sizeof ustime, "<td align=center>%Y-%m-%d %T", gmtime((time_t *) &eggStatistics[i].firstPacket));
	      }
	      if (eggtable[i].url == NULL) {
		  url1[0] = url2[0] = 0;
	      } else {
                  sprintf(url1, "<a href=\"%s\">", eggtable[i].url);
                  strcpy(url2, "</a>");
	      }
              fprintf(hf, "<tr><td>%s%s%s<td align=center>%d%s%s<td align=center>%s\n\
",
		      url1,
		      eggtable[i].name,
		      url2,
		      eggtable[i].id,
		      ustime,
		      udate,
                      eggtable[i].setup ? "Yes" : "No"
	      );

	      if (eggStatistics[i].trials == 0) {
                  fprintf(hf, "<td align=center>&nbsp;<td align=center>&nbsp;<td align=center>&nbsp;<td align=center>&nbsp;\n");
	      } else {
		  uint32 coverage;

                  /* The following expression computes "coverage" for
		     the egg--the percentage of samples received
		     to the number prescribed by the protocol
		     for the time in which the egg has been in contact. */
		  coverage = (100L * eggStatistics[i].trials * protocol.samp_rec) /
		      (protocol.sec_rec * (now - eggStatistics[i].firstPacket));
		  if (coverage > 100) {
		      coverage = 100;
		  }

                  fprintf(hf, "<td align=center>%ld<td align=center>%ld<td align=center>%ld%%<td align=center>%.3f\n",
			  eggStatistics[i].trials,
			  eggStatistics[i].missing,
			  coverage,
			  eggStatistics[i].sum / eggStatistics[i].trials
		  );
	      }
	    }


            fprintf(hf, "\
</table>\n\
</center>\n\
<center>\n\
<h4>Egg Status Table Fields</h4>\n\
<table width=\"75%%\" cellpadding=3>\n\
<tr><td valign=top><b>Name</b> <td>Egg name<p>\n\
<tr><td valign=top><b>Number</b> <td>Egg number.  Thousands digit indicates type of random event generator.<p>\n\
<tr><td valign=top><b>Up Since</b> <td>Time and date in first packet received from egg since basket started.<p>\n\
<tr><td valign=top><b>Last Packet</b> <td>Time and date of most recently received packet from egg.<p>\n\
<tr><td valign=top><b>Active</b> <td>Has egg ever communicated with this basket since it was started?<p>\n\
<tr><td valign=top><b>Trials</b> <td>Number of trials received from this egg.<p>\n\
<tr><td valign=top><b>Missed</b> <td>Number of scheduled trials missed by egg due to synchronisation problems.<p>\n\
<tr><td valign=top><b>Coverage</b> <td>Percent of trials collected from this egg since basket\n\
    started compared to the number scheduled by the protocol.<p>\n\
<tr><td valign=top><b>Mean</b> <td>Arithmetic mean of trials performed by this egg.\n\
</table>\n\
</center>\n\
</body>\n\
</html>\n");
	    fclose(hf);
#ifdef DEBUG
            fprintf(stderr, "Updating HTML file %s at %s",
		htmlFile, asctime(gmtime((time_t *) &now)));
#endif
	}
    }
}

static void Usage(void) {
  printf("Usage: %s [-i interface] [-p port]\n", pgmname);
  exit(-1);
}

/*  Main program.  */

int main(int argc, char *argv[]) {
  char			*pktbuf;
  uint16		pkttype, pktsize, pkt_eggid;
  int			sdlisten, res;
  struct sockaddr_in	rhost;
  ReqPacket		rpkt;
  SettingsPacket	spkt;
  int			i, isbasket, isegg, thisegg;
  char			*remname;
#ifdef SIGACTION_WORKING
  struct sigaction	sigact;
#endif
#ifdef EGG_DYNAMIC 
  uint32		ipvalidate, ipmatch, ipmask;
#endif

  pgmname = argv[0];
  argv++; argc--;

  /* Defaults correspond to original usage */
  myport = BASKETPORT;
  myaddr = NULL;

  while (argc) {
    if (argv[0][0] == '-') {
      switch(argv[0][1]) {
      case 'i':	/* Specify interface */
	if (argc < 2) Usage();
	myaddr = argv[1];
	argv++; argc--;
	break;
      case 'p':	/* Specify port */
	if (argc < 2) Usage();
	myport = atoi(argv[1]);
	if (myport < 0) Usage();
	argv++; argc--;
	break;
      default:	/* Oops. */
	Usage();
      }
    } else {
      Usage();
    }
    argv++; argc--;
  }

  LoadRCFile();
  LoadEggStats();

#ifdef Solaris
  /* Solaris has an eccentric definition of sigaction() which
     doesn't seem to even work according to Sun's own
     documentation.  (sa_flags is a 4 element array of
     unsigned longs, with no mention of how one stores the
     flags into it.)  Let's just use plain old signal for
     the moment. */
  signal(SIGHUP, handle_sighup);
#else
#ifdef SIGACTION_WORKING
  sigact.sa_handler = handle_sighup;
  sigact.sa_mask = 0;
  sigact.sa_flags = 0;
  sigact.sa_restorer = NULL;
  sigaction(SIGHUP, &sigact, NULL);
#else
  signal(SIGHUP, handle_sighup);
#endif
#endif

  /* No connection at outset */
  sdlisten = -1;

  /* Initialize storage system. */
  InitStorage("%Y-%m/%Y-%m-%d-$b");

  upsince = getzulutime(NULL);

  fprintf(stderr, "Starting basket %s.\n", Version);

  /* Inner loop is currently single thread:
     1A. Wait for connection.  When a connection comes in,
	 packet has been decrypted and verified.
     1B. Validate connection.
     1C. Reply to connection. */
  while (1) {
    /* If no current listening connection, try to get one,
       and if that fails, sleep and try later. */

    if (sdlisten < 0) {
      /* Get a listening socket at BASKETPORT.
         Only allow our */
      sdlisten = InitNetwork(myaddr, myport);
      if (sdlisten < 0) {
	sleep(1);
	continue;
      }
    }

    /* 1A. Wait for connection. */
    res = NetListen(sdlisten, &pktbuf, &rhost, TRUE);
    if (res < 0) {
      fprintf(stderr, "NetListen error: %d\n", res);
      continue;
    }
    
    /* 1B. Validate connection. 
       Remote host address is in rhost.  Make sure this is either an
       egg or a basket, and set the flag appropriately. */
#ifdef EGG_DYNAMIC
    ipvalidate = ntohl(rhost.sin_addr.s_addr);
#endif

    isegg = isbasket = 0;
    for (i = 0; i < numeggs; i++) {
#ifdef EGG_DYNAMIC
      /* When mask == 32:	ipmask = 0xFFFFFFFF (all significant)
	      mask == 16:	ipmask = 0xFFFF0000 (only top half sig)
	      mask ==  0:	ipmask = 0x00000000 (none significant) */
      ipmask = (0xFFFFFFFF << (32-eggtable[i].netmask));
      ipmatch = ntohl(eggtable[i].ipaddr.sin_addr.s_addr);
      if ((ipvalidate & ipmask) == (ipmatch & ipmask)) {
	isegg = eggtable[i].id;
	thisegg = i;
	remname = eggtable[i].name;
      }      
#else
      if (!memcmp(&(rhost.sin_addr),
		  &(eggtable[i].ipaddr.sin_addr),
		  sizeof(rhost.sin_addr))) {
	isegg = eggtable[i].id;
	thisegg = i;
	remname = eggtable[i].name;
      }
#endif
    }
    for (i = 0; i < numbaskets; i++) {
      if (!memcmp(&(rhost.sin_addr),
		  &(baskettable[i].ipaddr.sin_addr),
		  sizeof(rhost.sin_addr))) {
	isbasket = 1;
	remname = baskettable[i].name;
      }
    }

    if (!isegg && !isbasket) {
      fprintf(stderr, "Attempt to connect from unknown source: %s\n",
	      sockaddr2dquad(&rhost));
      continue;
    }

    memcpy(&pkttype, pktbuf, sizeof(uint16));
    memcpy(&pktsize, pktbuf+sizeof(uint16), sizeof(uint16));
    pkttype = ntohs(pkttype);
    pktsize = ntohs(pktsize);
    if (pktsize < 6) {
      fprintf(stderr, "Packet doesn't contain an egg id, why?\n");
      continue;
    }
    pkt_eggid = ntohs(*((uint16 *) (pktbuf + 4)));

    /* Am I hearing from who I think I am? */
#if EGG_DYNAMIC
    /* Have to assume the best case before testing: that the
       pkt_eggid is actually correct. */
    for (i = 0; i < numeggs; i++) if (eggtable[i].id == pkt_eggid) break;

    if (i == numeggs) {
      /* Must be a spoof, and possibly an attempt to crash software too. */
      fprintf(stderr, "%s: Egg spoofing?  %s reporting itself as %d\n",
	      pgmname, hl2dquad(ipvalidate), pkt_eggid);
      continue;
    }

    ipmask = (0xFFFFFFFF << (32-eggtable[i].netmask));
    ipmatch = ntohl(eggtable[i].ipaddr.sin_addr.s_addr);
    if ((ipvalidate & ipmask) != (ipmatch & ipmask)) {
      fprintf(stderr, "%s: Egg spoofing?  %s reporting itself as %d\n",
	      pgmname, hl2dquad(ipvalidate), pkt_eggid);
      continue;
    } else {
      isegg = eggtable[i].id;
      thisegg = i;
      remname = eggtable[i].name;
    }      
#else
    if (isegg != pkt_eggid) {
      fprintf(stderr, "%s: Egg spoofing?  %d reporting itself as %d\n",
	      pgmname, isegg, pkt_eggid);
      continue;
    }
#endif

    /* 1D. Reply to connection. */
    switch(pkttype) {
    case DATA_PACKET:
      /* Put it away. */
      BasketReceiveDataPacket(pktbuf, isegg, thisegg, &rhost);

      break;
    case REQ_PACKET:
      /* Is this how baskets communicate? */
      break;
    case AWAKE_PACKET:
#ifdef PACKETDUMP
      {
	uint32 egg_now_time;

	memcpy(&egg_now_time, pktbuf + 6, sizeof egg_now_time);
	egg_now_time = ntohl(egg_now_time);
	fprintf(stderr, "Awake egg %d (%s) at its %lu: %s",
		pkt_eggid, eggtable[thisegg].name, egg_now_time,
		asctime(gmtime((time_t *) &egg_now_time)));
      }
#endif

      /* Calculate last data received from this egg, decide what
	 to ask for. */
      MakeRequest(&rpkt, isegg, eggtable[thisegg].lastupd);
      rhost.sin_port = htons(EGGPORT);
      res = NetTalk(&rhost, (char *)&rpkt, TRUE);
      if (res < 0) fprintf(stderr, "NetTalk error %d.\n", res);

      /* If setup looks like it's out of date, correct it. */
      if (!eggtable[thisegg].setup) {
	MakeSettings(&spkt, thisegg);
	rhost.sin_port = htons(EGGPORT);
	res = NetTalk(&rhost, (char *)&spkt, TRUE);
        if (res < 0) fprintf(stderr, "NetTalk error %d.\n", res);
      }
      break;

    case SETTINGS_PACKET:
      /* I don't have settings, Bozo. */
      break;
    }
    free(pktbuf);
  }
}

/*  handle_sighup  --  Catch SIGHUP and reload configuraton files.  */

static void handle_sighup(int arg) {
  /* Reread the RC file */
  LoadRCFile();
  LoadEggStats();
#ifdef DEBUG
  fprintf(stderr, "Received SIGHUP: reloading configuration files.\n");
#endif
#ifndef SIGACTION_WORKING
  signal(SIGHUP, handle_sighup);      /* Reset signal handler for bottom-feeder signal() */
#endif
}

/*  LoadRCFile	--  Load basket configuration file.  */

static void LoadRCFile(void) {
  FILE *fp;
  char linebuf[200];
  char *myargv[MAX_PARSE], *tp;
  int myargc, lcount, p, i;

  if ((fp = fopen(".basketrc", "r")) == NULL) {
    if ((fp = fopen("~/.basketrc", "r")) == NULL) {
      if ((fp = fopen("/etc/basketrc", "r")) == NULL) {
        fprintf(stderr, "%s: Couldn't find a basket RC file.\n", pgmname);
	exit(-1);
      }
    }
  }

  numeggs = numbaskets = 0;
  lcount = 0;

  while(fgets(linebuf, 200, fp) != NULL) {
    /* Comments and blank lines ignored. */
    lcount++;
    if (*linebuf == '#' || *linebuf == '\n') continue;
    Parse(linebuf, &myargc, myargv);
    if (myargc == 0) continue;
    if (!strcmp(myargv[0], "EGG")) {
      if (myargc < 7 || myargc > 8) {
        fprintf(stderr, "%s: RC error, line %d, poor %s format\n", 
		pgmname, lcount, myargv[0]);
	exit(-1);
      }
	
      eggtable[numeggs].name = mallocpy(myargv[1]);
      eggtable[numeggs].id = atoi(myargv[2]);
      dquad2sockaddr(&(eggtable[numeggs].ipaddr), 
		     &(eggtable[numeggs].netmask),
		     myargv[3]);
      eggtable[numeggs].primbasket = mallocpy(myargv[4]);
      eggtable[numeggs].conntype = (!strcmp(myargv[5], "PERM"))?CONN_PERM:CONN_DND;
      eggtable[numeggs].connival = atoi(myargv[6]);
      eggtable[numeggs].url = NULL;
      if ((myargc > 7) && (strcmp(myargv[7], ".") != 0)) {
	eggtable[numeggs].url = mallocpy(myargv[7]);
      }
      numeggs++;
    } else if (!strcmp(myargv[0], "BASKET")) {
      if (myargc != 3) {
        fprintf(stderr, "%s: RC error, line %d, poor %s format\n", 
		pgmname, lcount, myargv[0]);
	exit(-1);
      }
      baskettable[numbaskets].name = mallocpy(myargv[1]);
      dquad2sockaddr(&(baskettable[numbaskets].ipaddr), NULL, myargv[2]);
      numbaskets++;
    } else if (!strcmp(myargv[0], "PROTOCOL")) {
      if (myargc != 5) {
        fprintf(stderr, "%s: RC error, line %d, poor %s format\n", 
		pgmname, lcount, myargv[0]);
	exit(-1);
      }
      protocol.samp_rec = atoi(myargv[1]);
      protocol.sec_rec = atoi(myargv[2]);
      protocol.rec_pkt = atoi(myargv[3]);
      protocol.trialsz = atoi(myargv[4]);
    } else if (!strcmp(myargv[0], "HTML")) {
      if (myargc != 3) {
        fprintf(stderr, "%s: RC error, line %d, poor %s format\n", 
		pgmname, lcount, myargv[0]);
	exit(-1);
      }
      htmlInterval = atoi(myargv[1]);
      strcpy(htmlFile, myargv[2]);
#ifdef DEBUG
      fprintf(stderr, "Updating HTML file %s every %d seconds.\n", htmlFile, htmlInterval);
#endif
    } else if (!strcmp(myargv[0], "PORT") ||
	       !strcmp(myargv[0], "INTERFACE")) {
      if (myargc != 2) {
        fprintf(stderr, "%s: RC error, line %d, poor %s format\n", 
		pgmname, lcount, myargv[0]);
	exit(-1);
      }
      tp = (char *) malloc(200);
      for (*tp = 0, p = 1; p < myargc; p++) {
	strcat(tp, myargv[p]);
        if (p < myargc-1) strcat(tp, " ");
      }

      if (!strcmp(myargv[0], "PORT")) {
	if (atoi(tp) <= 0) {
	  fprintf(stderr, "%s: RC error, ignoring bad port %d, using %d\n",
		  pgmname, atoi(tp), myport);
	} else {
	  myport = atoi(tp);
	}
	free(tp);
      } else if (!strcmp(myargv[0], "INTERFACE")) myaddr = tp;
    } else {
      fprintf(stderr, "%s: RC error, %s is unknown keyword\n", pgmname, myargv[0]);
      exit(-1);
    }
  }
  fclose(fp);

#if REPORT > 0
  /* Report a list of the eggs that we are accepting. */
  for (i = 0; i < numeggs; i++) {
    fprintf(stderr, "Egg %2d:\tid = %4d\tip=%s/%d\n", 
	    i, eggtable[i].id, 
	    sockaddr2dquad(&(eggtable[i].ipaddr)),
	    eggtable[i].netmask);
  }
#endif
}

/*  BasketReceiveDataPacket  --  Process a data packet received frm an egg.  */

static int32 BasketReceiveDataPacket(char *pkt, int16 isegg, int16 thisegg,
			      struct sockaddr_in *rhost) {
  EggHeader		*dpktp;
  EggCarton		result;
  ReqPacket		rpkt;
  uint16		offset, rec;
  uint32		latest;
  int32 		res;
#ifdef DEBUG
  uint16		i;
#endif

  EggHeader dpk;
  char *pktP = pkt;
  
  /* Unpack the portable header into a host-order and aligned
     EggHeader packet. */
  
  dpktp = &dpk;
  unpackShort(dpk.type);
  unpackShort(dpk.pktsize);
  unpackShort(dpk.eggid);
  unpackShort(dpk.samp_rec);
  unpackShort(dpk.sec_rec);
  unpackShort(dpk.rec_pkt);
  unpackByte(dpk.trialsz);
  unpackShort(dpk.numrec);

  /* This spoofing test is probably completely redundant now. */
#ifndef EGG_DYNAMIC      
  /* Am I hearing from an egg, and is it who I think it is? */
  if (isegg && isegg != dpktp->eggid) {
    fprintf(stderr, "%s: Egg spoofing?  %d reporting itself as %d\n",
	    pgmname, isegg, dpktp->eggid);
    return -1;
  }
#endif

  /* Save the packet header in the result EggCarton buffer. */

  memcpy(&(result.hdr), dpktp, sizeof(EggHeader));
  offset = sizeof(EggHeader);

#ifdef DEBUG
  {
    uint32 stamp = 0;
    char *spktp = pktP;

    if (result.hdr.numrec > 0) {
      unpackLong(stamp);
    }
    pktP = spktp;
    fprintf(stderr, "Received packet: %d records from egg %d (%s).\n",
      result.hdr.numrec, dpktp->eggid, eggtable[thisegg].name);
  }
#endif

  /* Unpack the individual data records from the packet and
     transcribe to the result EggCarton.  */
  
  for (latest = 0, rec = 0; rec < result.hdr.numrec; rec++) {
    unpackLong(result.records[rec].timestamp);	 /* Record timestamp */
    if (result.records[rec].timestamp > latest)
      latest = result.records[rec].timestamp;
#ifdef DEBUG
    fprintf(stderr, "  %10lu  ", result.records[rec].timestamp);
#endif
    /* Assumes sizeof(trial) = 1 */
    unpackBytes(&(result.records[rec].trials), result.hdr.samp_rec);
#ifdef DEBUG
    for (i = 0; i < result.hdr.samp_rec; i++) {
      fprintf(stderr, "%3d ", result.records[rec].trials[i]);
    }
    fprintf(stderr, "%s", asctime(gmtime((time_t *) &result.records[rec].timestamp)));
#endif
  }

  /* Save the packet in the basket database file. */
  
  if ((res = SavePacket(&result)) < 0) return res;

  /* If we're generating an HTML report, update the per-basket
     statistics based on the content of this packet. */

  if (htmlReport) {
      int r, t;
      uint32 ptrials = 0, psum = 0, dropped = 0;

      /* Update the egg status for HTML report. */

      for (r = 0; r < result.hdr.rec_pkt; r++) {
	  for (t = 0; t < result.hdr.samp_rec; t++) {
	      if (result.records[r].trials[t] != EGG_MISSING_DATA) {
		  ptrials++;
		  psum += result.records[r].trials[t];
	      } else {
		  dropped++;
	      }
	  }
      }
      eggStatistics[thisegg].trials += ptrials;
      eggStatistics[thisegg].sum += psum;

      /* If this is the first packet received from the egg, and
         the first sample in it is missing, don't count missing
         samples against this egg, since it's probable this is
	 the first packet collected since the egg started, which
	 will contain initial missing samples for time before the
	 egg started. */

      if (eggStatistics[thisegg].firstPacket != 0) {
	  eggStatistics[thisegg].missing += dropped;
      } else {
	  eggStatistics[thisegg].firstPacket = result.records[0].timestamp;
      }
  }

  if (latest > eggtable[thisegg].lastupd) {
    eggtable[thisegg].lastupd = latest;
    SaveEggStats();
    updateHTML();		      /* Update HTML status file, if warranted */
    if (latest < getzulutime(NULL)) {
      /* Get more data if it looks like there should be more. */
      MakeRequest(&rpkt, isegg, eggtable[thisegg].lastupd);
      rhost->sin_port = htons(EGGPORT);
      res = NetTalk(rhost, (char *)&rpkt, TRUE);
      if (res < 0) fprintf(stderr, "NetTalk error %d.\n", (int)res);
    }
  }

  /* If the protocol now in effect differs from that of the
     packet just received from the egg, mark the egg as not
     set-up, and thus in need of a SETTINGS_PACKET to reset
     its protocol. */

  eggtable[thisegg].setup = (protocol.samp_rec == dpktp->samp_rec &&
			     protocol.sec_rec == dpktp->sec_rec &&
			     protocol.rec_pkt == dpktp->rec_pkt &&
			     protocol.trialsz == dpktp->trialsz);

  return ERR_NONE;
}

/*  MakeRequest  --  Assemble a request packet for a given egg,
		     specifying, in whence, the time and date
		     of the last sample received from that egg.  */

static void MakeRequest(ReqPacket *pkt, uint16 eggid, uint32 whence) {
  char *pktP = (char *) pkt;

  packShort(REQ_PACKET);
  packShort((4 * sizeof(uint16)) + sizeof(uint32));
  packShort(eggid);
  packLong(whence);
}

/*  MakeSettings  --  Create a settings packet.  This packet,
		      sent periodically to connected eggs,
		      informs them of any change in protocol.  */

static void MakeSettings(SettingsPacket *pkt, uint16 eggid) {
  char *pktP = (char *) pkt;

  packShort(SETTINGS_PACKET);
  packShort((7 * sizeof(uint16)) + sizeof(uint32) + sizeof(trial));
  packShort(eggid);
  packLong(getzulutime(NULL));
  packShort(protocol.samp_rec);
  packShort(protocol.sec_rec);
  packShort(protocol.rec_pkt);
  packByte(protocol.trialsz);
}

/*  LoadEggStats  --  Initialise in-memory egg status table
		      from the EGGSTATS file.  */

static void LoadEggStats(void) {
  FILE *fp;
  char linebuf[200], *myargv[MAX_PARSE], *name;
  int i, f, myargc, id, setup;
  int32 lastupd;

  /* Default stats */
  for (i = 0; i < numeggs; i++) {
    eggtable[i].lastupd = 0;
    eggtable[i].setup = 0;
  }
  
  if ((fp = fopen(EGGSTATS, "r")) == NULL) return;
  
  while(fgets(linebuf, 200, fp) != NULL) {
    /* Comments and blank lines ignored. */
    if (*linebuf == '#' || *linebuf == '\n') continue;
    Parse(linebuf, &myargc, myargv);
    if (myargc == 0) continue;
    if (!strcmp(myargv[0], "STAT") && myargc == 5) {
      name = myargv[1];
      id = atoi(myargv[2]);
      lastupd = atol(myargv[3]);
      setup = atoi(myargv[4]);
      for (f = 0, i = 0; i < numeggs; i++) {
	if (eggtable[i].id == id &&
	    !strcmp(eggtable[i].name, name)) {
	  eggtable[i].lastupd = lastupd;
	  eggtable[i].setup = setup;
	  f = 1;
	}
      }
      if (!f) fprintf(stderr, "%s: Egg '%s' no longer hosted.\n", pgmname, name);
    }
  }
  fclose(fp);
}

/*  SaveEggStats  --  Dump in-memory egg status to the EGGSTATS file.  */

static void SaveEggStats(void) {
  FILE *fp;
  int i;

  if ((fp = fopen(EGGSTATS, "w")) == NULL) {
    /* Couldn't write stats! */  
    fprintf(stderr, "%s: Couldn't save egg stats!\n", pgmname);
    return;
  }

  fprintf(fp, "# Status lines of form:\n");
  fprintf(fp, "#   STAT <eggname> <eggid> <lastupdzulu> <setupvalid>\n");
  for (i = 0; i < numeggs; i++) {
    fprintf(fp, "STAT %s %d %ld %d\n",
	    eggtable[i].name,
	    eggtable[i].id,
	    eggtable[i].lastupd,
	    eggtable[i].setup);
  }
  fclose(fp);
}
