------------------------------------------------------------------------------
--  Ada95 Interface to Oracle RDBMS                                         --
--  Copyright (C) 2000-2007 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-connections.ads,v 1.14 2007/07/18 08:19:25 vagul Exp $

with Ada.Containers.Indefinite_Hashed_Maps;
with Ada.Strings.Hash;
with OCI.Environments;
with OCI.Thick.Servers;

package OCI.Thick.Connections is

   use OCI.Thick.Servers;

   type Connection is new Handle_Reference with private;

   function Logon
     (Server_Name : in String;
      User        : in String;
      Password    : in String) return Connection;

   function Logon (Connect : String) return Connection;

   function Logon
     (DB       : in Server;
      User     : in String;
      Password : in String) return Connection;

   procedure Split_Connect_String
     (Connect        : in     String;
      User_First     :    out Positive;
      User_Last      :    out Positive;
      Password_First :    out Positive;
      Password_Last  :    out Natural;
      Server_First   :    out Positive;
      Server_Last    :    out Natural);

   procedure Logoff (Connect : in out Connection);

   procedure Set_Blocking (Connect : in out Connection; Mode : in Boolean);

   function Is_Blocking (Connect : in Connection) return Boolean;

   function User_Name (Connect : in Connection) return String;

   function Server_Version (Connect : in Connection) return String;

   procedure Commit (Connect : in Connection);

   procedure Rollback (Connect : in Connection);

   function Rollback (Connect : in Connection) return Boolean;
   --  Return false in error on rollback operation.
   --  Use it if you do not want to cach exception just on Rollback.
   --  It is convinient when connection is be broken, and we could not
   --  recognise it from first error code.

   procedure Break (It : Connection);
   procedure Reset (It : Connection);

   function Get_Connection (Context : OCIExtProcContext) return Connection;
   --  get Connection for extproc callbacks

   function Get_Type
     (Connect : in Connection; Name : in String) return OCIType;
   --  Need to object type creation

private

   package Type_Caches is new Ada.Containers.Indefinite_Hashed_Maps
     (Key_Type        => String,
      Element_Type    => OCIType,
      Hash            => Ada.Strings.Hash,
      Equivalent_Keys => "=",
      "="             => System."=");

   type Type_Cache_Access is access all Type_Caches.Map;

   type Connection is new Handle_Reference with record
      DB          : Server;
      Sessn       : OCISession := OCISession (Empty_Handle);
      Environment : OCI.Environments.Thread_Environment;
      Type_Cache  : Type_Cache_Access;
   end record;

   procedure Destroy (Object : in out Connection);

end OCI.Thick.Connections;
