------------------------------------------------------------------------------
--  Ada95 Interface to Oracle RDBMS                                         --
--  Copyright (C) 2006 Maxim Reznik                                         --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

with OCI.Thread;
with OCI.Environments;
with System.Address_To_Access_Conversions;

package body OCI.Thick.Notifications.DB_Change is

   function Get_Table
     (Object : Change_Descriptor;
      Index  : Positive)
      return OCIColl;

   function Get_Table_Collection (Object : Change_Descriptor) return OCIColl;

   function Get_Attr_Ub4  is new Get_Attr_G (Ub4);
   function Get_Attr_Coll is new Get_Attr_G (OCIColl);

   ---------------
   -- Associate --
   ---------------

   procedure Associate
     (Item        : in out Change_Subscription;
      Stmt        : in out Statements.Statement)
   is
   begin
      Set_Attr
        (Handle (Stmt),
         OCI_HTYPE_STMT,
         OCI_ATTR_CHNF_REGHANDLE,
         Value => OCIHandle (Item.Handle));

      Statements.Execute (Stmt);

      Set_Attr
        (Handle (Stmt),
         OCI_HTYPE_STMT,
         OCI_ATTR_CHNF_REGHANDLE,
         Empty_Handle);
   end Associate;

   -----------------
   -- Change_Kind --
   -----------------

   function Change_Kind (Object : Change_Descriptor) return Change_Kinds is
      Code : constant Ub4 := Get_Attr_Ub4 (OCIHandle (Object.Descriptor),
                                           OCI_DTYPE_CHDES,
                                           OCI_ATTR_CHDES_NFYTYPE);
   begin
      return Change_Kinds'Val (Code);
   end Change_Kind;

   ------------
   -- Create --
   ------------

   procedure Create
     (Item        : in out Change_Subscription;
      Need_Rowids : in Boolean := False)
   is
      use type Ub4;
      Yes : aliased constant Bool := OCI.Lib.TRUE;
   begin
      Create (Item, OCI_SUBSCR_NAMESPACE_DBCHANGE);

      if Need_Rowids then
         Set_Attr
           (OCIHandle (Item.Handle),
            OCI_HTYPE_SUBSCRIPTION,
            OCI_ATTR_CHNF_ROWIDS,
            Value => Yes'Address);
      end if;
   end Create;

   ---------------
   -- Get_Table --
   ---------------

   function Get_Table
     (Object : Change_Descriptor;
      Index  : Positive)
     return OCIColl
   is
      package Convert is new
        System.Address_To_Access_Conversions (OCIColl);

      Collection : constant OCIColl := Get_Table_Collection (Object);
      Exists     : aliased Bool;
      Elem       : aliased DVoid;
   begin
      Check_Error (OCICollGetElem (Thread.Environment,
                                   Thread.Error,
                                   Collection,
                                   Sb4 (Index - 1),
                                   Exists'Access,
                                   Elem'Access));

      return Convert.To_Pointer (Elem).all;
   end Get_Table;

   --------------------------
   -- Get_Table_Collection --
   --------------------------

   function Get_Table_Collection (Object : Change_Descriptor) return OCIColl is
      Collection : constant OCIColl :=
        Get_Attr_Coll (OCIHandle (Object.Descriptor),
                       OCI_DTYPE_CHDES,
                       OCI_ATTR_CHDES_TABLE_CHANGES);
   begin
      return Collection;
   end Get_Table_Collection;

   ---------------------
   -- Internal_Notify --
   ---------------------

   procedure Internal_Notify
     (Item       : in out Change_Subscription;
      Descriptor : in     OCI.Lib.DVoid;
      Buffer     : in     Buffer_Type)
   is
      pragma Unreferenced (Buffer);

      Data : Change_Descriptor;
   begin
      Data.Descriptor := Descriptor;
      Notify (Change_Subscription'Class (Item), Data);
   end Internal_Notify;

   ------------
   -- Notify --
   ------------

   procedure Notify
     (Item   : in out Change_Subscription;
      Data   : in     Change_Descriptor)
   is
      pragma Unreferenced (Item);
      pragma Unreferenced (Data);
   begin
      --  To be overloaded
      null;
   end Notify;

   -----------------
   -- Table_Count --
   -----------------

   function Table_Count (Object : Change_Descriptor) return Natural is
      Collection : constant OCIColl := Get_Table_Collection (Object);
      Size       : aliased Sb4;
   begin
      if Collection = OCIColl (Empty_Handle) then
         return 0;
      else
         Check_Error (OCICollSize (Thread.Environment,
                                   Thread.Error,
                                   Collection,
                                   Size'Access));
         return Natural (Size);
      end if;
   end Table_Count;

   ----------------
   -- Table_Name --
   ----------------

   function Table_Name
     (Object : Change_Descriptor;
      Index  : Positive)
      return String
   is
      Table : constant OCIColl := Get_Table (Object, Index);
   begin
      return Get_Attr (OCIHandle (Table),
                       OCI_DTYPE_TABLE_CHDES,
                       OCI_ATTR_CHDES_TABLE_NAME);
   end Table_Name;

   ---------------------
   -- Table_Operation --
   ---------------------

   function Table_Operation
     (Object : Change_Descriptor;
      Index  : Positive)
      return Operation_Kinds
   is
      Table : constant OCIColl := Get_Table (Object, Index);
      Code  : constant Ub4 := Get_Attr_Ub4 (OCIHandle (Table),
                                            OCI_DTYPE_TABLE_CHDES,
                                            OCI_ATTR_CHDES_TABLE_OPFLAGS);
   begin
      return Operation_Kinds (Code);
   end Table_Operation;

   ----------------
   -- Table_Rows --
   ----------------

   function Table_Rows
     (Object : Change_Descriptor;
      Index  : Positive)
      return Row_Array
   is
      use Ada.Strings.Unbounded;

      package Convert is new
        System.Address_To_Access_Conversions (OCIHandle);

      Table : constant OCIColl := Get_Table (Object, Index);
      Rows  : constant OCIColl
        := Get_Attr_Coll (OCIHandle (Table),
                          OCI_DTYPE_TABLE_CHDES,
                          OCI_ATTR_CHDES_TABLE_ROW_CHANGES);
      Size  : aliased Sb4;
   begin
      if Rows = OCIColl (Empty_Handle) then
         return (1 .. 0 => (Null_Unbounded_String, 0));
      else
         Check_Error (OCICollSize (Thread.Environment,
                                   Thread.Error,
                                   Rows,
                                   Size'Access));

         declare
            Length : aliased UWord := UWord (Size);
            Exists : array (1 .. Size) of aliased Bool := (others => 0);
            Elem   : array (1 .. Size) of aliased DVoid;
            Result : Row_Array (1 .. Natural (Size));
            Index  : Positive := 1;
            Handle : OCIHandle;
         begin
            Check_Error (OCICollGetElemArray (Thread.Environment,
                                              Thread.Error,
                                              Rows,
                                              0,
                                              Exists (1)'Access,
                                              Elem (1)'Access,
                                              Length => Length'Access));
            for J in Elem'Range loop
               Handle := Convert.To_Pointer (Elem (J)).all;

               Result (Index).Id := To_Unbounded_String
                 (Get_Attr (Handle,
                            OCI_DTYPE_ROW_CHDES,
                            OCI_ATTR_CHDES_ROW_ROWID));

               Result (Index).Kind := Operation_Kinds
                 (Get_Attr_Ub4 (Handle,
                                OCI_DTYPE_ROW_CHDES,
                                OCI_ATTR_CHDES_ROW_OPFLAGS));
               Index := Index + 1;
            end loop;

            return Result;
         end;
      end if;
   end Table_Rows;

begin
   OCI.Environments.Set_Create_Mode_Flag (OCI_OBJECT);
end OCI.Thick.Notifications.DB_Change;
