/*!
   \file lib/ogsf/mc33_table.h

   \brief OGSF library -

   GRASS OpenGL gsurf OGSF Library

   LookUpTable for the MarchingCubes 33 Algorithm
   Thomas Lewiner, thomas.lewiner@polytechnique.org, Math Dept, PUC-Rio

   (C) 1999-2008 by the GRASS Development Team

   This program is free software under the
   GNU General Public License (>=v2).
   Read the file COPYING that comes with GRASS
   for details.

   \author Tomas Paudits, 2004
 */

#ifndef _CELL_TABLE_H
#define _CELL_TABLE_H

typedef struct {
    unsigned char nedges;
    unsigned char edges[13];
    unsigned char npolys;
    unsigned char polys[36];
} CELL_ENTRY;

/************************************************************************/
/* cube cases with appropriate configuration */
static const char cases[256][2] = {
    {0, -1},  {1, 0},   {1, 1},   {2, 0},  {1, 2},   {3, 0},   {2, 3},
    {5, 0},   {1, 3},   {2, 1},   {3, 3},  {5, 1},   {2, 5},   {5, 4},
    {5, 9},   {8, 0},   {1, 4},   {2, 2},  {3, 4},   {5, 2},   {4, 2},
    {6, 2},   {6, 9},   {11, 0},  {3, 8},  {5, 5},   {7, 3},   {9, 1},
    {6, 16},  {14, 3},  {12, 12}, {5, 24}, {1, 5},   {3, 1},   {2, 4},
    {5, 3},   {3, 6},   {7, 0},   {5, 10}, {9, 0},   {4, 3},   {6, 4},
    {6, 11},  {14, 1},  {6, 17},  {12, 4}, {11, 6},  {5, 25},  {2, 8},
    {5, 7},   {5, 12},  {8, 1},   {6, 18}, {12, 5},  {14, 7},  {5, 28},
    {6, 21},  {11, 4},  {12, 15}, {5, 30}, {10, 5},  {6, 32},  {6, 39},
    {2, 12},  {1, 6},   {4, 0},   {3, 5},  {6, 0},   {2, 6},   {6, 3},
    {5, 11},  {14, 0},  {3, 9},   {6, 5},  {7, 4},   {12, 1},  {5, 14},
    {11, 3},  {9, 4},   {5, 26},  {3, 10}, {6, 6},   {7, 5},   {12, 2},
    {6, 19},  {10, 1},  {12, 13}, {6, 24}, {7, 7},   {12, 9},  {13, 1},
    {7, 9},   {12, 20}, {6, 33},  {7, 13}, {3, 12},  {2, 10},  {6, 7},
    {5, 13},  {11, 2},  {5, 16},  {12, 7}, {8, 3},   {5, 29},  {6, 22},
    {10, 2},  {12, 17}, {6, 27},  {14, 9}, {6, 34},  {5, 39},  {2, 14},
    {5, 20},  {14, 5},  {9, 5},   {5, 32}, {11, 10}, {6, 35},  {5, 41},
    {2, 16},  {12, 23}, {6, 37},  {7, 14}, {3, 16},  {6, 46},  {4, 6},
    {3, 21},  {1, 8},   {1, 7},   {3, 2},  {4, 1},   {6, 1},   {3, 7},
    {7, 1},   {6, 10},  {12, 0},  {2, 7},  {5, 6},   {6, 12},  {11, 1},
    {5, 15},  {9, 2},   {14, 6},  {5, 27}, {2, 9},   {5, 8},   {6, 13},
    {14, 2},  {6, 20},  {12, 6},  {10, 3}, {6, 25},  {5, 18},  {8, 2},
    {12, 16}, {5, 31},  {11, 9},  {5, 34}, {6, 40},  {2, 13},  {3, 11},
    {7, 2},   {6, 14},  {12, 3},  {7, 6},  {13, 0},  {12, 14}, {7, 8},
    {6, 23},  {12, 10}, {10, 4},  {6, 28}, {12, 21}, {7, 10},  {6, 41},
    {3, 13},  {5, 21},  {9, 3},   {11, 8}, {5, 33},  {12, 22}, {7, 11},
    {6, 42},  {3, 14},  {14, 11}, {5, 36}, {6, 44},  {2, 17},  {6, 47},
    {3, 18},  {4, 7},   {1, 9},   {2, 11}, {6, 8},   {6, 15},  {10, 0},
    {5, 17},  {12, 8},  {11, 7},  {6, 26}, {5, 19},  {14, 4},  {12, 18},
    {6, 29},  {8, 4},   {5, 35},  {5, 40}, {2, 15},  {5, 22},  {11, 5},
    {12, 19}, {6, 30},  {14, 10}, {6, 36}, {6, 43},  {4, 4},   {9, 7},
    {5, 37},  {7, 15},  {3, 17},  {5, 44}, {2, 19},  {3, 22},  {1, 10},
    {5, 23},  {12, 11}, {14, 8},  {6, 31}, {9, 6},   {7, 12},  {5, 42},
    {3, 15},  {11, 11}, {6, 38},  {6, 45}, {4, 5},   {5, 45},  {3, 19},
    {2, 21},  {1, 11},  {8, 5},   {5, 38}, {5, 43},  {2, 18},  {5, 46},
    {3, 20},  {2, 22},  {1, 12},  {5, 47}, {2, 20},  {3, 23},  {1, 13},
    {2, 23},  {1, 14},  {1, 15},  {0, -1}};

/************************************************************************/
/* case, subcase tiling offset in tiling table */

#define OFFSET_T1         0
#define OFFSET_T2         16
#define OFFSET_T3_1       64
#define OFFSET_T3_2       40
#define OFFSET_T4_1       88
#define OFFSET_T4_2       96
#define OFFSET_T5         104
#define OFFSET_T6_1_1     200
#define OFFSET_T6_1_2     248
#define OFFSET_T6_2       152
#define OFFSET_T7_1       424
#define OFFSET_T7_2_S1    376
#define OFFSET_T7_2_S2    392
#define OFFSET_T7_2_S3    408
#define OFFSET_T7_3_S1    328
#define OFFSET_T7_3_S2    344
#define OFFSET_T7_3_S3    360
#define OFFSET_T7_4_1     296
#define OFFSET_T7_4_2     312
#define OFFSET_T8         440
#define OFFSET_T9         446
#define OFFSET_T10_1_1_S1 466
#define OFFSET_T10_1_1_S2 472
#define OFFSET_T10_1_2    478
#define OFFSET_T10_2_S1   454
#define OFFSET_T10_2_S2   460
#define OFFSET_T11        484
#define OFFSET_T12_1_1_S1 544
#define OFFSET_T12_1_1_S2 568
#define OFFSET_T12_1_2    592
#define OFFSET_T12_2_S1   496
#define OFFSET_T12_2_S2   520
#define OFFSET_T13_1_S1   616
#define OFFSET_T13_1_S2   714
#define OFFSET_T13_2_S1   618
#define OFFSET_T13_2_S2   702
#define OFFSET_T13_3_S1   630
#define OFFSET_T13_3_S2   678
#define OFFSET_T13_4      654
#define OFFSET_T13_5_1    662
#define OFFSET_T13_5_2    670
#define OFFSET_T14        716

/************************************************************************/
/* lookup table for mc 33 - all possible cube tilings */
static const CELL_ENTRY cell_table[] = {
    {
        3,
        {
            0,
            3,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        1,
        {
            0, 8, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        3,
        {
            0,
            1,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        1,
        {
            0, 1, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        3,
        {
            1,
            2,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        1,
        {
            1, 2, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        3,
        {
            2,
            3,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        1,
        {
            3, 11, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        3,
        {
            4,
            7,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        1,
        {
            4, 7, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        3,
        {
            4,
            5,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        1,
        {
            9, 5, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        3,
        {
            5,
            6,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        1,
        {
            10, 6, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        3,
        {
            6,
            7,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        1,
        {
            7, 6, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        3,
        {
            6,
            7,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        1,
        {
            7, 11, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        3,
        {
            5,
            6,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        1,
        {
            10, 5, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        3,
        {
            4,
            5,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        1,
        {
            9, 4, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        3,
        {
            4,
            7,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        1,
        {
            4, 8, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        3,
        {
            2,
            3,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        1,
        {
            3, 2, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        3,
        {
            1,
            2,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        1,
        {
            1, 10, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        3,
        {
            0,
            1,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        1,
        {
            0, 9, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        3,
        {
            0,
            3,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        1,
        {
            0, 3, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            1,
            3,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            1, 8, 3, 9, 8, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            0,
            2,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            0, 11, 2, 8, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            0,
            3,
            4,
            7,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            4, 3, 0, 7, 3, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            0,
            2,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            9, 2, 10, 0, 2, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            0,
            1,
            4,
            5,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            0, 5, 4, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            1,
            3,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            3, 10, 1, 11, 10, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            1,
            2,
            5,
            6,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            1, 6, 5, 2, 6, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            2,
            3,
            6,
            7,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            7, 2, 3, 6, 2, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            5,
            7,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            9, 7, 8, 5, 7, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            4,
            6,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            6, 8, 4, 11, 8, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            4,
            6,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            10, 4, 9, 6, 4, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            5,
            7,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            11, 5, 10, 7, 5, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            5,
            7,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            11, 10, 5, 7, 11, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            4,
            6,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            10, 9, 4, 6, 10, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            4,
            6,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            6, 4, 8, 11, 6, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            5,
            7,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            9, 8, 7, 5, 9, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            2,
            3,
            6,
            7,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            7, 3, 2, 6, 7, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            1,
            2,
            5,
            6,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            1, 5, 6, 2, 1, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            1,
            3,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            3, 1, 10, 11, 3, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            0,
            1,
            4,
            5,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            0, 4, 5, 1, 0, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            0,
            2,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            9, 10, 2, 0, 9, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            0,
            3,
            4,
            7,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            4, 0, 3, 7, 4, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            0,
            2,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            0, 2, 11, 8, 0, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            1,
            3,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            1, 3, 8, 9, 1, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            2,
            3,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            2, 3, 10, 3, 8, 10, 0, 1, 10, 0, 10, 8, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            3,
            4,
            5,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            8, 4, 3, 4, 5, 3, 9, 0, 3, 9, 3, 5, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            3,
            6,
            7,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            7, 8, 6, 8, 0, 6, 3, 11, 6, 3, 6, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            2,
            3,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            3, 0, 11, 0, 9, 11, 1, 2, 11, 1, 11, 9, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            4,
            7,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            4, 9, 7, 9, 1, 7, 0, 8, 7, 0, 7, 1, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            5,
            6,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            10, 1, 6, 1, 0, 6, 9, 0, 6, 9, 6, 5, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            1,
            2,
            4,
            5,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            5, 10, 4, 10, 2, 4, 1, 9, 4, 1, 4, 2, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            1,
            2,
            6,
            7,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            11, 2, 7, 2, 1, 7, 10, 6, 7, 10, 7, 1, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            2,
            3,
            4,
            7,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            11, 7, 2, 7, 4, 2, 8, 3, 2, 8, 2, 4, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            2,
            3,
            5,
            6,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            6, 11, 5, 11, 3, 5, 2, 10, 5, 2, 5, 3, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0,  0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            4,
            5,
            6,
            7,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            7, 6, 8, 6, 10, 8, 5, 4, 8, 5, 8, 10, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            4,
            5,
            6,
            7,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            6, 5, 11, 5, 9, 11, 4, 7, 11, 4, 11, 9, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            4,
            5,
            6,
            7,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            6, 5, 11, 5, 9, 11, 4, 7, 11, 4, 11, 9, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            4,
            5,
            6,
            7,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            7, 6, 8, 6, 10, 8, 5, 4, 8, 5, 8, 10, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            2,
            3,
            5,
            6,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            6, 11, 5, 11, 3, 5, 2, 10, 5, 2, 5, 3, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0,  0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            2,
            3,
            4,
            7,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            11, 7, 2, 7, 4, 2, 8, 3, 2, 8, 2, 4, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            1,
            2,
            6,
            7,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            11, 2, 7, 2, 1, 7, 10, 6, 7, 10, 7, 1, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            1,
            2,
            4,
            5,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            5, 10, 4, 10, 2, 4, 1, 9, 4, 1, 4, 2, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            5,
            6,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            10, 1, 6, 1, 0, 6, 9, 0, 6, 9, 6, 5, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            4,
            7,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            4, 9, 7, 9, 1, 7, 0, 8, 7, 0, 7, 1, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            2,
            3,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            3, 0, 11, 0, 9, 11, 1, 2, 11, 1, 11, 9, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            3,
            6,
            7,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            7, 8, 6, 8, 0, 6, 3, 11, 6, 3, 6, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            3,
            4,
            5,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            8, 4, 3, 4, 5, 3, 9, 0, 3, 9, 3, 5, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            2,
            3,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            2, 3, 10, 3, 8, 10, 0, 1, 10, 0, 10, 8, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            2,
            3,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            0, 8, 3, 1, 2, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            3,
            4,
            5,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            9, 5, 4, 0, 8, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            3,
            6,
            7,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            3, 0, 8, 11, 7, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            2,
            3,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            1, 9, 0, 2, 3, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            4,
            7,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            0, 1, 9, 8, 4, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            5,
            6,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            9, 0, 1, 5, 10, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            1,
            2,
            4,
            5,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            1, 2, 10, 9, 5, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            1,
            2,
            6,
            7,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            10, 1, 2, 6, 11, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            2,
            3,
            4,
            7,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            8, 4, 7, 3, 11, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            2,
            3,
            5,
            6,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            2, 3, 11, 10, 6, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            4,
            5,
            6,
            7,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            5, 10, 6, 4, 7, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            4,
            5,
            6,
            7,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            4, 9, 5, 7, 6, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            4,
            5,
            6,
            7,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            4, 9, 5, 7, 6, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            4,
            5,
            6,
            7,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            5, 10, 6, 4, 7, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            2,
            3,
            5,
            6,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            2, 3, 11, 10, 6, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            2,
            3,
            4,
            7,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            8, 4, 7, 3, 11, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            1,
            2,
            6,
            7,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            10, 1, 2, 6, 11, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            1,
            2,
            4,
            5,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            1, 2, 10, 9, 5, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            5,
            6,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            9, 0, 1, 5, 10, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            4,
            7,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            0, 1, 9, 8, 4, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            2,
            3,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            1, 9, 0, 2, 3, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            3,
            6,
            7,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            3, 0, 8, 11, 7, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            3,
            4,
            5,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            9, 5, 4, 0, 8, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            2,
            3,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            0, 8, 3, 1, 2, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            3,
            5,
            6,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            0, 8, 3, 5, 10, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            6,
            7,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            0, 1, 9, 11, 7, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            1,
            2,
            4,
            7,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            1, 2, 10, 8, 4, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            2,
            3,
            4,
            5,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            9, 5, 4, 2, 3, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            2,
            3,
            4,
            5,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            9, 5, 4, 2, 3, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            1,
            2,
            4,
            7,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            1, 2, 10, 8, 4, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            6,
            7,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            0, 1, 9, 11, 7, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            3,
            5,
            6,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            0, 8, 3, 5, 10, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            3,
            5,
            6,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        6,
        {
            0, 5, 8, 5, 8, 6, 8, 6, 3, 6, 3, 10, 3, 10, 0, 10, 0, 5,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            6,
            7,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        6,
        {
            1, 6, 9, 6, 9, 7, 9, 7, 0, 7, 0, 11, 0, 11, 1, 11, 1, 6,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,
        },
    },
    {
        6,
        {
            1,
            2,
            4,
            7,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        6,
        {
            2, 7, 10, 7, 10, 4, 10, 4, 1, 4, 1, 8, 1, 8, 2, 8, 2, 7,
            0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            2,
            3,
            4,
            5,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        6,
        {
            3, 4, 11, 4, 11, 5, 11, 5, 2, 5, 2, 9, 2, 9, 3, 9, 3, 4,
            0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            2,
            3,
            4,
            5,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        6,
        {
            3, 4, 11, 4, 11, 5, 11, 5, 2, 5, 2, 9, 2, 9, 3, 9, 3, 4,
            0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            1,
            2,
            4,
            7,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        6,
        {
            2, 7, 10, 7, 10, 4, 10, 4, 1, 4, 1, 8, 1, 8, 2, 8, 2, 7,
            0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            6,
            7,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        6,
        {
            1, 6, 9, 6, 9, 7, 9, 7, 0, 7, 0, 11, 0, 11, 1, 11, 1, 6,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,
        },
    },
    {
        6,
        {
            0,
            3,
            5,
            6,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        6,
        {
            0, 5, 8, 5, 8, 6, 8, 6, 3, 6, 3, 10, 3, 10, 0, 10, 0, 5,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,
        },
    },
    {
        5,
        {
            2,
            3,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            2, 8, 3, 2, 10, 8, 10, 9, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            1,
            2,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            1, 11, 2, 1, 9, 11, 9, 8, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0, 0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            1,
            3,
            4,
            7,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            4, 1, 9, 4, 7, 1, 7, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            1,
            3,
            4,
            5,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            8, 5, 4, 8, 3, 5, 3, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            0,
            1,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            0, 10, 1, 0, 8, 10, 8, 11, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0, 0, 0,  0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            0,
            2,
            4,
            7,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            11, 4, 7, 11, 2, 4, 2, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            0,
            2,
            6,
            7,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            7, 0, 8, 7, 6, 0, 6, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            0,
            3,
            5,
            7,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            9, 3, 0, 9, 5, 3, 5, 7, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            0,
            3,
            4,
            6,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            3, 6, 11, 3, 0, 6, 0, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            0,
            3,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            3, 9, 0, 3, 11, 9, 11, 10, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            0,
            2,
            4,
            5,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            5, 2, 10, 5, 4, 2, 4, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            0,
            2,
            5,
            6,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            9, 6, 5, 9, 0, 6, 0, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            0,
            1,
            5,
            7,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            0, 7, 8, 0, 1, 7, 1, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            0,
            1,
            4,
            6,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            10, 0, 1, 10, 6, 0, 6, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            1,
            3,
            5,
            6,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            6, 3, 11, 6, 5, 3, 5, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            1,
            3,
            6,
            7,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            10, 7, 6, 10, 1, 7, 1, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            1,
            2,
            4,
            6,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            1, 4, 9, 1, 2, 4, 2, 6, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            1,
            2,
            5,
            7,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            11, 1, 2, 11, 7, 1, 7, 5, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            2,
            3,
            4,
            6,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            8, 2, 3, 8, 4, 2, 4, 6, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            2,
            3,
            5,
            7,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            2, 5, 10, 2, 3, 5, 3, 7, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            6,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            7, 10, 6, 7, 8, 10, 8, 9, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0, 0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            5,
            6,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            6, 9, 5, 6, 11, 9, 11, 8, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            4,
            5,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            5, 8, 4, 5, 10, 8, 10, 11, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            4,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            4, 11, 7, 4, 9, 11, 9, 10, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0, 0, 0,  0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            4,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            4, 7, 11, 4, 11, 9, 9, 11, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0,  0, 0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            4,
            5,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            5, 4, 8, 5, 8, 10, 10, 8, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            5,
            6,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            6, 5, 9, 6, 9, 11, 11, 9, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            6,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            7, 6, 10, 7, 10, 8, 8, 10, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            2,
            3,
            5,
            7,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            2, 10, 5, 2, 5, 3, 3, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            2,
            3,
            4,
            6,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            8, 3, 2, 8, 2, 4, 4, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            1,
            2,
            5,
            7,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            11, 2, 1, 11, 1, 7, 7, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            1,
            2,
            4,
            6,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            1, 9, 4, 1, 4, 2, 2, 4, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            1,
            3,
            6,
            7,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            10, 6, 7, 10, 7, 1, 1, 7, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            1,
            3,
            5,
            6,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            6, 11, 3, 6, 3, 5, 5, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            0,
            1,
            4,
            6,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            10, 1, 0, 10, 0, 6, 6, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            0,
            1,
            5,
            7,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            0, 8, 7, 0, 7, 1, 1, 7, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            0,
            2,
            5,
            6,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            9, 5, 6, 9, 6, 0, 0, 6, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            0,
            2,
            4,
            5,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            5, 10, 2, 5, 2, 4, 4, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            0,
            3,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            3, 0, 9, 3, 9, 11, 11, 9, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            0,
            3,
            4,
            6,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            3, 11, 6, 3, 6, 0, 0, 6, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            0,
            3,
            5,
            7,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            9, 0, 3, 9, 3, 5, 5, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            0,
            2,
            6,
            7,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            7, 8, 0, 7, 0, 6, 6, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            0,
            2,
            4,
            7,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            11, 7, 4, 11, 4, 2, 2, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            0,
            1,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            0, 1, 10, 0, 10, 8, 8, 10, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0,  0, 0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            1,
            3,
            4,
            5,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            8, 4, 5, 8, 5, 3, 3, 5, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            1,
            3,
            4,
            7,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            4, 9, 1, 4, 1, 7, 7, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            1,
            2,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            1, 2, 11, 1, 11, 9, 9, 11, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        5,
        {
            2,
            3,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            2, 3, 8, 2, 8, 10, 10, 8, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            3,
            5,
            6,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            1, 10, 3, 10, 3, 6, 3, 6, 8, 6, 8, 5, 8, 5, 9, 0, 0, 0,
            0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            3,
            6,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            3, 11, 1, 11, 1, 6, 1, 6, 9, 6, 9, 7, 9, 7, 8, 0, 0, 0,
            0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            0, 1, 4, 1, 4, 10, 4, 10, 7, 10, 7, 2, 7, 2, 3, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            2, 3, 6, 3, 6, 8, 6, 8, 5, 8, 5, 0, 5, 0, 1, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            2,
            4,
            5,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            0, 9, 2, 9, 2, 5, 2, 5, 11, 5, 11, 4, 11, 4, 8, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            2,
            5,
            6,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            2, 10, 0, 10, 0, 5, 0, 5, 8, 5, 8, 6, 8, 6, 11, 0, 0, 0,
            0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            4, 5, 0, 5, 0, 10, 0, 10, 3, 10, 3, 6, 3, 6, 7, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            3,
            4,
            6,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            4, 8, 6, 8, 6, 3, 6, 3, 10, 3, 10, 0, 10, 0, 9, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            3,
            5,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            7, 8, 5, 8, 5, 0, 5, 0, 10, 0, 10, 3, 10, 3, 11, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,  0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            2,
            4,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            0, 8, 2, 8, 2, 7, 2, 7, 10, 7, 10, 4, 10, 4, 9, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            2,
            6,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            2, 11, 0, 11, 0, 7, 0, 7, 9, 7, 9, 6, 9, 6, 10, 0, 0, 0,
            0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            1, 2, 5, 2, 5, 11, 5, 11, 4, 11, 4, 3, 4, 3, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            3, 0, 7, 0, 7, 9, 7, 9, 6, 9, 6, 1, 6, 1, 2, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            4,
            6,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            4, 9, 6, 9, 6, 1, 6, 1, 11, 1, 11, 0, 11, 0, 8, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            5, 6, 1, 6, 1, 11, 1, 11, 0, 11, 0, 7, 0, 7, 4, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            5,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            5, 9, 7, 9, 7, 0, 7, 0, 11, 0, 11, 1, 11, 1, 10, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,  0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            3,
            4,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            3, 8, 1, 8, 1, 4, 1, 4, 10, 4, 10, 7, 10, 7, 11, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,  0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            3,
            4,
            5,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            1, 9, 3, 9, 3, 4, 3, 4, 11, 4, 11, 5, 11, 5, 10, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,  0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            2,
            5,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            5, 10, 7, 10, 7, 2, 7, 2, 8, 2, 8, 1, 8, 1, 9, 0, 0, 0,
            0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            6, 7, 2, 7, 2, 8, 2, 8, 1, 8, 1, 4, 1, 4, 5, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            2,
            4,
            6,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            6, 10, 4, 10, 4, 1, 4, 1, 8, 1, 8, 2, 8, 2, 11, 0, 0, 0,
            0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        7,
        {
            2,
            3,
            5,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            7, 11, 5, 11, 5, 2, 5, 2, 9, 2, 9, 3, 9, 3, 8, 0, 0, 0,
            0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            2,
            3,
            4,
            6,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            6, 11, 4, 11, 4, 3, 4, 3, 9, 3, 9, 2, 9, 2, 10, 0, 0, 0,
            0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        7,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            7, 4, 3, 4, 3, 9, 3, 9, 2, 9, 2, 5, 2, 5, 6, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            7, 4, 3, 4, 3, 9, 3, 9, 2, 9, 2, 5, 2, 5, 6, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            2,
            3,
            4,
            6,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            6, 11, 4, 11, 4, 3, 4, 3, 9, 3, 9, 2, 9, 2, 10, 0, 0, 0,
            0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        7,
        {
            2,
            3,
            5,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            7, 11, 5, 11, 5, 2, 5, 2, 9, 2, 9, 3, 9, 3, 8, 0, 0, 0,
            0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            2,
            4,
            6,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            6, 10, 4, 10, 4, 1, 4, 1, 8, 1, 8, 2, 8, 2, 11, 0, 0, 0,
            0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            6, 7, 2, 7, 2, 8, 2, 8, 1, 8, 1, 4, 1, 4, 5, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            2,
            5,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            5, 10, 7, 10, 7, 2, 7, 2, 8, 2, 8, 1, 8, 1, 9, 0, 0, 0,
            0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            3,
            4,
            5,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            1, 9, 3, 9, 3, 4, 3, 4, 11, 4, 11, 5, 11, 5, 10, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,  0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            3,
            4,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            3, 8, 1, 8, 1, 4, 1, 4, 10, 4, 10, 7, 10, 7, 11, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,  0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            5,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            5, 9, 7, 9, 7, 0, 7, 0, 11, 0, 11, 1, 11, 1, 10, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,  0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            5, 6, 1, 6, 1, 11, 1, 11, 0, 11, 0, 7, 0, 7, 4, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            4,
            6,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            4, 9, 6, 9, 6, 1, 6, 1, 11, 1, 11, 0, 11, 0, 8, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            3, 0, 7, 0, 7, 9, 7, 9, 6, 9, 6, 1, 6, 1, 2, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            1, 2, 5, 2, 5, 11, 5, 11, 4, 11, 4, 3, 4, 3, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            2,
            6,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            2, 11, 0, 11, 0, 7, 0, 7, 9, 7, 9, 6, 9, 6, 10, 0, 0, 0,
            0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            2,
            4,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            0, 8, 2, 8, 2, 7, 2, 7, 10, 7, 10, 4, 10, 4, 9, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            3,
            5,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            7, 8, 5, 8, 5, 0, 5, 0, 10, 0, 10, 3, 10, 3, 11, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,  0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            3,
            4,
            6,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            4, 8, 6, 8, 6, 3, 6, 3, 10, 3, 10, 0, 10, 0, 9, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            4, 5, 0, 5, 0, 10, 0, 10, 3, 10, 3, 6, 3, 6, 7, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            2,
            5,
            6,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            2, 10, 0, 10, 0, 5, 0, 5, 8, 5, 8, 6, 8, 6, 11, 0, 0, 0,
            0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            2,
            4,
            5,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            0, 9, 2, 9, 2, 5, 2, 5, 11, 5, 11, 4, 11, 4, 8, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            2, 3, 6, 3, 6, 8, 6, 8, 5, 8, 5, 0, 5, 0, 1, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            0, 1, 4, 1, 4, 10, 4, 10, 7, 10, 7, 2, 7, 2, 3, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            3,
            6,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            3, 11, 1, 11, 1, 6, 1, 6, 9, 6, 9, 7, 9, 7, 8, 0, 0, 0,
            0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            3,
            5,
            6,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        5,
        {
            1, 10, 3, 10, 3, 6, 3, 6, 8, 6, 8, 5, 8, 5, 9, 0, 0, 0,
            0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            3,
            5,
            6,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            10, 5, 6, 3, 1, 8, 1, 8, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            3,
            6,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            11, 7, 6, 1, 3, 9, 3, 9, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            1, 2, 10, 4, 0, 7, 0, 7, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            3, 0, 8, 6, 2, 5, 2, 5, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            2,
            4,
            5,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            9, 4, 5, 2, 0, 11, 0, 11, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            2,
            5,
            6,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            10, 6, 5, 0, 2, 8, 2, 8, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            5, 6, 10, 0, 4, 3, 4, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            3,
            4,
            6,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            8, 0, 3, 6, 4, 10, 4, 10, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            3,
            5,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            8, 3, 0, 5, 7, 10, 7, 10, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            2,
            4,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            8, 4, 7, 2, 0, 10, 0, 10, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            2,
            6,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            11, 6, 7, 0, 2, 9, 2, 9, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            2, 3, 11, 5, 1, 4, 1, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            0, 1, 9, 7, 3, 6, 3, 6, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            4,
            6,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            9, 0, 1, 6, 4, 11, 4, 11, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            6, 7, 11, 1, 5, 0, 5, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            5,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            9, 1, 0, 7, 5, 11, 5, 11, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            3,
            4,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            8, 7, 4, 1, 3, 10, 3, 10, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            3,
            4,
            5,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            9, 5, 4, 3, 1, 11, 1, 11, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            2,
            5,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            10, 1, 2, 7, 5, 8, 5, 8, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            7, 4, 8, 2, 6, 1, 6, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            2,
            4,
            6,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            10, 2, 1, 4, 6, 8, 6, 8, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            2,
            3,
            5,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            11, 3, 2, 5, 7, 9, 7, 9, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            2,
            3,
            4,
            6,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            11, 2, 3, 4, 6, 9, 6, 9, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            4, 5, 9, 3, 7, 2, 7, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            4, 5, 9, 3, 7, 2, 7, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            2,
            3,
            4,
            6,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            11, 2, 3, 4, 6, 9, 6, 9, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            2,
            3,
            5,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            11, 3, 2, 5, 7, 9, 7, 9, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            2,
            4,
            6,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            10, 2, 1, 4, 6, 8, 6, 8, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            7, 4, 8, 2, 6, 1, 6, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            2,
            5,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            10, 1, 2, 7, 5, 8, 5, 8, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            3,
            4,
            5,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            9, 5, 4, 3, 1, 11, 1, 11, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            3,
            4,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            8, 7, 4, 1, 3, 10, 3, 10, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            5,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            9, 1, 0, 7, 5, 11, 5, 11, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            6, 7, 11, 1, 5, 0, 5, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            4,
            6,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            9, 0, 1, 6, 4, 11, 4, 11, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            0, 1, 9, 7, 3, 6, 3, 6, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            2, 3, 11, 5, 1, 4, 1, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            2,
            6,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            11, 6, 7, 0, 2, 9, 2, 9, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            2,
            4,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            8, 4, 7, 2, 0, 10, 0, 10, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            3,
            5,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            8, 3, 0, 5, 7, 10, 7, 10, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            3,
            4,
            6,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            8, 0, 3, 6, 4, 10, 4, 10, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            5, 6, 10, 0, 4, 3, 4, 3, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            2,
            5,
            6,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            10, 6, 5, 0, 2, 8, 2, 8, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            2,
            4,
            5,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            9, 4, 5, 2, 0, 11, 0, 11, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            3, 0, 8, 6, 2, 5, 2, 5, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            1, 2, 10, 4, 0, 7, 0, 7, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            3,
            6,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            11, 7, 6, 1, 3, 9, 3, 9, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            3,
            5,
            6,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        3,
        {
            10, 5, 6, 3, 1, 8, 1, 8, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            3,
            5,
            6,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            1, 10, 3,  10, 3, 6, 3, 6, 8, 6, 8, 5, 8, 5, 9, 5, 9, 1,
            5, 1,  10, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            3,
            6,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            3, 11, 1,  11, 1, 6, 1, 6, 9, 6, 9, 7, 9, 7, 8, 7, 8, 3,
            7, 3,  11, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            0, 1, 4, 1, 4, 10, 4, 10, 7, 10, 7, 2, 7, 2, 3, 2, 3, 0,
            2, 0, 1, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            2, 3, 6, 3, 6, 8, 6, 8, 5, 8, 5, 0, 5, 0, 1, 0, 1, 2,
            0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            2,
            4,
            5,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            0, 9, 2, 9, 2, 5, 2, 5, 11, 5, 11, 4, 11, 4, 8, 4, 8, 0,
            4, 0, 9, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            2,
            5,
            6,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            2, 10, 0,  10, 0, 5, 0, 5, 8, 5, 8, 6, 8, 6, 11, 6, 11, 2,
            6, 2,  10, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0,
        },
    },
    {
        7,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            4, 5, 0, 5, 0, 10, 0, 10, 3, 10, 3, 6, 3, 6, 7, 6, 7, 4,
            6, 4, 5, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            3,
            4,
            6,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            4, 8, 6, 8, 6, 3, 6, 3, 10, 3, 10, 0, 10, 0, 9, 0, 9, 4,
            0, 4, 8, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            3,
            5,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            7, 8, 5, 8, 5, 0, 5, 0, 10, 0, 10, 3, 10, 3, 11, 3, 11, 7,
            3, 7, 8, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,  0, 0,  0,
        },
    },
    {
        7,
        {
            0,
            2,
            4,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            0, 8, 2, 8, 2, 7, 2, 7, 10, 7, 10, 4, 10, 4, 9, 4, 9, 0,
            4, 0, 8, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            2,
            6,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            2, 11, 0,  11, 0, 7, 0, 7, 9, 7, 9, 6, 9, 6, 10, 6, 10, 2,
            6, 2,  11, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            1, 2, 5, 2, 5, 11, 5, 11, 4, 11, 4, 3, 4, 3, 0, 3, 0, 1,
            3, 1, 2, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            3, 0, 7, 0, 7, 9, 7, 9, 6, 9, 6, 1, 6, 1, 2, 1, 2, 3,
            1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            4,
            6,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            4, 9, 6, 9, 6, 1, 6, 1, 11, 1, 11, 0, 11, 0, 8, 0, 8, 4,
            0, 4, 9, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            5, 6, 1, 6, 1, 11, 1, 11, 0, 11, 0, 7, 0, 7, 4, 7, 4, 5,
            7, 5, 6, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            5,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            5, 9, 7, 9, 7, 0, 7, 0, 11, 0, 11, 1, 11, 1, 10, 1, 10, 5,
            1, 5, 9, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,  0, 0,  0,
        },
    },
    {
        7,
        {
            1,
            3,
            4,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            3, 8, 1, 8, 1, 4, 1, 4, 10, 4, 10, 7, 10, 7, 11, 7, 11, 3,
            7, 3, 8, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,  0, 0,  0,
        },
    },
    {
        7,
        {
            1,
            3,
            4,
            5,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            1, 9, 3, 9, 3, 4, 3, 4, 11, 4, 11, 5, 11, 5, 10, 5, 10, 1,
            5, 1, 9, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,  0, 0,  0,
        },
    },
    {
        7,
        {
            1,
            2,
            5,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            5, 10, 7,  10, 7, 2, 7, 2, 8, 2, 8, 1, 8, 1, 9, 1, 9, 5,
            1, 5,  10, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            6, 7, 2, 7, 2, 8, 2, 8, 1, 8, 1, 4, 1, 4, 5, 4, 5, 6,
            4, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            2,
            4,
            6,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            6, 10, 4,  10, 4, 1, 4, 1, 8, 1, 8, 2, 8, 2, 11, 2, 11, 6,
            2, 6,  10, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0,
        },
    },
    {
        7,
        {
            2,
            3,
            5,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            7, 11, 5,  11, 5, 2, 5, 2, 9, 2, 9, 3, 9, 3, 8, 3, 8, 7,
            3, 7,  11, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            2,
            3,
            4,
            6,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            6, 11, 4,  11, 4, 3, 4, 3, 9, 3, 9, 2, 9, 2, 10, 2, 10, 6,
            2, 6,  11, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0,
        },
    },
    {
        7,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            7, 4, 3, 4, 3, 9, 3, 9, 2, 9, 2, 5, 2, 5, 6, 5, 6, 7,
            5, 7, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            7, 4, 3, 4, 3, 9, 3, 9, 2, 9, 2, 5, 2, 5, 6, 5, 6, 7,
            5, 7, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            2,
            3,
            4,
            6,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            6, 11, 4,  11, 4, 3, 4, 3, 9, 3, 9, 2, 9, 2, 10, 2, 10, 6,
            2, 6,  11, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0,
        },
    },
    {
        7,
        {
            2,
            3,
            5,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            7, 11, 5,  11, 5, 2, 5, 2, 9, 2, 9, 3, 9, 3, 8, 3, 8, 7,
            3, 7,  11, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            2,
            4,
            6,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            6, 10, 4,  10, 4, 1, 4, 1, 8, 1, 8, 2, 8, 2, 11, 2, 11, 6,
            2, 6,  10, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0,
        },
    },
    {
        7,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            6, 7, 2, 7, 2, 8, 2, 8, 1, 8, 1, 4, 1, 4, 5, 4, 5, 6,
            4, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            2,
            5,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            5, 10, 7,  10, 7, 2, 7, 2, 8, 2, 8, 1, 8, 1, 9, 1, 9, 5,
            1, 5,  10, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            3,
            4,
            5,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            1, 9, 3, 9, 3, 4, 3, 4, 11, 4, 11, 5, 11, 5, 10, 5, 10, 1,
            5, 1, 9, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,  0, 0,  0,
        },
    },
    {
        7,
        {
            1,
            3,
            4,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            3, 8, 1, 8, 1, 4, 1, 4, 10, 4, 10, 7, 10, 7, 11, 7, 11, 3,
            7, 3, 8, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,  0, 0,  0,
        },
    },
    {
        7,
        {
            0,
            1,
            5,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            5, 9, 7, 9, 7, 0, 7, 0, 11, 0, 11, 1, 11, 1, 10, 1, 10, 5,
            1, 5, 9, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,  0, 0,  0,
        },
    },
    {
        7,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            5, 6, 1, 6, 1, 11, 1, 11, 0, 11, 0, 7, 0, 7, 4, 7, 4, 5,
            7, 5, 6, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            4,
            6,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            4, 9, 6, 9, 6, 1, 6, 1, 11, 1, 11, 0, 11, 0, 8, 0, 8, 4,
            0, 4, 9, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            3, 0, 7, 0, 7, 9, 7, 9, 6, 9, 6, 1, 6, 1, 2, 1, 2, 3,
            1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            1, 2, 5, 2, 5, 11, 5, 11, 4, 11, 4, 3, 4, 3, 0, 3, 0, 1,
            3, 1, 2, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            2,
            6,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            2, 11, 0,  11, 0, 7, 0, 7, 9, 7, 9, 6, 9, 6, 10, 6, 10, 2,
            6, 2,  11, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0,
        },
    },
    {
        7,
        {
            0,
            2,
            4,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            0, 8, 2, 8, 2, 7, 2, 7, 10, 7, 10, 4, 10, 4, 9, 4, 9, 0,
            4, 0, 8, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            3,
            5,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            7, 8, 5, 8, 5, 0, 5, 0, 10, 0, 10, 3, 10, 3, 11, 3, 11, 7,
            3, 7, 8, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,  0, 0,  0,
        },
    },
    {
        7,
        {
            0,
            3,
            4,
            6,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            4, 8, 6, 8, 6, 3, 6, 3, 10, 3, 10, 0, 10, 0, 9, 0, 9, 4,
            0, 4, 8, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            4, 5, 0, 5, 0, 10, 0, 10, 3, 10, 3, 6, 3, 6, 7, 6, 7, 4,
            6, 4, 5, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            2,
            5,
            6,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            2, 10, 0,  10, 0, 5, 0, 5, 8, 5, 8, 6, 8, 6, 11, 6, 11, 2,
            6, 2,  10, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0,
        },
    },
    {
        7,
        {
            0,
            2,
            4,
            5,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            0, 9, 2, 9, 2, 5, 2, 5, 11, 5, 11, 4, 11, 4, 8, 4, 8, 0,
            4, 0, 9, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            8,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            2, 3, 6, 3, 6, 8, 6, 8, 5, 8, 5, 0, 5, 0, 1, 0, 1, 2,
            0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            0, 1, 4, 1, 4, 10, 4, 10, 7, 10, 7, 2, 7, 2, 3, 2, 3, 0,
            2, 0, 1, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            3,
            6,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            3, 11, 1,  11, 1, 6, 1, 6, 9, 6, 9, 7, 9, 7, 8, 7, 8, 3,
            7, 3,  11, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        7,
        {
            1,
            3,
            5,
            6,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        7,
        {
            1, 10, 3,  10, 3, 6, 3, 6, 8, 6, 8, 5, 8, 5, 9, 5, 9, 1,
            5, 1,  10, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        5,
        {
            8, 4, 3, 4, 3, 10, 3, 10, 2, 4, 10, 5, 0, 1, 9, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            10, 6, 1, 6, 1, 8, 1, 8, 0, 6, 8, 7, 2, 3, 11, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            11, 3, 6, 3, 6, 9, 6, 9, 5, 3, 9, 0, 7, 4, 8, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            11, 7, 2, 7, 2, 9, 2, 9, 1, 7, 9, 4, 3, 0, 8, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            9, 5, 0, 5, 0, 11, 0, 11, 3, 5, 11, 6, 1, 2, 10, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        5,
        {
            8, 0, 7, 0, 7, 10, 7, 10, 6, 0, 10, 1, 4, 5, 9, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            9, 1, 4, 1, 4, 11, 4, 11, 7, 1, 11, 2, 5, 6, 10, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            10, 2, 5, 2, 5, 8, 5, 8, 4, 2, 8, 3, 6, 7, 11, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            10, 2, 5, 2, 5, 8, 5, 8, 4, 2, 8, 3, 6, 7, 11, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            9, 1, 4, 1, 4, 11, 4, 11, 7, 1, 11, 2, 5, 6, 10, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        5,
        {
            8, 0, 7, 0, 7, 10, 7, 10, 6, 0, 10, 1, 4, 5, 9, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            9, 5, 0, 5, 0, 11, 0, 11, 3, 5, 11, 6, 1, 2, 10, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            11, 7, 2, 7, 2, 9, 2, 9, 1, 7, 9, 4, 3, 0, 8, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            11, 3, 6, 3, 6, 9, 6, 9, 5, 3, 9, 0, 7, 4, 8, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            10, 6, 1, 6, 1, 8, 1, 8, 0, 6, 8, 7, 2, 3, 11, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        5,
        {
            8, 4, 3, 4, 3, 10, 3, 10, 2, 4, 10, 5, 0, 1, 9, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        9,
        {
            8, 4, 9, 4, 9, 5, 9, 5, 10, 9, 10, 1, 10, 1, 2, 1, 2, 0,
            2, 0, 3, 0, 3, 8, 0, 8, 9,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        9,
        {
            10, 6, 11, 6, 11, 7,  11, 7,  8,  11, 8, 3, 8, 3, 0, 3, 0, 2,
            0,  2, 1,  2, 1,  10, 2,  10, 11, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        9,
        {
            11, 3, 8, 3, 8, 0,  8, 0,  9, 8, 9, 4, 9, 4, 5, 4, 5, 7,
            5,  7, 6, 7, 6, 11, 7, 11, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        9,
        {
            11, 7, 8, 7, 8, 4,  8, 4,  9, 8, 9, 0, 9, 0, 1, 0, 1, 3,
            1,  3, 2, 3, 2, 11, 3, 11, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        9,
        {
            9, 5, 10, 5, 10, 6, 10, 6, 11, 10, 11, 2, 11, 2, 3, 2, 3, 1,
            3, 1, 0,  1, 0,  9, 1,  9, 10, 0,  0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        9,
        {
            8, 0, 9, 0, 9, 1, 9, 1, 10, 9, 10, 5, 10, 5, 6, 5, 6, 4,
            6, 4, 7, 4, 7, 8, 4, 8, 9,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        9,
        {
            9, 1, 10, 1, 10, 2, 10, 2, 11, 10, 11, 6, 11, 6, 7, 6, 7, 5,
            7, 5, 4,  5, 4,  9, 5,  9, 10, 0,  0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        9,
        {
            10, 2, 11, 2, 11, 3,  11, 3,  8,  11, 8, 7, 8, 7, 4, 7, 4, 6,
            4,  6, 5,  6, 5,  10, 6,  10, 11, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        9,
        {
            10, 2, 11, 2, 11, 3,  11, 3,  8,  11, 8, 7, 8, 7, 4, 7, 4, 6,
            4,  6, 5,  6, 5,  10, 6,  10, 11, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        9,
        {
            9, 1, 10, 1, 10, 2, 10, 2, 11, 10, 11, 6, 11, 6, 7, 6, 7, 5,
            7, 5, 4,  5, 4,  9, 5,  9, 10, 0,  0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        9,
        {
            8, 0, 9, 0, 9, 1, 9, 1, 10, 9, 10, 5, 10, 5, 6, 5, 6, 4,
            6, 4, 7, 4, 7, 8, 4, 8, 9,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        9,
        {
            9, 5, 10, 5, 10, 6, 10, 6, 11, 10, 11, 2, 11, 2, 3, 2, 3, 1,
            3, 1, 0,  1, 0,  9, 1,  9, 10, 0,  0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        9,
        {
            11, 7, 8, 7, 8, 4,  8, 4,  9, 8, 9, 0, 9, 0, 1, 0, 1, 3,
            1,  3, 2, 3, 2, 11, 3, 11, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        9,
        {
            11, 3, 8, 3, 8, 0,  8, 0,  9, 8, 9, 4, 9, 4, 5, 4, 5, 7,
            5,  7, 6, 7, 6, 11, 7, 11, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        9,
        {
            10, 6, 11, 6, 11, 7,  11, 7,  8,  11, 8, 3, 8, 3, 0, 3, 0, 2,
            0,  2, 1,  2, 1,  10, 2,  10, 11, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        9,
        {
            8, 4, 9, 4, 9, 5, 9, 5, 10, 9, 10, 1, 10, 1, 2, 1, 2, 0,
            2, 0, 3, 0, 3, 8, 0, 8, 9,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            8,
            9,
            10,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 2, 10, 12, 10, 5, 12, 5, 4, 12, 4, 8, 12, 8, 3, 12, 3, 0,
            12, 0, 9,  12, 9,  1, 12, 1, 2, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            8,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 0, 8,  12, 8,  7, 12, 7, 6, 12, 6, 10, 12, 10, 1, 12, 1, 2,
            12, 2, 11, 12, 11, 3, 12, 3, 0, 0,  0, 0,  0,  0,  0, 0,  0, 0,
        },
    },
    {
        10,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 5, 9, 12, 9, 0, 12, 0, 3, 12, 3, 11, 12, 11, 6, 12, 6, 7,
            12, 7, 8, 12, 8, 4, 12, 4, 5, 0,  0, 0,  0,  0,  0, 0,  0, 0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            8,
            9,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 1, 9, 12, 9, 4, 12, 4, 7, 12, 7, 11, 12, 11, 2, 12, 2, 3,
            12, 3, 8, 12, 8, 0, 12, 0, 1, 0,  0, 0,  0,  0,  0, 0,  0, 0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 3, 11, 12, 11, 6, 12, 6, 5, 12, 5, 9, 12, 9, 0, 12, 0, 1,
            12, 1, 10, 12, 10, 2, 12, 2, 3, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        10,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 6, 10, 12, 10, 1, 12, 1, 0, 12, 0, 8, 12, 8, 7, 12, 7, 4,
            12, 4, 9,  12, 9,  5, 12, 5, 6, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        10,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 7, 11, 12, 11, 2, 12, 2, 1, 12, 1, 9, 12, 9, 4, 12, 4, 5,
            12, 5, 10, 12, 10, 6, 12, 6, 7, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        10,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 4, 8,  12, 8,  3, 12, 3, 2, 12, 2, 10, 12, 10, 5, 12, 5, 6,
            12, 6, 11, 12, 11, 7, 12, 7, 4, 0,  0, 0,  0,  0,  0, 0,  0, 0,
        },
    },
    {
        10,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 4, 8,  12, 8,  3, 12, 3, 2, 12, 2, 10, 12, 10, 5, 12, 5, 6,
            12, 6, 11, 12, 11, 7, 12, 7, 4, 0,  0, 0,  0,  0,  0, 0,  0, 0,
        },
    },
    {
        10,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 7, 11, 12, 11, 2, 12, 2, 1, 12, 1, 9, 12, 9, 4, 12, 4, 5,
            12, 5, 10, 12, 10, 6, 12, 6, 7, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        10,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 6, 10, 12, 10, 1, 12, 1, 0, 12, 0, 8, 12, 8, 7, 12, 7, 4,
            12, 4, 9,  12, 9,  5, 12, 5, 6, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 3, 11, 12, 11, 6, 12, 6, 5, 12, 5, 9, 12, 9, 0, 12, 0, 1,
            12, 1, 10, 12, 10, 2, 12, 2, 3, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            8,
            9,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 1, 9, 12, 9, 4, 12, 4, 7, 12, 7, 11, 12, 11, 2, 12, 2, 3,
            12, 3, 8, 12, 8, 0, 12, 0, 1, 0,  0, 0,  0,  0,  0, 0,  0, 0,
        },
    },
    {
        10,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 5, 9, 12, 9, 0, 12, 0, 3, 12, 3, 11, 12, 11, 6, 12, 6, 7,
            12, 7, 8, 12, 8, 4, 12, 4, 5, 0,  0, 0,  0,  0,  0, 0,  0, 0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            8,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 0, 8,  12, 8,  7, 12, 7, 6, 12, 6, 10, 12, 10, 1, 12, 1, 2,
            12, 2, 11, 12, 11, 3, 12, 3, 0, 0,  0, 0,  0,  0,  0, 0,  0, 0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            8,
            9,
            10,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 2, 10, 12, 10, 5, 12, 5, 4, 12, 4, 8, 12, 8, 3, 12, 3, 0,
            12, 0, 9,  12, 9,  1, 12, 1, 2, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            8,
            9,
            10,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 5, 4, 12, 4, 8, 12, 8, 3, 12, 3, 2, 12, 2, 10, 12, 10, 1,
            12, 1, 0, 12, 0, 9, 12, 9, 5, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            8,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 7, 6, 12, 6, 10, 12, 10, 1, 12, 1, 0, 12, 0, 8, 12, 8, 3,
            12, 3, 2, 12, 2, 11, 12, 11, 7, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        10,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 0, 3, 12, 3, 11, 12, 11, 6, 12, 6, 5, 12, 5, 9, 12, 9, 4,
            12, 4, 7, 12, 7, 8,  12, 8,  0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            8,
            9,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 4, 7, 12, 7, 11, 12, 11, 2, 12, 2, 1, 12, 1, 9, 12, 9, 0,
            12, 0, 3, 12, 3, 8,  12, 8,  4, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 6, 5, 12, 5, 9,  12, 9,  0, 12, 0, 3, 12, 3, 11, 12, 11, 2,
            12, 2, 1, 12, 1, 10, 12, 10, 6, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        10,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 1, 0, 12, 0, 8, 12, 8, 7, 12, 7, 6, 12, 6, 10, 12, 10, 5,
            12, 5, 4, 12, 4, 9, 12, 9, 1, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        10,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 2, 1, 12, 1, 9,  12, 9,  4, 12, 4, 7, 12, 7, 11, 12, 11, 6,
            12, 6, 5, 12, 5, 10, 12, 10, 2, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        10,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 3, 2, 12, 2, 10, 12, 10, 5, 12, 5, 4, 12, 4, 8, 12, 8, 7,
            12, 7, 6, 12, 6, 11, 12, 11, 3, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        10,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 3, 2, 12, 2, 10, 12, 10, 5, 12, 5, 4, 12, 4, 8, 12, 8, 7,
            12, 7, 6, 12, 6, 11, 12, 11, 3, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        10,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 2, 1, 12, 1, 9,  12, 9,  4, 12, 4, 7, 12, 7, 11, 12, 11, 6,
            12, 6, 5, 12, 5, 10, 12, 10, 2, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        10,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 1, 0, 12, 0, 8, 12, 8, 7, 12, 7, 6, 12, 6, 10, 12, 10, 5,
            12, 5, 4, 12, 4, 9, 12, 9, 1, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 6, 5, 12, 5, 9,  12, 9,  0, 12, 0, 3, 12, 3, 11, 12, 11, 2,
            12, 2, 1, 12, 1, 10, 12, 10, 6, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            8,
            9,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 4, 7, 12, 7, 11, 12, 11, 2, 12, 2, 1, 12, 1, 9, 12, 9, 0,
            12, 0, 3, 12, 3, 8,  12, 8,  4, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        10,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 0, 3, 12, 3, 11, 12, 11, 6, 12, 6, 5, 12, 5, 9, 12, 9, 4,
            12, 4, 7, 12, 7, 8,  12, 8,  0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            8,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 7, 6, 12, 6, 10, 12, 10, 1, 12, 1, 0, 12, 0, 8, 12, 8, 3,
            12, 3, 2, 12, 2, 11, 12, 11, 7, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            8,
            9,
            10,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 5, 4, 12, 4, 8, 12, 8, 3, 12, 3, 2, 12, 2, 10, 12, 10, 1,
            12, 1, 0, 12, 0, 9, 12, 9, 5, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            8,
            9,
            10,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 4, 5, 12, 5, 10, 12, 10, 2, 12, 2, 3, 12, 3, 8, 12, 8, 0,
            12, 0, 1, 12, 1, 9,  12, 9,  4, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            8,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 6, 7, 12, 7, 8,  12, 8,  0, 12, 0, 1, 12, 1, 10, 12, 10, 2,
            12, 2, 3, 12, 3, 11, 12, 11, 6, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        10,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 3, 0, 12, 0, 9, 12, 9, 5, 12, 5, 6, 12, 6, 11, 12, 11, 7,
            12, 7, 4, 12, 4, 8, 12, 8, 3, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            8,
            9,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 7, 4, 12, 4, 9, 12, 9, 1, 12, 1, 2, 12, 2, 11, 12, 11, 3,
            12, 3, 0, 12, 0, 8, 12, 8, 7, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 5, 6, 12, 6, 11, 12, 11, 3, 12, 3, 0, 12, 0, 9, 12, 9, 1,
            12, 1, 2, 12, 2, 10, 12, 10, 5, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        10,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 0, 1, 12, 1, 10, 12, 10, 6, 12, 6, 7, 12, 7, 8, 12, 8, 4,
            12, 4, 5, 12, 5, 9,  12, 9,  0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        10,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 1, 2, 12, 2, 11, 12, 11, 7, 12, 7, 4, 12, 4, 9, 12, 9, 5,
            12, 5, 6, 12, 6, 10, 12, 10, 1, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        10,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 2, 3, 12, 3, 8,  12, 8,  4, 12, 4, 5, 12, 5, 10, 12, 10, 6,
            12, 6, 7, 12, 7, 11, 12, 11, 2, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        10,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 2, 3, 12, 3, 8,  12, 8,  4, 12, 4, 5, 12, 5, 10, 12, 10, 6,
            12, 6, 7, 12, 7, 11, 12, 11, 2, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        10,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 1, 2, 12, 2, 11, 12, 11, 7, 12, 7, 4, 12, 4, 9, 12, 9, 5,
            12, 5, 6, 12, 6, 10, 12, 10, 1, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        10,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 0, 1, 12, 1, 10, 12, 10, 6, 12, 6, 7, 12, 7, 8, 12, 8, 4,
            12, 4, 5, 12, 5, 9,  12, 9,  0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 5, 6, 12, 6, 11, 12, 11, 3, 12, 3, 0, 12, 0, 9, 12, 9, 1,
            12, 1, 2, 12, 2, 10, 12, 10, 5, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            8,
            9,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 7, 4, 12, 4, 9, 12, 9, 1, 12, 1, 2, 12, 2, 11, 12, 11, 3,
            12, 3, 0, 12, 0, 8, 12, 8, 7, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        10,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 3, 0, 12, 0, 9, 12, 9, 5, 12, 5, 6, 12, 6, 11, 12, 11, 7,
            12, 7, 4, 12, 4, 8, 12, 8, 3, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            8,
            10,
            11,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 6, 7, 12, 7, 8,  12, 8,  0, 12, 0, 1, 12, 1, 10, 12, 10, 2,
            12, 2, 3, 12, 3, 11, 12, 11, 6, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        10,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            8,
            9,
            10,
            12,
            0,
            0,
            0,
        },
        9,
        {
            12, 4, 5, 12, 5, 10, 12, 10, 2, 12, 2, 3, 12, 3, 8, 12, 8, 0,
            12, 0, 1, 12, 1, 9,  12, 9,  4, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        5,
        {
            1, 2, 10, 8, 4, 3, 4, 3, 5, 3, 5, 0, 5, 0, 9, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            3, 0, 8, 10, 6, 1, 6, 1, 7, 1, 7, 2, 7, 2, 11, 0, 0, 0,
            0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            4, 5, 9, 11, 3, 6, 3, 6, 0, 6, 0, 7, 0, 7, 8, 0, 0, 0,
            0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            0, 1, 9, 11, 7, 2, 7, 2, 4, 2, 4, 3, 4, 3, 8, 0, 0, 0,
            0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            2, 3, 11, 9, 5, 0, 5, 0, 6, 0, 6, 1, 6, 1, 10, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        5,
        {
            5, 6, 10, 8, 0, 7, 0, 7, 1, 7, 1, 4, 1, 4, 9, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            6, 7, 11, 9, 1, 4, 1, 4, 2, 4, 2, 5, 2, 5, 10, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            7, 4, 8, 10, 2, 5, 2, 5, 3, 5, 3, 6, 3, 6, 11, 0, 0, 0,
            0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            7, 4, 8, 10, 2, 5, 2, 5, 3, 5, 3, 6, 3, 6, 11, 0, 0, 0,
            0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            6, 7, 11, 9, 1, 4, 1, 4, 2, 4, 2, 5, 2, 5, 10, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        5,
        {
            5, 6, 10, 8, 0, 7, 0, 7, 1, 7, 1, 4, 1, 4, 9, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            2, 3, 11, 9, 5, 0, 5, 0, 6, 0, 6, 1, 6, 1, 10, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            0, 1, 9, 11, 7, 2, 7, 2, 4, 2, 4, 3, 4, 3, 8, 0, 0, 0,
            0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            4, 5, 9, 11, 3, 6, 3, 6, 0, 6, 0, 7, 0, 7, 8, 0, 0, 0,
            0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            3, 0, 8, 10, 6, 1, 6, 1, 7, 1, 7, 2, 7, 2, 11, 0, 0, 0,
            0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        5,
        {
            1, 2, 10, 8, 4, 3, 4, 3, 5, 3, 5, 0, 5, 0, 9, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        5,
        {
            3, 0, 8, 9, 1, 4, 1, 4, 2, 4, 2, 5, 2, 5, 10, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            1, 2, 10, 11, 3, 6, 3, 6, 0, 6, 0, 7, 0, 7, 8, 0, 0, 0,
            0, 0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            6, 7, 11, 8, 4, 3, 4, 3, 5, 3, 5, 0, 5, 0, 9, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            2, 3, 11, 8, 0, 7, 0, 7, 1, 7, 1, 4, 1, 4, 9, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            0, 1, 9, 10, 2, 5, 2, 5, 3, 5, 3, 6, 3, 6, 11, 0, 0, 0,
            0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        5,
        {
            7, 4, 8, 9, 5, 0, 5, 0, 6, 0, 6, 1, 6, 1, 10, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            4, 5, 9, 10, 6, 1, 6, 1, 7, 1, 7, 2, 7, 2, 11, 0, 0, 0,
            0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            5, 6, 10, 11, 7, 2, 7, 2, 4, 2, 4, 3, 4, 3, 8, 0, 0, 0,
            0, 0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            5, 6, 10, 11, 7, 2, 7, 2, 4, 2, 4, 3, 4, 3, 8, 0, 0, 0,
            0, 0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            4, 5, 9, 10, 6, 1, 6, 1, 7, 1, 7, 2, 7, 2, 11, 0, 0, 0,
            0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        5,
        {
            7, 4, 8, 9, 5, 0, 5, 0, 6, 0, 6, 1, 6, 1, 10, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            0, 1, 9, 10, 2, 5, 2, 5, 3, 5, 3, 6, 3, 6, 11, 0, 0, 0,
            0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            2, 3, 11, 8, 0, 7, 0, 7, 1, 7, 1, 4, 1, 4, 9, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            6, 7, 11, 8, 4, 3, 4, 3, 5, 3, 5, 0, 5, 0, 9, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            1, 2, 10, 11, 3, 6, 3, 6, 0, 6, 0, 7, 0, 7, 8, 0, 0, 0,
            0, 0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        5,
        {
            3, 0, 8, 9, 1, 4, 1, 4, 2, 4, 2, 5, 2, 5, 10, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        5,
        {
            4, 5, 9, 1, 10, 0, 10, 0, 8, 10, 8, 2, 8, 2, 3, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            6, 7, 11, 3, 8, 2, 8, 2, 10, 8, 10, 0, 10, 0, 1, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            3, 0, 8, 4, 9, 7, 9, 7, 11, 9, 11, 5, 11, 5, 6, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            7, 4, 8, 0, 9, 3, 9, 3, 11, 9, 11, 1, 11, 1, 2, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            5, 6, 10, 2, 11, 1, 11, 1, 9, 11, 9, 3, 9, 3, 0, 0, 0, 0,
            0, 0, 0,  0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        5,
        {
            0, 1, 9, 5, 10, 4, 10, 4, 8, 10, 8, 6, 8, 6, 7, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            1, 2, 10, 6, 11, 5, 11, 5, 9, 11, 9, 7, 9, 7, 4, 0, 0, 0,
            0, 0, 0,  0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            2, 3, 11, 7, 8, 6, 8, 6, 10, 8, 10, 4, 10, 4, 5, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            2, 3, 11, 7, 8, 6, 8, 6, 10, 8, 10, 4, 10, 4, 5, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            1, 2, 10, 6, 11, 5, 11, 5, 9, 11, 9, 7, 9, 7, 4, 0, 0, 0,
            0, 0, 0,  0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        5,
        {
            0, 1, 9, 5, 10, 4, 10, 4, 8, 10, 8, 6, 8, 6, 7, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            5, 6, 10, 2, 11, 1, 11, 1, 9, 11, 9, 3, 9, 3, 0, 0, 0, 0,
            0, 0, 0,  0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            7, 4, 8, 0, 9, 3, 9, 3, 11, 9, 11, 1, 11, 1, 2, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            3, 0, 8, 4, 9, 7, 9, 7, 11, 9, 11, 5, 11, 5, 6, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        5,
        {
            6, 7, 11, 3, 8, 2, 8, 2, 10, 8, 10, 0, 10, 0, 1, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        5,
        {
            4, 5, 9, 1, 10, 0, 10, 0, 8, 10, 8, 2, 8, 2, 3, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        3,
        {
            4, 5, 9, 10, 1, 2, 0, 3, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        3,
        {
            6, 7, 11, 8, 3, 0, 2, 1, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        3,
        {
            3, 0, 8, 9, 4, 5, 7, 6, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        3,
        {
            7, 4, 8, 9, 0, 1, 3, 2, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        3,
        {
            5, 6, 10, 11, 2, 3, 1, 0, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        3,
        {
            0, 1, 9, 10, 5, 6, 4, 7, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        3,
        {
            1, 2, 10, 11, 6, 7, 5, 4, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        3,
        {
            2, 3, 11, 8, 7, 4, 6, 5, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        3,
        {
            2, 3, 11, 8, 7, 4, 6, 5, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            1,
            2,
            4,
            5,
            6,
            7,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        3,
        {
            1, 2, 10, 11, 6, 7, 5, 4, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        3,
        {
            0, 1, 9, 10, 5, 6, 4, 7, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            5,
            6,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        3,
        {
            5, 6, 10, 11, 2, 3, 1, 0, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        3,
        {
            7, 4, 8, 9, 0, 1, 3, 2, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            11,
            0,
            0,
            0,
            0,
        },
        3,
        {
            3, 0, 8, 9, 4, 5, 7, 6, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            6,
            7,
            8,
            10,
            11,
            0,
            0,
            0,
            0,
        },
        3,
        {
            6, 7, 11, 8, 3, 0, 2, 1, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            8,
            9,
            10,
            0,
            0,
            0,
            0,
        },
        3,
        {
            4, 5, 9, 10, 1, 2, 0, 3, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            9, 8, 10, 10, 8, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            1,
            3,
            5,
            7,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            1, 5, 3, 3, 5, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            0,
            2,
            4,
            6,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            0, 4, 2, 4, 6, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            0,
            2,
            4,
            6,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            0, 2, 4, 4, 2, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            1,
            3,
            5,
            7,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            1, 3, 5, 3, 7, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        4,
        {
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        2,
        {
            9, 10, 8, 10, 11, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            2,
            3,
            4,
            5,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            2, 10, 5, 3, 2, 5, 3, 5, 4, 3, 4, 8, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            1,
            2,
            4,
            7,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            4, 7, 11, 9, 4, 11, 9, 11, 2, 9, 2, 1, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            6,
            7,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            10, 7, 6, 1, 7, 10, 1, 8, 7, 1, 0, 8, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            3,
            5,
            6,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            3, 6, 11, 0, 6, 3, 0, 5, 6, 0, 9, 5, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            3,
            5,
            6,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            3, 11, 6, 0, 3, 6, 0, 6, 5, 0, 5, 9, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            6,
            7,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            10, 6, 7, 1, 10, 7, 1, 7, 8, 1, 8, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            1,
            2,
            4,
            7,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            4, 11, 7, 9, 11, 4, 9, 2, 11, 9, 1, 2, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            2,
            3,
            4,
            5,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            2, 5, 10, 3, 5, 2, 3, 4, 5, 3, 8, 4, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            1,
            3,
            5,
            7,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 5,  9, 12, 9, 8, 12, 8, 3, 12, 3, 1, 12, 1, 10, 12, 10, 11,
            12, 11, 7, 12, 7, 5, 0,  0, 0, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 1, 0, 12, 0, 4, 12, 4, 7, 12, 7, 3, 12, 3, 2, 12, 2, 6,
            12, 6, 5, 12, 5, 1, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            2,
            4,
            6,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 8, 4,  12, 4,  6, 12, 6, 10, 12, 10, 9, 12, 9, 0, 12, 0, 2,
            12, 2, 11, 12, 11, 8, 0,  0, 0,  0,  0,  0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            2,
            4,
            6,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 9, 4,  12, 4,  6, 12, 6, 11, 12, 11, 8, 12, 8, 0, 12, 0, 2,
            12, 2, 10, 12, 10, 9, 0,  0, 0,  0,  0,  0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 3, 0, 12, 0, 4, 12, 4, 5, 12, 5, 1, 12, 1, 2, 12, 2, 6,
            12, 6, 7, 12, 7, 3, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            1,
            3,
            5,
            7,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 5, 10, 12, 10, 11, 12, 11, 3, 12, 3, 1, 12, 1, 9, 12, 9, 8,
            12, 8, 7,  12, 7,  5,  0,  0,  0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            1,
            3,
            5,
            7,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 7,  8, 12, 8, 9, 12, 9, 1, 12, 1, 3, 12, 3, 11, 12, 11, 10,
            12, 10, 5, 12, 5, 7, 0,  0, 0, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 5, 4, 12, 4, 0, 12, 0, 3, 12, 3, 7, 12, 7, 6, 12, 6, 2,
            12, 2, 1, 12, 1, 5, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            2,
            4,
            6,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 11, 6, 12, 6, 4,  12, 4, 9, 12, 9, 10, 12, 10, 2, 12, 2, 0,
            12, 0,  8, 12, 8, 11, 0,  0, 0, 0,  0, 0,  0,  0,  0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            2,
            4,
            6,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 10, 6, 12, 6, 4,  12, 4, 8, 12, 8, 11, 12, 11, 2, 12, 2, 0,
            12, 0,  9, 12, 9, 10, 0,  0, 0, 0,  0, 0,  0,  0,  0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 7, 4, 12, 4, 0, 12, 0, 1, 12, 1, 5, 12, 5, 6, 12, 6, 2,
            12, 2, 3, 12, 3, 7, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            1,
            3,
            5,
            7,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 7, 11, 12, 11, 10, 12, 10, 1, 12, 1, 3, 12, 3, 8, 12, 8, 9,
            12, 9, 5,  12, 5,  7,  0,  0,  0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        8,
        {
            1,
            3,
            5,
            7,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            5, 10, 7, 10, 7, 11, 9, 1, 8, 1, 8, 3, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            1, 2, 5, 2, 5, 6, 0, 3, 4, 3, 4, 7, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            2,
            4,
            6,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            8, 0, 11, 0, 11, 2, 4, 9, 6, 9, 6, 10, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            2,
            4,
            6,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            9, 0, 10, 0, 10, 2, 4, 8, 6, 8, 6, 11, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            3, 2, 7, 2, 7, 6, 0, 1, 4, 1, 4, 5, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            1,
            3,
            5,
            7,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            5, 9, 7, 9, 7, 8, 10, 1, 11, 1, 11, 3, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            1,
            3,
            5,
            7,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            5, 9, 7, 9, 7, 8, 10, 1, 11, 1, 11, 3, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            3, 2, 7, 2, 7, 6, 0, 1, 4, 1, 4, 5, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            2,
            4,
            6,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            9, 0, 10, 0, 10, 2, 4, 8, 6, 8, 6, 11, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            2,
            4,
            6,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            8, 0, 11, 0, 11, 2, 4, 9, 6, 9, 6, 10, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            1, 2, 5, 2, 5, 6, 0, 3, 4, 3, 4, 7, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            1,
            3,
            5,
            7,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            5, 10, 7, 10, 7, 11, 9, 1, 8, 1, 8, 3, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            1,
            3,
            5,
            7,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            3,  11, 7, 3,  7, 8,  7, 8, 9, 7, 9, 5, 9, 5, 10, 9, 10, 1,
            10, 1,  3, 10, 3, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            7, 6, 5, 7, 5, 4, 5, 4, 0, 5, 0, 1, 0, 1, 2, 0, 2, 3,
            2, 3, 7, 2, 7, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            2,
            4,
            6,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            10, 2, 11, 10, 11, 6, 11, 6, 4, 11, 4, 8, 4, 8, 0, 4, 0, 9,
            0,  9, 10, 0,  10, 2, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            2,
            4,
            6,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            11, 2, 10, 11, 10, 6, 10, 6, 4, 10, 4, 9, 4, 9, 0, 4, 0, 8,
            0,  8, 11, 0,  11, 2, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            5, 6, 7, 5, 7, 4, 7, 4, 0, 7, 0, 3, 0, 3, 2, 0, 2, 1,
            2, 1, 5, 2, 5, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            1,
            3,
            5,
            7,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            3, 8, 7, 3, 7, 11, 7, 11, 10, 7, 10, 5, 10, 5, 9, 10, 9, 1,
            9, 1, 3, 9, 3, 8,  0, 0,  0,  0, 0,  0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        6,
        {
            2,
            3,
            4,
            7,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            2, 10, 9, 2, 9, 7, 2, 7, 3, 7, 9, 4, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            1,
            2,
            6,
            7,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            1, 6, 2, 1, 8, 6, 1, 9, 8, 8, 7, 6, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            1,
            3,
            4,
            6,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            8, 3, 1, 8, 1, 6, 8, 6, 4, 6, 1, 10, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            5,
            6,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            0, 8, 11, 0, 11, 5, 0, 5, 1, 5, 11, 6, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            2,
            5,
            7,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            9, 5, 7, 9, 7, 2, 9, 2, 0, 2, 7, 11, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            3,
            4,
            5,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            5, 0, 4, 5, 11, 0, 5, 10, 11, 11, 3, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0, 0,  0,  0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            3,
            4,
            5,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            5, 4, 0, 5, 0, 11, 5, 11, 10, 11, 0, 3, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0,  0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            2,
            5,
            7,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            9, 7, 5, 9, 2, 7, 9, 0, 2, 2, 11, 7, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            5,
            6,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            0, 11, 8, 0, 5, 11, 0, 1, 5, 5, 6, 11, 0, 0, 0, 0, 0, 0,
            0, 0,  0, 0, 0, 0,  0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            1,
            3,
            4,
            6,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            8, 1, 3, 8, 6, 1, 8, 4, 6, 6, 10, 1, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            1,
            2,
            6,
            7,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            1, 2, 6, 1, 6, 8, 1, 8, 9, 8, 6, 7, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            2,
            3,
            4,
            7,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            2, 9, 10, 2, 7, 9, 2, 3, 7, 7, 4, 9, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        9,
        {
            2,
            3,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 8, 9, 12, 9, 10, 12, 10, 2, 12, 2, 3, 12, 3, 11, 12, 11, 6,
            12, 6, 7, 12, 7, 8,  0,  0,  0, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        9,
        {
            1,
            2,
            5,
            6,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 11, 8, 12, 8, 9,  12, 9, 1, 12, 1, 2, 12, 2, 10, 12, 10, 5,
            12, 5,  6, 12, 6, 11, 0,  0, 0, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        9,
        {
            1,
            3,
            4,
            5,
            6,
            7,
            9,
            10,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 1, 3,  12, 3,  7, 12, 7, 4, 12, 4, 9, 12, 9, 5, 12, 5, 6,
            12, 6, 10, 12, 10, 1, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            1,
            3,
            4,
            5,
            6,
            7,
            8,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 3, 1, 12, 1, 5, 12, 5, 6, 12, 6, 11, 12, 11, 7, 12, 7, 4,
            12, 4, 8, 12, 8, 3, 0,  0, 0, 0,  0, 0,  0,  0,  0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            4,
            5,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 10, 11, 12, 11, 8,  12, 8, 0, 12, 0, 1, 12, 1, 9, 12, 9, 4,
            12, 4,  5,  12, 5,  10, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            5,
            7,
            9,
            10,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 5, 7, 12, 7, 3, 12, 3, 2, 12, 2, 10, 12, 10, 1, 12, 1, 0,
            12, 0, 9, 12, 9, 5, 0,  0, 0, 0,  0, 0,  0,  0,  0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            6,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 6, 4,  12, 4,  0, 12, 0, 1, 12, 1, 10, 12, 10, 2, 12, 2, 3,
            12, 3, 11, 12, 11, 6, 0,  0, 0, 0,  0, 0,  0,  0,  0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            6,
            8,
            9,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 4, 6, 12, 6, 2, 12, 2, 3, 12, 3, 8, 12, 8, 0, 12, 0, 1,
            12, 1, 9, 12, 9, 4, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            5,
            7,
            8,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 7, 5,  12, 5,  1, 12, 1, 0, 12, 0, 8, 12, 8, 3, 12, 3, 2,
            12, 2, 11, 12, 11, 7, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            2,
            4,
            5,
            6,
            7,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 2, 0,  12, 0,  4, 12, 4, 5, 12, 5, 10, 12, 10, 6, 12, 6, 7,
            12, 7, 11, 12, 11, 2, 0,  0, 0, 0,  0, 0,  0,  0,  0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            2,
            4,
            5,
            6,
            7,
            8,
            9,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 0, 2, 12, 2, 6, 12, 6, 7, 12, 7, 8, 12, 8, 4, 12, 4, 5,
            12, 5, 9, 12, 9, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            3,
            4,
            7,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 9, 10, 12, 10, 11, 12, 11, 7, 12, 7, 4, 12, 4, 8, 12, 8, 3,
            12, 3, 0,  12, 0,  9,  0,  0,  0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            3,
            4,
            7,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 9, 10, 12, 10, 11, 12, 11, 7, 12, 7, 4, 12, 4, 8, 12, 8, 3,
            12, 3, 0,  12, 0,  9,  0,  0,  0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            2,
            4,
            5,
            6,
            7,
            8,
            9,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 0, 2, 12, 2, 6, 12, 6, 7, 12, 7, 8, 12, 8, 4, 12, 4, 5,
            12, 5, 9, 12, 9, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            2,
            4,
            5,
            6,
            7,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 2, 0,  12, 0,  4, 12, 4, 5, 12, 5, 10, 12, 10, 6, 12, 6, 7,
            12, 7, 11, 12, 11, 2, 0,  0, 0, 0,  0, 0,  0,  0,  0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            5,
            7,
            8,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 7, 5,  12, 5,  1, 12, 1, 0, 12, 0, 8, 12, 8, 3, 12, 3, 2,
            12, 2, 11, 12, 11, 7, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            6,
            8,
            9,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 4, 6, 12, 6, 2, 12, 2, 3, 12, 3, 8, 12, 8, 0, 12, 0, 1,
            12, 1, 9, 12, 9, 4, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            6,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 6, 4,  12, 4,  0, 12, 0, 1, 12, 1, 10, 12, 10, 2, 12, 2, 3,
            12, 3, 11, 12, 11, 6, 0,  0, 0, 0,  0, 0,  0,  0,  0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            5,
            7,
            9,
            10,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 5, 7, 12, 7, 3, 12, 3, 2, 12, 2, 10, 12, 10, 1, 12, 1, 0,
            12, 0, 9, 12, 9, 5, 0,  0, 0, 0,  0, 0,  0,  0,  0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            4,
            5,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 10, 11, 12, 11, 8,  12, 8, 0, 12, 0, 1, 12, 1, 9, 12, 9, 4,
            12, 4,  5,  12, 5,  10, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            1,
            3,
            4,
            5,
            6,
            7,
            8,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 3, 1, 12, 1, 5, 12, 5, 6, 12, 6, 11, 12, 11, 7, 12, 7, 4,
            12, 4, 8, 12, 8, 3, 0,  0, 0, 0,  0, 0,  0,  0,  0, 0,  0, 0,
        },
    },
    {
        9,
        {
            1,
            3,
            4,
            5,
            6,
            7,
            9,
            10,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 1, 3,  12, 3,  7, 12, 7, 4, 12, 4, 9, 12, 9, 5, 12, 5, 6,
            12, 6, 10, 12, 10, 1, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            1,
            2,
            5,
            6,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 11, 8, 12, 8, 9,  12, 9, 1, 12, 1, 2, 12, 2, 10, 12, 10, 5,
            12, 5,  6, 12, 6, 11, 0,  0, 0, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        9,
        {
            2,
            3,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 8, 9, 12, 9, 10, 12, 10, 2, 12, 2, 3, 12, 3, 11, 12, 11, 6,
            12, 6, 7, 12, 7, 8,  0,  0,  0, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        9,
        {
            2,
            3,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 2, 11, 12, 11, 7, 12, 7, 6, 12, 6, 10, 12, 10, 9, 12, 9, 8,
            12, 8, 3,  12, 3,  2, 0,  0, 0, 0,  0, 0,  0,  0,  0, 0,  0, 0,
        },
    },
    {
        9,
        {
            1,
            2,
            5,
            6,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 1,  10, 12, 10, 6, 12, 6, 5, 12, 5, 9, 12, 9, 8, 12, 8, 11,
            12, 11, 2,  12, 2,  1, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            1,
            3,
            4,
            5,
            6,
            7,
            9,
            10,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 4, 5, 12, 5, 10, 12, 10, 6, 12, 6, 7, 12, 7, 3, 12, 3, 1,
            12, 1, 9, 12, 9, 4,  0,  0,  0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            1,
            3,
            4,
            5,
            6,
            7,
            8,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 6, 7,  12, 7,  8, 12, 8, 4, 12, 4, 5, 12, 5, 1, 12, 1, 3,
            12, 3, 11, 12, 11, 6, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            4,
            5,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 0,  9, 12, 9, 5, 12, 5, 4, 12, 4, 8, 12, 8, 11, 12, 11, 10,
            12, 10, 1, 12, 1, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            5,
            7,
            9,
            10,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 2, 1,  12, 1,  9, 12, 9, 0, 12, 0, 3, 12, 3, 7, 12, 7, 5,
            12, 5, 10, 12, 10, 2, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            6,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 1, 2,  12, 2,  11, 12, 11, 3, 12, 3, 0, 12, 0, 4, 12, 4, 6,
            12, 6, 10, 12, 10, 1,  0,  0,  0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            6,
            8,
            9,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 3, 0, 12, 0, 9, 12, 9, 1, 12, 1, 2, 12, 2, 6, 12, 6, 4,
            12, 4, 8, 12, 8, 3, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            5,
            7,
            8,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 0, 3, 12, 3, 11, 12, 11, 2, 12, 2, 1, 12, 1, 5, 12, 5, 7,
            12, 7, 8, 12, 8, 0,  0,  0,  0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            2,
            4,
            5,
            6,
            7,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 5, 6,  12, 6,  11, 12, 11, 7, 12, 7, 4, 12, 4, 0, 12, 0, 2,
            12, 2, 10, 12, 10, 5,  0,  0,  0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            2,
            4,
            5,
            6,
            7,
            8,
            9,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 7, 4, 12, 4, 9, 12, 9, 5, 12, 5, 6, 12, 6, 2, 12, 2, 0,
            12, 0, 8, 12, 8, 7, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            3,
            4,
            7,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 7, 8, 12, 8, 0, 12, 0, 3, 12, 3, 11, 12, 11, 10, 12, 10, 9,
            12, 9, 4, 12, 4, 7, 0,  0, 0, 0,  0, 0,  0,  0,  0,  0,  0,  0,
        },
    },
    {
        9,
        {
            0,
            3,
            4,
            7,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 7, 8, 12, 8, 0, 12, 0, 3, 12, 3, 11, 12, 11, 10, 12, 10, 9,
            12, 9, 4, 12, 4, 7, 0,  0, 0, 0,  0, 0,  0,  0,  0,  0,  0,  0,
        },
    },
    {
        9,
        {
            0,
            2,
            4,
            5,
            6,
            7,
            8,
            9,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 7, 4, 12, 4, 9, 12, 9, 5, 12, 5, 6, 12, 6, 2, 12, 2, 0,
            12, 0, 8, 12, 8, 7, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            2,
            4,
            5,
            6,
            7,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 5, 6,  12, 6,  11, 12, 11, 7, 12, 7, 4, 12, 4, 0, 12, 0, 2,
            12, 2, 10, 12, 10, 5,  0,  0,  0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            5,
            7,
            8,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 0, 3, 12, 3, 11, 12, 11, 2, 12, 2, 1, 12, 1, 5, 12, 5, 7,
            12, 7, 8, 12, 8, 0,  0,  0,  0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            6,
            8,
            9,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 3, 0, 12, 0, 9, 12, 9, 1, 12, 1, 2, 12, 2, 6, 12, 6, 4,
            12, 4, 8, 12, 8, 3, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            4,
            6,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 1, 2,  12, 2,  11, 12, 11, 3, 12, 3, 0, 12, 0, 4, 12, 4, 6,
            12, 6, 10, 12, 10, 1,  0,  0,  0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            2,
            3,
            5,
            7,
            9,
            10,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 2, 1,  12, 1,  9, 12, 9, 0, 12, 0, 3, 12, 3, 7, 12, 7, 5,
            12, 5, 10, 12, 10, 2, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            0,
            1,
            4,
            5,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 0,  9, 12, 9, 5, 12, 5, 4, 12, 4, 8, 12, 8, 11, 12, 11, 10,
            12, 10, 1, 12, 1, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        9,
        {
            1,
            3,
            4,
            5,
            6,
            7,
            8,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 6, 7,  12, 7,  8, 12, 8, 4, 12, 4, 5, 12, 5, 1, 12, 1, 3,
            12, 3, 11, 12, 11, 6, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            1,
            3,
            4,
            5,
            6,
            7,
            9,
            10,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 4, 5, 12, 5, 10, 12, 10, 6, 12, 6, 7, 12, 7, 3, 12, 3, 1,
            12, 1, 9, 12, 9, 4,  0,  0,  0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            1,
            2,
            5,
            6,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 1,  10, 12, 10, 6, 12, 6, 5, 12, 5, 9, 12, 9, 8, 12, 8, 11,
            12, 11, 2,  12, 2,  1, 0,  0, 0, 0,  0, 0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        9,
        {
            2,
            3,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
            0,
            0,
            0,
            0,
        },
        8,
        {
            12, 2, 11, 12, 11, 7, 12, 7, 6, 12, 6, 10, 12, 10, 9, 12, 9, 8,
            12, 8, 3,  12, 3,  2, 0,  0, 0, 0,  0, 0,  0,  0,  0, 0,  0, 0,
        },
    },
    {
        8,
        {
            2,
            3,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            7, 6, 11, 2, 3, 10, 3, 10, 8, 10, 8, 9, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            1,
            2,
            5,
            6,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            6, 5, 10, 1, 2, 9, 2, 9, 11, 9, 11, 8, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            1,
            3,
            4,
            5,
            6,
            7,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            10, 6, 5, 4, 9, 7, 9, 7, 1, 7, 1, 3, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            1,
            3,
            4,
            5,
            6,
            7,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            11, 6, 7, 4, 8, 5, 8, 5, 3, 5, 3, 1, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            4,
            5,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            5, 4, 9, 0, 1, 8, 1, 8, 10, 8, 10, 11, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0,  0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            5,
            7,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            10, 2, 1, 0, 9, 3, 9, 3, 5, 3, 5, 7, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            4,
            6,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            10, 1, 2, 3, 11, 0, 11, 0, 6, 0, 6, 4, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            4,
            6,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            8, 3, 0, 1, 9, 2, 9, 2, 4, 2, 4, 6, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            5,
            7,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            8, 0, 3, 2, 11, 1, 11, 1, 7, 1, 7, 5, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            2,
            4,
            5,
            6,
            7,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            10, 5, 6, 7, 11, 4, 11, 4, 2, 4, 2, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            2,
            4,
            5,
            6,
            7,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            9, 5, 4, 7, 8, 6, 8, 6, 0, 6, 0, 2, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            3,
            4,
            7,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            0, 3, 8, 7, 4, 11, 4, 11, 9, 11, 9, 10, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            3,
            4,
            7,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            4, 7, 8, 3, 0, 11, 0, 11, 9, 11, 9, 10, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            2,
            4,
            5,
            6,
            7,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            8, 7, 4, 5, 9, 6, 9, 6, 0, 6, 0, 2, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            2,
            4,
            5,
            6,
            7,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            11, 7, 6, 5, 10, 4, 10, 4, 2, 4, 2, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            5,
            7,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            11, 2, 3, 0, 8, 1, 8, 1, 7, 1, 7, 5, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            4,
            6,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            9, 1, 0, 3, 8, 2, 8, 2, 4, 2, 4, 6, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            4,
            6,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            11, 3, 2, 1, 10, 0, 10, 0, 6, 0, 6, 4, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            5,
            7,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            9, 0, 1, 2, 10, 3, 10, 3, 5, 3, 5, 7, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            4,
            5,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            1, 0, 9, 4, 5, 8, 5, 8, 10, 8, 10, 11, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0,  0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            1,
            3,
            4,
            5,
            6,
            7,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            8, 4, 7, 6, 11, 5, 11, 5, 3, 5, 3, 1, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            1,
            3,
            4,
            5,
            6,
            7,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            9, 4, 5, 6, 10, 7, 10, 7, 1, 7, 1, 3, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            1,
            2,
            5,
            6,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            2, 1, 10, 5, 6, 9, 6, 9, 11, 9, 11, 8, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            2,
            3,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            3, 2, 11, 6, 7, 10, 7, 10, 8, 10, 8, 9, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            2,
            3,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            3, 2, 11, 6, 7, 10, 7, 10, 8, 10, 8, 9, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            1,
            2,
            5,
            6,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            2, 1, 10, 5, 6, 9, 6, 9, 11, 9, 11, 8, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            1,
            3,
            4,
            5,
            6,
            7,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            9, 4, 5, 6, 10, 7, 10, 7, 1, 7, 1, 3, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            1,
            3,
            4,
            5,
            6,
            7,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            8, 4, 7, 6, 11, 5, 11, 5, 3, 5, 3, 1, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            4,
            5,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            1, 0, 9, 4, 5, 8, 5, 8, 10, 8, 10, 11, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0,  0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            5,
            7,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            9, 0, 1, 2, 10, 3, 10, 3, 5, 3, 5, 7, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            4,
            6,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            11, 3, 2, 1, 10, 0, 10, 0, 6, 0, 6, 4, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            4,
            6,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            9, 1, 0, 3, 8, 2, 8, 2, 4, 2, 4, 6, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            5,
            7,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            11, 2, 3, 0, 8, 1, 8, 1, 7, 1, 7, 5, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            2,
            4,
            5,
            6,
            7,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            11, 7, 6, 5, 10, 4, 10, 4, 2, 4, 2, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            2,
            4,
            5,
            6,
            7,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            8, 7, 4, 5, 9, 6, 9, 6, 0, 6, 0, 2, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            3,
            4,
            7,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            4, 7, 8, 3, 0, 11, 0, 11, 9, 11, 9, 10, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            3,
            4,
            7,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            0, 3, 8, 7, 4, 11, 4, 11, 9, 11, 9, 10, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            2,
            4,
            5,
            6,
            7,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            9, 5, 4, 7, 8, 6, 8, 6, 0, 6, 0, 2, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            2,
            4,
            5,
            6,
            7,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            10, 5, 6, 7, 11, 4, 11, 4, 2, 4, 2, 0, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            5,
            7,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            8, 0, 3, 2, 11, 1, 11, 1, 7, 1, 7, 5, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            4,
            6,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            8, 3, 0, 1, 9, 2, 9, 2, 4, 2, 4, 6, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            4,
            6,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            10, 1, 2, 3, 11, 0, 11, 0, 6, 0, 6, 4, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            5,
            7,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            10, 2, 1, 0, 9, 3, 9, 3, 5, 3, 5, 7, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            4,
            5,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            5, 4, 9, 0, 1, 8, 1, 8, 10, 8, 10, 11, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0,  0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            1,
            3,
            4,
            5,
            6,
            7,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            11, 6, 7, 4, 8, 5, 8, 5, 3, 5, 3, 1, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            1,
            3,
            4,
            5,
            6,
            7,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            10, 6, 5, 4, 9, 7, 9, 7, 1, 7, 1, 3, 0, 0, 0, 0, 0, 0,
            0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            1,
            2,
            5,
            6,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            6, 5, 10, 1, 2, 9, 2, 9, 11, 9, 11, 8, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            2,
            3,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            7, 6, 11, 2, 3, 10, 3, 10, 8, 10, 8, 9, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0,  0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            2,
            3,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            11, 3, 7,  3, 7,  8, 7, 8, 9, 7, 9, 6, 9, 6, 10, 6, 10, 2,
            6,  2, 11, 2, 11, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0,
        },
    },
    {
        8,
        {
            1,
            2,
            5,
            6,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            10, 2, 6,  2, 6,  11, 6, 11, 8, 6, 8, 5, 8, 5, 9, 5, 9, 1,
            5,  1, 10, 1, 10, 2,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            1,
            3,
            4,
            5,
            6,
            7,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            5, 9, 10, 9, 10, 1, 10, 1, 3, 10, 3, 6, 3, 6, 7, 6, 7, 4,
            6, 4, 5,  4, 5,  9, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            1,
            3,
            4,
            5,
            6,
            7,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            7, 8, 11, 8, 11, 3, 11, 3, 1, 11, 1, 6, 1, 6, 5, 6, 5, 4,
            6, 4, 7,  4, 7,  8, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            4,
            5,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            9, 1, 5, 1, 5, 10, 5, 10, 11, 5, 11, 4, 11, 4, 8, 4, 8, 0,
            4, 0, 9, 0, 9, 1,  0, 0,  0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            5,
            7,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            1, 9, 10, 9, 10, 5, 10, 5, 7, 10, 7, 2, 7, 2, 3, 2, 3, 0,
            2, 0, 1,  0, 1,  9, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            4,
            6,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            2, 11, 10, 11, 10, 6,  10, 6, 4, 10, 4, 1, 4, 1, 0, 1, 0, 3,
            1, 3,  2,  3,  2,  11, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            4,
            6,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            0, 9, 8, 9, 8, 4, 8, 4, 6, 8, 6, 3, 6, 3, 2, 3, 2, 1,
            3, 1, 0, 1, 0, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            5,
            7,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            3, 11, 8, 11, 8, 7,  8, 7, 5, 8, 5, 0, 5, 0, 1, 0, 1, 2,
            0, 2,  3, 2,  3, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            2,
            4,
            5,
            6,
            7,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            6, 11, 10, 11, 10, 2,  10, 2, 0, 10, 0, 5, 0, 5, 4, 5, 4, 7,
            5, 7,  6,  7,  6,  11, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            2,
            4,
            5,
            6,
            7,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            4, 8, 9, 8, 9, 0, 9, 0, 2, 9, 2, 5, 2, 5, 6, 5, 6, 7,
            5, 7, 4, 7, 4, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            3,
            4,
            7,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            8, 4, 0, 4, 0, 9, 0, 9, 10, 0, 10, 3, 10, 3, 11, 3, 11, 7,
            3, 7, 8, 7, 8, 4, 0, 0, 0,  0, 0,  0, 0,  0, 0,  0, 0,  0,
        },
    },
    {
        8,
        {
            0,
            3,
            4,
            7,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            8, 0, 4, 0, 4, 9, 4, 9, 10, 4, 10, 7, 10, 7, 11, 7, 11, 3,
            7, 3, 8, 3, 8, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,  0, 0,  0,
        },
    },
    {
        8,
        {
            0,
            2,
            4,
            5,
            6,
            7,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            4, 9, 8, 9, 8, 0, 8, 0, 2, 8, 2, 7, 2, 7, 6, 7, 6, 5,
            7, 5, 4, 5, 4, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            2,
            4,
            5,
            6,
            7,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            6, 10, 11, 10, 11, 2,  11, 2, 0, 11, 0, 7, 0, 7, 4, 7, 4, 5,
            7, 5,  6,  5,  6,  10, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            5,
            7,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            3, 8, 11, 8, 11, 7, 11, 7, 5, 11, 5, 2, 5, 2, 1, 2, 1, 0,
            2, 0, 3,  0, 3,  8, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            4,
            6,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            0, 8, 9, 8, 9, 4, 9, 4, 6, 9, 6, 1, 6, 1, 2, 1, 2, 3,
            1, 3, 0, 3, 0, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            4,
            6,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            2, 10, 11, 10, 11, 6,  11, 6, 4, 11, 4, 3, 4, 3, 0, 3, 0, 1,
            3, 1,  2,  1,  2,  10, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            2,
            3,
            5,
            7,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            1, 10, 9, 10, 9, 5,  9, 5, 7, 9, 7, 0, 7, 0, 3, 0, 3, 2,
            0, 2,  1, 2,  1, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            0,
            1,
            4,
            5,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            9, 5, 1, 5, 1, 10, 1, 10, 11, 1, 11, 0, 11, 0, 8, 0, 8, 4,
            0, 4, 9, 4, 9, 5,  0, 0,  0,  0, 0,  0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            1,
            3,
            4,
            5,
            6,
            7,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            7, 11, 8, 11, 8, 3,  8, 3, 1, 8, 1, 4, 1, 4, 5, 4, 5, 6,
            4, 6,  7, 6,  7, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            1,
            3,
            4,
            5,
            6,
            7,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            5, 10, 9, 10, 9, 1,  9, 1, 3, 9, 3, 4, 3, 4, 7, 4, 7, 6,
            4, 6,  5, 6,  5, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            1,
            2,
            5,
            6,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            10, 6, 2,  6, 2,  11, 2, 11, 8, 2, 8, 1, 8, 1, 9, 1, 9, 5,
            1,  5, 10, 5, 10, 6,  0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        8,
        {
            2,
            3,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
        },
        8,
        {
            11, 7, 3,  7, 3,  8, 3, 8, 9, 3, 9, 2, 9, 2, 10, 2, 10, 6,
            2,  6, 11, 6, 11, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        4,
        {
            6, 7, 11, 1, 2, 10, 0, 3, 8, 4, 5, 9, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        4,
        {
            7, 4, 8, 2, 3, 11, 1, 0, 9, 5, 6, 10, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            1, 2, 10, 6, 7, 11, 8, 4, 3, 4, 3, 5, 3, 5, 0, 5, 0, 9,
            0, 0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            0, 3, 8, 6, 7, 11, 9, 1, 4, 1, 4, 2, 4, 2, 5, 2, 5, 10,
            0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            4, 5, 9, 0, 3, 8, 10, 6, 1, 6, 1, 7, 1, 7, 2, 7, 2, 11,
            0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            4, 5, 9, 1, 2, 10, 11, 3, 6, 3, 6, 0, 6, 0, 7, 0, 7, 8,
            0, 0, 0, 0, 0, 0,  0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            4, 5, 9, 6, 7, 11, 1, 10, 0, 10, 0, 8, 10, 8, 2, 8, 2, 3,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            1, 2, 10, 3, 0, 8, 4, 9, 7, 9, 7, 11, 9, 11, 5, 11, 5, 6,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            2, 3, 11, 7, 4, 8, 9, 5, 0, 5, 0, 6, 0, 6, 1, 6, 1, 10,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            1, 0, 9, 7, 4, 8, 10, 2, 5, 2, 5, 3, 5, 3, 6, 3, 6, 11,
            0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            5, 6, 10, 1, 0, 9, 11, 7, 2, 7, 2, 4, 2, 4, 3, 4, 3, 8,
            0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            5, 6, 10, 2, 3, 11, 8, 0, 7, 0, 7, 1, 7, 1, 4, 1, 4, 9,
            0, 0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            5, 6, 10, 7, 4, 8, 2, 11, 1, 11, 1, 9, 11, 9, 3, 9, 3, 0,
            0, 0, 0,  0, 0, 0, 0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            2, 3, 11, 0, 1, 9, 5, 10, 4, 10, 4, 8, 10, 8, 6, 8, 6, 7,
            0, 0, 0,  0, 0, 0, 0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            6,  7, 11, 12, 2, 10, 12, 10, 5, 12, 5, 4, 12, 4, 8, 12, 8, 3,
            12, 3, 0,  12, 0, 9,  12, 9,  1, 12, 1, 2, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            1,  2, 10, 12, 5, 9, 12, 9, 0, 12, 0, 3, 12, 3, 11, 12, 11, 6,
            12, 6, 7,  12, 7, 8, 12, 8, 4, 12, 4, 5, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            6,  7,  11, 12, 5, 4, 12, 4, 8, 12, 8, 3, 12, 3, 2, 12, 2, 10,
            12, 10, 1,  12, 1, 0, 12, 0, 9, 12, 9, 5, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            1,  2,  10, 12, 3, 0, 12, 0, 9, 12, 9, 5, 12, 5, 6, 12, 6, 11,
            12, 11, 7,  12, 7, 4, 12, 4, 8, 12, 8, 3, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            0,  3, 8, 12, 7, 11, 12, 11, 2, 12, 2, 1, 12, 1, 9, 12, 9, 4,
            12, 4, 5, 12, 5, 10, 12, 10, 6, 12, 6, 7, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            6,  7, 11, 12, 4, 5, 12, 5, 10, 12, 10, 2, 12, 2, 3, 12, 3, 8,
            12, 8, 0,  12, 0, 1, 12, 1, 9,  12, 9,  4, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            0,  3,  8, 12, 2, 1, 12, 1, 9,  12, 9,  4, 12, 4, 7, 12, 7, 11,
            12, 11, 6, 12, 6, 5, 12, 5, 10, 12, 10, 2, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            4,  5, 9, 12, 0, 8,  12, 8,  7, 12, 7, 6, 12, 6, 10, 12, 10, 1,
            12, 1, 2, 12, 2, 11, 12, 11, 3, 12, 3, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            4,  5, 9, 12, 7, 6, 12, 6, 10, 12, 10, 1, 12, 1, 0, 12, 0, 8,
            12, 8, 3, 12, 3, 2, 12, 2, 11, 12, 11, 7, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            0,  3, 8, 12, 1, 2, 12, 2, 11, 12, 11, 7, 12, 7, 4, 12, 4, 9,
            12, 9, 5, 12, 5, 6, 12, 6, 10, 12, 10, 1, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            4,  5,  9, 12, 6, 7, 12, 7, 8,  12, 8,  0, 12, 0, 1, 12, 1, 10,
            12, 10, 2, 12, 2, 3, 12, 3, 11, 12, 11, 6, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            1,  2, 10, 12, 0, 3, 12, 3, 11, 12, 11, 6, 12, 6, 5, 12, 5, 9,
            12, 9, 4,  12, 4, 7, 12, 7, 8,  12, 8,  0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            7,  4, 8, 12, 3, 11, 12, 11, 6, 12, 6, 5, 12, 5, 9, 12, 9, 0,
            12, 0, 1, 12, 1, 10, 12, 10, 2, 12, 2, 3, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            2,  3, 11, 12, 6, 10, 12, 10, 1, 12, 1, 0, 12, 0, 8, 12, 8, 7,
            12, 7, 4,  12, 4, 9,  12, 9,  5, 12, 5, 6, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            7,  4,  8, 12, 6, 5, 12, 5, 9,  12, 9,  0, 12, 0, 3, 12, 3, 11,
            12, 11, 2, 12, 2, 1, 12, 1, 10, 12, 10, 6, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            2,  3, 11, 12, 0, 1, 12, 1, 10, 12, 10, 6, 12, 6, 7, 12, 7, 8,
            12, 8, 4,  12, 4, 5, 12, 5, 9,  12, 9,  0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            0,  1, 9, 12, 4, 8,  12, 8,  3, 12, 3, 2, 12, 2, 10, 12, 10, 5,
            12, 5, 6, 12, 6, 11, 12, 11, 7, 12, 7, 4, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            7,  4, 8, 12, 5, 6, 12, 6, 11, 12, 11, 3, 12, 3, 0, 12, 0, 9,
            12, 9, 1, 12, 1, 2, 12, 2, 10, 12, 10, 5, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            1,  0, 9, 12, 3, 2, 12, 2, 10, 12, 10, 5, 12, 5, 4, 12, 4, 8,
            12, 8, 7, 12, 7, 6, 12, 6, 11, 12, 11, 3, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            5,  6, 10, 12, 1, 9, 12, 9, 4, 12, 4, 7, 12, 7, 11, 12, 11, 2,
            12, 2, 3,  12, 3, 8, 12, 8, 0, 12, 0, 1, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            5,  6, 10, 12, 4, 7, 12, 7, 11, 12, 11, 2, 12, 2, 1, 12, 1, 9,
            12, 9, 0,  12, 0, 3, 12, 3, 8,  12, 8,  4, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            1,  0,  9, 12, 2, 3, 12, 3, 8,  12, 8,  4, 12, 4, 5, 12, 5, 10,
            12, 10, 6, 12, 6, 7, 12, 7, 11, 12, 11, 2, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            5,  6,  10, 12, 7, 4, 12, 4, 9, 12, 9, 1, 12, 1, 2, 12, 2, 11,
            12, 11, 3,  12, 3, 0, 12, 0, 8, 12, 8, 7, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            2,  3,  11, 12, 1, 0, 12, 0, 8, 12, 8, 7, 12, 7, 6, 12, 6, 10,
            12, 10, 5,  12, 5, 4, 12, 4, 9, 12, 9, 1, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        12,
        {
            12, 2, 10, 12, 10, 5, 12, 5, 6, 12, 6, 11, 12, 11, 7, 12, 7, 4,
            12, 4, 8,  12, 8,  3, 12, 3, 0, 12, 0, 9,  12, 9,  1, 12, 1, 2,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        12,
        {
            12, 3, 11, 12, 11, 6, 12, 6, 7, 12, 7, 8,  12, 8,  4, 12, 4, 5,
            12, 5, 9,  12, 9,  0, 12, 0, 1, 12, 1, 10, 12, 10, 2, 12, 2, 3,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        12,
        {
            12, 1, 9,  12, 9,  4, 12, 4, 5, 12, 5, 10, 12, 10, 6, 12, 6, 7,
            12, 7, 11, 12, 11, 2, 12, 2, 3, 12, 3, 8,  12, 8,  0, 12, 0, 1,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        12,
        {
            12, 0, 8,  12, 8,  7, 12, 7, 4, 12, 4, 9,  12, 9,  5, 12, 5, 6,
            12, 6, 10, 12, 10, 1, 12, 1, 2, 12, 2, 11, 12, 11, 3, 12, 3, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        12,
        {
            12, 3, 11, 12, 11, 6, 12, 6, 7, 12, 7, 8,  12, 8,  4, 12, 4, 5,
            12, 5, 9,  12, 9,  0, 12, 0, 1, 12, 1, 10, 12, 10, 2, 12, 2, 3,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        12,
        {
            12, 0, 8,  12, 8,  7, 12, 7, 4, 12, 4, 9,  12, 9,  5, 12, 5, 6,
            12, 6, 10, 12, 10, 1, 12, 1, 2, 12, 2, 11, 12, 11, 3, 12, 3, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        12,
        {
            12, 2, 10, 12, 10, 5, 12, 5, 6, 12, 6, 11, 12, 11, 7, 12, 7, 4,
            12, 4, 8,  12, 8,  3, 12, 3, 0, 12, 0, 9,  12, 9,  1, 12, 1, 2,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        12,
        {
            12, 1, 9,  12, 9,  4, 12, 4, 5, 12, 5, 10, 12, 10, 6, 12, 6, 7,
            12, 7, 11, 12, 11, 2, 12, 2, 3, 12, 3, 8,  12, 8,  0, 12, 0, 1,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            6, 7, 11, 1, 0, 9, 2, 3, 10, 3, 10, 5, 3, 5, 8, 5, 8, 4,
            0, 0, 0,  0, 0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            2, 1, 10, 7, 4, 8, 3, 0, 11, 0, 11, 6, 0, 6, 9, 6, 9, 5,
            0, 0, 0,  0, 0, 0, 0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            0, 3, 8, 5, 6, 10, 1, 2, 9, 2, 9, 4, 2, 4, 11, 4, 11, 7,
            0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            4, 5, 9, 3, 2, 11, 0, 1, 8, 1, 8, 7, 1, 7, 10, 7, 10, 6,
            0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            7, 4, 8, 2, 1, 10, 3, 0, 11, 0, 11, 6, 0, 6, 9, 6, 9, 5,
            0, 0, 0, 0, 0, 0,  0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            3, 2, 11, 4, 5, 9, 0, 1, 8, 1, 8, 7, 1, 7, 10, 7, 10, 6,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            1, 0, 9, 6, 7, 11, 2, 3, 10, 3, 10, 5, 3, 5, 8, 5, 8, 4,
            0, 0, 0, 0, 0, 0,  0, 0, 0,  0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            5, 6, 10, 0, 3, 8, 1, 2, 9, 2, 9, 4, 2, 4, 11, 4, 11, 7,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        10,
        {
            0,  1,  9, 7,  4, 8, 7, 8, 3, 7, 3, 11, 3, 11, 2, 11, 2, 10,
            11, 10, 6, 10, 6, 5, 6, 5, 7, 5, 7, 4,  0, 0,  0, 0,  0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        10,
        {
            7,  4, 8, 2, 3, 11, 2, 11, 6, 2, 6, 10, 6, 10, 5, 10, 5, 9,
            10, 9, 1, 9, 1, 0,  1, 0,  2, 0, 2, 3,  0, 0,  0, 0,  0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        10,
        {
            5, 6,  10, 0,  1, 9, 0, 9, 4, 0, 4, 8, 4, 8, 7, 8, 7, 11,
            8, 11, 3,  11, 3, 2, 3, 2, 0, 2, 0, 1, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        10,
        {
            2, 3, 11, 5, 6, 10, 5, 10, 1, 5, 1, 9, 1, 9, 0, 9, 0, 8,
            9, 8, 4,  4, 8, 7,  4, 7,  5, 7, 5, 6, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        10,
        {
            1, 2,  10, 4,  5, 9, 4, 9, 0, 4, 0, 8, 0, 8, 3, 8, 3, 11,
            8, 11, 7,  11, 7, 6, 7, 6, 4, 6, 4, 5, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        10,
        {
            4,  5,  9, 3,  0, 8, 3, 8, 7, 3, 7, 11, 7, 11, 6, 11, 6, 10,
            11, 10, 2, 10, 2, 1, 2, 1, 3, 1, 3, 0,  0, 0,  0, 0,  0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        10,
        {
            6, 7, 11, 1, 2, 10, 1, 10, 5, 1, 5, 9, 5, 9, 4, 9, 4, 8,
            9, 8, 0,  8, 0, 3,  0, 3,  1, 3, 1, 2, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        10,
        {
            3,  0, 8, 6, 7, 11, 6, 11, 2, 6, 2, 10, 2, 10, 1, 10, 1, 9,
            10, 9, 5, 5, 9, 4,  5, 4,  6, 4, 6, 7,  0, 0,  0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            3,  2, 11, 12, 7, 8, 12, 8, 0, 12, 0, 1, 12, 1, 10, 12, 10, 6,
            12, 6, 5,  12, 5, 9, 12, 9, 4, 12, 4, 7, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            6,  5, 10, 12, 2, 11, 12, 11, 7, 12, 7, 4, 12, 4, 9, 12, 9, 1,
            12, 1, 0,  12, 0, 8,  12, 8,  3, 12, 3, 2, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            5,  6,  10, 12, 7, 4, 12, 4, 9, 12, 9, 1, 12, 1, 2, 12, 2, 11,
            12, 11, 3,  12, 3, 0, 12, 0, 8, 12, 8, 7, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            2,  3,  11, 12, 1, 0, 12, 0, 8, 12, 8, 7, 12, 7, 6, 12, 6, 10,
            12, 10, 5,  12, 5, 4, 12, 4, 9, 12, 9, 1, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            7,  4, 8, 12, 3, 11, 12, 11, 6, 12, 6, 5, 12, 5, 9, 12, 9, 0,
            12, 0, 1, 12, 1, 10, 12, 10, 2, 12, 2, 3, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            7,  4,  8, 12, 6, 5, 12, 5, 9,  12, 9,  0, 12, 0, 3, 12, 3, 11,
            12, 11, 2, 12, 2, 1, 12, 1, 10, 12, 10, 6, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            2,  3, 11, 12, 0, 1, 12, 1, 10, 12, 10, 6, 12, 6, 7, 12, 7, 8,
            12, 8, 4,  12, 4, 5, 12, 5, 9,  12, 9,  0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            1,  0, 9, 12, 4, 8,  12, 8,  3, 12, 3, 2, 12, 2, 10, 12, 10, 5,
            12, 5, 6, 12, 6, 11, 12, 11, 7, 12, 7, 4, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            7,  4, 8, 12, 5, 6, 12, 6, 11, 12, 11, 3, 12, 3, 0, 12, 0, 9,
            12, 9, 1, 12, 1, 2, 12, 2, 10, 12, 10, 5, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            1,  0, 9, 12, 3, 2, 12, 2, 10, 12, 10, 5, 12, 5, 4, 12, 4, 8,
            12, 8, 7, 12, 7, 6, 12, 6, 11, 12, 11, 3, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            5,  6, 10, 12, 4, 7, 12, 7, 11, 12, 11, 2, 12, 2, 1, 12, 1, 9,
            12, 9, 0,  12, 0, 3, 12, 3, 8,  12, 8,  4, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            0,  1,  9, 12, 2, 3, 12, 3, 8,  12, 8,  4, 12, 4, 5, 12, 5, 10,
            12, 10, 6, 12, 6, 7, 12, 7, 11, 12, 11, 2, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            0,  3, 8, 12, 4, 9,  12, 9,  1, 12, 1, 2, 12, 2, 11, 12, 11, 7,
            12, 7, 6, 12, 6, 10, 12, 10, 5, 12, 5, 4, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            7,  6, 11, 12, 3, 8, 12, 8, 4, 12, 4, 5, 12, 5, 10, 12, 10, 2,
            12, 2, 1,  12, 1, 9, 12, 9, 0, 12, 0, 3, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            6,  7, 11, 12, 4, 5, 12, 5, 10, 12, 10, 2, 12, 2, 3, 12, 3, 8,
            12, 8, 0,  12, 0, 1, 12, 1, 9,  12, 9,  4, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            3,  0,  8, 12, 2, 1, 12, 1, 9,  12, 9,  4, 12, 4, 7, 12, 7, 11,
            12, 11, 6, 12, 6, 5, 12, 5, 10, 12, 10, 2, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            4,  5, 9, 12, 0, 8,  12, 8,  7, 12, 7, 6, 12, 6, 10, 12, 10, 1,
            12, 1, 2, 12, 2, 11, 12, 11, 3, 12, 3, 0, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            4,  5, 9, 12, 7, 6, 12, 6, 10, 12, 10, 1, 12, 1, 0, 12, 0, 8,
            12, 8, 3, 12, 3, 2, 12, 2, 11, 12, 11, 7, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            3,  0, 8, 12, 1, 2, 12, 2, 11, 12, 11, 7, 12, 7, 4, 12, 4, 9,
            12, 9, 5, 12, 5, 6, 12, 6, 10, 12, 10, 1, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            2,  1, 10, 12, 5, 9, 12, 9, 0, 12, 0, 3, 12, 3, 11, 12, 11, 6,
            12, 6, 7,  12, 7, 8, 12, 8, 4, 12, 4, 5, 0,  0, 0,  0,  0,  0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            4,  5,  9, 12, 6, 7, 12, 7, 8,  12, 8,  0, 12, 0, 1, 12, 1, 10,
            12, 10, 2, 12, 2, 3, 12, 3, 11, 12, 11, 6, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            2,  1, 10, 12, 0, 3, 12, 3, 11, 12, 11, 6, 12, 6, 5, 12, 5, 9,
            12, 9, 4,  12, 4, 7, 12, 7, 8,  12, 8,  0, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            6,  7,  11, 12, 5, 4, 12, 4, 8, 12, 8, 3, 12, 3, 2, 12, 2, 10,
            12, 10, 1,  12, 1, 0, 12, 0, 9, 12, 9, 5, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        13,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            12,
        },
        10,
        {
            1,  2,  10, 12, 3, 0, 12, 0, 9, 12, 9, 5, 12, 5, 6, 12, 6, 11,
            12, 11, 7,  12, 7, 4, 12, 4, 8, 12, 8, 3, 0,  0, 0, 0,  0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            5, 6, 10, 2, 3, 11, 8, 0, 7, 0, 7, 1, 7, 1, 4, 1, 4, 9,
            0, 0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            2, 3, 11, 7, 4, 8, 9, 5, 0, 5, 0, 6, 0, 6, 1, 6, 1, 10,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            1, 0, 9, 7, 4, 8, 10, 2, 5, 2, 5, 3, 5, 3, 6, 3, 6, 11,
            0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            5, 6, 10, 1, 0, 9, 11, 7, 2, 7, 2, 4, 2, 4, 3, 4, 3, 8,
            0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            5, 6, 10, 7, 4, 8, 2, 11, 1, 11, 1, 9, 11, 9, 3, 9, 3, 0,
            0, 0, 0,  0, 0, 0, 0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            2, 3, 11, 0, 1, 9, 5, 10, 4, 10, 4, 8, 10, 8, 6, 8, 6, 7,
            0, 0, 0,  0, 0, 0, 0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            6, 7, 11, 3, 0, 8, 9, 1, 4, 1, 4, 2, 4, 2, 5, 2, 5, 10,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            3, 0, 8, 4, 5, 9, 10, 6, 1, 6, 1, 7, 1, 7, 2, 7, 2, 11,
            0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            2, 1, 10, 4, 5, 9, 11, 3, 6, 3, 6, 0, 6, 0, 7, 0, 7, 8,
            0, 0, 0,  0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            6, 7, 11, 2, 1, 10, 8, 4, 3, 4, 3, 5, 3, 5, 0, 5, 0, 9,
            0, 0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            6, 7, 11, 4, 5, 9, 3, 8, 2, 8, 2, 10, 8, 10, 0, 10, 0, 1,
            0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        6,
        {
            3, 0, 8, 1, 2, 10, 6, 11, 5, 11, 5, 9, 11, 9, 7, 9, 7, 4,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        4,
        {
            7, 4, 8, 2, 3, 11, 1, 0, 9, 5, 6, 10, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0,
        },
    },
    {
        12,
        {
            0,
            1,
            2,
            3,
            4,
            5,
            6,
            7,
            8,
            9,
            10,
            11,
            0,
        },
        4,
        {
            6, 7, 11, 1, 2, 10, 0, 3, 8, 4, 5, 9, 0, 0, 0, 0, 0, 0,
            0, 0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            2,
            3,
            5,
            6,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            5, 9, 8, 5, 8, 2, 5, 2, 6, 3, 2, 8, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            1,
            2,
            4,
            5,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            2, 1, 5, 2, 5, 8, 2, 8, 11, 4, 8, 5, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            1,
            3,
            4,
            6,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            9, 4, 6, 9, 6, 3, 9, 3, 1, 11, 3, 6, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            4,
            7,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            1, 11, 10, 1, 4, 11, 1, 0, 4, 7, 11, 4, 0, 0, 0, 0, 0, 0,
            0, 0,  0,  0, 0, 0,  0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            2,
            5,
            7,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            8, 2, 0, 8, 5, 2, 8, 7, 5, 10, 2, 5, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            3,
            6,
            7,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            0, 7, 3, 0, 10, 7, 0, 9, 10, 6, 7, 10, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0,  0, 0, 0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            3,
            6,
            7,
            9,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            0, 3, 7, 0, 7, 10, 0, 10, 9, 6, 10, 7, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0,  0, 0,  0, 0, 0,  0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            2,
            5,
            7,
            8,
            10,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            8, 0, 2, 8, 2, 5, 8, 5, 7, 10, 5, 2, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            0,
            1,
            4,
            7,
            10,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            1, 10, 11, 1, 11, 4, 1, 4, 0, 7, 4, 11, 0, 0, 0, 0, 0, 0,
            0, 0,  0,  0, 0,  0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            1,
            3,
            4,
            6,
            9,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            9, 6, 4, 9, 3, 6, 9, 1, 3, 11, 6, 3, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            1,
            2,
            4,
            5,
            8,
            11,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            2, 5, 1, 2, 8, 5, 2, 11, 8, 4, 5, 8, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
    {
        6,
        {
            2,
            3,
            5,
            6,
            8,
            9,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
        4,
        {
            5, 8, 9, 5, 2, 8, 5, 6, 2, 3, 8, 2, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        },
    },
};

/************************************************************************/
/* case offset in test table */

#define OFFSET_TEST3  0
#define OFFSET_TEST4  24
#define OFFSET_TEST6  32
#define OFFSET_TEST7  80
#define OFFSET_TEST10 96
#define OFFSET_TEST12 102
#define OFFSET_TEST13 126

/************************************************************************/
/* faces and interior to test */
static const char test[128][7] = {
    /* case 3 */
    {
        5,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        1,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        4,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        5,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        1,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        2,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        2,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        3,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        4,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        3,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        6,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        6,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -6,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -6,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -3,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -4,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -3,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -2,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -2,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -1,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -5,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -4,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -1,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -5,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    /* case 4 */
    {
        7,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        7,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        7,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        7,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -7,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -7,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -7,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -7,
        0,
        0,
        0,
        0,
        0,
        0,
    },
    /* case 6 */
    {
        2,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        4,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        5,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        5,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        1,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        3,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        6,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        1,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        4,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        1,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        3,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        5,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        5,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        1,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        6,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        2,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        4,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        2,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        2,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        6,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        3,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        4,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        3,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        6,
        7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -6,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -3,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -4,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -3,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -6,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -2,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -2,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -4,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -2,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -6,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -1,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -5,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -5,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -3,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -1,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -4,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -1,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -6,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -3,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -1,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -5,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -5,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -4,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    {
        -2,
        -7,
        0,
        0,
        0,
        0,
        0,
    },
    /* case 7 */
    {
        1,
        2,
        5,
        7,
        0,
        0,
        0,
    },
    {
        3,
        4,
        5,
        7,
        0,
        0,
        0,
    },
    {
        4,
        1,
        6,
        7,
        0,
        0,
        0,
    },
    {
        4,
        1,
        5,
        7,
        0,
        0,
        0,
    },
    {
        2,
        3,
        5,
        7,
        0,
        0,
        0,
    },
    {
        1,
        2,
        6,
        7,
        0,
        0,
        0,
    },
    {
        2,
        3,
        6,
        7,
        0,
        0,
        0,
    },
    {
        3,
        4,
        6,
        7,
        0,
        0,
        0,
    },
    {
        -3,
        -4,
        -6,
        -7,
        0,
        0,
        0,
    },
    {
        -2,
        -3,
        -6,
        -7,
        0,
        0,
        0,
    },
    {
        -1,
        -2,
        -6,
        -7,
        0,
        0,
        0,
    },
    {
        -2,
        -3,
        -5,
        -7,
        0,
        0,
        0,
    },
    {
        -4,
        -1,
        -5,
        -7,
        0,
        0,
        0,
    },
    {
        -4,
        -1,
        -6,
        -7,
        0,
        0,
        0,
    },
    {
        -3,
        -4,
        -5,
        -7,
        0,
        0,
        0,
    },
    {
        -1,
        -2,
        -5,
        -7,
        0,
        0,
        0,
    },
    /* case 10 */
    {
        2,
        4,
        7,
        0,
        0,
        0,
        0,
    },
    {
        5,
        6,
        7,
        0,
        0,
        0,
        0,
    },
    {
        1,
        3,
        7,
        0,
        0,
        0,
        0,
    },
    {
        1,
        3,
        7,
        0,
        0,
        0,
        0,
    },
    {
        5,
        6,
        7,
        0,
        0,
        0,
        0,
    },
    {
        2,
        4,
        7,
        0,
        0,
        0,
        0,
    },
    /* case 12 */
    {
        4,
        3,
        7,
        0,
        0,
        0,
        0,
    },
    {
        3,
        2,
        7,
        0,
        0,
        0,
        0,
    },
    {
        2,
        6,
        7,
        0,
        0,
        0,
        0,
    },
    {
        6,
        4,
        7,
        0,
        0,
        0,
        0,
    },
    {
        2,
        1,
        7,
        0,
        0,
        0,
        0,
    },
    {
        5,
        2,
        7,
        0,
        0,
        0,
        0,
    },
    {
        5,
        3,
        7,
        0,
        0,
        0,
        0,
    },
    {
        5,
        1,
        7,
        0,
        0,
        0,
        0,
    },
    {
        5,
        4,
        7,
        0,
        0,
        0,
        0,
    },
    {
        6,
        3,
        7,
        0,
        0,
        0,
        0,
    },
    {
        1,
        6,
        7,
        0,
        0,
        0,
        0,
    },
    {
        1,
        4,
        7,
        0,
        0,
        0,
        0,
    },
    {
        4,
        1,
        7,
        0,
        0,
        0,
        0,
    },
    {
        6,
        1,
        7,
        0,
        0,
        0,
        0,
    },
    {
        3,
        6,
        7,
        0,
        0,
        0,
        0,
    },
    {
        4,
        5,
        7,
        0,
        0,
        0,
        0,
    },
    {
        1,
        5,
        7,
        0,
        0,
        0,
        0,
    },
    {
        3,
        5,
        7,
        0,
        0,
        0,
        0,
    },
    {
        2,
        5,
        7,
        0,
        0,
        0,
        0,
    },
    {
        1,
        2,
        7,
        0,
        0,
        0,
        0,
    },
    {
        4,
        6,
        7,
        0,
        0,
        0,
        0,
    },
    {
        6,
        2,
        7,
        0,
        0,
        0,
        0,
    },
    {
        2,
        3,
        7,
        0,
        0,
        0,
        0,
    },
    {
        3,
        4,
        7,
        0,
        0,
        0,
        0,
    },
    /* case 13 */
    {
        1,
        2,
        3,
        4,
        5,
        6,
        7,
    },
    {
        2,
        3,
        4,
        1,
        5,
        6,
        7,
    },
};

static const char subconfig7[8] = {
    7, 6, 5, 3, 4, 2, 1, 0,
};

static const char subconfig13[64] = {
    0,  1,  2,  7,  3,  -1, 11, -1, 4,  8,  -1, -1, 14, -1, -1, -1,
    5,  9,  12, 23, 15, -1, 21, 38, 17, 20, -1, 36, 26, 33, 30, 44,
    6,  10, 13, 19, 16, -1, 25, 37, 18, 24, -1, 35, 22, 32, 29, 43,
    -1, -1, -1, 34, -1, -1, 28, 42, -1, 31, -1, 41, 27, 40, 39, 45,
};

/************************************************************************/
/* for each cube edge appropriate cube vertices */
static const unsigned char edge_vert[12][2] = {
    {
        0,
        1,
    },
    {
        1,
        2,
    },
    {
        3,
        2,
    },
    {
        0,
        3,
    },
    {
        4,
        5,
    },
    {
        5,
        6,
    },
    {
        7,
        6,
    },
    {
        4,
        7,
    },
    {
        0,
        4,
    },
    {
        1,
        5,
    },
    {
        2,
        6,
    },
    {
        3,
        7,
    },
};

/************************************************************************/
/* position of cube edges relative to cube */
static const unsigned char edge_vert_pos[12][5] = {
    {X, Y, 0, Z, 0}, {Y, X, 1, Z, 0}, {X, Y, 1, Z, 0}, {Y, X, 0, Z, 0},
    {X, Y, 0, Z, 1}, {Y, X, 1, Z, 1}, {X, Y, 1, Z, 1}, {Y, X, 0, Z, 1},
    {Z, X, 0, Y, 0}, {Z, X, 1, Y, 0}, {Z, X, 1, Y, 1}, {Z, X, 0, Y, 1},
};

#endif /* _CELL_TABLE_H */
