#!/usr/bin/env python

import os, os.path, re, shutil, sys

sys.path.append("debian/lib/python")

from debian_ibm3270.debian import VersionIbm3270
from debian_linux.debian import Changelog
from debian_linux.patches import PatchSeries

class Main(object):
    def __init__(self, input_tar, override_version):
        self.log = sys.stdout.write

        self.input_tar = input_tar

        changelog = Changelog(version = VersionIbm3270)[0]
        source = changelog.source
        version = changelog.version

        if override_version:
            version = VersionIbm3270('%s-undef' % override_version)

        self.version_dfsg = version.ibm3270_dfsg
        if self.version_dfsg is None:
            self.version_dfsg = '0'

        self.log('Using source name %s, version %s, dfsg %s\n' % (source, version.upstream, self.version_dfsg))

        self.orig = '%s-%s' % (source, version.upstream)
        self.orig_tar = '%s_%s.orig.tar.gz' % (source, version.upstream)

    def __call__(self):
        import tempfile
        self.dir = tempfile.mkdtemp(prefix = 'genorig', dir = 'debian')
        try:
            self.upstream()
            self.move()
            #self.debian_patch()
            self.tar()
        finally:
            shutil.rmtree(self.dir)

    def upstream(self):
        self.log("Extracting tarball %s\n" % self.input_tar)
        match = re.match(r'(^|.*/).*\.(?P<extension>(tar\.(bz2|gz)|tbz2|tgz))?$', self.input_tar)
        if not match:
            raise RuntimeError("Can't identify name of tarball")
        os.mkdir(os.path.join(self.dir, self.orig))
        cmdline = ['tar -xf', self.input_tar, '-C', os.path.join(self.dir, self.orig)]
        if match.group('extension') in ('tar.bz2', 'tbz2'):
            cmdline.append('-j')
        elif match.group('extension') in ('tar.gz', 'tgz'):
            cmdline.append('-z')
        if os.spawnv(os.P_WAIT, '/bin/sh', ['sh', '-c', ' '.join(cmdline)]):
            raise RuntimeError("Can't extract tarball")

    def move(self):
        root_real = os.path.join(self.dir, self.orig)
        for root, dirs, files in os.walk(root_real):
            for dir in dirs:
                dir_list = dir.split('-')
                if len(dir_list) == 2 and dir_list[1] == '3.3':
                    os.rename(os.path.join(root, dir), os.path.join(root, dir_list[0]))
            del dirs[:]
        for root, dirs, files in os.walk(root_real):
            for dir in dirs:
                os.chmod(os.path.join(root, dir), 0755)
            for file in files:
                f = os.path.join(root, file)
                mode = os.access(f, os.X_OK) and 0755 or 0644
                os.chmod(f, mode)

    def debian_patch(self):
        name = "orig-" + self.version_dfsg
        self.log("Patching source with debian patch (series %s)\n" % name)
        fp = file("debian/patches/series/" + name)
        series = PatchSeries(name, "debian/patches", fp)
        series(dir = os.path.join(self.dir, self.orig))

    def tar(self):
        out = os.path.join("../orig", self.orig_tar)
        try:
            os.mkdir("../orig")
        except OSError: pass
        try:
            os.stat(out)
            raise RuntimeError("Destination already exists")
        except OSError: pass
        self.log("Generate tarball %s\n" % out)
        cmdline = ['tar -czf', out, '-C', self.dir, self.orig]
        try:
            if os.spawnv(os.P_WAIT, '/bin/sh', ['sh', '-c', ' '.join(cmdline)]):
                raise RuntimeError("Can't patch source")
            os.chmod(out, 0644)
        except:
            try:
                os.unlink(out)
            except OSError:
                pass
            raise

if __name__ == '__main__':
    from optparse import OptionParser
    parser = OptionParser(usage = "%prog [OPTION]... TAR [PATCH]")
    parser.add_option("-v", "--version", dest = "version", help = "Override version", metavar = "VERSION")
    options, args = parser.parse_args()
    Main(args[0], options.version)()
