/** @file
  The Key Management Service (KMS) protocol as defined in the UEFI 2.3.1 specification is to
  provides services to generate, store, retrieve, and manage cryptographic keys.
  The intention is to specify a simple generic protocol that could be used for many implementations.

  A driver implementing the protocol may need to provide basic key service that consists of a
  key store and cryptographic key generation capability. It may connect to an external key
  server over the network, or to a Hardware Security Module (HSM) attached to the system it
  runs on, or anything else that is capable of providing the key management service.

  Copyright (c) 2011, Intel Corporation. All rights reserved.<BR>
  This program and the accompanying materials are licensed and made available under
  the terms and conditions of the BSD License that accompanies this distribution.
  The full text of the license may be found at
  http://opensource.org/licenses/bsd-license.php.

  THE PROGRAM IS DISTRIBUTED UNDER THE BSD LICENSE ON AN "AS IS" BASIS,
  WITHOUT WARRANTIES OR REPRESENTATIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED.

**/

#ifndef __KMS_H__
#define __KMS_H__

#define EFI_KMS_PROTOCOL_GUID \
  { \
    0xEC3A978D, 0x7C4E, 0x48FA, {0x9A, 0xBE, 0x6A, 0xD9, 0x1C, 0xC8, 0xF8, 0x11 } \
  }

typedef struct _EFI_KMS_PROTOCOL EFI_KMS_PROTOCOL;

//
// Where appropriate, EFI_KMS_DATA_TYPE values may be combined using a bitwise 'OR'
// operation to indicate support for multiple data types.
//
#define EFI_KMS_DATA_TYPE_NONE      0
#define EFI_KMS_DATA_TYPE_BINARY    1
#define EFI_KMS_DATA_TYPE_ASCII     2
#define EFI_KMS_DATA_TYPE_UNICODE   4
#define EFI_KMS_DATA_TYPE_UTF8      8


//
// The key formats recognized by the KMS protocol are defined by an EFI_GUID which specifies
// a (key-algorithm, key-size) pair. The names of these GUIDs are in the format
// EFI_KMS_KEY_(key-algorithm)_(key-size)_GUID, where the key-size is expressed in bits.
// The key formats recognized fall into three categories, generic (no algorithm), hash algorithms,
// and encrypted algorithms.
//

///
/// The following GUIDs define formats that contain generic key data of a specific size in bits,
/// but which is not associated with any specific key algorithm(s).
///@{
#define EFI_KMS_FORMAT_GENERIC_128_GUID \
  { \
    0xec8a3d69, 0x6ddf, 0x4108, {0x94, 0x76, 0x73, 0x37, 0xfc, 0x52, 0x21, 0x36 } \
  }
#define EFI_KMS_FORMAT_GENERIC_160_GUID \
  { \
    0xa3b3e6f8, 0xefca, 0x4bc1, {0x88, 0xfb, 0xcb, 0x87, 0x33, 0x9b, 0x25, 0x79 } \
  }
#define EFI_KMS_FORMAT_GENERIC_256_GUID \
  { \
    0x70f64793, 0xc323, 0x4261, {0xac, 0x2c, 0xd8, 0x76, 0xf2, 0x7c, 0x53, 0x45 } \
  }
#define EFI_KMS_FORMAT_GENERIC_512_GUID \
  { \
    0x978fe043, 0xd7af, 0x422e, {0x8a, 0x92, 0x2b, 0x48, 0xe4, 0x63, 0xbd, 0xe6 } \
  }
#define EFI_KMS_FORMAT_GENERIC_1024_GUID \
  { \
    0x43be0b44, 0x874b, 0x4ead, {0xb0, 0x9c, 0x24, 0x1a, 0x4f, 0xbd, 0x7e, 0xb3 } \
  }
#define EFI_KMS_FORMAT_GENERIC_2048_GUID \
  { \
    0x40093f23, 0x630c, 0x4626, {0x9c, 0x48, 0x40, 0x37, 0x3b, 0x19, 0xcb, 0xbe } \
  }
#define EFI_KMS_FORMAT_GENERIC_3072_GUID \
  { \
    0xb9237513, 0x6c44, 0x4411, {0xa9, 0x90, 0x21, 0xe5, 0x56, 0xe0, 0x5a, 0xde } \
  }
///@}

///
/// These GUIDS define key data formats that contain data generated by basic hash algorithms
/// with no cryptographic properties.
///@{
#define EFI_KMS_FORMAT_MD2_128_GUID \
  { \
    0x78be11c4, 0xee44, 0x4a22, {0x9f, 0x05, 0x03, 0x85, 0x2e, 0xc5, 0xc9, 0x78 } \
  }
#define EFI_KMS_FORMAT_MDC2_128_GUID \
  { \
    0xf7ad60f8, 0xefa8, 0x44a3, {0x91, 0x13, 0x23, 0x1f, 0x39, 0x9e, 0xb4, 0xc7 } \
  }
#define EFI_KMS_FORMAT_MD4_128_GUID \
  { \
    0xd1c17aa1, 0xcac5, 0x400f, {0xbe, 0x17, 0xe2, 0xa2, 0xae, 0x06, 0x67, 0x7c } \
  }
#define EFI_KMS_FORMAT_MDC4_128_GUID \
  { \
    0x3fa4f847, 0xd8eb, 0x4df4, {0xbd, 0x49, 0x10, 0x3a, 0x0a, 0x84, 0x7b, 0xbc } \
  }
#define EFI_KMS_FORMAT_MD5_128_GUID \
  { \
    0xdcbc3662, 0x9cda, 0x4b52, {0xa0, 0x4c, 0x82, 0xeb, 0x1d, 0x23, 0x48, 0xc7 } \
  }
#define EFI_KMS_FORMAT_MD5SHA_128_GUID \
  { \
    0x1c178237, 0x6897, 0x459e, {0x9d, 0x36, 0x67, 0xce, 0x8e, 0xf9, 0x4f, 0x76 } \
  }
#define EFI_KMS_FORMAT_SHA1_160_GUID \
  { \
    0x453c5e5a, 0x482d, 0x43f0, {0x87, 0xc9, 0x59, 0x41, 0xf3, 0xa3, 0x8a, 0xc2 } \
  }
#define EFI_KMS_FORMAT_SHA256_256_GUID \
  { \
    0x6bb4f5cd, 0x8022, 0x448d, {0xbc, 0x6d, 0x77, 0x1b, 0xae, 0x93, 0x5f, 0xc6 } \
  }
#define EFI_KMS_FORMAT_SHA512_512_GUID \
  { \
    0x2f240e12, 0xe14d, 0x475c, {0x83, 0xb0, 0xef, 0xff, 0x22, 0xd7, 0x7b, 0xe7 } \
  }
///@}

///
/// These GUIDs define key data formats that contain data generated by cryptographic key algorithms.
/// There may or may not be a separate data hashing algorithm associated with the key algorithm.
///@{
#define EFI_KMS_FORMAT_AESXTS_128_GUID \
  { \
    0x4776e33f, 0xdb47, 0x479a, {0xa2, 0x5f, 0xa1, 0xcd, 0x0a, 0xfa, 0xb3, 0x8b } \
  }
#define EFI_KMS_FORMAT_AESXTS_256_GUID \
  { \
    0xdc7e8613, 0xc4bb, 0x4db0, {0x84, 0x62, 0x13, 0x51, 0x13, 0x57, 0xab, 0xe2 } \
  }
#define EFI_KMS_FORMAT_AESCBC_128_GUID \
  { \
    0xa0e8ee6a, 0x0e92, 0x44d4, {0x86, 0x1b, 0x0e, 0xaa, 0x4a, 0xca, 0x44, 0xa2 } \
  }
#define EFI_KMS_FORMAT_AESCBC_256_GUID \
  { \
    0xd7e69789, 0x1f68, 0x45e8, {0x96, 0xef, 0x3b, 0x64, 0x07, 0xa5, 0xb2, 0xdc } \
  }
#define EFI_KMS_FORMAT_RSASHA1_1024_GUID \
  { \
    0x56417bed, 0x6bbe, 0x4882, {0x86, 0xa0, 0x3a, 0xe8, 0xbb, 0x17, 0xf8, 0xf9 } \
  }
#define EFI_KMS_FORMAT_RSASHA1_2048_GUID \
  { \
    0xf66447d4, 0x75a6, 0x463e, {0xa8, 0x19, 0x07, 0x7f, 0x2d, 0xda, 0x05, 0xe9 } \
  }
#define EFI_KMS_FORMAT_RSASHA256_2048_GUID \
  { \
    0xa477af13, 0x877d, 0x4060, {0xba, 0xa1, 0x25, 0xd1, 0xbe, 0xa0, 0x8a, 0xd3 } \
  }
#define EFI_KMS_FORMAT_RSASHA256_3072_GUID \
  { \
    0x4e1356c2,  0xeed, 0x463f, {0x81, 0x47, 0x99, 0x33, 0xab, 0xdb, 0xc7, 0xd5 } \
  }
///@}

#define EFI_KMS_ATTRIBUTE_TYPE_NONE             0x00
#define EFI_KMS_ATTRIBUTE_TYPE_INTEGER          0x01
#define EFI_KMS_ATTRIBUTE_TYPE_LONG_INTEGER     0x02
#define EFI_KMS_ATTRIBUTE_TYPE_BIG_INTEGER      0x03
#define EFI_KMS_ATTRIBUTE_TYPE_ENUMERATION      0x04
#define EFI_KMS_ATTRIBUTE_TYPE_BOOLEAN          0x05
#define EFI_KMS_ATTRIBUTE_TYPE_BYTE_STRING      0x06
#define EFI_KMS_ATTRIBUTE_TYPE_TEXT_STRING      0x07
#define EFI_KMS_ATTRIBUTE_TYPE_DATE_TIME        0x08
#define EFI_KMS_ATTRIBUTE_TYPE_INTERVAL         0x09
#define EFI_KMS_ATTRIBUTE_TYPE_STRUCTURE        0x0A
#define EFI_KMS_ATTRIBUTE_TYPE_DYNAMIC          0x0B

typedef struct {
  ///
  /// The size in bytes for the client identifier.
  ///
  UINT16        ClientIdSize;
  ///
  /// Pointer to a valid client identifier.
  ///
  VOID          *ClientId;
  ///
  /// The client name string type used by this client. The string type set here must be one of
  /// the string types reported in the ClientNameStringTypes field of the KMS protocol. If the
  /// KMS does not support client names, this field should be set to EFI_KMS_DATA_TYPE_NONE.
  ///
  UINT8         ClientNameType;
  ///
  /// The size in characters for the client name. This field will be ignored if
  /// ClientNameStringType is set to EFI_KMS_DATA_TYPE_NONE. Otherwise, it must contain
  /// number of characters contained in the ClientName field.
  ///
  UINT8         ClientNameCount;
  ///
  /// Pointer to a client name. This field will be ignored if ClientNameStringType is set to
  /// EFI_KMS_DATA_TYPE_NONE. Otherwise, it must point to a valid string of the specified type.
  ///
  VOID          *ClientName;
} EFI_KMS_CLIENT_INFO;

typedef struct {
  ///
  /// The size of the KeyIdentifier field in bytes. This field is limited to the range 0 to 255.
  ///
  UINT8         KeyIdentifierSize;
  ///
  /// Pointer to an array of KeyIdentifierType elements.
  ///
  VOID          *KeyIdentifier;
  ///
  /// An EFI_GUID which specifies the algorithm and key value size for this key.
  ///
  EFI_GUID      KeyFormat;
  ///
  /// Pointer to a key value for a key specified by the KeyFormat field. A NULL value for this
  /// field indicates that no key is available.
  ///
  VOID          *KeyValue;
  ///
  /// Specifies the results of KMS operations performed with this descriptor. This field is used
  /// to indicate the status of individual operations when a KMS function is called with multiple
  /// EFI_KMS_KEY_DESCRIPTOR structures.
  /// KeyStatus codes returned for the individual key requests are:
  ///   EFI_SUCCESS             Successfully processed this key.
  ///   EFI_WARN_STALE_DATA     Successfully processed this key, however, the key's parameters
  ///                           exceed internal policies/limits and should be replaced.
  ///   EFI_COMPROMISED_DATA    Successfully processed this key, but the key may have been
  ///                           compromised and must be replaced.
  ///   EFI_UNSUPPORTED         Key format is not supported by the service.
  ///   EFI_OUT_OF_RESOURCES    Could not allocate resources for the key processing.
  ///   EFI_TIMEOUT             Timed out waiting for device or key server.
  ///   EFI_DEVICE_ERROR        Device or key server error.
  ///   EFI_INVALID_PARAMETER   KeyFormat is invalid.
  ///   EFI_NOT_FOUND           The key does not exist on the KMS.
  ///
  EFI_STATUS    KeyStatus;
} EFI_KMS_KEY_DESCRIPTOR;

typedef struct {
  ///
  /// Part of a tag-type-length triplet that identifies the KeyAttributeData formatting. The
  /// definition of the value is outside the scope of this standard and may be defined by the KMS.
  ///
  UINT16        Tag;
  ///
  /// Part of a tag-type-length triplet that identifies the KeyAttributeData formatting. The
  /// definition of the value is outside the scope of this standard and may be defined by the KMS.
  ///
  UINT16        Type;
  ///
  /// Length in bytes of the KeyAttributeData.
  ///
  UINT32        Length;
  ///
  /// An array of bytes to hold the attribute data associated with the KeyAttributeIdentifier.
  ///
  UINT8         KeyAttributeData[1];
} EFI_KMS_DYNAMIC_FIELD;

typedef struct {
  ///
  /// The number of members in the EFI_KMS_DYNAMIC_ATTRIBUTE structure.
  ///
  UINT32                    FieldCount;
  ///
  /// An array of EFI_KMS_DYNAMIC_FIELD structures.
  ///
  EFI_KMS_DYNAMIC_FIELD     Field[1];
} EFI_KMS_DYNAMIC_ATTRIBUTE;

typedef struct {
  ///
  /// The data type used for the KeyAttributeIdentifier field. Values for this field are defined
  /// by the EFI_KMS_DATA_TYPE constants, except that EFI_KMS_DATA_TYPE_BINARY is not
  /// valid for this field.
  ///
  UINT8         KeyAttributeIdentifierType;
  ///
  /// The length of the KeyAttributeIdentifier field in units defined by KeyAttributeIdentifierType
  /// field. This field is limited to the range 0 to 255.
  ///
  UINT8         KeyAttributeIdentifierCount;
  ///
  /// Pointer to an array of KeyAttributeIdentifierType elements. For string types, there must
  /// not be a null-termination element at the end of the array.
  ///
  VOID          *KeyAttributeIdentifier;
  ///
  /// The instance number of this attribute. If there is only one instance, the value is set to
  /// one. If this value is set to 0xFFFF (all binary 1's) then this field should be ignored if an
  /// output or treated as a wild card matching any value if it is an input. If the attribute is
  /// stored with this field, it will match any attribute request regardless of the setting of the
  /// field in the request. If set to 0xFFFF in the request, it will match any attribute with the
  /// same KeyAttributeIdentifier.
  ///
  UINT16        KeyAttributeInstance;
  ///
  /// The data type of the KeyAttributeValue (e.g. struct, bool, etc.). See the list of
  /// KeyAttributeType definitions.
  ///
  UINT16        KeyAttributeType;
  ///
  /// The size in bytes of the KeyAttribute field. A value of zero for this field indicates that no
  /// key attribute value is available.
  ///
  UINT16        KeyAttributeValueSize;
  ///
  /// Pointer to a key attribute value for the attribute specified by the KeyAttributeIdentifier
  /// field. If the KeyAttributeValueSize field is zero, then this field must be NULL.
  ///
  VOID          *KeyAttributeValue;
  ///
  /// KeyAttributeStatusSpecifies the results of KMS operations performed with this attribute.
  /// This field is used to indicate the status of individual operations when a KMS function is
  /// called with multiple EFI_KMS_KEY_ATTRIBUTE structures.
  /// KeyAttributeStatus codes returned for the individual key attribute requests are:
  ///   EFI_SUCCESS             Successfully processed this request.
  ///   EFI_WARN_STALE_DATA     Successfully processed this request, however, the key's
  ///                           parameters exceed internal policies/limits and should be replaced.
  ///   EFI_COMPROMISED_DATA    Successfully processed this request, but the key may have been
  ///                           compromised and must be replaced.
  ///   EFI_UNSUPPORTED         Key attribute format is not supported by the service.
  ///   EFI_OUT_OF_RESOURCES    Could not allocate resources for the request processing.
  ///   EFI_TIMEOUT             Timed out waiting for device or key server.
  ///   EFI_DEVICE_ERROR        Device or key server error.
  ///   EFI_INVALID_PARAMETER   A field in the EFI_KMS_KEY_ATTRIBUTE structure is invalid.
  ///   EFI_NOT_FOUND           The key attribute does not exist on the KMS.
  ///
  EFI_STATUS    KeyAttributeStatus;
} EFI_KMS_KEY_ATTRIBUTE;

/**
  Get the current status of the key management service.

  @param[in]      This              Pointer to the EFI_KMS_PROTOCOL instance.

  @retval EFI_SUCCESS               The KMS is ready for use.
  @retval EFI_NOT_READY             No connection to the KMS is available.
  @retval EFI_NO_MAPPING            No valid connection configuration exists for the KMS.
  @retval EFI_NO_RESPONSE           No response was received from the KMS.
  @retval EFI_DEVICE_ERROR          An error occurred when attempting to access the KMS.
  @retval EFI_INVALID_PARAMETER     This is NULL.

**/
typedef
EFI_STATUS
(EFIAPI *EFI_KMS_GET_SERVICE_STATUS) (
  IN EFI_KMS_PROTOCOL           *This
  );

/**
  Register client information with the supported KMS.

  @param[in]      This              Pointer to the EFI_KMS_PROTOCOL instance.
  @param[in]      Client            Pointer to a valid EFI_KMS_CLIENT_INFO structure.
  @param[in, out] ClientDataSize    Pointer to the size, in bytes, of an arbitrary block of 
                                    data specified by the ClientData parameter. This
                                    parameter may be NULL, in which case the ClientData
                                    parameter will be ignored and no data will be
                                    transferred to or from the KMS. If the parameter is
                                    not NULL, then ClientData must be a valid pointer.
                                    If the value pointed to is 0, no data will be transferred
                                    to the KMS, but data may be returned by the KMS.
                                    For all non-zero values *ClientData will be transferred
                                    to the KMS, which may also return data to the caller.
                                    In all cases, the value upon return to the caller will
                                    be the size of the data block returned to the caller,
                                    which will be zero if no data is returned from the KMS.
  @param[in, out] ClientData        Pointer to a pointer to an arbitrary block of data of
                                    *ClientDataSize that is to be passed directly to the
                                    KMS if it supports the use of client data. This 
                                    parameter may be NULL if and only if the 
                                    ClientDataSize parameter is also NULL. Upon return to
                                    the caller, *ClientData points to a block of data of 
                                    *ClientDataSize that was returned from the KMS. 
                                    If the returned value for *ClientDataSize is zero,
                                    then the returned value for *ClientData must be NULL
                                    and should be ignored by the caller. The KMS protocol
                                    consumer is responsible for freeing all valid buffers
                                    used for client data regardless of whether they are
                                    allocated by the caller for input to the function or by
                                    the implementation for output back to the caller.

  @retval EFI_SUCCESS               The client information has been accepted by the KMS.
  @retval EFI_NOT_READY             No connection to the KMS is available.
  @retval EFI_NO_RESPONSE           There was no response from the device or the key server.
  @retval EFI_ACCESS_DENIED         Access was denied by the device or the key server.
  @retval EFI_DEVICE_ERROR          An error occurred when attempting to access the KMS.
  @retval EFI_OUT_OF_RESOURCES      Required resources were not available to perform the function.
  @retval EFI_INVALID_PARAMETER     This is NULL.
  @retval EFI_UNSUPPORTED           The KMS does not support the use of client identifiers.

**/
typedef
EFI_STATUS
(EFIAPI *EFI_KMS_REGISTER_CLIENT) (
  IN EFI_KMS_PROTOCOL           *This,
  IN EFI_KMS_CLIENT_INFO        *Client,
  IN OUT UINTN                  *ClientDataSize OPTIONAL,
  IN OUT VOID                   **ClientData OPTIONAL
  ); 

/**
  Request that the KMS generate one or more new keys and associate them with key identifiers.
  The key value(s) is returned to the caller.

  @param[in]      This               Pointer to the EFI_KMS_PROTOCOL instance.
  @param[in]      Client             Pointer to a valid EFI_KMS_CLIENT_INFO structure.
  @param[in, out] KeyDescriptorCount Pointer to a count of the number of key descriptors to be
                                     processed by this operation. On return, this number
                                     will be updated with the number of key descriptors
                                     successfully processed.
  @param[in, out] KeyDescriptors     Pointer to an array of EFI_KMS_KEY_DESCRIPTOR
                                     structures which describe the keys to be generated.
                                     On input, the KeyIdentifierSize and the KeyIdentifier
                                     may specify an identifier to be used for the key,
                                     but this is not required. The KeyFormat field must
                                     specify a key format GUID reported as supported by
                                     the KeyFormats field of the EFI_KMS_PROTOCOL.
                                     The value for this field in the first key descriptor will
                                     be considered the default value for subsequent key
                                     descriptors requested in this operation if those key
                                     descriptors have a NULL GUID in the key format field.
                                     On output, the KeyIdentifierSize and KeyIdentifier fields
                                     will specify an identifier for the key which will be either
                                     the original identifier if one was provided, or an identifier
                                     generated either by the KMS or the KMS protocol
                                     implementation. The KeyFormat field will be updated
                                     with the GUID used to generate the key if it was a
                                     NULL GUID, and the KeyValue field will contain a pointer
                                     to memory containing the key value for the generated
                                     key. Memory for both the KeyIdentifier and the KeyValue
                                     fields will be allocated with the BOOT_SERVICES_DATA
                                     type and must be freed by the caller when it is no longer
                                     needed. Also, the KeyStatus field must reflect the result
                                     of the request relative to that key.
  @param[in, out] ClientDataSize     Pointer to the size, in bytes, of an arbitrary block of 
                                     data specified by the ClientData parameter. This
                                     parameter may be NULL, in which case the ClientData
                                     parameter will be ignored and no data will be
                                     transferred to or from the KMS. If the parameter is
                                     not NULL, then ClientData must be a valid pointer.
                                     If the value pointed to is 0, no data will be transferred
                                     to the KMS, but data may be returned by the KMS.
                                     For all non-zero values *ClientData will be transferred
                                     to the KMS, which may also return data to the caller.
                                     In all cases, the value upon return to the caller will
                                     be the size of the data block returned to the caller,
                                     which will be zero if no data is returned from the KMS.
  @param[in, out] ClientData         Pointer to a pointer to an arbitrary block of data of
                                     *ClientDataSize that is to be passed directly to the
                                     KMS if it supports the use of client data. This 
                                     parameter may be NULL if and only if the 
                                     ClientDataSize parameter is also NULL. Upon return to
                                     the caller, *ClientData points to a block of data of 
                                     *ClientDataSize that was returned from the KMS. 
                                     If the returned value for *ClientDataSize is zero,
                                     then the returned value for *ClientData must be NULL
                                     and should be ignored by the caller. The KMS protocol
                                     consumer is responsible for freeing all valid buffers
                                     used for client data regardless of whether they are
                                     allocated by the caller for input to the function or by
                                     the implementation for output back to the caller.

  @retval EFI_SUCCESS                Successfully generated and retrieved all requested keys.
  @retval EFI_UNSUPPORTED            This function is not supported by the KMS. --OR--
                                     One (or more) of the key requests submitted is not supported by
                                     the KMS. Check individual key request(s) to see which ones
                                     may have been processed.
  @retval EFI_OUT_OF_RESOURCES       Required resources were not available to perform the function.
  @retval EFI_TIMEOUT                Timed out waiting for device or key server. Check individual key
                                     request(s) to see which ones may have been processed.
  @retval EFI_ACCESS_DENIED          Access was denied by the device or the key server; OR a
                                     ClientId is required by the server and either no id was
                                     provided or an invalid id was provided.
  @retval EFI_DEVICE_ERROR           An error occurred when attempting to access the KMS. Check
                                     individual key request(s) to see which ones may have been
                                     processed.
  @retval EFI_INVALID_PARAMETER      This is NULL, ClientId is required but it is NULL,
                                     KeyDescriptorCount is NULL, or Keys is NULL.
  @retval EFI_NOT_FOUND              One or more EFI_KMS_KEY_DESCRIPTOR structures
                                     could not be processed properly. KeyDescriptorCount
                                     contains the number of structures which were successfully
                                     processed. Individual structures will reflect the status of the
                                     processing for that structure.

**/
typedef
EFI_STATUS
(EFIAPI *EFI_KMS_CREATE_KEY) (
  IN EFI_KMS_PROTOCOL           *This,
  IN EFI_KMS_CLIENT_INFO        *Client,
  IN OUT UINT16                 *KeyDescriptorCount,
  IN OUT EFI_KMS_KEY_DESCRIPTOR *KeyDescriptors,
  IN OUT UINTN                  *ClientDataSize OPTIONAL,
  IN OUT VOID                   **ClientData OPTIONAL
  );

/**
  Retrieve an existing key.

  @param[in]      This               Pointer to the EFI_KMS_PROTOCOL instance.
  @param[in]      Client             Pointer to a valid EFI_KMS_CLIENT_INFO structure.
  @param[in, out] KeyDescriptorCount Pointer to a count of the number of key descriptors to be
                                     processed by this operation. On return, this number
                                     will be updated with the number of key descriptors
                                     successfully processed.
  @param[in, out] KeyDescriptors     Pointer to an array of EFI_KMS_KEY_DESCRIPTOR
                                     structures which describe the keys to be retrieved
                                     from the KMS.
                                     On input, the KeyIdentifierSize and the KeyIdentifier
                                     must specify an identifier to be used to retrieve a
                                     specific key. All other fields in the descriptor should
                                     be NULL.
                                     On output, the KeyIdentifierSize and KeyIdentifier fields
                                     will be unchanged, while the KeyFormat and KeyValue
                                     fields will be updated values associated with this key
                                     identifier. Memory for the KeyValue field will be 
                                     allocated with the BOOT_SERVICES_DATA type and
                                     must be freed by the caller when it is no longer needed.
                                     Also, the KeyStatus field will reflect the result of the
                                     request relative to the individual key descriptor.
  @param[in, out] ClientDataSize     Pointer to the size, in bytes, of an arbitrary block of 
                                     data specified by the ClientData parameter. This
                                     parameter may be NULL, in which case the ClientData
                                     parameter will be ignored and no data will be
                                     transferred to or from the KMS. If the parameter is
                                     not NULL, then ClientData must be a valid pointer.
                                     If the value pointed to is 0, no data will be transferred
                                     to the KMS, but data may be returned by the KMS.
                                     For all non-zero values *ClientData will be transferred
                                     to the KMS, which may also return data to the caller.
                                     In all cases, the value upon return to the caller will
                                     be the size of the data block returned to the caller,
                                     which will be zero if no data is returned from the KMS.
  @param[in, out] ClientData         Pointer to a pointer to an arbitrary block of data of
                                     *ClientDataSize that is to be passed directly to the
                                     KMS if it supports the use of client data. This 
                                     parameter may be NULL if and only if the 
                                     ClientDataSize parameter is also NULL. Upon return to
                                     the caller, *ClientData points to a block of data of 
                                     *ClientDataSize that was returned from the KMS. 
                                     If the returned value for *ClientDataSize is zero,
                                     then the returned value for *ClientData must be NULL
                                     and should be ignored by the caller. The KMS protocol
                                     consumer is responsible for freeing all valid buffers
                                     used for client data regardless of whether they are
                                     allocated by the caller for input to the function or by
                                     the implementation for output back to the caller.

  @retval EFI_SUCCESS                Successfully retrieved all requested keys.
  @retval EFI_OUT_OF_RESOURCES       Could not allocate resources for the method processing.
  @retval EFI_TIMEOUT                Timed out waiting for device or key server. Check individual key
                                     request(s) to see which ones may have been processed.
  @retval EFI_BUFFER_TOO_SMALL       If multiple keys are associated with a single identifier, and the
                                     KeyValue buffer does not contain enough structures
                                     (KeyDescriptorCount) to contain all the key data, then
                                     the available structures will be filled and
                                     KeyDescriptorCount will be updated to indicate the
                                     number of keys which could not be processed.
  @retval EFI_ACCESS_DENIED          Access was denied by the device or the key server; OR a
                                     ClientId is required by the server and either none or an
                                     invalid id was provided.
  @retval EFI_DEVICE_ERROR           Device or key server error. Check individual key request(s) to
                                     see which ones may have been processed.
  @retval EFI_INVALID_PARAMETER      This is NULL, ClientId is required but it is NULL,
                                     KeyDescriptorCount is NULL, or Keys is NULL.
  @retval EFI_NOT_FOUND              One or more EFI_KMS_KEY_DESCRIPTOR structures
                                     could not be processed properly. KeyDescriptorCount
                                     contains the number of structures which were successfully
                                     processed. Individual structures will reflect the status of the
                                     processing for that structure.
  @retval EFI_UNSUPPORTED            The implementation/KMS does not support this function.

**/
typedef
EFI_STATUS
(EFIAPI *EFI_KMS_GET_KEY) (
  IN EFI_KMS_PROTOCOL           *This,
  IN EFI_KMS_CLIENT_INFO        *Client,
  IN OUT UINT16                 *KeyDescriptorCount,
  IN OUT EFI_KMS_KEY_DESCRIPTOR *KeyDescriptors,
  IN OUT UINTN                  *ClientDataSize OPTIONAL,
  IN OUT VOID                   **ClientData OPTIONAL
  );

/**
  Add a new key.

  @param[in]      This               Pointer to the EFI_KMS_PROTOCOL instance.
  @param[in]      Client             Pointer to a valid EFI_KMS_CLIENT_INFO structure.
  @param[in, out] KeyDescriptorCount Pointer to a count of the number of key descriptors to be
                                     processed by this operation. On normal return, this
                                     number will be updated with the number of key
                                     descriptors successfully processed.
  @param[in, out] KeyDescriptors     Pointer to an array of EFI_KMS_KEY_DESCRIPTOR
                                     structures which describe the keys to be added.
                                     On input, the KeyId field for first key must contain
                                     valid identifier data to be used for adding a key to
                                     the KMS. The values for these fields in this key
                                     definition will be considered default values for
                                     subsequent keys requested in this operation. A value
                                     of 0 in any subsequent KeyId field will be replaced
                                     with the current default value. The KeyFormat and
                                     KeyValue fields for each key to be added must contain
                                     consistent values to be associated with the given KeyId.
                                     On return, the KeyStatus field will reflect the result
                                     of the operation for each key request.
  @param[in, out] ClientDataSize     Pointer to the size, in bytes, of an arbitrary block of 
                                     data specified by the ClientData parameter. This
                                     parameter may be NULL, in which case the ClientData
                                     parameter will be ignored and no data will be
                                     transferred to or from the KMS. If the parameter is
                                     not NULL, then ClientData must be a valid pointer.
                                     If the value pointed to is 0, no data will be transferred
                                     to the KMS, but data may be returned by the KMS.
                                     For all non-zero values *ClientData will be transferred
                                     to the KMS, which may also return data to the caller.
                                     In all cases, the value upon return to the caller will
                                     be the size of the data block returned to the caller,
                                     which will be zero if no data is returned from the KMS.
  @param[in, out] ClientData         Pointer to a pointer to an arbitrary block of data of
                                     *ClientDataSize that is to be passed directly to the
                                     KMS if it supports the use of client data. This 
                                     parameter may be NULL if and only if the 
                                     ClientDataSize parameter is also NULL. Upon return to
                                     the caller, *ClientData points to a block of data of 
                                     *ClientDataSize that was returned from the KMS. 
                                     If the returned value for *ClientDataSize is zero,
                                     then the returned value for *ClientData must be NULL
                                     and should be ignored by the caller. The KMS protocol
                                     consumer is responsible for freeing all valid buffers
                                     used for client data regardless of whether they are
                                     allocated by the caller for input to the function or by
                                     the implementation for output back to the caller.

  @retval EFI_SUCCESS                Successfully added all requested keys.
  @retval EFI_OUT_OF_RESOURCES       Could not allocate required resources.
  @retval EFI_TIMEOUT                Timed out waiting for device or key server. Check individual key
                                     request(s) to see which ones may have been processed.
  @retval EFI_BUFFER_TOO_SMALL       If multiple keys are associated with a single identifier, and the
                                     KeyValue buffer does not contain enough structures
                                     (KeyDescriptorCount) to contain all the key data, then
                                     the available structures will be filled and
                                     KeyDescriptorCount will be updated to indicate the
                                     number of keys which could not be processed
  @retval EFI_ACCESS_DENIED          Access was denied by the device or the key server; OR a
                                     ClientId is required by the server and either none or an
                                     invalid id was provided.
  @retval EFI_DEVICE_ERROR           Device or key server error. Check individual key request(s) to
                                     see which ones may have been processed.
  @retval EFI_INVALID_PARAMETER      This is NULL, ClientId is required but it is NULL,
                                     KeyDescriptorCount is NULL, or Keys is NULL.
  @retval EFI_NOT_FOUND              One or more EFI_KMS_KEY_DESCRIPTOR structures
                                     could not be processed properly. KeyDescriptorCount
                                     contains the number of structures which were successfully
                                     processed. Individual structures will reflect the status of the
                                     processing for that structure.
  @retval EFI_UNSUPPORTED            The implementation/KMS does not support this function.

**/
typedef
EFI_STATUS
(EFIAPI *EFI_KMS_ADD_KEY) (
  IN EFI_KMS_PROTOCOL           *This,
  IN EFI_KMS_CLIENT_INFO        *Client,
  IN OUT UINT16                 *KeyDescriptorCount,
  IN OUT EFI_KMS_KEY_DESCRIPTOR *KeyDescriptors,
  IN OUT UINTN                  *ClientDataSize OPTIONAL,
  IN OUT VOID                   **ClientData OPTIONAL
  );

/**
  Delete an existing key from the KMS database.

  @param[in]      This               Pointer to the EFI_KMS_PROTOCOL instance.
  @param[in]      Client             Pointer to a valid EFI_KMS_CLIENT_INFO structure.
  @param[in, out] KeyDescriptorCount Pointer to a count of the number of key descriptors to be
                                     processed by this operation. On normal return, this
                                     number will be updated with the number of key
                                     descriptors successfully processed.
  @param[in, out] KeyDescriptors     Pointer to an array of EFI_KMS_KEY_DESCRIPTOR
                                     structures which describe the keys to be deleted.
                                     On input, the KeyId field for first key must contain
                                     valid identifier data to be used for adding a key to
                                     the KMS. The values for these fields in this key
                                     definition will be considered default values for
                                     subsequent keys requested in this operation. A value
                                     of 0 in any subsequent KeyId field will be replaced
                                     with the current default value. The KeyFormat and
                                     KeyValue fields are ignored, but should be 0.
                                     On return, the KeyStatus field will reflect the result
                                     of the operation for each key request.
  @param[in, out] ClientDataSize     Pointer to the size, in bytes, of an arbitrary block of 
                                     data specified by the ClientData parameter. This
                                     parameter may be NULL, in which case the ClientData
                                     parameter will be ignored and no data will be
                                     transferred to or from the KMS. If the parameter is
                                     not NULL, then ClientData must be a valid pointer.
                                     If the value pointed to is 0, no data will be transferred
                                     to the KMS, but data may be returned by the KMS.
                                     For all non-zero values *ClientData will be transferred
                                     to the KMS, which may also return data to the caller.
                                     In all cases, the value upon return to the caller will
                                     be the size of the data block returned to the caller,
                                     which will be zero if no data is returned from the KMS.
  @param[in, out] ClientData         Pointer to a pointer to an arbitrary block of data of
                                     *ClientDataSize that is to be passed directly to the
                                     KMS if it supports the use of client data. This 
                                     parameter may be NULL if and only if the 
                                     ClientDataSize parameter is also NULL. Upon return to
                                     the caller, *ClientData points to a block of data of 
                                     *ClientDataSize that was returned from the KMS. 
                                     If the returned value for *ClientDataSize is zero,
                                     then the returned value for *ClientData must be NULL
                                     and should be ignored by the caller. The KMS protocol
                                     consumer is responsible for freeing all valid buffers
                                     used for client data regardless of whether they are
                                     allocated by the caller for input to the function or by
                                     the implementation for output back to the caller.

  @retval EFI_SUCCESS                Successfully deleted all requested keys.
  @retval EFI_OUT_OF_RESOURCES       Could not allocate required resources.
  @retval EFI_TIMEOUT                Timed out waiting for device or key server. Check individual key
                                     request(s) to see which ones may have been processed.
  @retval EFI_ACCESS_DENIED          Access was denied by the device or the key server; OR a
                                     ClientId is required by the server and either none or an
                                     invalid id was provided.
  @retval EFI_DEVICE_ERROR           Device or key server error. Check individual key request(s) to
                                     see which ones may have been processed.
  @retval EFI_INVALID_PARAMETER      This is NULL, ClientId is required but it is NULL,
                                     KeyDescriptorCount is NULL, or Keys is NULL.
  @retval EFI_NOT_FOUND              One or more EFI_KMS_KEY_DESCRIPTOR structures
                                     could not be processed properly. KeyDescriptorCount
                                     contains the number of structures which were successfully
                                     processed. Individual structures will reflect the status of the
                                     processing for that structure.
  @retval EFI_UNSUPPORTED            The implementation/KMS does not support this function.

**/
typedef
EFI_STATUS
(EFIAPI *EFI_KMS_DELETE_KEY) (
  IN EFI_KMS_PROTOCOL           *This,
  IN EFI_KMS_CLIENT_INFO        *Client,
  IN OUT UINT16                 *KeyDescriptorCount,
  IN OUT EFI_KMS_KEY_DESCRIPTOR *KeyDescriptors,
  IN OUT UINTN                  *ClientDataSize OPTIONAL,
  IN OUT VOID                   **ClientData OPTIONAL
  );

/**
  Get one or more attributes associated with a specified key identifier.
  If none are found, the returned attributes count contains a value of zero.

  @param[in]      This               Pointer to the EFI_KMS_PROTOCOL instance.
  @param[in]      Client             Pointer to a valid EFI_KMS_CLIENT_INFO structure.
  @param[in]      KeyIdentifierSize  Pointer to the size in bytes of the KeyIdentifier variable.
  @param[in]      KeyIdentifier      Pointer to the key identifier associated with this key.
  @param[in, out] KeyAttributesCount Pointer to the number of EFI_KMS_KEY_ATTRIBUTE
                                     structures associated with the Key identifier. If none
                                     are found, the count value is zero on return.
                                     On input this value reflects the number of KeyAttributes
                                     that may be returned.
                                     On output, the value reflects the number of completed
                                     KeyAttributes structures found.
  @param[in, out] KeyAttributes      Pointer to an array of EFI_KMS_KEY_ATTRIBUTE
                                     structures associated with the Key Identifier.
                                     On input, the fields in the structure should be NULL.
                                     On output, the attribute fields will have updated values
                                     for attributes associated with this key identifier.
  @param[in, out] ClientDataSize     Pointer to the size, in bytes, of an arbitrary block of 
                                     data specified by the ClientData parameter. This
                                     parameter may be NULL, in which case the ClientData
                                     parameter will be ignored and no data will be
                                     transferred to or from the KMS. If the parameter is
                                     not NULL, then ClientData must be a valid pointer.
                                     If the value pointed to is 0, no data will be transferred
                                     to the KMS, but data may be returned by the KMS.
                                     For all non-zero values *ClientData will be transferred
                                     to the KMS, which may also return data to the caller.
                                     In all cases, the value upon return to the caller will
                                     be the size of the data block returned to the caller,
                                     which will be zero if no data is returned from the KMS.
  @param[in, out] ClientData         Pointer to a pointer to an arbitrary block of data of
                                     *ClientDataSize that is to be passed directly to the
                                     KMS if it supports the use of client data. This 
                                     parameter may be NULL if and only if the 
                                     ClientDataSize parameter is also NULL. Upon return to
                                     the caller, *ClientData points to a block of data of 
                                     *ClientDataSize that was returned from the KMS. 
                                     If the returned value for *ClientDataSize is zero,
                                     then the returned value for *ClientData must be NULL
                                     and should be ignored by the caller. The KMS protocol
                                     consumer is responsible for freeing all valid buffers
                                     used for client data regardless of whether they are
                                     allocated by the caller for input to the function or by
                                     the implementation for output back to the caller.

  @retval EFI_SUCCESS                Successfully retrieved all key attributes.
  @retval EFI_OUT_OF_RESOURCES       Could not allocate resources for the method processing.
  @retval EFI_TIMEOUT                Timed out waiting for device or key server. Check individual key
                                     attribute request(s) to see which ones may have been
                                     processed.
  @retval EFI_BUFFER_TOO_SMALL       If multiple key attributes are associated with a single identifier,
                                     and the KeyAttributes buffer does not contain enough
                                     structures (KeyAttributesCount) to contain all the key
                                     attributes data, then the available structures will be filled and
                                     KeyAttributesCount will be updated to indicate the
                                     number of key attributes which could not be processed.
  @retval EFI_ACCESS_DENIED          Access was denied by the device or the key server; OR a
                                     ClientId is required by the server and either none or an
                                     invalid id was provided.
  @retval EFI_DEVICE_ERROR           Device or key server error. Check individual key attribute
                                     request(s) (i.e. key attribute status for each) to see which ones
                                     may have been processed.
  @retval EFI_INVALID_PARAMETER      This is NULL, ClientId is required but it is NULL,
                                     KeyIdentifierSize is NULL , or KeyIdentifier
                                     is NULL, or KeyAttributes is NULL, or
                                     KeyAttributesSize is NULL.
  @retval EFI_NOT_FOUND              The KeyIdentifier could not be found.
                                     KeyAttributesCount contains zero. Individual
                                     structures will reflect the status of the processing for that
                                     structure.
  @retval EFI_UNSUPPORTED            The implementation/KMS does not support this function.

**/
typedef
EFI_STATUS
(EFIAPI *EFI_KMS_GET_KEY_ATTRIBUTES) (
  IN EFI_KMS_PROTOCOL           *This,
  IN EFI_KMS_CLIENT_INFO        *Client,
  IN UINT8                      *KeyIdentifierSize,
  IN CONST VOID                 *KeyIdentifier,
  IN OUT UINT16                 *KeyAttributesCount,
  IN OUT EFI_KMS_KEY_ATTRIBUTE  *KeyAttributes,
  IN OUT UINTN                  *ClientDataSize OPTIONAL,
  IN OUT VOID                   **ClientData OPTIONAL
  );

/**
  Add one or more attributes to a key specified by a key identifier.

  @param[in]      This               Pointer to the EFI_KMS_PROTOCOL instance.
  @param[in]      Client             Pointer to a valid EFI_KMS_CLIENT_INFO structure.
  @param[in]      KeyIdentifierSize  Pointer to the size in bytes of the KeyIdentifier variable.
  @param[in]      KeyIdentifier      Pointer to the key identifier associated with this key.
  @param[in, out] KeyAttributesCount Pointer to the number of EFI_KMS_KEY_ATTRIBUTE
                                     structures to associate with the Key. On normal returns,
                                     this number will be updated with the number of key
                                     attributes successfully processed.
  @param[in, out] KeyAttributes      Pointer to an array of EFI_KMS_KEY_ATTRIBUTE
                                     structures providing the attribute information to
                                     associate with the key.
                                     On input, the values for the fields in the structure
                                     are completely filled in.
                                     On return the KeyAttributeStatus field will reflect the
                                     result of the operation for each key attribute request.
  @param[in, out] ClientDataSize     Pointer to the size, in bytes, of an arbitrary block of 
                                     data specified by the ClientData parameter. This
                                     parameter may be NULL, in which case the ClientData
                                     parameter will be ignored and no data will be
                                     transferred to or from the KMS. If the parameter is
                                     not NULL, then ClientData must be a valid pointer.
                                     If the value pointed to is 0, no data will be transferred
                                     to the KMS, but data may be returned by the KMS.
                                     For all non-zero values *ClientData will be transferred
                                     to the KMS, which may also return data to the caller.
                                     In all cases, the value upon return to the caller will
                                     be the size of the data block returned to the caller,
                                     which will be zero if no data is returned from the KMS.
  @param[in, out] ClientData         Pointer to a pointer to an arbitrary block of data of
                                     *ClientDataSize that is to be passed directly to the
                                     KMS if it supports the use of client data. This 
                                     parameter may be NULL if and only if the 
                                     ClientDataSize parameter is also NULL. Upon return to
                                     the caller, *ClientData points to a block of data of 
                                     *ClientDataSize that was returned from the KMS. 
                                     If the returned value for *ClientDataSize is zero,
                                     then the returned value for *ClientData must be NULL
                                     and should be ignored by the caller. The KMS protocol
                                     consumer is responsible for freeing all valid buffers
                                     used for client data regardless of whether they are
                                     allocated by the caller for input to the function or by
                                     the implementation for output back to the caller.

  @retval EFI_SUCCESS                Successfully added all requested key attributes.
  @retval EFI_OUT_OF_RESOURCES       Could not allocate required resources.
  @retval EFI_TIMEOUT                Timed out waiting for device or key server. Check individual key
                                     attribute request(s) to see which ones may have been
                                     processed.
  @retval EFI_BUFFER_TOO_SMALL       If multiple keys attributes are associated with a single key
                                     identifier, and the attributes buffer does not contain
                                     enough structures (KeyAttributesCount) to contain all
                                     the data, then the available structures will be filled and
                                     KeyAttributesCount will be updated to indicate the
                                     number of key attributes which could not be processed. The
                                     status of each key attribute is also updated indicating success or
                                     failure for that attribute in case there are other errors for those
                                     attributes that could be processed.
  @retval EFI_ACCESS_DENIED          Access was denied by the device or the key server; OR a
                                     ClientId is required by the server and either none or an
                                     invalid id was provided.
  @retval EFI_DEVICE_ERROR           Device or key server error. Check individual key attribute
                                     request(s) (i.e. key attribute status for each) to see which ones
                                     may have been processed.
  @retval EFI_INVALID_PARAMETER      This is NULL, ClientId is required but it is NULL,
                                     KeyAttributesCount is NULL, or KeyAttributes
                                     is NULL, or KeyIdentifierSize is NULL, or
                                     KeyIdentifer is NULL.
  @retval EFI_NOT_FOUND              The KeyIdentifier could not be found. On return the
                                     KeyAttributesCount contains the number of attributes
                                     processed. Individual structures will reflect the status of the
                                     processing for that structure.
  @retval EFI_UNSUPPORTED            The implementation/KMS does not support this function.

**/
typedef
EFI_STATUS
(EFIAPI *EFI_KMS_ADD_KEY_ATTRIBUTES) (
  IN EFI_KMS_PROTOCOL           *This,
  IN EFI_KMS_CLIENT_INFO        *Client,
  IN UINT8                      *KeyIdentifierSize,
  IN CONST VOID                 *KeyIdentifier,
  IN OUT UINT16                 *KeyAttributesCount,
  IN OUT EFI_KMS_KEY_ATTRIBUTE  *KeyAttributes,
  IN OUT UINTN                  *ClientDataSize OPTIONAL,
  IN OUT VOID                   **ClientData OPTIONAL
  );

/**
  Delete attributes to a key specified by a key identifier.

  @param[in]      This               Pointer to the EFI_KMS_PROTOCOL instance.
  @param[in]      Client             Pointer to a valid EFI_KMS_CLIENT_INFO structure.
  @param[in]      KeyIdentifierSize  Pointer to the size in bytes of the KeyIdentifier variable.
  @param[in]      KeyIdentifier      Pointer to the key identifier associated with this key.
  @param[in, out] KeyAttributesCount Pointer to the number of EFI_KMS_KEY_ATTRIBUTE
                                     structures to associate with the Key.
                                     On input, the count value is one or more.
                                     On normal returns, this number will be updated with
                                     the number of key attributes successfully processed.
  @param[in, out] KeyAttributes      Pointer to an array of EFI_KMS_KEY_ATTRIBUTE
                                     structures providing the attribute information to
                                     associate with the key.
                                     On input, the values for the fields in the structure
                                     are completely filled in.
                                     On return the KeyAttributeStatus field will reflect the
                                     result of the operation for each key attribute request.
  @param[in, out] ClientDataSize     Pointer to the size, in bytes, of an arbitrary block of 
                                     data specified by the ClientData parameter. This
                                     parameter may be NULL, in which case the ClientData
                                     parameter will be ignored and no data will be
                                     transferred to or from the KMS. If the parameter is
                                     not NULL, then ClientData must be a valid pointer.
                                     If the value pointed to is 0, no data will be transferred
                                     to the KMS, but data may be returned by the KMS.
                                     For all non-zero values *ClientData will be transferred
                                     to the KMS, which may also return data to the caller.
                                     In all cases, the value upon return to the caller will
                                     be the size of the data block returned to the caller,
                                     which will be zero if no data is returned from the KMS.
  @param[in, out] ClientData         Pointer to a pointer to an arbitrary block of data of
                                     *ClientDataSize that is to be passed directly to the
                                     KMS if it supports the use of client data. This 
                                     parameter may be NULL if and only if the 
                                     ClientDataSize parameter is also NULL. Upon return to
                                     the caller, *ClientData points to a block of data of 
                                     *ClientDataSize that was returned from the KMS. 
                                     If the returned value for *ClientDataSize is zero,
                                     then the returned value for *ClientData must be NULL
                                     and should be ignored by the caller. The KMS protocol
                                     consumer is responsible for freeing all valid buffers
                                     used for client data regardless of whether they are
                                     allocated by the caller for input to the function or by
                                     the implementation for output back to the caller.

  @retval EFI_SUCCESS                Successfully deleted all requested key attributes.
  @retval EFI_OUT_OF_RESOURCES       Could not allocate required resources.
  @retval EFI_TIMEOUT                Timed out waiting for device or key server. Check individual key
                                     attribute request(s) to see which ones may have been
                                     processed.
  @retval EFI_ACCESS_DENIED          Access was denied by the device or the key server; OR a
                                     ClientId is required by the server and either none or an
                                     invalid id was provided.
  @retval EFI_DEVICE_ERROR           Device or key server error. Check individual key attribute
                                     request(s) (i.e. key attribute status for each) to see which ones
                                     may have been processed.
  @retval EFI_INVALID_PARAMETER      This is NULL, ClientId is required but it is NULL,
                                     KeyAttributesCount is NULL, or
                                     KeyAttributes is NULL, or KeyIdentifierSize
                                     is NULL, or KeyIdentifer is NULL.
  @retval EFI_NOT_FOUND              The KeyIdentifier could not be found or the attribute
                                     could not be found. On return the KeyAttributesCount
                                     contains the number of attributes processed. Individual
                                     structures will reflect the status of the processing for that
                                     structure.
  @retval EFI_UNSUPPORTED            The implementation/KMS does not support this function.

**/
typedef
EFI_STATUS
(EFIAPI *EFI_KMS_DELETE_KEY_ATTRIBUTES) (
  IN EFI_KMS_PROTOCOL           *This,
  IN EFI_KMS_CLIENT_INFO        *Client,
  IN UINT8                      *KeyIdentifierSize,
  IN CONST VOID                 *KeyIdentifier,
  IN OUT UINT16                 *KeyAttributesCount,
  IN OUT EFI_KMS_KEY_ATTRIBUTE  *KeyAttributes,
  IN OUT UINTN                  *ClientDataSize OPTIONAL,
  IN OUT VOID                   **ClientData OPTIONAL
  );

/**
  Retrieve one or more key that has matched all of the specified key attributes.

  @param[in]      This               Pointer to the EFI_KMS_PROTOCOL instance.
  @param[in]      Client             Pointer to a valid EFI_KMS_CLIENT_INFO structure.
  @param[in, out] KeyAttributesCount Pointer to a count of the number of key attribute structures
                                     that must be matched for each returned key descriptor.
                                     On input the count value is one or more.
                                     On normal returns, this number will be updated with
                                     the number of key attributes successfully processed.
  @param[in, out] KeyAttributes      Pointer to an array of EFI_KMS_KEY_ATTRIBUTE
                                     structure to search for.
                                     On input, the values for the fields in the structure are
                                     completely filled in.
                                     On return the KeyAttributeStatus field will reflect the
                                     result of the operation for each key attribute request.
  @param[in, out] KeyDescriptorCount Pointer to a count of the number of key descriptors matched
                                     by this operation.
                                     On entry, this number will be zero.
                                     On return, this number will be updated to the number
                                     of key descriptors successfully found.
  @param[in, out] KeyDescriptors     Pointer to an array of EFI_KMS_KEY_DESCRIPTOR
                                     structures which describe the keys from the KMS
                                     having the KeyAttribute(s) specified.
                                     On input, this pointer will be NULL.
                                     On output, the array will contain an
                                     EFI_KMS_KEY_DESCRIPTOR structure for each key
                                     meeting the search criteria. Memory for the array
                                     and all KeyValue fields will be allocated with the
                                     EfiBootServicesData type and must be freed by the
                                     caller when it is no longer needed. Also, the KeyStatus
                                     field of each descriptor will reflect the result of the
                                     request relative to that key descriptor.
  @param[in, out] ClientDataSize     Pointer to the size, in bytes, of an arbitrary block of 
                                     data specified by the ClientData parameter. This
                                     parameter may be NULL, in which case the ClientData
                                     parameter will be ignored and no data will be
                                     transferred to or from the KMS. If the parameter is
                                     not NULL, then ClientData must be a valid pointer.
                                     If the value pointed to is 0, no data will be transferred
                                     to the KMS, but data may be returned by the KMS.
                                     For all non-zero values *ClientData will be transferred
                                     to the KMS, which may also return data to the caller.
                                     In all cases, the value upon return to the caller will
                                     be the size of the data block returned to the caller,
                                     which will be zero if no data is returned from the KMS.
  @param[in, out] ClientData         Pointer to a pointer to an arbitrary block of data of
                                     *ClientDataSize that is to be passed directly to the
                                     KMS if it supports the use of client data. This 
                                     parameter may be NULL if and only if the 
                                     ClientDataSize parameter is also NULL. Upon return to
                                     the caller, *ClientData points to a block of data of 
                                     *ClientDataSize that was returned from the KMS. 
                                     If the returned value for *ClientDataSize is zero,
                                     then the returned value for *ClientData must be NULL
                                     and should be ignored by the caller. The KMS protocol
                                     consumer is responsible for freeing all valid buffers
                                     used for client data regardless of whether they are
                                     allocated by the caller for input to the function or by
                                     the implementation for output back to the caller.

  @retval EFI_SUCCESS                Successfully retrieved all requested keys.
  @retval EFI_OUT_OF_RESOURCES       Could not allocate required resources.
  @retval EFI_TIMEOUT                Timed out waiting for device or key server. Check individual key
                                     attribute request(s) to see which ones may have been
                                     processed.
  @retval EFI_BUFFER_TOO_SMALL       If multiple keys are associated with the attribute(s), and the
                                     KeyValue buffer does not contain enough structures
                                     (KeyDescriptorCount) to contain all the key data, then
                                     the available structures will be filled and
                                     KeyDescriptorCount will be updated to indicate the
                                     number of keys which could not be processed.
  @retval EFI_ACCESS_DENIED          Access was denied by the device or the key server; OR a
                                     ClientId is required by the server and either none or an
                                     invalid id was provided.
  @retval EFI_DEVICE_ERROR           Device or key server error. Check individual key attribute
                                     request(s) (i.e. key attribute status for each) to see which ones
                                     may have been processed.
  @retval EFI_INVALID_PARAMETER      This is NULL, ClientId is required but it is NULL,
                                     KeyDescriptorCount is NULL, or
                                     KeyDescriptors is NULL or KeyAttributes is
                                     NULL, or KeyAttributesCount is NULL.
  @retval EFI_NOT_FOUND              One or more EFI_KMS_KEY_ATTRIBUTE structures could
                                     not be processed properly. KeyAttributeCount contains
                                     the number of structures which were successfully processed.
                                     Individual structures will reflect the status of the processing for
                                     that structure.
  @retval EFI_UNSUPPORTED            The implementation/KMS does not support this function.

**/
typedef
EFI_STATUS
(EFIAPI *EFI_KMS_GET_KEY_BY_ATTRIBUTES) (
  IN EFI_KMS_PROTOCOL           *This,
  IN EFI_KMS_CLIENT_INFO        *Client,
  IN OUT UINTN                  *KeyAttributeCount,
  IN OUT EFI_KMS_KEY_ATTRIBUTE  *KeyAttributes,
  IN OUT UINTN                  *KeyDescriptorCount,
  IN OUT EFI_KMS_KEY_DESCRIPTOR *KeyDescriptors,
  IN OUT UINTN                  *ClientDataSize OPTIONAL,
  IN OUT VOID                   **ClientData OPTIONAL
  );

///
/// The Key Management Service (KMS) protocol provides services to generate, store, retrieve,
/// and manage cryptographic keys.
///
struct _EFI_KMS_PROTOCOL {
  ///
  /// Get the current status of the key management service. If the implementation has not yet
  /// connected to the KMS, then a call to this function will initiate a connection. This is the
  /// only function that is valid for use prior to the service being marked available.
  ///
  EFI_KMS_GET_SERVICE_STATUS        GetServiceStatus;
  ///
  /// Register a specific client with the KMS.
  ///
  EFI_KMS_REGISTER_CLIENT           RegisterClient;
  ///
  /// Request the generation of a new key and retrieve it.
  ///
  EFI_KMS_CREATE_KEY                CreateKey;
  ///
  /// Retrieve an existing key.
  ///
  EFI_KMS_GET_KEY                   GetKey;
  ///
  /// Add a local key to KMS database. If there is an existing key with this key identifier in the
  /// KMS database, it will be replaced with the new key.
  ///
  EFI_KMS_ADD_KEY                   AddKey;
  ///
  /// Delete an existing key from the KMS database.
  ///
  EFI_KMS_DELETE_KEY                DeleteKey;
  ///
  /// Get attributes for an existing key in the KMS database.
  ///
  EFI_KMS_GET_KEY_ATTRIBUTES        GetKeyAttributes;
  ///
  /// Add attributes to an existing key in the KMS database.
  ///
  EFI_KMS_ADD_KEY_ATTRIBUTES        AddKeyAttributes;
  ///
  /// Delete attributes for an existing key in the KMS database.
  ///
  EFI_KMS_DELETE_KEY_ATTRIBUTES     DeleteKeyAttributes;
  ///
  /// Get existing key(s) with the specified attributes.
  ///
  EFI_KMS_GET_KEY_BY_ATTRIBUTES     GetKeyByAttributes;
  ///
  /// The version of this EFI_KMS_PROTOCOL structure. This must be set to 0x00020040 for
  /// the initial version of this protocol.
  ///
  UINT32                            ProtocolVersion;
  ///
  /// Optional GUID used to identify a specific KMS. This GUID may be supplied by the provider,
  /// by the implementation, or may be null. If is null, then the ServiceName must not be null.
  ///
  EFI_GUID                          ServiceId;
  ///
  /// Optional pointer to a unicode string which may be used to identify the KMS or provide
  /// other information about the supplier.
  ///
  CHAR16                            *ServiceName;
  ///
  /// Optional 32-bit value which may be used to indicate the version of the KMS provided by
  /// the supplier.
  ///
  UINT32                            ServiceVersion;
  ///
  /// TRUE if and only if the service is active and available for use. To avoid unnecessary
  /// delays in POST, this protocol may be installed without connecting to the service. In this
  /// case, the first call to the GetServiceStatus () function will cause the implementation to
  /// connect to the supported service and mark it as available. The capabilities of this service
  /// as defined in the reminder of this protocol are not guaranteed to be valid until the service
  /// has been marked available.
  ///
  BOOLEAN                           ServiceAvailable;
  ///
  /// TRUE if and only if the service supports client identifiers. Client identifiers may be used
  /// for auditing, access control or any other purpose specific to the implementation.
  ///
  BOOLEAN                           ClientIdSupported;
  ///
  /// TRUE if and only if the service requires a client identifier in order to process key requests.
  /// FALSE otherwise.
  ///
  BOOLEAN                           ClientIdRequired;
  ///
  /// The maximum size in bytes for the client identifier.
  ///
  UINT16                            ClientIdMaxSize;
  ///
  /// The client name string type(s) supported by the KMS service. If client names are not
  /// supported, this field will be set the EFI_KMS_DATA_TYPE_NONE. Otherwise, it will be set
  /// to the inclusive 'OR' of all client name formats supported. Client names may be used for
  /// auditing, access control or any other purpose specific to the implementation.
  ///
  UINT8                             ClientNameStringTypes;
  ///
  /// TRUE if only if the KMS requires a client name to be supplied to the service.
  /// FALSE otherwise.
  ///
  BOOLEAN                           ClientNameRequired;
  ///
  /// The maximum number of characters allowed for the client name.
  ///
  UINT16                            ClientNameMaxCount;
  ///
  /// TRUE if and only if the service supports arbitrary client data requests. The use of client
  /// data requires the caller to have specific knowledge of the individual KMS service and
  /// should be used only if absolutely necessary.
  /// FALSE otherwise.
  ///
  BOOLEAN                           ClientDataSupported;
  ///
  /// The maximum size in bytes for the client data. If the maximum data size is not specified
  /// by the KMS or it is not known, then this field must be filled with all ones.
  ///
  UINTN                             ClientDataMaxSize;
  ///
  /// TRUE if variable length key identifiers are supported.
  /// FALSE if a fixed length key identifier is supported.
  ///
  BOOLEAN                           KeyIdVariableLenSupported;
  ///
  /// If KeyIdVariableLenSupported is TRUE, this is the maximum supported key identifier length
  /// in bytes. Otherwise this is the fixed length of key identifier supported. Key ids shorter
  /// than the fixed length will be padded on the right with blanks.
  ///
  UINTN                             KeyIdMaxSize;
  ///
  /// The number of key format/size GUIDs returned in the KeyFormats field.
  ///
  UINTN                             KeyFormatsCount;
  ///
  /// A pointer to an array of EFI_GUID values which specify key formats/sizes supported by
  /// this KMS. Each format/size pair will be specified by a separate EFI_GUID. At least one
  /// key format/size must be supported. All formats/sizes with the same hashing algorithm
  /// must be contiguous in the array, and for each hashing algorithm, the key sizes must be in
  /// ascending order. See "Related Definitions" for GUIDs which identify supported key formats/sizes.
  /// This list of GUIDs supported by the KMS is not required to be exhaustive, and the KMS
  /// may provide support for additional key formats/sizes. Users may request key information
  /// using an arbitrary GUID, but any GUID not recognized by the implementation or not
  /// supported by the KMS will return an error code of EFI_UNSUPPORTED
  ///
  EFI_GUID                          *KeyFormats;
  ///
  /// TRUE if key attributes are supported.
  /// FALSE if key attributes are not supported.
  ///
  BOOLEAN                           KeyAttributesSupported;
  ///
  /// The key attribute identifier string type(s) supported by the KMS service. If key attributes
  /// are not supported, this field will be set to EFI_KMS_DATA_TYPE_NONE. Otherwise, it will
  /// be set to the inclusive 'OR' of all key attribute identifier string types supported.
  /// EFI_KMS_DATA_TYPE_BINARY is not valid for this field.
  ///
  UINT8                             KeyAttributeIdStringTypes;
  UINT16                            KeyAttributeIdMaxCount;
  ///
  /// The number of predefined KeyAttributes structures returned in the KeyAttributes
  /// parameter. If the KMS does not support predefined key attributes, or if it does not
  /// provide a method to obtain predefined key attributes data, then this field must be zero.
  ///
  UINTN                             KeyAttributesCount;
  ///
  /// A pointer to an array of KeyAttributes structures which contains the predefined
  /// attributes supported by this KMS. Each structure must contain a valid key attribute
  /// identifier and should provide any other information as appropriate for the attribute,
  /// including a default value if one exists. This variable must be set to NULL if the
  /// KeyAttributesCount variable is zero. It must point to a valid buffer if the
  /// KeyAttributesCount variable is non-zero.
  /// This list of predefined attributes is not required to be exhaustive, and the KMS may
  /// provide additional predefined attributes not enumerated in this list. The implementation
  /// does not distinguish between predefined and used defined attributes, and therefore,
  /// predefined attributes not enumerated will still be processed to the KMS.
  ///
  EFI_KMS_KEY_ATTRIBUTE             *KeyAttributes;
};

extern EFI_GUID gEfiKmsFormatGeneric128Guid;
extern EFI_GUID gEfiKmsFormatGeneric160Guid;
extern EFI_GUID gEfiKmsFormatGeneric256Guid;
extern EFI_GUID gEfiKmsFormatGeneric512Guid;
extern EFI_GUID gEfiKmsFormatGeneric1024Guid;
extern EFI_GUID gEfiKmsFormatGeneric2048Guid;
extern EFI_GUID gEfiKmsFormatGeneric3072Guid;
extern EFI_GUID gEfiKmsFormatMd2128Guid;
extern EFI_GUID gEfiKmsFormatMdc2128Guid;
extern EFI_GUID gEfiKmsFormatMd4128Guid;
extern EFI_GUID gEfiKmsFormatMdc4128Guid;
extern EFI_GUID gEfiKmsFormatMd5128Guid;
extern EFI_GUID gEfiKmsFormatMd5sha128Guid;
extern EFI_GUID gEfiKmsFormatSha1160Guid;
extern EFI_GUID gEfiKmsFormatSha256256Guid;
extern EFI_GUID gEfiKmsFormatSha512512Guid;
extern EFI_GUID gEfiKmsFormatAesxts128Guid;
extern EFI_GUID gEfiKmsFormatAesxts256Guid;
extern EFI_GUID gEfiKmsFormatAescbc128Guid;
extern EFI_GUID gEfiKmsFormatAescbc256Guid;
extern EFI_GUID gEfiKmsFormatRsasha11024Guid;
extern EFI_GUID gEfiKmsFormatRsasha12048Guid;
extern EFI_GUID gEfiKmsFormatRsasha2562048Guid;
extern EFI_GUID gEfiKmsFormatRsasha2563072Guid;
extern EFI_GUID gEfiKmsProtocolGuid;

#endif
