/*
 * Copyright 2021-2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.schema;

import java.util.Objects;

import static javax.xml.XMLConstants.W3C_XML_SCHEMA_INSTANCE_NS_URI;

/**
 * An XML namespace identified by a URI.
 */
public class Namespace {

	/**
	 * W3C XML Schema Instance namespace ({@value javax.xml.XMLConstants#W3C_XML_SCHEMA_INSTANCE_NS_URI}).
	 */
	public static final Namespace XML_SCHEMA_INSTANCE = Namespace.of(W3C_XML_SCHEMA_INSTANCE_NS_URI);

	/**
	 * Open Test Reporting core namespace
	 */
	public static final Namespace REPORTING_CORE = Namespace.of("https://schemas.opentest4j.org/reporting/core/0.1.0");

	/**
	 * Open Test Reporting events namespace
	 */
	public static final Namespace REPORTING_EVENTS = Namespace.of(
		"https://schemas.opentest4j.org/reporting/events/0.1.0");

	/**
	 * Open Test Reporting Java namespace
	 */
	public static final Namespace REPORTING_JAVA = Namespace.of("https://schemas.opentest4j.org/reporting/java/0.1.0");

	/**
	 * Open Test Reporting hierarchical namespace
	 */
	public static final Namespace REPORTING_HIERARCHY = Namespace.of(
		"https://schemas.opentest4j.org/reporting/hierarchy/0.1.0");

	/**
	 * Create the namespace with the supplied URI.
	 *
	 * @param uri the namespace URI
	 * @return namespace with the supplied URI
	 */
	public static Namespace of(String uri) {
		return new Namespace(uri);
	}

	private final String uri;

	private Namespace(String uri) {
		this.uri = uri;
	}

	/**
	 * @return the URI that identifies this namespace
	 */
	public String getUri() {
		return uri;
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}
		if (o == null || getClass() != o.getClass()) {
			return false;
		}
		Namespace namespace = (Namespace) o;
		return uri.equals(namespace.uri);
	}

	@Override
	public int hashCode() {
		return Objects.hash(uri);
	}

	@Override
	public String toString() {
		return String.format("Namespace{uri='%s'}", uri);
	}
}
