/***************************************************************************
 *   Copyright (C) 2005-2013 by Georg Hennig                               *
 *   Email: georg.hennig@web.de                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <QDateTime>

#include <KGlobal>
#include <KLocale>

#include "kmdcodec.h"
#include "kfileitemext.h"

#include "klistviewitemdups.h"

#include <version.h>

KListViewItemDups::KListViewItemDups( QTreeWidget *parent, KFileItemExt *_item )
	: QTreeWidgetItem( (QTreeWidget*)parent ), item( _item ), m_filter( NONE )
{
	m_color = Qt::black;
	m_my_blue = QColor( 15, 25, 200 );
	m_use_color = true;

	QString date = item->getKFileItemExtModificationTimeString();

	setTextAlignment( 2, Qt::AlignRight );

	// Fill the item with data
	KUrl tmp_url = item->url(); // This is necessary to display a pretty url on non-local file systems.
	tmp_url.setPath( "" );
	setText( 0, tmp_url.pathOrUrl() + item->url().upUrl().path( KUrl::RemoveTrailingSlash ) );
	setText( 1, KIO::decodeFileName( item->url().fileName() ) );
	setText( 2, /*KGlobal::locale()->formatNumber( item_tmp.size(), 0 )*/KIO::convertSize( item->size() ) );
	setText( 3, date );
	setText( 4, item->permissionsString() );
	setText( 5, item->user() );
	setText( 6, item->group() );

	// put the icon into the leftmost column
	item->determineMimeType();
	setIcon( 0, item->pixmap( 16 ) );

	updateColor();
}

KListViewItemDups::~KListViewItemDups()
{
}

bool KListViewItemDups::operator< ( const QTreeWidgetItem &other ) const
{
	int column = treeWidget()->sortColumn();

	switch ( column )
	{
		case 2:  // Returns size in bytes. Used for sorting
			return (uint)item->size() < (uint)((KListViewItemDups*)&other)->item->size();
		case 3:  // Returns time in secs from 1/1/1970. Used for sorting
			return item->getKFileItemExtModificationTimeT() < ((KListViewItemDups*)&other)->item->getKFileItemExtModificationTimeT();
	}
	return text( column ) < other.text( column );

}

void KListViewItemDups::setUseColor( bool use_color )
{
	m_use_color = use_color;

	if ( m_color != Qt::black && m_use_color )
	{
		QBrush textbrush( m_color );
		QBrush backgroundbrush( QColor( 255-m_color.red(), 255-m_color.green(), 255-m_color.blue() ) );

		int i;
		for ( i=0; i<columnCount(); i++ )
		{
			setForeground( i, textbrush );
		}
	}
	else
	{
		int i;
		for ( i=0; i<columnCount(); i++ )
		{
			setForeground( i, QBrush() );
		}
	}
}

bool KListViewItemDups::useColor()
{
	return m_use_color;
}

bool KListViewItemDups::isDuplicatesPair()
{
	KFileItemExt *tmp_item = item->duplicates_size;
	int count = 1;
	bool hasother = false;
	while ( tmp_item ) // check whether this duplicate list contains more than two files or duplicates within one directory.
	{
		if ( count > 2 ) break; // If there are more than two files, we surely have a duplicate that is no "brother" file.

		if ( KUrl::relativePath( item->virtual_parent_path.path( KUrl::RemoveTrailingSlash ), item->url().upUrl().path( KUrl::RemoveTrailingSlash ) ) ==
			KUrl::relativePath( tmp_item->virtual_parent_path.path( KUrl::RemoveTrailingSlash ), tmp_item->url().upUrl().path( KUrl::RemoveTrailingSlash ) ) &&
			item->url().fileName() == tmp_item->url().fileName() )
		{
			hasother = true;
		}

		count++;
		tmp_item = tmp_item->duplicates_size;
	}

	return ( count > 2 || count < 2 || !hasother ) ? false : true;
}

void KListViewItemDups::updateColor()
{
	m_color = isDuplicatesPair() ? Qt::black : m_my_blue;
}

void KListViewItemDups::addFilter( FILTER filter )
{
	m_filter |= filter;
}

void KListViewItemDups::removeFilter( FILTER filter )
{
	if ( m_filter & filter ) m_filter ^= filter;
}

void KListViewItemDups::setVisibility()
{
	setHidden( ( m_filter == NONE ) ? false : true );
}
