/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   _lcmaps_x509_utils.h
    \brief  This file contains utility functions needed for x509 credential
            handling (openssl)
    \author Martijn Steenbakkers for EGEE
    This file contains the headers for utility functions to retrieve
    info from pem-encoded, x509 credentials/contexts
    Contains code contributed by Andrew McNab.
*/

#ifndef _LCMAPS_X509_UTILS_H
#define _LCMAPS_X509_UTILS_H

/******************************************************************************
                             Include header files
******************************************************************************/
#include <openssl/x509.h>

/******************************************************************************
 *                 Module definition
 *****************************************************************************/

#ifndef LCMAPS_X509_UTILS_C
#   define LCMAPS_X509_UTILS_STORAGE_CLASS extern
#else
#   define LCMAPS_X509_UTILS_STORAGE_CLASS
#endif

/******************************************************************************
Function:       lcmaps_x509_to_dn()
Description:    Get the name of the X509 cert
Parameters:
                px509: x5009 certificate
Returns:        dn (which should be freed)
******************************************************************************/
LCMAPS_X509_UTILS_STORAGE_CLASS char * lcmaps_x509_to_dn(
        X509 *  px509
);

/******************************************************************************
Function:       lcmaps_x509_chain_to_dn()
Description:    Get the base DN of the X509 proxy cert
Parameters:
                px509: x509 certificate
                certstack: certificate chain
Returns:        dn (which should be freed)
******************************************************************************/
LCMAPS_X509_UTILS_STORAGE_CLASS char * lcmaps_x509_chain_to_dn(
        X509 *  px509, STACK_OF(X509) *certstack
);

/******************************************************************************
Function:       lcmaps_pem_string_to_x509
Description:    Reads a X509 certificate from a PEM-encoded string
Parameters:
                px: pointer to a X509 certificate
                cert_string: PEM-encoded string 
Returns:        0 on success, non-zero otherwise
******************************************************************************/
LCMAPS_X509_UTILS_STORAGE_CLASS int lcmaps_pem_string_to_x509(
    X509 **px,
    char *certstring
);

/******************************************************************************
Function:       lcmaps_pem_string_to_x509_chain
Description:    Creates a dynamically allocated stack of X509 certificate objects 
                by walking through the PEM-encoded X509 certificates. 
                Copied from gridsite (Andrew McNab), original name:
                    GRSTx509StringToChain()
Parameters:
                certstack: pointer to a stack of X509 certificates
                cert_string: PEM-encoded string 
Returns:        0 on success, non-zero otherwise
******************************************************************************/
LCMAPS_X509_UTILS_STORAGE_CLASS int lcmaps_pem_string_to_x509_chain(
    STACK_OF(X509) **certstack,
    char *certstring
);


int lcmaps_x509IsCA (X509 *cert);
void lcmaps_x509_free_chain(STACK_OF(X509) **);


/******************************************************************************
Function:   cgul_x509IsCA
Description:
    Tests if the X509 * cert is a CA certificate or not
Parameters:
    A X509 pointer
Returns:
    0      : Not a CA cert
    1      : This is a CA cert
******************************************************************************/
LCMAPS_X509_UTILS_STORAGE_CLASS int cgul_x509IsCA (X509 *cert);

/******************************************************************************
Function:       cgul_x509_select_eec_from_chain()
Description:    Get the End-Entity Certificate from a certificate chain
Parameters:
                certstack: certificate chain
Returns:        pointer to X509 EEC
******************************************************************************/
LCMAPS_X509_UTILS_STORAGE_CLASS X509 * cgul_x509_select_eec_from_chain (STACK_OF(X509) * chain);

/******************************************************************************
Function:       cgul_x509_select_final_cert_from_chain()
Description:    Get the Final Delegation from a certificate chain
Parameters:
                certstack: certificate chain
Returns:        pointer to X509 final delegation
******************************************************************************/
LCMAPS_X509_UTILS_STORAGE_CLASS X509 * cgul_x509_select_final_cert_from_chain (STACK_OF(X509) * chain);

/******************************************************************************
Function:       cgul_x509_chain_to_subject_dn()
Description:    Get the Subject DN of the End-Entity Certificate of the X509 cert
Parameters:
                certstack: certificate chain
Returns:        Subject DN string (which should be freed)
******************************************************************************/
LCMAPS_X509_UTILS_STORAGE_CLASS char * cgul_x509_chain_to_subject_dn(STACK_OF(X509) * chain);

/******************************************************************************
Function:       cgul_x509_chain_to_issuer_dn()
Description:    Get the Issuer DN of the End-Entity Certificate of the X509 cert
Parameters:
                certstack: certificate chain
Returns:        Issuer DN string (which should be freed)
******************************************************************************/
LCMAPS_X509_UTILS_STORAGE_CLASS char * cgul_x509_chain_to_issuer_dn(STACK_OF(X509) * chain);


#endif /* _LCMAPS_X509_UTILS_H */

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps/src/grid_credential_handling/x509_handling/_lcmaps_x509_utils.h,v $
    $Date: 2010-12-16 16:29:48 +0100 (Thu, 16 Dec 2010) $
    $Revision: 11755 $
    $Author: dennisvd $
******************************************************************************/
