package tests::ImportJobTest;

use strict;

use base qw/ Lire::Test::TestCase /;

use Lire::ImportJob;
use POSIX qw/strftime/;
use Lire::I18N qw/set_fh_encoding/;
use Lire::Utils qw/tempfile/;
use Lire::PluginManager;
use Lire::DlfStore;
use Lire::DlfConverter;
use Lire::Test::Mock;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->{'spec'} = $self->lire_default_config_spec();
    $self->{'www_plugin'} = new_proxy Lire::Test::Mock( 'Lire::DlfConverter' );
    $self->{'www_plugin'}->set_result( 'name' => 'www' );

    return $self;
}

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->{'cfg'}{'_lr_config_spec'} = $self->{'spec'};
    $self->set_up_plugin_mgr();
    Lire::PluginManager->register_plugin( $self->{'www_plugin'} );

    return;
}

sub tear_down {
    my $self = $_[0];

    Lire::Test::Mock->reset_factories();

    unlink $self->{'tmpfile'}
      if $self->{'tmpfile'};

    $self->SUPER::tear_down();

    return;
}

sub test_new {
    my $self = $_[0];

    my %args = ( 'period' => "unique",
                 'pattern' => "/var/log/messages",
                 'converter' => "www",
                 'encoding' => 'ISO-8859-1',
                 'filter' => 'cat',
                 'name' => "test",
               );
    my $job = new Lire::ImportJob( "test", %args,
                                   'converter_config' => {} );
    $self->assert_isa( 'Lire::ImportJob', $job );

    foreach my $a ( keys %args ) {
        no strict 'refs';
        $self->assert_str_equals( $args{$a}, $job->$a() );
    }
    $self->assert_deep_equals( {}, $job->converter_config() );
}

sub test_file {
    my $self = $_[0];

    my $job = new Lire::ImportJob( "test" );
    $self->assert( UNIVERSAL::isa( $job, "Lire::ImportJob"),
                   "constructor didn't returned an instance of Lire::ImportJob"
                 );

    $job->pattern( '/var/log/messages%Y%m' );
    my $file = $job->file;
    $self->assert_not_null( $file, "file() returned undef" );
    $self->assert_equals( strftime( '/var/log/messages%Y%m', localtime ),
                          $file );
}

sub test_new_from_config {
    my $self = $_[0];

    my $job = $self->{'spec'}->get( 'import_jobs' )->get( 'import_job' )->instance();
    $job->get( 'name' )->set( 'aJob' );
    $job->get( 'filter' )->set( '' );
    $job->get( 'log_encoding' )->set( 'UTF-8' );
    $job->get( 'log_file' )->set( 'a file' );
    $job->get( 'period' )->set( 'hourly' );
    $job->get( 'service' )->set_plugin( 'www' );

    my $jobObject = $job->as_value();
    $self->assert_isa( 'Lire::ImportJob', $jobObject );
    $self->assert_str_equals( 'aJob', $jobObject->{'_name'} );
    $self->assert_str_equals( 'UTF-8', $jobObject->{'_encoding'} );
    $self->assert_str_equals( 'www', $jobObject->{'_converter'} );
    $self->assert_str_equals( 'a file', $jobObject->{'_pattern'} );
    $self->assert_str_equals( 'hourly', $jobObject->{'_period'} );
    $self->assert_deep_equals( {}, $jobObject->{'_converter_config'} );
    $self->assert_null( $jobObject->{'_filter'} );
}

sub test_log_fh_filter {
    my $self = $_[0];

    my ( $fh, $name ) = tempfile( "log-utf8XXXXXX" );
    $self->{'tmpfile'} = $name;
    print $fh "Line 1\nLine 2\n";
    close $fh;

    my $job = new Lire::ImportJob( 'test', 'converter' => 'line',
                                   'pattern' => $name,
                                   'filter' => 'grep "Line 1"' );
    my $job_fh = $job->log_fh();
    $self->assert_deep_equals( [ "Line 1\n" ], [ <$job_fh> ] );

    $job->filter( 'cat | grep "Line"' );
    $job_fh = $job->log_fh();
    $self->assert_deep_equals( [ "Line 1\n", "Line 2\n" ], [ <$job_fh> ] );
}

sub test_log_fh_encoding {
    my $self = $_[0];

    my $unistr = "This an A with a bar: \x{0100}\n";
    my ( $fh, $name ) = tempfile( "log-utf8XXXXXX" );
    $self->{'tmpfile'} = $name;
    set_fh_encoding( $fh, 'utf8' );
    print $fh $unistr;
    close $fh;

    my $job = new Lire::ImportJob( "test", converter => 'line',
                                   'pattern' => $name,
                                   'encoding' => 'utf8' );

    my $job_fh = $job->log_fh();
    my $string = <$job_fh>;
    $self->assert_str_equals( $unistr, $string );
}

sub test_run {
    my $self = $_[0];

    $self->set_up_plugin_mgr();
    my $line = new_proxy Lire::Test::Mock( 'Lire::DlfConverter' );
    $line->set_result('name' => 'line',
                      'schemas' => sub { ( 'test', 'test-extended' ) } );
    Lire::PluginManager->register_plugin( $line );

    my $job = new Lire::ImportJob( "test", converter => 'line',
                                   'pattern' => "test.log" );
    my $store = new Lire::Test::Mock( 'Lire::DlfStore',
                                      'has_dlf_stream' => sub { $_[1] eq 'test' } );
    Lire::Test::Mock->set_mock_factory( 'Lire::DlfConverterProcess',
                                        'run_import_job' => '',
                                        'job_id' => 'my_id',
                                      );
    $job->run( $store );

    my $process = Lire::Test::Mock->mock_instances( 'Lire::DlfConverterProcess' );
    $self->assert_num_equals( 1, scalar @$process );
    $self->assert_str_equals( $store, $process->[0]->dlf_store() );
    $self->assert_num_equals( 1, $process->[0]->invocation_count( 'run_import_job' ) );

    $self->assert_num_equals( 1, $store->invocation_count( 'run_analysers' ) );
    $self->assert_deep_equals( [ $store, 'test', 'my_id' ],
                               $store->get_invocation( 'run_analysers', 0 ) );
}

1;
