/*=============================================================================
** Lynkeos
** $Id: corelation.c,v 1.6 2005/01/27 23:13:27 j-etienne Exp $
**-----------------------------------------------------------------------------
**
**  Created by Jean-Etienne LAMIAUD on Apr 30, 1998
**  Copyright (c) 1998,2003-2005. Jean-Etienne LAMIAUD
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
**
**-----------------------------------------------------------------------------
*/
#include <stdlib.h>
#include <assert.h>

#include "corelation.h"

/*=============================================================================
** correlate_spectrums
**-----------------------------------------------------------------------------
**
** Purpose     : correlate on precomputed spectrums
**
**-----------------------------------------------------------------------------
**
** Entry       : 
**               
**
** Output      : 
**
**=============================================================================
*/
void correlate_spectrums( FFT_DATA s1, FFT_DATA s2, FFT_DATA r )
{
   u_short x, y, c;
   u_long i;

   /* Fourier transform of correlation is the product of s1 by s2 conjugate */
   for( i = 0; i < r.nplanes*r.h*(r.w/2+1); i++ )
      r.spectrum[i] = s1.spectrum[i] * ~s2.spectrum[i];

   fourier_inverse( r, NULL, NULL );

   /* Transpose the quadrants, to bring origin at the center */
   for( y = 0; y < r.h/2; y++ )
   {
      for( x = 0; x < r.w/2; x++ )
      {
         for( c = 0; c < r.nplanes; c++ )
         {
            REAL r1 = *colorValue(r,x,y,c),
                 r2 = *colorValue(r,x,y+r.h/2,c);

            /* Exchange NW & SE quadrant */
            *colorValue(r,x,y,c) = *colorValue(r,x+r.w/2,y+r.h/2,c);
            *colorValue(r,x+r.w/2,y+r.h/2,c) = r1;

            /* Exchange NE et SW quadrant */
            *colorValue(r,x,y+r.h/2,c) = *colorValue(r,x+r.w/2,y,c);
            *colorValue(r,x+r.w/2,y,c) = r2;
         }
      }
   }
}

/*=============================================================================
** correlate
**-----------------------------------------------------------------------------
**
** Purpose     : Correlation of two images
**
**-----------------------------------------------------------------------------
**
** Entry       : Two images in an array of COMPLEX
**               
**
** Output      : Correlation in an array of COMPLEX (although the i part is null)
**
**=============================================================================
*/
void correlate( FFT_DATA s1, FFT_DATA s2, FFT_DATA r )
{
   /* Transform both images */
   fourier( s1 );
   fourier( s2 );

   /* Perform the correlation */
   correlate_spectrums( s1, s2, r );
}

/*=============================================================================
** corelation_peak
**-----------------------------------------------------------------------------
**
** Purpose     : Search for corelation peak in the corelation data
**
**-----------------------------------------------------------------------------
**
** Entry       : Corelation result
**
** Output      : 
**
**=============================================================================
*/
void corelation_peak( FFT_DATA result, CORRELATION_PEAK *peak )
{
   u_short x, y, c; 
   double sum, module_max, module_min;
   double xp, yp, s_x2, s_y2;
   u_long nb_pixel;
   REAL r;

   assert( peak != NULL );

   for( c = 0; c < result.nplanes; c++ )
   {
      /* Search for min and max */
      module_max = 0.0;
      module_min = HUGE;
      
      for( y = 0; y < result.h; y++ )
      {
         for( x = 0; x < result.w; x++ )
         {
            r = *colorValue(result,x,y,c);

            if ( r > module_max )
               module_max = r;
            if ( r < module_min )
               module_min = r;
         }
      }

      /* Locate the peak as the barycenter of pixels above (max-min)/sqrt(2) */
      xp = 0.0;
      yp = 0.0;
      s_x2 = 0.0;
      s_y2 = 0.0;
      sum = 0.0;
      nb_pixel = 0;

      for( y = 0; y < result.h; y++ )
      {
         for( x = 0; x < result.w; x++ )
         {
            double module;
            r = *colorValue(result,x,y,c);
            module = r - module_min;
            
            if ( module > (module_max-module_min)*0.707 )
            {
               xp += (x-result.w/2)*module;
               yp += (y-result.h/2)*module;
               s_x2 += (x-result.w/2)*(x-result.w/2)*module;
               s_y2 += (y-result.h/2)*(y-result.h/2)*module;
               sum += module;
               nb_pixel++;
            }
         }
      }

      /* Present the results */
      xp /= sum;
      yp /= sum;
      peak[c].val = module_max - module_min;
      peak[c].x = xp;
      peak[c].y = yp;
      peak[c].sigma_x = sqrt(s_x2/sum - xp*xp);
      peak[c].sigma_y = sqrt(s_y2/sum - yp*yp);
   }
}
