/*
 *  Copyright (c) 2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "TimeMachine.h"

#include <algorithm>

#include "Debug.h"
#include "Value.h"
#include "String.h"
#include "Utils_p.h"

#include "TimeController.h"
#include "TimeValue_p.h"
#include "Type.h"
#include "Metadata/ParameterEntry.h"

using namespace GTLCore;

struct TimeMachine::Private {
  std::map< GTLCore::String, TimeValue* > timeValues;
  std::list<const TimeController*> controllers;
  void registerController(const TimeController* controler);
  static TimeValue* minMaxToNumericalTimeValue(const GTLCore::Type* _type, const TimeController* controler, const GTLCore::Value& min, const GTLCore::Value& max);
};

void TimeMachine::Private::registerController(const TimeController* controler)
{
  if(std::find(controllers.begin(), controllers.end(), controler) == controllers.end())
  {
    controllers.push_back(controler);
  }
}

TimeValue* TimeMachine::Private::minMaxToNumericalTimeValue(const GTLCore::Type* _type, const TimeController* controler, const GTLCore::Value& min, const GTLCore::Value& max)
{
  GTL_ASSERT(_type->isNumber());
  GTL_ASSERT(min.type()->isNumber());
  GTL_ASSERT(max.type()->isNumber());
  switch(_type->dataType())
  {
    case Type::INTEGER32:
      return new TimeNumericValue<gtl_int32>(min.asInt32(), max.asInt32(), controler);
      break;
    case Type::FLOAT32:
      return new TimeNumericValue<float>(min.asFloat32(), max.asFloat32(), controler);
      break;
    default:
      GTL_ABORT("Unimplemented");
  }
}

TimeMachine::TimeMachine() : d(new Private)
{
}

TimeMachine::~TimeMachine()
{
  deleteAll(d->controllers);
  for(std::map<GTLCore::String, TimeValue*>::iterator it = d->timeValues.begin();
      it != d->timeValues.end(); ++it)
  {
    delete it->second;
  }
  delete d;
}

void TimeMachine::startControlling(const Metadata::ParameterEntry* _entry, const TimeController* _timeController)
{
  stopControlling(_entry->name());
  d->timeValues[_entry->name()] = Private::minMaxToNumericalTimeValue(_entry->type(), _timeController, _entry->minimumValue(), _entry->maximumValue());
}

void TimeMachine::startControlling(const Metadata::ParameterEntry* _entry, const std::vector<const TimeController*>& _timeControllers)
{
  stopControlling(_entry->name());
  if(_timeControllers.size() == 1)
  {
    startControlling(_entry, _timeControllers.front());
  } else {
    if(_entry->type() == Type::Color)
    {
      GTL_ASSERT(_timeControllers.size() == 4);
      d->timeValues[_entry->name()] = new TimeColorValue(_timeControllers[0], _timeControllers[1], _timeControllers[2], _timeControllers[3] );
    } else {
      GTL_ASSERT(_entry->type()->dataType() == Type::VECTOR);
      GTL_ASSERT(_entry->type()->vectorSize() == _timeControllers.size());
      const GTLCore::Type* embeddedType = _entry->type()->embeddedType();
      std::vector< TimeValue* > values;
      for (std::size_t i = 0; i < _timeControllers.size(); ++i)
      {
        values.push_back(Private::minMaxToNumericalTimeValue(embeddedType, _timeControllers[i], _entry->minimumValue().asArray()[i], _entry->maximumValue().asArray()[i] ) );
      }
      d->timeValues[_entry->name()] = new TimeVectorArrayValue(_entry->type(), values);
    }
  }
}

void TimeMachine::stopControlling(const String&  _name)
{
  delete d->timeValues[_name];
  d->timeValues.erase(_name);
}

std::map< GTLCore::String, GTLCore::Value> TimeMachine::parametersAt(double t) const
{
  std::map< GTLCore::String, GTLCore::Value> m;
  for(std::map<GTLCore::String, TimeValue*>::iterator it = d->timeValues.begin();
      it != d->timeValues.end(); ++it)
  {
    m[it->first] = it->second->value(t);
  }
  return m;
}
