/*
 *  Copyright (c) 2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Transform.h"

#include <cmath>

#include "Debug.h"
#include "Region.h"

using namespace GTLCore;

struct Transform::Private {
  enum Type {
      None      = 0x00,
      Translate = 0x01,
      Scale     = 0x02,
      Rotate    = 0x04,
      Shear     = 0x08
  };
  Type type;
  double m11, m12, m13, m21, m22, m23, m31, m32;
  template<typename _T_>
  inline void map(_T_ x, _T_ y, _T_& x_o, _T_& y_o);
};

template<typename _T_>
void Transform::Private::map(_T_ x, _T_ y, _T_& x_o, _T_& y_o)
{
  switch(type) {
    case Private::None:
      x_o = _T_(x);
      y_o = _T_(y);
      return;
    case Private::Translate:
      x_o = _T_(x + m13);
      y_o = _T_(y + m23);
      return;
    case Private::Scale:
      x_o = _T_(m11 * x + m13);
      y_o = _T_(m22 * y + m23);
      return;
    case Private::Rotate:
    case Private::Shear:
      x_o = _T_(m11 * x + m12 * y + m13);
      y_o = _T_(m21 * x + m22 * y + m23);
      return;
  }
  GTL_ABORT("Unsupported");
}

Transform::Transform() : d(new Private)
{
  d->type = Private::None;
  d->m11 = 1.0;
  d->m12 = 0.0;
  d->m13 = 0.0;
  d->m21 = 0.0;
  d->m22 = 1.0;
  d->m23 = 0.0;
  d->m31 = 0.0;
  d->m32 = 0.0;
}

Transform::Transform(const Transform& _transform) : d(new Private(*_transform.d))
{
}

Transform& Transform::operator=(const Transform& _transform)
{
  *d = *_transform.d;
  return *this;
}

Transform::~Transform()
{
  delete d;
}

void Transform::rotate(double _angle)
{
  if( _angle == 0)
    return;

  double sina = sin(_angle);
  double cosa = cos(_angle);

  switch(d->type) {
    case Private::None:
    case Private::Translate:
      d->m11 = cosa;
      d->m12 = -sina;
      d->m21 = sina;
      d->m22 = cosa;
      break;
    case Private::Scale:
    {
      double tm11 = cosa * d->m11;
      double tm12 = -sina * d->m22;
      double tm21 = sina * d->m11;
      double tm22 = cosa * d->m22;
      d->m11 = tm11; d->m12 = tm12;
      d->m21 = tm21; d->m22 = tm22;
      break;
    }
    case Private::Rotate:
    case Private::Shear: {
      double tm11 = cosa * d->m11 - sina * d->m21;
      double tm12 = cosa * d->m12 - sina * d->m22;
      double tm21 = sina * d->m11 + cosa * d->m21;
      double tm22 = sina * d->m12 + cosa * d->m22;
      d->m11 = tm11; d->m12 = tm12;
      d->m21 = tm21; d->m22 = tm22;
      break;
    }
  }
  if ( d->type < Private::Rotate)
      d->type = Private::Rotate;
}

void Transform::translate(double _dx, double _dy)
{
  d->m13 += _dx;
  d->m23 += _dy;
  if ( d->type < Private::Translate)
      d->type = Private::Translate;
}

void Transform::scale(double sx, double sy)
{
  switch(d->type)
  {
    case Private::None:
    case Private::Translate:
      d->m11 = sx;
      d->m22 = sy;
      break;
    case Private::Rotate:
    case Private::Shear:
      d->m21 *= sx;
      d->m12 *= sy;
    case Private::Scale:
      d->m11 *= sx;
      d->m22 *= sy;
      break;
  }
  if ( d->type < Private::Scale)
      d->type = Private::Scale;
}

void Transform::shear(double sh, double sv)
{
  switch(d->type)
  {
    case Private::None:
    case Private::Translate:
      d->m12 = sh;
      d->m21 = sv;
      break;
    case Private::Scale:
      d->m12 = sh * d->m11;
      d->m21 = sv * d->m22;
      break;
    case Private::Rotate:
    case Private::Shear:
      double tm11 = sv * d->m21;
      double tm22 = sh * d->m12;
      double tm12 = sh * d->m11;
      double tm21 = sv * d->m22;
      d->m11 += tm11; d->m12 += tm12;
      d->m21 += tm21; d->m22 += tm22;
      break;
  }
  if ( d->type < Private::Shear)
      d->type = Private::Shear;
}

Transform Transform::invert() const
{
  Transform inv;
  switch(d->type) {
    case Private::None:
      break;
    case Private::Translate:
      inv.d->m13 = -d->m13;
      inv.d->m23 = -d->m23;
      break;
    case Private::Scale:
      inv.d->m11 = 1.0 / d->m11;
      inv.d->m22 = 1.0 / d->m22;
      inv.d->m13 = -d->m13 * inv.d->m11;
      inv.d->m23 = -d->m23 * inv.d->m22;
      break;
    case Private::Rotate:
    case Private::Shear:
    {
      double det = 1.0 / (d->m11* d->m22 - d->m12*d->m21);
      inv.d->m11 =   d->m22 * det;
      inv.d->m12 = - d->m12 * det;
      inv.d->m21 = - d->m21 * det;
      inv.d->m22 =   d->m11 * det;
      inv.d->m13 = (d->m12 * d->m23 - d->m22 * d->m13)*det;
      inv.d->m23 = (d->m21 * d->m13 - d->m11 * d->m23)*det;
    }
  }
  inv.d->type = d->type;
  return inv;
}

void Transform::map(double x, double y, double& x_o, double& y_o) const
{
  d->map<double>(x, y, x_o, y_o);
}

void Transform::map(float x, float y, float& x_o, float& y_o) const
{
  d->map<float>(x, y, x_o, y_o);
}

bool minMax1(float a, float b, float c, float d, float& min, float& max)
{
  if( a >= b and a >= c and a >= d)
  {
    max = a;
    if( b <= c and b <= d)
    {
      min = b;
    } else if (c <= d) {
      min = c;
    } else {
      min = d;
    }
    return true;
  }
  return false;
}

inline void minMax(float a, float b, float c, float d, float& min, float& max)
{
  if(minMax1(a,b,c,d,min,max)) return;
  if(minMax1(b,a,c,d,min,max)) return;
  if(minMax1(c,a,b,c,min,max)) return;
  if(minMax1(d,a,b,c,min,max)) return;
  GTL_ABORT("Impossible");
}

RegionF Transform::map(const RegionF& region) const
{
  float x1, x2, x3, x4;
  float y1, y2, y3, y4;
  
  // Compute point mapping for corners
  d->map<float>(region.left(), region.top(), x1, y1);
  d->map<float>(region.right(), region.top(), x2, y2);
  d->map<float>(region.left(), region.bottom(), x3, y3);
  d->map<float>(region.right(), region.bottom(), x4, y4);
  
  // Compute maximum
  float xMin, xMax, yMin, yMax;
  minMax(x1,x2,x3,x4,xMin,xMax);
  minMax(y1,y2,y3,y4,yMin,yMax);
  
  return RegionF(xMin, yMin, xMax - xMin, yMax - yMin);
}

const Transform& Transform::operator*=(const Transform &m) 
{
  *this = *this * m;
  return *this;
}

Transform Transform::operator*(const Transform &m) const
{
  const Private::Type otherType = m.d->type;
  if (otherType == Private::None)
    return *this;

  const Private::Type thisType = d->type;

  if (thisType == Private::None)
    return m;

  Transform t;

  Private::Type type = thisType > otherType ? thisType : otherType;

  switch(type) {
    case Private::None:
      break;
    case Private::Translate:
      t.d->m13 = d->m13 + m.d->m13;
      t.d->m23 = d->m23 + m.d->m23;
      break;
    case Private::Scale:
    {
      double m11 = d->m11*m.d->m11;
      double m22 = d->m22*m.d->m22;
      double m31 = m.d->m13*d->m11 + d->m13;
      double m32 = m.d->m23*d->m22 + d->m23;
      t.d->m11 = m11;
      t.d->m22 = m22;
      t.d->m13 = m31; t.d->m23 = m32;
      break;
    }
    case Private::Rotate:
    case Private::Shear:
    {
      double m11 = m.d->m11*d->m11 + m.d->m21*d->m12;
      double m12 = m.d->m11*d->m21 + m.d->m21*d->m22;

      double m21 = m.d->m12*d->m11 + m.d->m22*d->m12;
      double m22 = m.d->m12*d->m21 + m.d->m22*d->m22;

      double m31 = m.d->m13*d->m11 + m.d->m23*d->m12 + d->m13;
      double m32 = m.d->m13*d->m21 + m.d->m23*d->m22 + d->m23;

      t.d->m11 = m11; t.d->m21 = m12;
      t.d->m12 = m21; t.d->m22 = m22;
      t.d->m13 = m31; t.d->m23 = m32;
      break;
    }
  }
  t.d->type = type;
  return t;
}
