///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/animation/AnimManager.h>
#include <core/utilities/ProgressIndicator.h>

#include "LAMMPSBinaryDumpParser.h"
#include <atomviz/atoms/AtomsObject.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(LAMMPSBinaryDumpParser, LAMMPSDumpParser)

/// This method looks for a special argument on the command line
/// that enables the use of an extended binary LAMMPS dump format
/// that can handle non-orthogonal simulation boxes.
static bool isModifiedBinaryDumpFormatEnabled()
{
	return true;
	Q_FOREACH(QString s, QCoreApplication::arguments()) {
		if(s == "--extended-dump-format")
			return true;
	}
	return false;
}

/******************************************************************************
* Default constructor.
******************************************************************************/
LAMMPSBinaryDumpParser::LAMMPSBinaryDumpParser(bool isLoading) : LAMMPSDumpParser(isLoading)
{
}

/******************************************************************************
* Parses the header of the given file and returns the number of data columns contained in the file.
******************************************************************************/
bool LAMMPSBinaryDumpParser::inspectFileHeader(const QString& filepath, int& numberOfColumns, QStringList& columnNames)
{
	// Open the input file for reading.
	QFile stream(filepath);
	if(!stream.open(QIODevice::ReadOnly))
		throw Exception(tr("Failed to open binary dump file %1: %2").arg(filepath, stream.errorString()));

	int ntimestep, natoms, nColumns, nchunk;
	double xlo, xhi, ylo, yhi, zlo, zhi, xy, xz, yz;

	VerboseLogger() << "Parsing binary dump file header" << endl;
	stream.read((char*)&ntimestep,sizeof(int));
	stream.read((char*)&natoms,sizeof(int));
	stream.read((char*)&xlo,sizeof(double));
	stream.read((char*)&xhi,sizeof(double));
	stream.read((char*)&ylo,sizeof(double));
	stream.read((char*)&yhi,sizeof(double));
	stream.read((char*)&zlo,sizeof(double));
	stream.read((char*)&zhi,sizeof(double));
	if(isModifiedBinaryDumpFormatEnabled()) {
		stream.read((char*)&xy,sizeof(double));
		stream.read((char*)&xz,sizeof(double));
		stream.read((char*)&yz,sizeof(double));
	}
	stream.read((char*)&nColumns,sizeof(int));
	stream.read((char*)&nchunk,sizeof(int));

	VerboseLogger() << "Timestep:" << ntimestep << endl;
	VerboseLogger() << "Number of atoms:" << natoms << endl;
	VerboseLogger() << "Number of columns:" << nColumns << endl;
	VerboseLogger() << "Number of chunks:" << nchunk << endl;

	if(natoms <= 0 || natoms > 100000000)
		throw Exception(tr("Invalid number of atoms in binary dump file: %1").arg(natoms));

	numberOfColumns = nColumns;

	return true;
}

/******************************************************************************
* Scans an atoms file for the time step frames contained therein.
******************************************************************************/
bool LAMMPSBinaryDumpParser::scanFileForTimeSteps(const QString& filepath, bool suppressDialogs)
{
	// Open the input file for reading.
	QFile stream(filepath);
	if(!stream.open(QIODevice::ReadOnly))
		throw Exception(tr("Failed to open binary dump file %1: %2").arg(filepath, stream.errorString()));

	int ntimestep, natoms, nColumns, nchunk;
	double xlo, xhi, ylo, yhi, zlo, zhi, xy, xz, yz;

	ProgressIndicator progress(QString(), stream.size() / (qint64)1000, suppressDialogs);

	int timeStepNumber = 0;
	while(!stream.atEnd()) {
		natoms = 0;
		qint64 byteOffset = stream.pos();
		stream.read((char*)&ntimestep,sizeof(int));
		stream.read((char*)&natoms,sizeof(int));
		stream.read((char*)&xlo,sizeof(double));
		stream.read((char*)&xhi,sizeof(double));
		stream.read((char*)&ylo,sizeof(double));
		stream.read((char*)&yhi,sizeof(double));
		stream.read((char*)&zlo,sizeof(double));
		stream.read((char*)&zhi,sizeof(double));
		if(isModifiedBinaryDumpFormatEnabled()) {
			stream.read((char*)&xy,sizeof(double));
			stream.read((char*)&xz,sizeof(double));
			stream.read((char*)&yz,sizeof(double));
		}
		stream.read((char*)&nColumns,sizeof(int));
		stream.read((char*)&nchunk,sizeof(int));

		if(natoms <= 0 || natoms > 100000000) throw Exception(tr("Invalid number of atoms in binary dump file."));

		// Create a new record for the time step.
		addTimeStep(filepath, byteOffset);
		timeStepNumber++;
		progress.setLabelText(tr("Scanning LAMMPS dump file (Frame %1)...").arg(timeStepNumber));

		// Loop over processor chunks in file
		QVector<double> dataValues;
		for(int chunki = 0; chunki < nchunk; chunki++) {
			int n;
			stream.read((char*)&n,sizeof(int));
			OVITO_ASSERT(n % nColumns == 0);

			// Update progress indicator.
			progress.setValue(stream.pos() / (qint64)1000);
			if(progress.isCanceled()) return false;

			stream.seek(stream.pos() + n * sizeof(double));
		}
	}
	return true;
}



/******************************************************************************
* Parses the atomic data of a single time step.
******************************************************************************/
EvaluationStatus LAMMPSBinaryDumpParser::loadTimeStep(AtomsObject* destination, int movieFrame, const QString& filename, streampos byteOffset, int lineNumber, bool suppressDialogs)
{
	CHECK_OBJECT_POINTER(destination);

	// Open the input file for reading.
	QFile stream(filename);
	if(!stream.open(QIODevice::ReadOnly))
		throw Exception(tr("Failed to open binary dump file %1: %2").arg(filename, stream.errorString()));

	// Seek to the byte offset where the requested movie frame is stored.
	if(byteOffset != streampos(0)) {
		if(!stream.seek(byteOffset))
			throw Exception(tr("Failed to seek to frame %1 in binary dump file %2").arg(movieFrame).arg(filename));
	}

	scoped_ptr<ProgressIndicator> progress;

	int ntimestep, natoms, nColumns, nchunk;
	double xlo, xhi, ylo, yhi, zlo, zhi, xy = 0, xz = 0, yz = 0;

	stream.read((char*)&ntimestep,sizeof(int));
	stream.read((char*)&natoms,sizeof(int));
	stream.read((char*)&xlo,sizeof(double));
	stream.read((char*)&xhi,sizeof(double));
	stream.read((char*)&ylo,sizeof(double));
	stream.read((char*)&yhi,sizeof(double));
	stream.read((char*)&zlo,sizeof(double));
	stream.read((char*)&zhi,sizeof(double));
	if(isModifiedBinaryDumpFormatEnabled()) {
		stream.read((char*)&xy,sizeof(double));
		stream.read((char*)&xz,sizeof(double));
		stream.read((char*)&yz,sizeof(double));
	}
	stream.read((char*)&nColumns,sizeof(int));
	stream.read((char*)&nchunk,sizeof(int));

	VerboseLogger() << "Reading binary LAMMPS Dump file:" << endl;
	VerboseLogger() << "timestep:" << ntimestep << endl;
	VerboseLogger() << "natomzhis:" << natoms << endl;
	VerboseLogger() << "xlo:" << xlo << endl;
	VerboseLogger() << "ylo:" << ylo << endl;
	VerboseLogger() << "zlo:" << zlo << endl;
	VerboseLogger() << "xhi:" << xhi << endl;
	VerboseLogger() << "yhi:" << yhi << endl;
	VerboseLogger() << "zhi:" << zhi << endl;
	VerboseLogger() << "xy:" << xy << endl;
	VerboseLogger() << "xz:" << xz << endl;
	VerboseLogger() << "yz:" << yz << endl;
	VerboseLogger() << "ncolumns:" << nColumns << endl;
	VerboseLogger() << "nchunks:" << nchunk << endl;

	if(natoms <= 0 || natoms > 100000000) throw Exception(tr("Invalid number of atoms in binary dump file."));

	progress.reset(new ProgressIndicator(tr("Loading LAMMPS dump file..."), natoms, suppressDialogs));

	if(xy == 0 && xz == 0 && yz == 0) {
		Box3 simBox(Point3(xlo, ylo, zlo), Point3(xhi, yhi, zhi));
		destination->simulationCell()->setBoxShape(simBox);
	}
	else {
		xlo -= min(min(min(xy, xz), xy+xz), 0.0);
		xhi -= max(max(max(xy, xz), xy+xz), 0.0);
		ylo -= min(yz, 0.0);
		yhi -= max(yz, 0.0);
		destination->simulationCell()->setCellShape(Point3(xlo, ylo, zlo), Vector3(xhi - xlo, 0, 0), Vector3(xy, yhi - ylo, 0), Vector3(xz, yz, zhi - zlo));
	}

	// Parse atom coordinates.
	destination->setAtomsCount(natoms);

	// Prepare the mapping between input file columns and data channels.
	DataRecordParserHelper recordParser(&columnMapping(), destination);

	int i = 0;	// The current atom

	// Loop over processor chunks in file
	QVector<double> dataValues;
	for(int chunki = 0; chunki < nchunk; chunki++) {

		int n;
		stream.read((char*)&n,sizeof(int));
		OVITO_ASSERT(n % nColumns == 0);

		VerboseLogger() << "Size of chunk " << chunki << ":" << n << endl;

		dataValues.resize(n);
		if(stream.read((char*)dataValues.data(), n * sizeof(double)) != n * sizeof(double))
			throw Exception(tr("Could not read enough bytes from the input file."));

		const double* iter = dataValues.constData();
		int nChunkAtoms = n / nColumns;
		for(; nChunkAtoms--; ++i, iter += nColumns) {

			// Update progress indicator.
			if((i % 1000) == 0 && progress) {
				progress->setValue(i);
				if(progress->isCanceled()) return EvaluationStatus(EvaluationStatus::EVALUATION_ERROR);
			}

			try {
				recordParser.storeAtom(i, iter, nColumns);
			}
			catch(Exception& ex) {
				throw ex.prependGeneralMessage(tr("Parsing error in LAMMPS binary dump file."));
			}

		}
	}

	DataChannel* posChannel = destination->getStandardDataChannel(DataChannel::PositionChannel);
	if(posChannel && posChannel->size() > 1) {
		// Rescale atoms if they are given in reduced coordinates.
		const Box3& boundingBox = recordParser.boundingBox();
		if(Box3(Point3(-0.05f), Point3(1.05f)).containsBox(boundingBox)) {
			VerboseLogger() << "Rescaling reduced coordinates in interval [0,1] to absolute box size." << endl;
			AffineTransformation simCell = destination->simulationCell()->cellMatrix();
			Point3* p = posChannel->dataPoint3();
			for(size_t i = posChannel->size(); i != 0; --i, ++p)
				*p = simCell * (*p);
		}
	}

	destination->invalidate();

	QString statusMessage = tr("%1 atoms at timestep %2").arg(natoms).arg(ntimestep);
	return EvaluationStatus(EvaluationStatus::EVALUATION_SUCCESS, statusMessage);
}

};	// End of namespace AtomViz
