/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL ES-CM 1.1 plugin
 *
 * Copyright © 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "pgmglesdrawable.h"

GST_DEBUG_CATEGORY_STATIC (pgm_gles_drawable_debug);
#define GST_CAT_DEFAULT pgm_gles_drawable_debug

static GstObjectClass *parent_class = NULL;

/* Private methods */

/* Update the position of the duplicated GL drawable. Since these values are
   accessed quite often, storing it in the GL drawable avoids putting locks
   on the generic drawable. */
static void
update_position (PgmGlesDrawable *glesdrawable)
{
  PgmDrawable *drawable = glesdrawable->drawable;

  GST_OBJECT_LOCK (drawable);
  glesdrawable->x = drawable->x;
  glesdrawable->y = drawable->y;
  glesdrawable->z = drawable->z;
  GST_OBJECT_UNLOCK (drawable);
}

/* Update the size of the duplicated GL drawable. Since these values are
   accessed quite often, storing it in the GL drawable avoids putting locks
   on the generic drawable. */
static void
update_size (PgmGlesDrawable *glesdrawable)
{
  PgmDrawable *drawable = glesdrawable->drawable;

  GST_OBJECT_LOCK (drawable);
  glesdrawable->width = drawable->width;
  glesdrawable->height = drawable->height;
  GST_OBJECT_UNLOCK (drawable);
}

/* GObject stuff */

PGM_DEFINE_DYNAMIC_TYPE_EXTENDED (PgmGlesDrawable, pgm_gles_drawable,
                                  GST_TYPE_OBJECT, G_TYPE_FLAG_ABSTRACT, {})

void
pgm_gles_drawable_register (GTypeModule *module)
{
  pgm_gles_drawable_register_type (module);
}


static void
pgm_gles_drawable_dispose (GObject *object)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (object);

  glesdrawable->drawable = NULL;
  glesdrawable->glesviewport = NULL;

  if (glesdrawable->transformation_matrix)
    pgm_mat4x4_free (glesdrawable->transformation_matrix);

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_gles_drawable_class_init (PgmGlesDrawableClass *klass)
{
  GObjectClass *gobject_class;

  GST_DEBUG_CATEGORY_INIT (pgm_gles_drawable_debug, "pgm_gles_drawable", 0,
                           "OpenGL ES plugin: PgmGlesDrawable");

  parent_class = g_type_class_peek_parent (klass);

  gobject_class = G_OBJECT_CLASS (klass);

  /* GObject virtual table */
  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_gles_drawable_dispose);
}

static void
pgm_gles_drawable_class_finalize (PgmGlesDrawableClass *klass)
{
  return;
}

static void
pgm_gles_drawable_init (PgmGlesDrawable *glesdrawable)
{
  /* glesdrawable->drawable = NULL; */
  /* glesdrawable->glesviewport = NULL; */
  /* glesdrawable->transformation_matrix = NULL; */
}

/* Public methods */

void
pgm_gles_drawable_sync (PgmGlesDrawable *glesdrawable)
{
  PgmGlesDrawableClass *klass;

  g_return_if_fail (PGM_IS_GLES_DRAWABLE (glesdrawable));

  update_size (glesdrawable);
  pgm_gles_drawable_set_position (glesdrawable);
  pgm_gles_drawable_set_bg_color (glesdrawable);
  pgm_gles_drawable_set_transformation_matrix (glesdrawable);

  klass = PGM_GLES_DRAWABLE_GET_CLASS (glesdrawable);

  if (klass->sync)
    klass->sync (glesdrawable);
}

void
pgm_gles_drawable_draw (PgmGlesDrawable *glesdrawable)
{
  static PgmGlesUshort indices[] = { 0, 1, 2, 2, 3, 0 };
  PgmGlesDrawableClass *klass;
  PgmDrawable *drawable;
  PgmGlesContextProcAddress *gles;
  gboolean visible;

  g_return_if_fail (PGM_IS_GLES_DRAWABLE (glesdrawable));
  g_return_if_fail (PGM_IS_DRAWABLE (glesdrawable->drawable));

  drawable = glesdrawable->drawable;

  /* Don't draw if not visible, or if the opacity is 0 */
  GST_OBJECT_LOCK (drawable);
  visible = (GST_OBJECT_FLAG_IS_SET (drawable, PGM_DRAWABLE_FLAG_VISIBLE));
  if (visible)
    visible = drawable->opacity;
  GST_OBJECT_UNLOCK (drawable);
  if (!visible)
    return;

  gles = glesdrawable->glesviewport->context->gles;

  /* Push and set the modelview matrix for the drawable if needed */
  if (!(glesdrawable->flags & PGM_GLES_DRAWABLE_IDENTITY_MATRIX))
    {
      gles->push_matrix ();
      gles->load_matrix_f (glesdrawable->transformation_matrix->m);
    }

  /* Batch background array if it's visible */
  if (glesdrawable->bg_color[3] != 0.0f)
    {
      gles->bind_texture (PGM_GLES_TEXTURE_2D, 0);
      gles->enable_client_state (PGM_GLES_VERTEX_ARRAY);
      gles->vertex_pointer (3, PGM_GLES_FLOAT, 0, glesdrawable->bg_vertex);
      gles->enable_client_state (PGM_GLES_COLOR_ARRAY);
      gles->color_pointer (4, PGM_GLES_FLOAT, 0, glesdrawable->bg_color);
      gles->draw_elements (PGM_GLES_TRIANGLES, 6,
                           PGM_GLES_UNSIGNED_SHORT, indices);
      gles->disable_client_state (PGM_GLES_VERTEX_ARRAY);
      gles->disable_client_state (PGM_GLES_COLOR_ARRAY);
    }

  klass = PGM_GLES_DRAWABLE_GET_CLASS (glesdrawable);

  if (klass && klass->draw)
    klass->draw (glesdrawable);

  /* Pop the modelview matrix if needed */
  if (!(glesdrawable->flags & PGM_GLES_DRAWABLE_IDENTITY_MATRIX))
    gles->pop_matrix ();
}

void
pgm_gles_drawable_regenerate (PgmGlesDrawable *glesdrawable)
{
  PgmGlesDrawableClass *klass;

  g_return_if_fail (PGM_IS_GLES_DRAWABLE (glesdrawable));

  klass = PGM_GLES_DRAWABLE_GET_CLASS (glesdrawable);

  if (klass->regenerate)
    klass->regenerate (glesdrawable);
}

void
pgm_gles_drawable_update_projection (PgmGlesDrawable *glesdrawable)
{
  PgmGlesDrawableClass *klass;

  g_return_if_fail (PGM_IS_GLES_DRAWABLE (glesdrawable));

  pgm_gles_drawable_set_position (glesdrawable);
  pgm_gles_drawable_set_size (glesdrawable);

  klass = PGM_GLES_DRAWABLE_GET_CLASS (glesdrawable);

  if (klass->update_projection)
    klass->update_projection (glesdrawable);
}

void
pgm_gles_drawable_set_visibility (PgmGlesDrawable *glesdrawable)
{
  PgmGlesDrawableClass *klass;

  g_return_if_fail (PGM_IS_GLES_DRAWABLE (glesdrawable));

  klass = PGM_GLES_DRAWABLE_GET_CLASS (glesdrawable);

  if (klass->set_visibility)
    klass->set_visibility (glesdrawable);
}

void
pgm_gles_drawable_set_size (PgmGlesDrawable *glesdrawable)
{
  PgmGlesDrawableClass *klass;

  g_return_if_fail (PGM_IS_GLES_DRAWABLE (glesdrawable));

  update_size (glesdrawable);

  glesdrawable->bg_vertex[3] = glesdrawable->bg_vertex[0] + glesdrawable->width;
  glesdrawable->bg_vertex[6] = glesdrawable->bg_vertex[9] + glesdrawable->width;
  glesdrawable->bg_vertex[10] = glesdrawable->bg_vertex[1] + glesdrawable->height;
  glesdrawable->bg_vertex[7] = glesdrawable->bg_vertex[4] + glesdrawable->height;

  klass = PGM_GLES_DRAWABLE_GET_CLASS (glesdrawable);

  if (klass->set_size)
    klass->set_size (glesdrawable);
}

void
pgm_gles_drawable_set_position (PgmGlesDrawable *glesdrawable)
{
  PgmGlesDrawableClass *klass;

  g_return_if_fail (PGM_IS_GLES_DRAWABLE (glesdrawable));

  update_position (glesdrawable);

  glesdrawable->bg_vertex[0] = glesdrawable->x;
  glesdrawable->bg_vertex[1] = glesdrawable->y;
  glesdrawable->bg_vertex[2] = glesdrawable->z;
  glesdrawable->bg_vertex[3] = glesdrawable->x + glesdrawable->width;
  glesdrawable->bg_vertex[4] = glesdrawable->y;
  glesdrawable->bg_vertex[5] = glesdrawable->z;
  glesdrawable->bg_vertex[6] = glesdrawable->x + glesdrawable->width;
  glesdrawable->bg_vertex[7] = glesdrawable->y + glesdrawable->height;
  glesdrawable->bg_vertex[8] = glesdrawable->z;
  glesdrawable->bg_vertex[9] = glesdrawable->x;
  glesdrawable->bg_vertex[10] = glesdrawable->y + glesdrawable->height;
  glesdrawable->bg_vertex[11] = glesdrawable->z;

  klass = PGM_GLES_DRAWABLE_GET_CLASS (glesdrawable);

  if (klass->set_position)
    klass->set_position (glesdrawable);
}

void
pgm_gles_drawable_set_transformation_matrix (PgmGlesDrawable *glesdrawable)
{
  PgmDrawable *drawable = glesdrawable->drawable;
  PgmGlesDrawableClass *klass;
  PgmMat4x4 matrix;

  g_return_if_fail (PGM_IS_GLES_DRAWABLE (glesdrawable));

  klass = PGM_GLES_DRAWABLE_GET_CLASS (glesdrawable);

  /* Store the matrix */
  GST_OBJECT_LOCK (drawable);
  pgm_mat4x4_set_from_mat4x4 (&matrix, drawable->transformation_matrix);
  GST_OBJECT_UNLOCK (drawable);
  glesdrawable->transformation_matrix = pgm_mat4x4_transpose (&matrix);

  /* Check identity so that we can avoid pushing the matrix at rendering */
  if (G_UNLIKELY (pgm_mat4x4_is_identity (glesdrawable->transformation_matrix)))
    glesdrawable->flags |= PGM_GLES_DRAWABLE_IDENTITY_MATRIX;
  else
    glesdrawable->flags &= ~PGM_GLES_DRAWABLE_IDENTITY_MATRIX;

  if (klass->set_transformation_matrix)
    klass->set_transformation_matrix (glesdrawable);
}

void
pgm_gles_drawable_set_fg_color (PgmGlesDrawable *glesdrawable)
{
  PgmGlesDrawableClass *klass;

  g_return_if_fail (PGM_IS_GLES_DRAWABLE (glesdrawable));

  klass = PGM_GLES_DRAWABLE_GET_CLASS (glesdrawable);

  if (klass->set_fg_color)
    klass->set_fg_color (glesdrawable);

}

void
pgm_gles_drawable_set_bg_color (PgmGlesDrawable *glesdrawable)
{
  PgmGlesFloat *bg_color = glesdrawable->bg_color;
  PgmGlesDrawableClass *klass;
  PgmDrawable *drawable;
  PgmGlesFloat color[4];

  g_return_if_fail (PGM_IS_GLES_DRAWABLE (glesdrawable));

  drawable = glesdrawable->drawable;

  color[0] = drawable->bg_r * INV_255;
  color[1] = drawable->bg_g * INV_255;
  color[2] = drawable->bg_b * INV_255;
  color[3] = drawable->bg_a * drawable->opacity * INV_255;

  GST_OBJECT_LOCK (drawable);
  bg_color[0] = color[0];
  bg_color[1] = color[1];
  bg_color[2] = color[2];
  bg_color[3] = color[3];
  bg_color[4] = color[0];
  bg_color[5] = color[1];
  bg_color[6] = color[2];
  bg_color[7] = color[3];
  bg_color[8] = color[0];
  bg_color[9] = color[1];
  bg_color[10] = color[2];
  bg_color[11] = color[3];
  bg_color[12] = color[0];
  bg_color[13] = color[1];
  bg_color[14] = color[2];
  bg_color[15] = color[3];
  GST_OBJECT_UNLOCK (drawable);

  klass = PGM_GLES_DRAWABLE_GET_CLASS (glesdrawable);

  if (klass->set_bg_color)
    klass->set_bg_color (glesdrawable);
}

void
pgm_gles_drawable_set_opacity (PgmGlesDrawable *glesdrawable)
{
  PgmGlesFloat *bg_color = glesdrawable->bg_color;
  PgmGlesDrawableClass *klass;
  PgmDrawable *drawable;
  PgmGlesFloat alpha;

  g_return_if_fail (PGM_IS_GLES_DRAWABLE (glesdrawable));

  klass = PGM_GLES_DRAWABLE_GET_CLASS (glesdrawable);

  drawable = glesdrawable->drawable;

  alpha = drawable->bg_a * drawable->opacity * SQR_INV_255;

  GST_OBJECT_LOCK (drawable);
  bg_color[3] = alpha;
  bg_color[7] = alpha;
  bg_color[11] = alpha;
  bg_color[15] = alpha;
  GST_OBJECT_UNLOCK (drawable);

  if (klass->set_opacity)
    klass->set_opacity (glesdrawable);
}
