% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bSpline.R
\name{bSpline}
\alias{bSpline}
\alias{ibs}
\alias{dbs}
\alias{bsp}
\title{B-Spline Basis for Polynomial Splines}
\usage{
bSpline(
  x,
  df = NULL,
  knots = NULL,
  degree = 3L,
  intercept = FALSE,
  Boundary.knots = NULL,
  periodic = FALSE,
  derivs = 0L,
  integral = FALSE,
  warn.outside = getOption("splines2.warn.outside", TRUE),
  ...
)

ibs(
  x,
  df = NULL,
  knots = NULL,
  degree = 3,
  intercept = FALSE,
  Boundary.knots = NULL,
  ...
)

dbs(
  x,
  derivs = 1L,
  df = NULL,
  knots = NULL,
  degree = 3,
  intercept = FALSE,
  Boundary.knots = NULL,
  ...
)

bsp(
  x,
  df = NULL,
  knots = NULL,
  degree = 3L,
  intercept = FALSE,
  Boundary.knots = NULL,
  periodic = FALSE,
  derivs = 0L,
  integral = FALSE,
  warn.outside = getOption("splines2.warn.outside", TRUE),
  ...
)
}
\arguments{
\item{x}{The predictor variable.  Missing values are allowed and will be
returned as they are.}

\item{df}{Degree of freedom that equals to the column number of the returned
matrix.  One can specify \code{df} rather than \code{knots}, then the
function chooses \code{df - degree - as.integer(intercept)} internal
knots at suitable quantiles of \code{x} ignoring missing values and
those \code{x} outside of the boundary.  For periodic splines, \code{df
- as.integer(intercept)} internal knots will be chosen at suitable
quantiles of \code{x} relative to the beginning of the cyclic intervals
they belong to (see Examples) and the number of internal knots must be
greater or equal to the specified \code{degree - 1}.  If internal knots
are specified via \code{knots}, the specified \code{df} will be ignored.}

\item{knots}{The internal breakpoints that define the splines.  The default
is \code{NULL}, which results in a basis for ordinary polynomial
regression.  Typical values are the mean or median for one knot,
quantiles for more knots.  For periodic splines, the number of knots
must be greater or equal to the specified \code{degree - 1}.
Duplicated internal knots are not allowed.}

\item{degree}{A nonnegative integer specifying the degree of the piecewise
polynomial. The default value is \code{3} for cubic splines. Zero degree
is allowed for piecewise constant basis functions.}

\item{intercept}{If \code{TRUE}, the complete basis matrix will be returned.
Otherwise, the first basis will be excluded from the output.}

\item{Boundary.knots}{Boundary points at which to anchor the splines.  By
default, they are the range of \code{x} excluding \code{NA}.  If both
\code{knots} and \code{Boundary.knots} are supplied, the basis
parameters do not depend on \code{x}. Data can extend beyond
\code{Boundary.knots}.  For periodic splines, the specified bounary
knots define the cyclic interval.}

\item{periodic}{A logical value.  If \code{TRUE}, the periodic splines will
be returned.  The default value is \code{FALSE}.}

\item{derivs}{A nonnegative integer specifying the order of derivatives of
splines basis function.  The default value is \code{0}.}

\item{integral}{A logical value.  If \code{TRUE}, the corresponding
integrals of spline basis functions will be returned.  The default value
is \code{FALSE}. For periodic splines, the integral of each basis is
integrated from the left boundary knot.}

\item{warn.outside}{A logical value indicating if a warning should be thrown
out when any \code{x} is outside the boundary.  This option can also be
set through \code{options("splines2.warn.outside")} after the package is
loaded.}

\item{...}{Optional arguments that are not used.}
}
\value{
A numeric matrix of \code{length(x)} rows and \code{df} columns if
    \code{df} is specified.  If \code{knots} are specified instead, the
    output matrix will consist of \code{length(knots) + degree +
    as.integer(intercept)} columns if \code{periodic = FALSE}, or
    \code{length(knots) + as.integer(intercept)} columns if \code{periodic =
    TRUE}.  Attributes that correspond to the arguments specified are
    returned for usage of other functions in this package.
}
\description{
Generates the spline basis matrix for B-splines representing the family of
piecewise polynomials with the specified interior knots, degree, and
boundary knots, evaluated at the values of \code{x}.
}
\details{
This function extends the \code{bs()} function in the \code{splines} package
for B-spline basis functions by allowing piecewise constant (left-closed and
right-open except on the right boundary) spline basis of degree zero.  In
addition, the function provides derivatives or integrals of the B-spline
basis functions when one specifies the arguments \code{derivs} or
\code{integral} appropriately.  The function constructs periodic B-splines
when \code{periodic} is \code{TRUE}.  All the implementations are based on
the closed-form recursion formula following De Boor (1978) and Wang and Yan
(2021).

The functions \code{ibs()} and \code{dbs()} are provided for convenience.
The former provides the integrals of B-splines and is equivalent to
\code{bSpline()} with \code{integral = TRUE}.  The latter produces the
derivatives of given order of B-splines and is equivalent to
\code{bSpline()} with default \code{derivs = 1}.  The function \code{bsp()}
is an alias of to encourage the use in a model formula.
}
\examples{
library(splines2)

set.seed(1)
x <- runif(100)
knots <- c(0.3, 0.5, 0.6) # internal knots

## cubic B-splines
bsMat <- bSpline(x, knots = knots, degree = 3, intercept = TRUE)
ibsMat <- update(bsMat, integral = TRUE) # the integrals
d1Mat <- deriv(bsMat)                    # the 1st derivaitves
d2Mat <- deriv(bsMat, 2)                 # the 2nd derivaitves

op <- par(mfrow = c(2, 2), mar = c(2.5, 2.5, 0.2, 0.1), mgp = c(1.5, 0.5, 0))
plot(bsMat, ylab = "Cubic B-splines")
plot(ibsMat, ylab = "The integrals")
plot(d1Mat, ylab = "The 1st derivatives")
plot(d2Mat, ylab = "The 2nd derivatives")

## evaluate at new values
predict(bsMat, c(0.125, 0.801))

## periodic B-splines
px <- seq(0, 3, 0.01)
pbsMat <- bSpline(px, knots = knots, Boundary.knots = c(0, 1),
                  intercept = TRUE, periodic = TRUE)
ipMat <- update(pbsMat, integral = TRUE)
dpMat <- deriv(pbsMat)
dp2Mat <- deriv(pbsMat, 2)

plot(pbsMat, ylab = "Periodic B-splines", mark_knots = "b")
plot(ipMat, ylab = "The integrals", mark_knots = "b")
plot(dpMat, ylab = "The 1st derivatives", mark_knots = "b")
plot(dp2Mat, ylab = "The 2nd derivatives", mark_knots = "b")
par(op) # reset to previous plotting settings
}
\references{
De Boor, Carl. (1978). \emph{A practical guide to splines}.
Vol. 27. New York: Springer-Verlag.

Wang, W., & Yan, J. (2021). \emph{Shape-restricted regression splines with R
package splines2}. Journal of Data Science, 19(3),498--517.
}
\seealso{
\code{\link{knots}} for extracting internal and boundary knots.
}
