/*
 * Database interaction routines for Cray XT/XE systems.
 *
 * Copyright (c) 2009-2011 Centro Svizzero di Calcolo Scientifico (CSCS)
 * Licensed under the GPLv2.
 */
#include "basil_mysql.h"
#include "basil_torus.h"

/** Read options from the appropriate my.cnf configuration file. */
static int get_options_from_default_conf(MYSQL *handle)
{
	const char **path, *default_conf_paths[] = {
		"/etc/my.cnf",
		"/etc/opt/cray/MySQL/my.cnf",
		NULL
	};

	for (path = default_conf_paths; *path; path++)
		if (access(*path, R_OK) == 0)
			break;
	if (*path == NULL)
		fatal("no readable 'my.cnf' found");
	return  mysql_options(handle, MYSQL_READ_DEFAULT_FILE, *path);
}

/**
 * cray_connect_sdb - Connect to the XTAdmin database on the SDB host
 */
MYSQL *cray_connect_sdb(void)
{
	MYSQL *handle = mysql_init(NULL);

	if (handle == NULL)
		return NULL;

	if (get_options_from_default_conf(handle) != 0) {
		error("can not get options from configuration file (%u) - %s",
		      mysql_errno(handle), mysql_error(handle));
		goto connect_failed;
	}

	if (mysql_real_connect(handle, "sdb", NULL, NULL, "XTAdmin",
						0, NULL, 0) == NULL) {
		error("can not connect (%u) - %s",
		      mysql_errno(handle), mysql_error(handle));
		goto connect_failed;
	}

	return handle;

connect_failed:
	mysql_close(handle);
	return NULL;
}

/*
 * Determine the number of cores available on a Cray CLE system.
 * @handle:	connected to sdb.XTAdmin database
 * Returns:
 *  o positive number on success
 *  o 0 if system is inhomogeneous
 *  o -1 on internal error
 */
int cray_num_compute_node_cores(MYSQL *handle)
{
	const char query[] =	"SELECT DISTINCT LOG2(coremask + 1) "
				"FROM attributes, processor "
				"WHERE processor_id   = nodeid "
				"AND   processor_type = 'compute'";
	MYSQL_STMT	*stmt;
	MYSQL_BIND	bind_cols[1];
	my_bool		is_null[1];
	my_bool		error[1];
	unsigned int	cores;
	int		nrows, retval = -1;

	memset(bind_cols, 0, sizeof(bind_cols));
	bind_cols[0].buffer_type = MYSQL_TYPE_LONG;
	bind_cols[0].buffer      = (char *)&cores;
	bind_cols[0].is_unsigned = true;
	bind_cols[0].is_null     = is_null;
	bind_cols[0].error       = error;

	stmt = prepare_stmt(handle, query, NULL, 0, bind_cols, 1);
	if (stmt == NULL)
		return -1;

	nrows = exec_stmt(stmt, query, bind_cols, 1);
	if (nrows > 1)
		retval = 0;
	else if (nrows == 1 && mysql_stmt_fetch(stmt) == 0)
		retval = cores;

	mysql_stmt_close(stmt);

	return retval;
}

/**
 * cray_is_gemini_system -  Figure out whether SeaStar (XT) or Gemini (XE)
 * @handle:	connected to sdb.XTAdmin database
 * Returns
 * -1 on error
 *  1 if on a Gemini system
 *  0 if on a SeaStar system
 */
int cray_is_gemini_system(MYSQL *handle)
{
	/*
	 * Assumptions that this 'is Gemini' test uses:
	 * --------------------------------------------
	 * - XT SeaStar systems have one SeaStar ASIC per node.
	 *   There are 4 nodes on a blade with 4 SeaStar ASICS, giving
	 *   4 distinct (X,Y,Z) coordinates per node, i.e. the total
	 *   node count equals the count of distinct coordinates.
	 * - XE Gemini systems connect pairs of nodes to a Gemini chip.
	 *   There are 4 nodes on a blade and 2 Gemini chips, nodes 0/1
	 *   are connected to Gemini chip 0, nodes 2/3 are connected to
	 *   Gemini chip 1. This configuration acts as if the nodes were
	 *   already connected in Y dimension, hence there are half as
	 *   many (X,Y,Z) coordinates than there are nodes in the system.
	 * - Coordinates can be NULL if the network chip is deactivated.
	 */
	return exec_boolean_query(handle,
		"SELECT COUNT(DISTINCT x_coord, y_coord, z_coord) < COUNT(*) "
		"FROM processor "
		"WHERE x_coord IS NOT NULL "
		"AND   y_coord IS NOT NULL "
		"AND   z_coord IS NOT NULL"
	);
}

/*
 * cray_torus_from_data_base - Obtain system configuration from database.
 * @handle:	connected to sdb.XTAdmin database
 * Returns torus information if ok, NULL on error.
 */
struct torus_info *cray_torus_from_database(MYSQL *handle)
{
	struct torus_info *ti;
	/*
	 * Caveats when querying node coordinates:
	 * - service nodes are not shown in the INVENTORY response
	 * - nodes whose state is not 'up' may have undefined coordinates
	 */
	const char query[] = "SELECT MAX(x_coord),"
			     "       MAX(y_coord),"
			     "       MAX(z_coord),"
			     "       MAX(cab_position),"
			     "       MAX(cab_row), "
			     "       COUNT(DISTINCT cab_position,cage) "
			     "FROM  processor";
	enum query_columns {
			COL_X_MAX,
			COL_Y_MAX,
			COL_Z_MAX,
			COL_CAB_MAX,
			COL_ROW_MAX,
			COL_CHASSIS,
			COLUMN_COUNT
	};
	MYSQL_BIND	bind[COLUMN_COUNT];
	MYSQL_STMT	*stmt;
	int		col_data[COLUMN_COUNT];
	my_bool		is_null[COLUMN_COUNT];
	my_bool		is_error[COLUMN_COUNT];
	int		i, is_gemini;

	ti = calloc(1, sizeof(*ti));
	if (ti == NULL)
		return NULL;

	is_gemini = cray_is_gemini_system(handle);
	if (is_gemini < 0) {
		error("can not determine whether this is a Gemini system");
		goto err;
	}

	memset(bind, 0, sizeof(bind));
	for (i = 0; i < COLUMN_COUNT; i ++) {
		bind[i].buffer_type = MYSQL_TYPE_LONG;
		bind[i].buffer	    = (char *)&col_data[i];
		bind[i].is_null	    = &is_null[i];
		bind[i].error	    = &is_error[i];
	}

	stmt = prepare_stmt(handle, query, NULL, 0, bind, COLUMN_COUNT);
	if (stmt == NULL)
		goto err;

	if (mysql_stmt_execute(stmt) || mysql_stmt_fetch(stmt)) {
		error("can not fetch data for '%s': %s",
		      query, mysql_stmt_error(stmt));
		goto err_close;
	}

	ti->cabling = torus_from_placement(col_data[COL_CAB_MAX],
					   col_data[COL_ROW_MAX]);
	if (ti->cabling == CABLING_UNKNOWN) {
		error("can not determine cabling topology");
		goto err_close;
	}
	/* Rows/cabs/chassis are counts, X/Y/Z are indices starting at 0. */
	ti->network	= is_gemini ? ASIC_XE_GEMINI : ASIC_XT_SEASTAR;
	ti->dimension	= col_data[COL_X_MAX] ? TORUS_3D : TORUS_2D;
	ti->cabs	= col_data[COL_CAB_MAX] + 1;
	ti->rows	= col_data[COL_ROW_MAX] + 1;
	ti->chassis	= col_data[COL_CHASSIS];
	ti->x_max	= col_data[COL_X_MAX];
	ti->y_max	= col_data[COL_Y_MAX];
	ti->z_max	= col_data[COL_Z_MAX];

	mysql_stmt_close(stmt);
	return ti;

err_close:
	mysql_stmt_close(stmt);
err:
	free(ti);
	return NULL;
}

/*
 * Determine maximum torus extensions
 * @handle:	connected to sdb.XTAdmin database
 * @x:		to return maximum X coordinate
 * @y:		to return maximum Y coordinate
 * @z:		to return maximum Z coordinate
 * Returns true if ok, false on error.
 */
bool cray_get_torus_dimensions(MYSQL *handle, int *x, int *y, int *z)
{
	struct torus_info *ti = cray_torus_from_database(handle);

	if (ti == NULL)
		return false;

	*x = ti->x_max;
	*y = ti->y_max;
	*z = ti->z_max;
	return true;
}
