-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (ErrorHandler.Conversions.ToString)
procedure ControlFlowError (Err_Num          : in     Error_Types.NumericError;
                            With_Explanation : in     Boolean;
                            E_Str            : in out E_Strings.T) is
   Err_Type : ErrorHandler.Control_Flow_Err_Type;

   procedure ControlFlowErrorExpl (E_Str : in out E_Strings.T)
   --# global in Err_Type;
   --# derives E_Str from *,
   --#                    Err_Type;
      is separate;
   -- Note that the parameter names for this subunit are chosen to make it as easy as
   --      possible to auto-generate the subunit from this, its parent, file.  The
   --      generation requires copying the case statement below, stripping out the
   --      current Append'Thing' statements and adding an Append_String for the
   --      explanatory text that is delineated by --! comments.

   procedure Append_Explanation
   --# global in     Err_Type;
   --#        in     With_Explanation;
   --#        in out E_Str;
   --# derives E_Str from *,
   --#                    Err_Type,
   --#                    With_Explanation;
   is
      Explanation_String : E_Strings.T := E_Strings.Empty_String;
   begin
      if With_Explanation then
         -- we need to at least look for an explanation
         ControlFlowErrorExpl (E_Str => Explanation_String);
         if E_Strings.Get_Length (E_Str => Explanation_String) > 0 then
            -- there actually is one
            E_Strings.Append_String (E_Str => E_Str,
                                     Str   => ErrorHandler.Explanation_Prefix);
            E_Strings.Append_Examiner_String (E_Str1 => E_Str,
                                              E_Str2 => Explanation_String);
            E_Strings.Append_String (E_Str => E_Str,
                                     Str   => ErrorHandler.Explanation_Postfix);
         end if;
      end if;
   end Append_Explanation;

begin
   Err_Type := ErrorHandler.Control_Flow_Err_Type'Val (Err_Num.ErrorNum - Error_Types.ControlFlowErrOffset);
   case Err_Type is

      -- HTML Directives
      --! <NameFormat> <"illegal-"><Name>
      --! <ErrorFormat> <"*** Illegal Structure : "><Name><" : "><Error>

      when ErrorHandler.Misplaced_Exit =>
         --! <Name> 1
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "An exit statement may not occur here");
         --! Exit statements must be of the form &quot;exit when c;&quot; where the closest
         --! enclosing statement is a loop or &quot;if c then S; exit;&quot; where the
         --! if statement has no else part and its closest enclosing
         --! statement is a loop.  See the SPARK Definition for details.

      when ErrorHandler.Misplaced_Return =>
         --! <Name> 2
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "A return statement may not occur here");
         --! A return statement may only occur as the last statement of a function.

      when ErrorHandler.Missing_Return =>
         --! <Name> 3
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "The last statement of this function is not a return statement");
         --! SPARK requires that the last statement of a function be a return statement.

      when ErrorHandler.Return_In_Proc =>
         --! <Name> 4
         E_Strings.Append_String (E_Str => E_Str,
                                  Str   => "Return statements may not occur in procedure subprograms");
   end case;

   Append_Explanation;
   E_Strings.Append_String (E_Str => E_Str,
                            Str   => ".");
end ControlFlowError;
