/* ====================================================================
 * Copyright (c) 2009       Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

#include "SingleProjectReaderTest.h"

// cppunit
#include <cppunit/TestSuite.h>
#include <cppunit/TestCaller.h>

// sc
#include "subcommander/settings/SingleProjectReader.h"
#include "subcommander/Project.h"
#include "util-test/AssertionTraitUuid.h"


///////////////////////////////////////////////////////////////////////

ConfigValue _( const char* key, const char* val )
{
  return ConfigValue(sc::String(key),sc::String(val));
}

void push( ConfigValues& v, const char* key, const char* val )
{
  v.push_back(_(key,val));
}

///////////////////////////////////////////////////////////////////////

void SingleProjectReaderTest::setUp()
{
  ConfigValues v;

  // a complete project
  push( v, "project.0.guid", "e7380f2c-5bf6-0310-a2d6-b2464cb4a0ca" );
  push( v, "project.0.name", "Subcommander" );
  push( v, "project.0.sortpos", "0" );
  {
    // with special items
    push( v, "project.0.trunk.name", "trunk" );
    push( v, "project.0.trunk.recursive", "true" );
    push( v, "project.0.trunk.revision", "head" );
    push( v, "project.0.trunk.sortpos", "-3" );
    push( v, "project.0.trunk.url",
      "http://subcommander.tigris.org/svn/subcommander/trunk" );
    push( v, "project.0.branches.name", "branches" );
    push( v, "project.0.branches.recursive", "true" );
    push( v, "project.0.branches.revision", "head" );
    push( v, "project.0.branches.sortpos", "-2" );
    push( v, "project.0.branches.url",
      "http://subcommander.tigris.org/svn/subcommander/branches" );
    push( v, "project.0.tags.name", "tags" );
    push( v, "project.0.tags.recursive", "true" );
    push( v, "project.0.tags.revision", "head" );
    push( v, "project.0.tags.sortpos", "-1" );
    push( v, "project.0.tags.url",
      "http://subcommander.tigris.org/svn/subcommander/tags" );
    // with working copy items
    push( v, "project.0.wc.0.name", "head" );
    push( v, "project.0.wc.0.autorefresh", "true" );
    push( v, "project.0.wc.0.autoupdate", "false" );
    push( v, "project.0.wc.0.recursive", "true" );
    push( v, "project.0.wc.0.sortpos", "0" );
    push( v, "project.0.wc.0.path",
      "C:/Development/Subcommander-Head" );
    {
      // with sub items
      push( v, "project.0.wc.0.subitem.0.type", "flat" );
      push( v, "project.0.wc.0.subitem.0.path",
        "C:/Development/Subcommander-Head" );
      push( v, "project.0.wc.0.subitem.1.type", "tree" );
      push( v, "project.0.wc.0.subitem.1.path",
        "C:/Development/Subcommander-Head/subcommander" );
    }
    push( v, "project.0.wc.1.name", "head-NoBuild" );
    push( v, "project.0.wc.1.autorefresh", "true" );
    push( v, "project.0.wc.1.autoupdate", "false" );
    push( v, "project.0.wc.1.recursive", "true" );
    push( v, "project.0.wc.1.sortpos", "1" );
    push( v, "project.0.wc.1.path",
      "C:/Development/Subcommander-Head-NoBuild" );

    push( v, "project.0.rp.0.name", "root" );
    push( v, "project.0.rp.0.recursive", "true" );
    push( v, "project.0.rp.0.revision", "head" );
    push( v, "project.0.rp.0.sortpos", "2" );
    push( v, "project.0.rp.0.url",
      "http://subcommander.tigris.org/svn/subcommander/" );
    push( v, "project.0.rp.1.name", "toor" );
    push( v, "project.0.rp.1.recursive", "true" );
    push( v, "project.0.rp.1.revision", "head" );
    push( v, "project.0.rp.1.sortpos", "3" );
    push( v, "project.0.rp.1.url",
      "http://subcommander.tigris.org/svn/subcommander/subcommander" );
  }

  _reader = new SingleProjectReader( 0, v );
}

void SingleProjectReaderTest::tearDown()
{
  delete _reader->getProject();
  delete _reader;
}

void SingleProjectReaderTest::testProjectRoot()
{
  _reader->read();
  Project* prj = _reader->getProject();

  CPPUNIT_ASSERT_EQUAL( Uuid("e7380f2c-5bf6-0310-a2d6-b2464cb4a0ca"),
    prj->getId() );
  CPPUNIT_ASSERT_EQUAL( sc::String("Subcommander"), prj->getName() );
  CPPUNIT_ASSERT_EQUAL( 0L, prj->getSortPos() );
}

void SingleProjectReaderTest::testTrunkItem()
{
  _reader->read();
  Project* prj = _reader->getProject();

  const Project::Item& item = prj->getTrunkItem();

  CPPUNIT_ASSERT_EQUAL( sc::String("trunk"), item.getName() );
  CPPUNIT_ASSERT_EQUAL( true, item.getRecursive() );
  //CPPUNIT_ASSERT_EQUAL( true, trunk.getRevision() );
  CPPUNIT_ASSERT_EQUAL( -3L, item.getSortPos() );
  CPPUNIT_ASSERT_EQUAL(
    sc::String("http://subcommander.tigris.org/svn/subcommander/trunk"),
    item.getSource() );
}

void SingleProjectReaderTest::testBranchesItem()
{
  _reader->read();
  Project* prj = _reader->getProject();

  const Project::Item& item = prj->getBranchesItem();

  CPPUNIT_ASSERT_EQUAL( sc::String("branches"), item.getName() );
  CPPUNIT_ASSERT_EQUAL( true, item.getRecursive() );
  //CPPUNIT_ASSERT_EQUAL( true, trunk.getRevision() );
  CPPUNIT_ASSERT_EQUAL( -2L, item.getSortPos() );
  CPPUNIT_ASSERT_EQUAL( sc::String(
    "http://subcommander.tigris.org/svn/subcommander/branches"),
    item.getSource() );
}

void SingleProjectReaderTest::testTagsItem()
{
  _reader->read();
  Project* prj = _reader->getProject();

  const Project::Item& item = prj->getTagsItem();

  CPPUNIT_ASSERT_EQUAL( sc::String("tags"), item.getName() );
  CPPUNIT_ASSERT_EQUAL( true, item.getRecursive() );
  //CPPUNIT_ASSERT_EQUAL( true, trunk.getRevision() );
  CPPUNIT_ASSERT_EQUAL( -1L, item.getSortPos() );
  CPPUNIT_ASSERT_EQUAL( sc::String(
    "http://subcommander.tigris.org/svn/subcommander/tags"),
    item.getSource() );
}

Project::Item findItemBySortPos( Project* prj, long sortPos )
{
  Project::Items items;
  prj->getItems(items);
  Project::Item item;
  for( Project::Items::iterator it = items.begin(); it != items.end();
    it++ )
  {
    if( (*it).getSortPos() == sortPos )
    {
      item = *it;
      break;
    }
  }
  return item;
}

void SingleProjectReaderTest::testWcItem0()
{
  _reader->read();
  Project* prj = _reader->getProject();

  Project::Item item = findItemBySortPos(prj,0);
  CPPUNIT_ASSERT_EQUAL( sc::String("head"), item.getName() );
  CPPUNIT_ASSERT_EQUAL( true, item.getAutoRefresh() );
  CPPUNIT_ASSERT_EQUAL( false, item.getAutoUpdate() );
  CPPUNIT_ASSERT_EQUAL( true, item.getRecursive() );
  CPPUNIT_ASSERT_EQUAL( 0L, item.getSortPos() );
  CPPUNIT_ASSERT_EQUAL( sc::String("C:/Development/Subcommander-Head"),
    item.getSource() );
}

void SingleProjectReaderTest::testWcItem1()
{
  _reader->read();
  Project* prj = _reader->getProject();

  Project::Item item = findItemBySortPos(prj,1);
  CPPUNIT_ASSERT_EQUAL( sc::String("head-NoBuild"), item.getName() );
  CPPUNIT_ASSERT_EQUAL( true, item.getAutoRefresh() );
  CPPUNIT_ASSERT_EQUAL( false, item.getAutoUpdate() );
  CPPUNIT_ASSERT_EQUAL( true, item.getRecursive() );
  CPPUNIT_ASSERT_EQUAL( 1L, item.getSortPos() );
  CPPUNIT_ASSERT_EQUAL( sc::String(
    "C:/Development/Subcommander-Head-NoBuild"), item.getSource() );
}

void SingleProjectReaderTest::testSubItemOfWcItem0()
{
  _reader->read();
  Project* prj = _reader->getProject();

  Project::Item item = findItemBySortPos(prj,0);

  Project::SubItems items;
  item.getSubItems(items);
  CPPUNIT_ASSERT_EQUAL( 2L, (long)items.size() );

  Project::SubItem subItem = items[0];
  CPPUNIT_ASSERT_EQUAL( sc::String(
    "C:/Development/Subcommander-Head"), subItem.getPath() );
  CPPUNIT_ASSERT_EQUAL( Project::SubItem_Flat, subItem.getType() );

  subItem = items[1];
  CPPUNIT_ASSERT_EQUAL( sc::String(
    "C:/Development/Subcommander-Head/subcommander"), subItem.getPath()
    );
  CPPUNIT_ASSERT_EQUAL( Project::SubItem_Tree, subItem.getType() );
}

void SingleProjectReaderTest::testSubItemOfWcItem1()
{
  _reader->read();
  Project* prj = _reader->getProject();

  Project::Item item = findItemBySortPos(prj,1);

  Project::SubItems items;
  item.getSubItems(items);
  CPPUNIT_ASSERT_EQUAL( 1L, (long)items.size() );

  Project::SubItem subItem = items[0];
  CPPUNIT_ASSERT_EQUAL( sc::String(
    "C:/Development/Subcommander-Head-NoBuild"), subItem.getPath() );
  CPPUNIT_ASSERT_EQUAL( Project::SubItem_Flat, subItem.getType() );
}

void SingleProjectReaderTest::testRpItem0()
{
  _reader->read();
  Project* prj = _reader->getProject();

  Project::Item item = findItemBySortPos(prj,2);
  CPPUNIT_ASSERT_EQUAL( sc::String("root"), item.getName() );
  CPPUNIT_ASSERT_EQUAL( true, item.getRecursive() );
  CPPUNIT_ASSERT_EQUAL( 2L, item.getSortPos() );
  CPPUNIT_ASSERT_EQUAL( 
    sc::String("http://subcommander.tigris.org/svn/subcommander"),
    item.getSource() );
}

void SingleProjectReaderTest::testRpItem1()
{
  _reader->read();
  Project* prj = _reader->getProject();

  Project::Item item = findItemBySortPos(prj,3);
  CPPUNIT_ASSERT_EQUAL( sc::String("toor"), item.getName() );
  CPPUNIT_ASSERT_EQUAL( true, item.getRecursive() );
  CPPUNIT_ASSERT_EQUAL( 3L, item.getSortPos() );
  CPPUNIT_ASSERT_EQUAL( sc::String(
    "http://subcommander.tigris.org/svn/subcommander/subcommander"),
    item.getSource() );
}
