/*
  @(#) $Id: unicodemap.c,v 1.14 2003/12/22 22:24:33 yeti Exp $
  unicode (i.e. ucs-2) data and conversion table managment

  Copyright (C) 2000-2003 David Necas (Yeti) <yeti@physics.muni.cz>
  Unicode data in this module come from various free on-line resources.

  This program is free software; you can redistribute it and/or modify it
  under the terms of version 2 of the GNU General Public License as published
  by the Free Software Foundation.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
*/
#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif /* HAVE_CONFIG_H */

#include "enca.h"
#include "internal.h"

/**
 * NO_CHR:
 *
 * Alias for ENCA_NOT_A_CHAR, which lines up with 0xabcd.
 **/
#define NO_CHR ENCA_NOT_A_CHAR

/**
 * EncaUnicodeMap:
 * @name: Charset name (any, Enca's is best).
 * @tstart: Table start, character codes less than tstart maps to themselves.
 * @map: The map itself.
 *
 * 8bit charset -> Unicode mapping.
 **/
struct _EncaUnicodeMap {
  const char *name;
  size_t tstart;
  const unsigned short int *map;
};

typedef struct _EncaUnicodeMap EncaUnicodeMap;

/* Local prototypes. */
static const EncaUnicodeMap* find_charset_map(int charset);

/* ISO-8859-2  */
static const unsigned short int unicode_map_iso88592[] = {
  0x00a0, 0x0104, 0x02d8, 0x0141, 0x00a4, 0x013d, 0x015a, 0x00a7, /* 0xa0 */
  0x00a8, 0x0160, 0x015e, 0x0164, 0x0179, 0x00ad, 0x017d, 0x017b, /* 0xa8 */
  0x00b0, 0x0105, 0x02db, 0x0142, 0x00b4, 0x013e, 0x015b, 0x02c7, /* 0xb0 */
  0x00b8, 0x0161, 0x015f, 0x0165, 0x017a, 0x02dd, 0x017e, 0x017c, /* 0xb8 */
  0x0154, 0x00c1, 0x00c2, 0x0102, 0x00c4, 0x0139, 0x0106, 0x00c7, /* 0xc0 */
  0x010c, 0x00c9, 0x0118, 0x00cb, 0x011a, 0x00cd, 0x00ce, 0x010e, /* 0xc8 */
  0x0110, 0x0143, 0x0147, 0x00d3, 0x00d4, 0x0150, 0x00d6, 0x00d7, /* 0xd0 */
  0x0158, 0x016e, 0x00da, 0x0170, 0x00dc, 0x00dd, 0x0162, 0x00df, /* 0xd8 */
  0x0155, 0x00e1, 0x00e2, 0x0103, 0x00e4, 0x013a, 0x0107, 0x00e7, /* 0xe0 */
  0x010d, 0x00e9, 0x0119, 0x00eb, 0x011b, 0x00ed, 0x00ee, 0x010f, /* 0xe8 */
  0x0111, 0x0144, 0x0148, 0x00f3, 0x00f4, 0x0151, 0x00f6, 0x00f7, /* 0xf0 */
  0x0159, 0x016f, 0x00fa, 0x0171, 0x00fc, 0x00fd, 0x0163, 0x02d9  /* 0xf8 */
};

/* ISO-8859-4  */
static const unsigned short int unicode_map_iso88594[] = {
  0x00a0, 0x0104, 0x0138, 0x0156, 0x00a4, 0x0128, 0x013b, 0x00a7, /* 0xa0 */
  0x00a8, 0x0160, 0x0112, 0x0122, 0x0166, 0x00ad, 0x017d, 0x00af, /* 0xa8 */
  0x00b0, 0x0105, 0x02db, 0x0157, 0x00b4, 0x0129, 0x013c, 0x02c7, /* 0xb0 */
  0x00b8, 0x0161, 0x0113, 0x0123, 0x0167, 0x014a, 0x017e, 0x014b, /* 0xb8 */
  0x0100, 0x00c1, 0x00c2, 0x00c3, 0x00c4, 0x00c5, 0x00c6, 0x012e, /* 0xc0 */
  0x010c, 0x00c9, 0x0118, 0x00cb, 0x0116, 0x00cd, 0x00ce, 0x012a, /* 0xc8 */
  0x0110, 0x0145, 0x014c, 0x0136, 0x00d4, 0x00d5, 0x00d6, 0x00d7, /* 0xd0 */
  0x00d8, 0x0172, 0x00da, 0x00db, 0x00dc, 0x0168, 0x016a, 0x00df, /* 0xd8 */
  0x0101, 0x00e1, 0x00e2, 0x00e3, 0x00e4, 0x00e5, 0x00e6, 0x012f, /* 0xe0 */
  0x010d, 0x00e9, 0x0119, 0x00eb, 0x0117, 0x00ed, 0x00ee, 0x012b, /* 0xe8 */
  0x0111, 0x0146, 0x014d, 0x0137, 0x00f4, 0x00f5, 0x00f6, 0x00f7, /* 0xf0 */
  0x00f8, 0x0173, 0x00fa, 0x00fb, 0x00fc, 0x0169, 0x016b, 0x02d9, /* 0xf8 */
};

/* ISO-8859-5  */
static const unsigned short int unicode_map_iso88595[] = {
  0x00a0, 0x0401, 0x0402, 0x0403, 0x0404, 0x0405, 0x0406, 0x0407, /* 0xa0 */
  0x0408, 0x0409, 0x040a, 0x040b, 0x040c, 0x00ad, 0x040e, 0x040f, /* 0xa8 */
  0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0416, 0x0417, /* 0xb0 */
  0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d, 0x041e, 0x041f, /* 0xb8 */
  0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426, 0x0427, /* 0xc0 */
  0x0428, 0x0429, 0x042a, 0x042b, 0x042c, 0x042d, 0x042e, 0x042f, /* 0xc8 */
  0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437, /* 0xd0 */
  0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d, 0x043e, 0x043f, /* 0xd8 */
  0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447, /* 0xe0 */
  0x0448, 0x0449, 0x044a, 0x044b, 0x044c, 0x044d, 0x044e, 0x044f, /* 0xe8 */
  0x2116, 0x0451, 0x0452, 0x0453, 0x0454, 0x0455, 0x0456, 0x0457, /* 0xf0 */
  0x0458, 0x0459, 0x045a, 0x045b, 0x045c, 0x00a7, 0x045e, 0x045f, /* 0xf8 */
};

/* ISO-8859-13  */
static const unsigned short int unicode_map_iso885913[] = {
  0x00a0, 0x201d, 0x00a2, 0x00a3, 0x00a4, 0x201e, 0x00a6, 0x00a7, /* 0xa0 */
  0x00d8, 0x00a9, 0x0156, 0x00ab, 0x00ac, 0x00ad, 0x00ae, 0x00c6, /* 0xa8 */
  0x00b0, 0x00b1, 0x00b2, 0x00b3, 0x201c, 0x00b5, 0x00b6, 0x00b7, /* 0xb0 */
  0x00f8, 0x00b9, 0x0157, 0x00bb, 0x00bc, 0x00bd, 0x00be, 0x00e6, /* 0xb8 */
  0x0104, 0x012e, 0x0100, 0x0106, 0x00c4, 0x00c5, 0x0118, 0x0112, /* 0xc0 */
  0x010c, 0x00c9, 0x0179, 0x0116, 0x0122, 0x0136, 0x012a, 0x013b, /* 0xc8 */
  0x0160, 0x0143, 0x0145, 0x00d3, 0x014c, 0x00d5, 0x00d6, 0x00d7, /* 0xd0 */
  0x0172, 0x0141, 0x015a, 0x016a, 0x00dc, 0x017b, 0x017d, 0x00df, /* 0xd8 */
  0x0105, 0x012f, 0x0101, 0x0107, 0x00e4, 0x00e5, 0x0119, 0x0113, /* 0xe0 */
  0x010d, 0x00e9, 0x017a, 0x0117, 0x0123, 0x0137, 0x012b, 0x013c, /* 0xe8 */
  0x0161, 0x0144, 0x0146, 0x00f3, 0x014d, 0x00f5, 0x00f6, 0x00f7, /* 0xf0 */
  0x0173, 0x0142, 0x015b, 0x016b, 0x00fc, 0x017c, 0x017e, 0x2019, /* 0xf8 */
};

/* ISO-8859-16  */
static const unsigned short int unicode_map_iso885916[] = {
  0x00a0, 0x0104, 0x0105, 0x0141, 0x20ac, 0x00ab, 0x0160, 0x00a7, /* 0xa0 */
  0x0161, 0x00a9, 0x0218, 0x201e, 0x0179, 0x00ad, 0x017a, 0x017b, /* 0xa8 */
  0x00b0, 0x00b1, 0x010c, 0x0142, 0x017d, 0x201d, 0x00b6, 0x00b7, /* 0xb0 */
  0x017e, 0x010d, 0x0219, 0x00bb, 0x0152, 0x0153, 0x0178, 0x017c, /* 0xb8 */
  0x00c0, 0x00c1, 0x00c2, 0x0102, 0x00c4, 0x0106, 0x00c6, 0x00c7, /* 0xc0 */
  0x00c8, 0x00c9, 0x00ca, 0x00cb, 0x00cc, 0x00cd, 0x00ce, 0x00cf, /* 0xc8 */
  0x0110, 0x0143, 0x00d2, 0x00d3, 0x00d4, 0x0150, 0x00d6, 0x015a, /* 0xd0 */
  0x0170, 0x00d9, 0x00da, 0x00db, 0x00dc, 0x0118, 0x021a, 0x00df, /* 0xd8 */
  0x00e0, 0x00e1, 0x00e2, 0x0103, 0x00e4, 0x0107, 0x00e6, 0x00e7, /* 0xe0 */
  0x00e8, 0x00e9, 0x00ea, 0x00eb, 0x00ec, 0x00ed, 0x00ee, 0x00ef, /* 0xe8 */
  0x0111, 0x0144, 0x00f2, 0x00f3, 0x00f4, 0x0151, 0x00f6, 0x015b, /* 0xf0 */
  0x0171, 0x00f9, 0x00fa, 0x00fb, 0x00fc, 0x0119, 0x021b, 0x00ff, /* 0xf8 */
};

/* Windows 1125  */
static const unsigned short int unicode_map_cp1125[] = {
  0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0416, 0x0417, /* 0x80 */
  0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d, 0x041e, 0x041f, /* 0x88 */
  0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426, 0x0427, /* 0x90 */
  0x0428, 0x0429, 0x042a, 0x042b, 0x042c, 0x042d, 0x042e, 0x042f, /* 0x98 */
  0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437, /* 0xa0 */
  0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d, 0x043e, 0x043f, /* 0xa8 */
  0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556, /* 0xb0 */
  0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b, 0x2510, /* 0xb8 */
  0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f, /* 0xc0 */
  0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567, /* 0xc8 */
  0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b, /* 0xd0 */
  0x256a, 0x2518, 0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580, /* 0xd8 */
  0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447, /* 0xe0 */
  0x0448, 0x0449, 0x044a, 0x044b, 0x044c, 0x044d, 0x044e, 0x044f, /* 0xe8 */
  0x0401, 0x0451, 0x0490, 0x0491, 0x0404, 0x0454, 0x0406, 0x0456, /* 0xf0 */
  0x0407, 0x0457, 0x00b7, 0x221a, 0x2116, 0x00a4, 0x25a0, 0x00a0, /* 0xf8 */
};

/* Windows 1250  */
static const unsigned short int unicode_map_cp1250[] = {
  NO_CHR, NO_CHR, 0x201a, NO_CHR, 0x201e, 0x2026, 0x2020, 0x2021, /* 0x80 */
  NO_CHR, 0x2030, 0x0160, 0x2039, 0x015a, 0x0164, 0x017d, 0x0179, /* 0x88 */
  NO_CHR, 0x2018, 0x2019, 0x201c, 0x201d, 0x2022, 0x2013, 0x2014, /* 0x90 */
  NO_CHR, 0x2122, 0x0161, 0x203a, 0x015b, 0x0165, 0x017e, 0x017a, /* 0x98 */
  0x00a0, 0x02c7, 0x02d8, 0x0141, 0x00a4, 0x0104, 0x00a6, 0x00a7, /* 0xa0 */
  0x00a8, 0x00a9, 0x015e, 0x00ab, 0x00ac, 0x00ad, 0x00ae, 0x017b, /* 0xa8 */
  0x00b0, 0x00b1, 0x02db, 0x0142, 0x00b4, 0x00b5, 0x00b6, 0x00b7, /* 0xb0 */
  0x00b8, 0x0105, 0x015f, 0x00bb, 0x013d, 0x02dd, 0x013e, 0x017c, /* 0xb8 */
  0x0154, 0x00c1, 0x00c2, 0x0102, 0x00c4, 0x0139, 0x0106, 0x00c7, /* 0xc0 */
  0x010c, 0x00c9, 0x0118, 0x00cb, 0x011a, 0x00cd, 0x00ce, 0x010e, /* 0xc8 */
  0x0110, 0x0143, 0x0147, 0x00d3, 0x00d4, 0x0150, 0x00d6, 0x00d7, /* 0xd0 */
  0x0158, 0x016e, 0x00da, 0x0170, 0x00dc, 0x00dd, 0x0162, 0x00df, /* 0xd8 */
  0x0155, 0x00e1, 0x00e2, 0x0103, 0x00e4, 0x013a, 0x0107, 0x00e7, /* 0xe0 */
  0x010d, 0x00e9, 0x0119, 0x00eb, 0x011b, 0x00ed, 0x00ee, 0x010f, /* 0xe8 */
  0x0111, 0x0144, 0x0148, 0x00f3, 0x00f4, 0x0151, 0x00f6, 0x00f7, /* 0xf0 */
  0x0159, 0x016f, 0x00fa, 0x0171, 0x00fc, 0x00fd, 0x0163, 0x02d9, /* 0xf8 */
};

/* Windows 1251  */
static const unsigned short int unicode_map_cp1251[] = {
  0x0402, 0x0403, 0x201a, 0x0453, 0x201e, 0x2026, 0x2020, 0x2021, /* 0x80 */
  0x20ac, 0x2030, 0x0409, 0x2039, 0x040a, 0x040c, 0x040b, 0x040f, /* 0x88 */
  0x0452, 0x2018, 0x2019, 0x201c, 0x201d, 0x2022, 0x2013, 0x2014, /* 0x90 */
  NO_CHR, 0x2122, 0x0459, 0x203a, 0x045a, 0x045c, 0x045b, 0x045f, /* 0x98 */
  0x00a0, 0x040e, 0x045e, 0x0408, 0x00a4, 0x0490, 0x00a6, 0x00a7, /* 0xa0 */
  0x0401, 0x00a9, 0x0404, 0x00ab, 0x00ac, 0x00ad, 0x00ae, 0x0407, /* 0xa8 */
  0x00b0, 0x00b1, 0x0406, 0x0456, 0x0491, 0x00b5, 0x00b6, 0x00b7, /* 0xb0 */
  0x0451, 0x2116, 0x0454, 0x00bb, 0x0458, 0x0405, 0x0455, 0x0457, /* 0xb8 */
  0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0416, 0x0417, /* 0xc0 */
  0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d, 0x041e, 0x041f, /* 0xc8 */
  0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426, 0x0427, /* 0xd0 */
  0x0428, 0x0429, 0x042a, 0x042b, 0x042c, 0x042d, 0x042e, 0x042f, /* 0xd8 */
  0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437, /* 0xe0 */
  0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d, 0x043e, 0x043f, /* 0xe8 */
  0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447, /* 0xf0 */
  0x0448, 0x0449, 0x044a, 0x044b, 0x044c, 0x044d, 0x044e, 0x044f, /* 0xf8 */
};

/* Windows 1257  */
static const unsigned short int unicode_map_cp1257[] = {
  0x20ac, NO_CHR, 0x201a, NO_CHR, 0x201e, 0x2026, 0x2020, 0x2021, /* 0x80 */
  NO_CHR, 0x2030, NO_CHR, 0x2039, NO_CHR, NO_CHR, NO_CHR, NO_CHR, /* 0x88 */
  NO_CHR, 0x2018, 0x2019, 0x201c, 0x201d, 0x2022, 0x2013, 0x2014, /* 0x90 */
  NO_CHR, 0x2122, NO_CHR, 0x203a, NO_CHR, NO_CHR, NO_CHR, NO_CHR, /* 0x98 */
  0x00a0, NO_CHR, 0x00a2, 0x00a3, 0x00a4, NO_CHR, 0x00a6, 0x00a7, /* 0xa0 */
  0x00d8, 0x00a9, 0x0156, 0x00ab, 0x00ac, 0x00ad, 0x00ae, 0x00c6, /* 0xa8 */
  0x00b0, 0x00b1, 0x00b2, 0x00b3, NO_CHR, 0x00b5, 0x00b6, 0x00b7, /* 0xb0 */
  0x00f8, 0x00b9, 0x0157, 0x00bb, 0x00bc, 0x00bd, 0x00be, 0x00e6, /* 0xb8 */
  0x0104, 0x012e, 0x0100, 0x0106, 0x00c4, 0x00c5, 0x0118, 0x0112, /* 0xc0 */
  0x010c, 0x00c9, 0x0179, 0x0116, 0x0122, 0x0136, 0x012a, 0x013b, /* 0xc8 */
  0x0160, 0x0143, 0x0145, 0x00d3, 0x014c, 0x00d5, 0x00d6, 0x00d7, /* 0xd0 */
  0x0172, 0x0141, 0x015a, 0x016a, 0x00dc, 0x017b, 0x017d, 0x00df, /* 0xd8 */
  0x0105, 0x012f, 0x0101, 0x0107, 0x00e4, 0x00e5, 0x0119, 0x0113, /* 0xe0 */
  0x010d, 0x00e9, 0x017a, 0x0117, 0x0123, 0x0137, 0x012b, 0x013c, /* 0xe8 */
  0x0161, 0x0144, 0x0146, 0x00f3, 0x014d, 0x00f5, 0x00f6, 0x00f7, /* 0xf0 */
  0x0173, 0x0142, 0x015b, 0x016b, 0x00fc, 0x017c, 0x017e, NO_CHR, /* 0xf8 */
};

/* IBM 852  */
static const unsigned short int unicode_map_ibm852[] = {
  0x00c7, 0x00fc, 0x00e9, 0x00e2, 0x00e4, 0x016f, 0x0107, 0x00e7, /* 0x80 */
  0x0142, 0x00eb, 0x0150, 0x0151, 0x00ee, 0x0179, 0x00c4, 0x0106, /* 0x88 */
  0x00c9, 0x0139, 0x013a, 0x00f4, 0x00f6, 0x013d, 0x013e, 0x015a, /* 0x90 */
  0x015b, 0x00d6, 0x00dc, 0x0164, 0x0165, 0x0141, 0x00d7, 0x010d, /* 0x98 */
  0x00e1, 0x00ed, 0x00f3, 0x00fa, 0x0104, 0x0105, 0x017d, 0x017e, /* 0xa0 */
  0x0118, 0x0119, 0x00ac, 0x017a, 0x010c, 0x015f, 0x00ab, 0x00bb, /* 0xa8 */
  0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x00c1, 0x00c2, 0x011a, /* 0xb0 */
  0x015e, 0x2563, 0x2551, 0x2557, 0x255d, 0x017b, 0x017c, 0x2510, /* 0xb8 */
  0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x0102, 0x0103, /* 0xc0 */
  0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x00a4, /* 0xc8 */
  0x0111, 0x0110, 0x010e, 0x00cb, 0x010f, 0x0147, 0x00cd, 0x00ce, /* 0xd0 */
  0x011b, 0x2518, 0x250c, 0x2588, 0x2584, 0x0162, 0x016e, 0x2580, /* 0xd8 */
  0x00d3, 0x00df, 0x00d4, 0x0143, 0x0144, 0x0148, 0x0160, 0x0161, /* 0xe0 */
  0x0154, 0x00da, 0x0155, 0x0170, 0x00fd, 0x00dd, 0x0163, 0x00b4, /* 0xe8 */
  0x00ad, 0x02dd, 0x02db, 0x02c7, 0x02d8, 0x00a7, 0x00f7, 0x00b8, /* 0xf0 */
  0x00b0, 0x00a8, 0x02d9, 0x0171, 0x0158, 0x0159, 0x25a0, 0x00a0, /* 0xf8 */
};

/* IBM 855  */
static const unsigned short int unicode_map_ibm855[] = {
  0x0452, 0x0402, 0x0453, 0x0403, 0x0451, 0x0401, 0x0454, 0x0404, /* 0x80 */
  0x0455, 0x0405, 0x0456, 0x0406, 0x0457, 0x0407, 0x0458, 0x0408, /* 0x88 */
  0x0459, 0x0409, 0x045a, 0x040a, 0x045b, 0x0093, 0x045c, 0x040c, /* 0x90 */
  0x045e, 0x040e, 0x045f, 0x040f, 0x044e, 0x042e, 0x044a, 0x042a, /* 0x98 */
  0x0430, 0x0410, 0x0431, 0x0411, 0x0446, 0x0426, 0x0434, 0x0414, /* 0xa0 */
  0x0435, 0x0415, 0x0444, 0x0424, 0x0433, 0x0413, 0x00ab, 0x00bb, /* 0xa8 */
  0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x0445, 0x0425, 0x0438, /* 0xb0 */
  0x0418, 0x2563, 0x2551, 0x2557, 0x255d, 0x0439, 0x0419, 0x2510, /* 0xb8 */
  0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x043a, 0x041a, /* 0xc0 */
  0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x00a4, /* 0xc8 */
  0x043b, 0x041b, 0x043c, 0x041c, 0x043d, 0x041d, 0x043e, 0x041e, /* 0xd0 */
  0x043f, 0x2518, 0x250c, 0x2588, 0x2584, 0x041f, 0x044f, 0x2580, /* 0xd8 */
  0x042f, 0x0440, 0x0420, 0x0441, 0x0421, 0x0442, 0x0422, 0x0443, /* 0xe0 */
  0x0423, 0x0436, 0x0416, 0x0432, 0x0412, 0x044c, 0x042c, 0x00b4, /* 0xe8 */
  0x00ad, 0x044b, 0x042b, 0x0437, 0x0417, 0x0448, 0x0428, 0x044d, /* 0xf0 */
  0x042d, 0x0449, 0x0429, 0x0447, 0x0427, NO_CHR, 0x25a0, 0x00a0, /* 0xf8 */
};

/* IBM 775  */
static const unsigned short int unicode_map_ibm775[] = {
  0x0106, 0x00fc, 0x00e9, 0x0101, 0x00e4, 0x0123, 0x00e5, 0x0107, /* 0x80 */
  0x0142, 0x0113, 0x0156, 0x0157, 0x012b, 0x0179, 0x00c4, 0x00c5, /* 0x88 */
  0x00c9, 0x00e6, 0x00c6, 0x014d, 0x00f6, 0x0122, 0x00a2, 0x015a, /* 0x90 */
  0x015b, 0x00d6, 0x00dc, 0x00f8, 0x00a3, 0x00d8, 0x00d7, 0x00a4, /* 0x98 */
  0x0100, 0x012a, 0x00f3, 0x017b, 0x017c, 0x017a, 0x201d, 0x00a6, /* 0xa0 */
  0x00a9, 0x00ae, 0x00ac, 0x00bd, 0x00bc, 0x0141, 0x00ab, 0x00bb, /* 0xa8 */
  0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x0104, 0x010c, 0x0118, /* 0xb0 */
  0x0116, 0x2563, 0x2551, 0x2557, 0x255d, 0x012e, 0x0160, 0x2510, /* 0xb8 */
  0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x0172, 0x016a, /* 0xc0 */
  0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x017d, /* 0xc8 */
  0x0105, 0x010d, 0x0119, 0x0117, 0x012f, 0x0161, 0x0173, 0x016b, /* 0xd0 */
  0x017e, 0x2518, 0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580, /* 0xd8 */
  0x00d3, 0x00df, 0x014c, 0x0143, 0x00f5, 0x00d5, 0x00b5, 0x0144, /* 0xe0 */
  0x0136, 0x0137, 0x013b, 0x013c, 0x0146, 0x0112, 0x0145, 0x2019, /* 0xe8 */
  0x00ad, 0x00b1, 0x201c, 0x00be, 0x00b6, 0x00a7, 0x00f7, 0x201e, /* 0xf0 */
  0x00b0, 0x2219, 0x00b7, 0x00b9, 0x00b3, 0x00b2, 0x25a0, 0x00a0, /* 0xf8 */
};

/* IBM 866  */
static const unsigned short int unicode_map_ibm866[] = {
  0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0416, 0x0417, /* 0x80 */
  0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d, 0x041e, 0x041f, /* 0x88 */
  0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426, 0x0427, /* 0x90 */
  0x0428, 0x0429, 0x042a, 0x042b, 0x042c, 0x042d, 0x042e, 0x042f, /* 0x98 */
  0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437, /* 0xa0 */
  0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d, 0x043e, 0x043f, /* 0xa8 */
  0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556, /* 0xb0 */
  0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b, 0x2510, /* 0xb8 */
  0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f, /* 0xc0 */
  0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567, /* 0xc8 */
  0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b, /* 0xd0 */
  0x256a, 0x2518, 0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580, /* 0xd8 */
  0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447, /* 0xe0 */
  0x0448, 0x0449, 0x044a, 0x044b, 0x044c, 0x044d, 0x044e, 0x044f, /* 0xe8 */
  0x0401, 0x0451, 0x0404, 0x0454, 0x0407, 0x0457, 0x040e, 0x045e, /* 0xf0 */
  0x00b0, 0x2219, 0x00b7, 0x221a, 0x2116, 0x00a4, 0x25a0, 0x00a0, /* 0xf8 */
};

/* Baltic  */
static const unsigned short int unicode_map_baltic[] = {
  0x00a0, 0x201d, 0x00a2, 0x00a3, 0x00a4, 0x00a5, 0x00a6, 0x00a7, /* 0xa0 */
  0x00d8, 0x00a9, 0x201e, 0x00ab, 0x00ac, 0x00ad, 0x00ae, 0x00c6, /* 0xa8 */
  0x00b0, 0x00b1, 0x00b2, 0x00b3, 0x00b4, 0x00b5, 0x00b6, 0x00b7, /* 0xb0 */
  0x00f8, 0x00b9, 0x201c, 0x00bb, 0x00bc, 0x00bd, 0x00be, 0x00e6, /* 0xb8 */
  0x012e, 0x0116, 0x0100, 0x0106, 0x00c4, 0x00c5, 0x0104, 0x010c, /* 0xc0 */
  0x0122, 0x00c9, 0x017d, 0x0118, 0x0112, 0x0136, 0x012a, 0x013b, /* 0xc8 */
  0x0141, 0x0145, 0x0143, 0x00d3, 0x014c, 0x00d5, 0x00d6, 0x00d7, /* 0xd0 */
  0x0172, 0x0160, 0x015a, 0x016a, 0x00dc, 0x017b, 0x0179, 0x00df, /* 0xd8 */
  0x012f, 0x0117, 0x0101, 0x0107, 0x00e4, 0x00e5, 0x0105, 0x010d, /* 0xe0 */
  0x0123, 0x00e9, 0x017e, 0x0119, 0x0113, 0x0137, 0x012b, 0x013c, /* 0xe8 */
  0x0142, 0x0146, 0x0144, 0x00f3, 0x014d, 0x00f5, 0x00f6, 0x00f7, /* 0xf0 */
  0x0173, 0x0161, 0x015b, 0x016b, 0x00fc, 0x017c, 0x017a, 0x0138, /* 0xf8 */
};

/* Kamenicky  */
static const unsigned short int unicode_map_keybcs2[] = {
  0x010c, 0x00fc, 0x00e9, 0x010f, 0x00e4, 0x010e, 0x0164, 0x010d, /* 0x80 */
  0x011b, 0x011a, 0x0139, 0x00cd, 0x013e, 0x013a, 0x00c4, 0x00c1, /* 0x88 */
  0x00c9, 0x017e, 0x017d, 0x00f4, 0x00f6, 0x00d3, 0x016f, 0x00da, /* 0x90 */
  0x00fd, 0x00d6, 0x00dc, 0x0160, 0x013d, 0x00dd, 0x0158, 0x0165, /* 0x98 */
  0x00e1, 0x00ed, 0x00f3, 0x00fa, 0x0148, 0x0147, 0x016e, 0x00d4, /* 0xa0 */
  0x0161, 0x0159, 0x0155, 0x0154, 0x00bc, 0x00a7, 0x00ab, 0x00bb, /* 0xa8 */
  0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556, /* 0xb0 */
  0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b, 0x2510, /* 0xb8 */
  0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f, /* 0xc0 */
  0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567, /* 0xc8 */
  0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b, /* 0xd0 */
  0x256a, 0x2518, 0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580, /* 0xd8 */
  0x03b1, 0x03b2, 0x0393, 0x03c0, 0x03a3, 0x03c3, 0x03bc, 0x03c4, /* 0xe0 */
  0x03a6, 0x0398, 0x03a9, 0x03b4, 0x221e, 0x2205, 0x03b5, 0x2229, /* 0xe8 */
  0x2261, 0x00b1, 0x2265, 0x2264, 0x2320, 0x2321, 0x00f7, 0x2248, /* 0xf0 */
  0x00b0, 0x00b7, 0x02d9, 0x221a, 0x207f, 0x00b2, 0x25a0, 0x00a0, /* 0xf8 */
};

/* Macintosh CE  */
static const unsigned short int unicode_map_macce[] = {
  0x00c4, 0x0100, 0x0101, 0x00c9, 0x0104, 0x00d6, 0x00dc, 0x00e1, /* 0x80 */
  0x0105, 0x010c, 0x00e4, 0x010d, 0x0106, 0x0107, 0x00e9, 0x0179, /* 0x88 */
  0x017a, 0x010e, 0x00ed, 0x010f, 0x0112, 0x0113, 0x0116, 0x00f3, /* 0x90 */
  0x0117, 0x00f4, 0x00f6, 0x00f5, 0x00fa, 0x011a, 0x011b, 0x00fc, /* 0x98 */
  0x2020, 0x00b0, 0x0118, 0x00a3, 0x00a7, 0x2022, 0x00b6, 0x00df, /* 0xa0 */
  0x00ae, 0x00a9, 0x2122, 0x0119, 0x00a8, 0x2260, 0x0123, 0x012e, /* 0xa8 */
  0x012f, 0x012a, 0x2264, 0x2265, 0x012b, 0x0136, 0x2202, 0x2211, /* 0xb0 */
  0x0142, 0x013b, 0x013c, 0x013d, 0x013e, 0x0139, 0x013a, 0x0145, /* 0xb8 */
  0x0146, 0x0143, 0x00ac, 0x221a, 0x0144, 0x0147, 0x2206, 0x00ab, /* 0xc0 */
  0x00bb, 0x2026, 0x00a0, 0x0148, 0x0150, 0x00d5, 0x0151, 0x014c, /* 0xc8 */
  0x2013, 0x2014, 0x201c, 0x201d, 0x2018, 0x2019, 0x00f7, 0x25ca, /* 0xd0 */
  0x014d, 0x0154, 0x0155, 0x0158, 0x2039, 0x203a, 0x0159, 0x0156, /* 0xd8 */
  0x0157, 0x0160, 0x201a, 0x201e, 0x0161, 0x015a, 0x015b, 0x00c1, /* 0xe0 */
  0x0164, 0x0165, 0x00cd, 0x017d, 0x017e, 0x016a, 0x00d3, 0x00d4, /* 0xe8 */
  0x016b, 0x016e, 0x00da, 0x016f, 0x0170, 0x0171, 0x0172, 0x0173, /* 0xf0 */
  0x00dd, 0x00fd, 0x0137, 0x017b, 0x0141, 0x017c, 0x0122, 0x02c7, /* 0xf8 */
};

/* Macintosh Cyrillic  */
static const unsigned short int unicode_map_maccyr[] = {
  0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0416, 0x0417, /* 0x80 */
  0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d, 0x041e, 0x041f, /* 0x88 */
  0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426, 0x0427, /* 0x90 */
  0x0428, 0x0429, 0x042a, 0x042b, 0x042c, 0x042d, 0x042e, 0x042f, /* 0x98 */
  0x2020, 0x00b0, 0x0490, 0x00a3, 0x00a7, 0x2022, 0x00b6, 0x0406, /* 0xa0 */
  0x00ae, 0x00a9, 0x2122, 0x0402, 0x0452, 0x2260, 0x0403, 0x0453, /* 0xa8 */
  0x221e, 0x00b1, 0x2264, 0x2265, 0x0456, 0x00b5, 0x0491, 0x0408, /* 0xb0 */
  0x0404, 0x0454, 0x0407, 0x0457, 0x0409, 0x0459, 0x040a, 0x045a, /* 0xb8 */
  0x0458, 0x0405, 0x00ac, 0x221a, 0x0192, 0x2248, 0x2206, 0x00ab, /* 0xc0 */
  0x00bb, 0x2026, 0x00a0, 0x040b, 0x045b, 0x040c, 0x045c, 0x0455, /* 0xc8 */
  0x2013, 0x2014, 0x201c, 0x201d, 0x2018, 0x2019, 0x00f7, 0x201e, /* 0xd0 */
  0x040e, 0x045e, 0x040f, 0x045f, 0x2116, 0x0401, 0x0451, 0x044f, /* 0xd8 */
  0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437, /* 0xe0 */
  0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d, 0x043e, 0x043f, /* 0xe8 */
  0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447, /* 0xf0 */
  0x0448, 0x0449, 0x044a, 0x044b, 0x044c, 0x044d, 0x044e, 0x20ac, /* 0xf8 */
};

/* ECMA Cyrillic  */
static const unsigned short int unicode_map_ecma113[] = {
  0x00a0, 0x0452, 0x0453, 0x0451, 0x0454, 0x0455, 0x0456, 0x0457, /* 0xa0 */
  0x0458, 0x0459, 0x045a, 0x045b, 0x045c, 0x00ad, 0x045e, 0x045f, /* 0xa8 */
  0x2116, 0x0402, 0x0403, 0x0401, 0x0404, 0x0405, 0x0406, 0x0407, /* 0xb0 */
  0x0408, 0x0409, 0x040a, 0x040b, 0x040c, 0x00a4, 0x040e, 0x040f, /* 0xb8 */
  0x044e, 0x0430, 0x0431, 0x0446, 0x0434, 0x0435, 0x0444, 0x0433, /* 0xc0 */
  0x0445, 0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d, 0x043e, /* 0xc8 */
  0x043f, 0x044f, 0x0440, 0x0441, 0x0442, 0x0443, 0x0436, 0x0432, /* 0xd0 */
  0x044c, 0x044b, 0x0437, 0x0448, 0x044d, 0x0449, 0x0447, 0x044a, /* 0xd8 */
  0x042e, 0x0410, 0x0411, 0x0426, 0x0414, 0x0415, 0x0424, 0x0413, /* 0xe0 */
  0x0425, 0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d, 0x041e, /* 0xe8 */
  0x041f, 0x042f, 0x0420, 0x0421, 0x0422, 0x0423, 0x0416, 0x0412, /* 0xf0 */
  0x042c, 0x042b, 0x0417, 0x0428, 0x042d, 0x0429, 0x0427, 0x042a, /* 0xf8 */
};

/* KOI8-CS2  */
static const unsigned short int unicode_map_koi8cs2[] = {
  0x00a0, NO_CHR, 0x00b4, NO_CHR, 0x007e, NO_CHR, 0x02d8, 0x02d9, /* 0xa0 */
  0x00a8, NO_CHR, 0x02da, 0x00b8, NO_CHR, 0x02dd, 0x02db, 0x02c7, /* 0xa8 */
  0x00a9, 0x00ae, 0x250c, 0x2510, 0x2514, 0x2518, 0x2500, 0x2193, /* 0xb0 */
  0x03a9, 0x00a7, 0x03b1, 0x03b3, 0x03b5, 0x03bc, 0x03c0, 0x03c9, /* 0xb8 */
  0x00e0, 0x00e1, 0x01ce, 0x010d, 0x010f, 0x011b, 0x0155, NO_CHR, /* 0xc0 */
  0x00fc, 0x00ed, 0x016f, 0x013a, 0x013e, 0x00f6, 0x0148, 0x00f3, /* 0xc8 */
  0x00f4, 0x00e4, 0x0159, 0x0161, 0x0165, 0x00fa, 0x00eb, 0x00e9, /* 0xd0 */
  0x0171, 0x00fd, 0x017e, NO_CHR, NO_CHR, 0x0151, 0x0117, 0x00df, /* 0xd8 */
  0x00c0, 0x00c1, 0x01cd, 0x010c, 0x010e, 0x011a, 0x0154, NO_CHR, /* 0xe0 */
  0x00dc, 0x00cd, 0x016e, 0x0139, 0x013d, 0x00d6, 0x0147, 0x00d3, /* 0xe8 */
  0x00d4, 0x00c4, 0x0158, 0x0160, 0x0164, 0x00da, 0x00cb, 0x00c9, /* 0xf0 */
  0x0170, 0x00dd, 0x017d, NO_CHR, NO_CHR, 0x0150, 0x0116, NO_CHR, /* 0xf8 */
};

/* KOI8-R  */
static const unsigned short int unicode_map_koi8r[] = {
  0x2500, 0x2502, 0x250c, 0x2510, 0x2514, 0x2518, 0x251c, 0x2524, /* 0x80 */
  0x252c, 0x2534, 0x253c, 0x2580, 0x2584, 0x2588, 0x258c, 0x2590, /* 0x88 */
  0x2591, 0x2592, 0x2593, 0x2320, 0x25a0, 0x2022, 0x221a, 0x2248, /* 0x90 */
  0x2264, 0x2265, 0x00a0, 0x2321, 0x00b0, 0x00b2, 0x00b7, 0x00f7, /* 0x98 */
  0x2550, 0x2551, 0x2552, 0x0451, 0x2553, 0x2554, 0x2555, 0x2556, /* 0xa0 */
  0x2557, 0x2558, 0x2559, 0x255a, 0x255b, 0x255c, 0x255d, 0x255e, /* 0xa8 */
  0x255f, 0x2560, 0x2561, 0x0401, 0x2562, 0x2563, 0x2564, 0x2565, /* 0xb0 */
  0x2566, 0x2567, 0x2568, 0x2569, 0x256a, 0x256b, 0x256c, 0x00a9, /* 0xb8 */
  0x044e, 0x0430, 0x0431, 0x0446, 0x0434, 0x0435, 0x0444, 0x0433, /* 0xc0 */
  0x0445, 0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d, 0x043e, /* 0xc8 */
  0x043f, 0x044f, 0x0440, 0x0441, 0x0442, 0x0443, 0x0436, 0x0432, /* 0xd0 */
  0x044c, 0x044b, 0x0437, 0x0448, 0x044d, 0x0449, 0x0447, 0x044a, /* 0xd8 */
  0x042e, 0x0410, 0x0411, 0x0426, 0x0414, 0x0415, 0x0424, 0x0413, /* 0xe0 */
  0x0425, 0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d, 0x041e, /* 0xe8 */
  0x041f, 0x042f, 0x0420, 0x0421, 0x0422, 0x0423, 0x0416, 0x0412, /* 0xf0 */
  0x042c, 0x042b, 0x0417, 0x0428, 0x042d, 0x0429, 0x0427, 0x042a, /* 0xf8 */
};

/* KOI8-U  */
static const unsigned short int unicode_map_koi8u[] = {
  0x2500, 0x2502, 0x250c, 0x2510, 0x2514, 0x2518, 0x251c, 0x2524, /* 0x80 */
  0x252c, 0x2534, 0x253c, 0x2580, 0x2584, 0x2588, 0x258c, 0x2590, /* 0x88 */
  0x2591, 0x2592, 0x2593, 0x2320, 0x25a0, 0x2219, 0x221a, 0x2248, /* 0x90 */
  0x2264, 0x2265, 0x00a0, 0x2321, 0x00b0, 0x00b2, 0x00b7, 0x00f7, /* 0x98 */
  0x2550, 0x2551, 0x2552, 0x0451, 0x0454, 0x2554, 0x0456, 0x0457, /* 0xa0 */
  0x2557, 0x2558, 0x2559, 0x255a, 0x255b, 0x0491, 0x255d, 0x255e, /* 0xa8 */
  0x255f, 0x2560, 0x2561, 0x0401, 0x0404, 0x2563, 0x0406, 0x0407, /* 0xb0 */
  0x2566, 0x2567, 0x2568, 0x2569, 0x256a, 0x0490, 0x256c, 0x00a9, /* 0xb8 */
  0x044e, 0x0430, 0x0431, 0x0446, 0x0434, 0x0435, 0x0444, 0x0433, /* 0xc0 */
  0x0445, 0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d, 0x043e, /* 0xc8 */
  0x043f, 0x044f, 0x0440, 0x0441, 0x0442, 0x0443, 0x0436, 0x0432, /* 0xd0 */
  0x044c, 0x044b, 0x0437, 0x0448, 0x044d, 0x0449, 0x0447, 0x044a, /* 0xd8 */
  0x042e, 0x0410, 0x0411, 0x0426, 0x0414, 0x0415, 0x0424, 0x0413, /* 0xe0 */
  0x0425, 0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d, 0x041e, /* 0xe8 */
  0x041f, 0x042f, 0x0420, 0x0421, 0x0422, 0x0423, 0x0416, 0x0412, /* 0xf0 */
  0x042c, 0x042b, 0x0417, 0x0428, 0x042d, 0x0429, 0x0427, 0x042a, /* 0xf8 */
};

/* KOI8-UNI  */
static const unsigned short int unicode_map_koi8uni[] = {
  0x2500, 0x2502, 0x250c, 0x2510, 0x2514, 0x2518, 0x251c, 0x2524, /* 0x80 */
  0x252c, 0x2534, 0x253c, 0x2580, 0x2584, 0x2588, 0x258c, 0x2590, /* 0x88 */
  0x2591, 0x2018, 0x2019, 0x201c, 0x201d, 0x2219, 0x2013, 0x2014, /* 0x90 */
  0x00a9, 0x2122, 0x00a0, 0x00bb, 0x00ae, 0x00ab, 0x00b7, 0x00a4, /* 0x98 */
  0x00a0, 0x0452, 0x0453, 0x0451, 0x0454, 0x0455, 0x0456, 0x0457, /* 0xa0 */
  0x0458, 0x0459, 0x045a, 0x045b, 0x045c, 0x0491, 0x045e, 0x045f, /* 0xa8 */
  0x2116, 0x0402, 0x0403, 0x0401, 0x0404, 0x0405, 0x0406, 0x0407, /* 0xb0 */
  0x0408, 0x0409, 0x040a, 0x040b, 0x040c, 0x0490, 0x040e, 0x040f, /* 0xb8 */
  0x044e, 0x0430, 0x0431, 0x0446, 0x0434, 0x0435, 0x0444, 0x0433, /* 0xc0 */
  0x0445, 0x0438, 0x0439, 0x043a, 0x043b, 0x043c, 0x043d, 0x043e, /* 0xc8 */
  0x043f, 0x044f, 0x0440, 0x0441, 0x0442, 0x0443, 0x0436, 0x0432, /* 0xd0 */
  0x044c, 0x044b, 0x0437, 0x0448, 0x044d, 0x0449, 0x0447, 0x044a, /* 0xd8 */
  0x042e, 0x0410, 0x0411, 0x0426, 0x0414, 0x0415, 0x0424, 0x0413, /* 0xe0 */
  0x0425, 0x0418, 0x0419, 0x041a, 0x041b, 0x041c, 0x041d, 0x041e, /* 0xe8 */
  0x041f, 0x042f, 0x0420, 0x0421, 0x0422, 0x0423, 0x0416, 0x0412, /* 0xf0 */
  0x042c, 0x042b, 0x0417, 0x0428, 0x042d, 0x0429, 0x0427, 0x042a, /* 0xf8 */
};

/* Cork  */
static const unsigned short int unicode_map_cork[] = {
  0x0060, 0x00b4, 0x005e, 0x007e, 0x00a8, 0x02dd, 0x02da, 0x02c7, /* 0x00 */
  0x02d8, 0x0009, 0x000a, 0x00b8, 0x02db, 0x000d, 0x2039, 0x203a, /* 0x08 */
  0x201c, 0x201d, 0x201e, 0x00ab, 0x00bb, 0x2013, 0x2014, 0x2080, /* 0x10 */
  0x0131, 0x002e, 0xfb00, 0xfb01, 0xfb02, 0xfb00, 0x0069, 0xfb00, /* 0x18 */
  0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027, /* 0x20 */
  0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f, /* 0x28 */
  0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, /* 0x30 */
  0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f, /* 0x38 */
  0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, /* 0x40 */
  0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, /* 0x48 */
  0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, /* 0x50 */
  0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f, /* 0x58 */
  0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, /* 0x60 */
  0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, /* 0x68 */
  0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, /* 0x70 */
  0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x007e, 0x2010, /* 0x78 */
  0x0102, 0x0104, 0x0106, 0x010c, 0x010e, 0x011a, 0x0118, 0x011e, /* 0x80 */
  0x0139, 0x013d, 0x0141, 0x0143, 0x0147, 0x014a, 0x0150, 0x0154, /* 0x88 */
  0x0158, 0x015a, 0x0160, 0x015e, 0x0164, 0x0162, 0x0170, 0x016e, /* 0x90 */
  0x0178, 0x0179, 0x017d, 0x017b, 0x0132, 0x0130, 0x0111, 0x00a7, /* 0x98 */
  0x0103, 0x0105, 0x0107, 0x010d, 0x010f, 0x011b, 0x0119, 0x011f, /* 0xa0 */
  0x013a, 0x013e, 0x0142, 0x0144, 0x0148, 0x014b, 0x0151, 0x0155, /* 0xa8 */
  0x0159, 0x015b, 0x0161, 0x015f, 0x0165, 0x0163, 0x0171, 0x016f, /* 0xb0 */
  0x00ff, 0x017a, 0x017e, 0x017c, 0x0133, 0x00a1, 0x00bf, 0x00a3, /* 0xb8 */
  0x00c0, 0x00c1, 0x00c2, 0x00c3, 0x00c4, 0x00c5, 0x00c6, 0x00c7, /* 0xc0 */
  0x00c8, 0x00c9, 0x00ca, 0x00cb, 0x00cc, 0x00cd, 0x00ce, 0x00cf, /* 0xc8 */
  0x00d0, 0x00d1, 0x00d2, 0x00d3, 0x00d4, 0x00d5, 0x00d6, 0x0152, /* 0xd0 */
  0x00d8, 0x00d9, 0x00da, 0x00db, 0x00dc, 0x00dd, 0x00de, 0x0053, /* 0xd8 */
  0x00e0, 0x00e1, 0x00e2, 0x00e3, 0x00e4, 0x00e5, 0x00e6, 0x00e7, /* 0xe0 */
  0x00e8, 0x00e9, 0x00ea, 0x00eb, 0x00ec, 0x00ed, 0x00ee, 0x00ef, /* 0xe8 */
  0x00f0, 0x00f1, 0x00f2, 0x00f3, 0x00f4, 0x00f5, 0x00f6, 0x0153, /* 0xf0 */
  0x00f8, 0x00f9, 0x00fa, 0x00fb, 0x00fc, 0x00fd, 0x00fe, 0x00df, /* 0xf8 */
};

#define MAKE_CS_LINE(name) \
  { #name, \
    0x0100 - sizeof(unicode_map_##name)/sizeof(unsigned short int), \
    unicode_map_##name }

/*
 * Charset maps with names assigned.
 * Since we don't know the id's at compile time, find_charset_map() finds
 * them when first needed.
 */
static const EncaUnicodeMap UNICODE_MAP[] = {
  MAKE_CS_LINE(iso88592),
  MAKE_CS_LINE(iso88594),
  MAKE_CS_LINE(iso88595),
  MAKE_CS_LINE(iso885913),
  MAKE_CS_LINE(iso885916),
  MAKE_CS_LINE(cp1125),
  MAKE_CS_LINE(cp1250),
  MAKE_CS_LINE(cp1251),
  MAKE_CS_LINE(cp1257),
  MAKE_CS_LINE(ibm852),
  MAKE_CS_LINE(ibm855),
  MAKE_CS_LINE(ibm775),
  MAKE_CS_LINE(ibm866),
  MAKE_CS_LINE(baltic),
  MAKE_CS_LINE(keybcs2),
  MAKE_CS_LINE(macce),
  MAKE_CS_LINE(maccyr),
  MAKE_CS_LINE(ecma113),
  MAKE_CS_LINE(koi8cs2),
  MAKE_CS_LINE(koi8r),
  MAKE_CS_LINE(koi8u),
  MAKE_CS_LINE(koi8uni),
  MAKE_CS_LINE(cork),
};

/**
 * enca_charset_has_ucs2_map:
 * @charset: An 8bit charset.
 *
 * Find whether we have charset -> UCS-2 map.
 *
 * Returns: Nonzero when the map is available, zero otherwise.
 **/
int
enca_charset_has_ucs2_map(int charset)
{
  return find_charset_map(charset) != NULL;
}

/**
 * enca_charset_ucs2_map:
 * @charset: An 8bit charset.
 * @buffer: Buffer to store the map to.
 *
 * Creates map from an 8bit charset to UCS-2.
 *
 * The buffer must be at least 256*sizeof(unsigned int) long to hold the map.
 *
 * Returns: Nonzero when the map was successfully created, zero otherwise.
 **/
int
enca_charset_ucs2_map(int charset,
                      unsigned int *buffer)
{
  unsigned int i;
  const EncaUnicodeMap *umap = find_charset_map(charset);

  if (umap == NULL)
    return 0;

  for (i = 0; i < umap->tstart; i++)
    buffer[i] = i;

  for (i = umap->tstart; i < 0x0100; i++)
    buffer[i] = umap->map[i - umap->tstart];

  return 1;
}

/**
 * find_charset_map:
 * @charset: An 8bit charset.
 *
 * Find charset -> UCS-2 map.
 *
 * Returns: Pointer to the map, #NULL when not found.
 **/
static const EncaUnicodeMap*
find_charset_map(int charset)
{
  static int charset_id[ELEMENTS(UNICODE_MAP)];
  static int charset_id_initialized = 0;
  size_t i;

  if (charset < 0)
    return NULL;

  if (!charset_id_initialized) {
    for (i = 0; i < ELEMENTS(UNICODE_MAP); i++) {
      charset_id[i] = enca_name_to_charset(UNICODE_MAP[i].name);
      assert(charset_id[i] != ENCA_CS_UNKNOWN);
    }
    charset_id_initialized = 1;
  }

  for (i = 0; i < ELEMENTS(UNICODE_MAP); i++) {
    if (charset_id[i] == charset)
      return UNICODE_MAP + i;
  }

  return NULL;
}

/**
 * enca_charsets_subset_identical:
 * @charset1: A charset.
 * @charset2: Another charset.
 * @counts: An array of size 0x100 containing character counts.
 *
 * Checks whether all characters with nonzero count have the same meaning
 * in both charsets.
 *
 * In other words, it checks whether conversion of sample containing only
 * these characters from @charset1 to @charset2 would be identity.
 *
 * Returns: Nonzero if charsets are identical on the subset, zero otherwise.
 **/
int
enca_charsets_subset_identical(int charset1,
                               int charset2,
                               const size_t *counts)
{
  const EncaUnicodeMap *umap1 = find_charset_map(charset1);
  const EncaUnicodeMap *umap2 = find_charset_map(charset2);
  size_t ts1, ts2;
  const unsigned short int *m1, *m2;
  size_t i;

  assert(umap1 != NULL);
  assert(umap2 != NULL);

  m1 = umap1->map;
  m2 = umap2->map;
  ts1 = umap1->tstart;
  ts2 = umap2->tstart;

  for (i = 0; i < 0x100; i++) {
    unsigned short int u1 = i < ts1 ? i : m1[i - ts1];
    unsigned short int u2 = i < ts2 ? i : m2[i - ts2];

    if (counts[i] > 0
        && u1 != u2
        && u1 != ENCA_NOT_A_CHAR
        && u2 != ENCA_NOT_A_CHAR)
      return 0;
  }

  return 1;
}

/***** Documentation *********************************************************/

/**
 * ENCA_NOT_A_CHAR:
 *
 * Not-a-character in unicode tables.
 **/

/* vim: ts=2 sw=2 et
 */
