// Define our add-on ID, which is needed to resolve paths for files within our extension
// and to be able to listen for notifications.
var ADDON_ID = "notifyToolsExample@jobisoft.de";

// Import any needed modules.
var { Services } = ChromeUtils.import("resource://gre/modules/Services.jsm");
var { ExtensionParent } = ChromeUtils.import("resource://gre/modules/ExtensionParent.jsm");

// Get our extension object.
let extension = ExtensionParent.GlobalManager.getExtension(ADDON_ID);

// Load notifyTools.
Services.scriptloader.loadSubScript(extension.rootURI.resolve("notifyTools/notifyTools.js"), window, "UTF-8");



// -- Define listeners for messages from the background script.

/**
 * NotifyTools currently is not 100% compatible with the behavior of
 * runtime.sendMessage. While runtime messaging is ignoring non-Promise return
 * values, NotifyTools only ignores <null>.
 * 
 * Why does this matter? Consider the following three listeners:
 * 
 * async function dominant_listener(data) {
 *  if (data.type == "A") {
 *    return { msg: "I should answer only type A" };
 *  }
 * }
 * 
 * function silent_listener(data) {
 *  if (data.type == "B") {
 *    return { msg: "I should answer only type B" };
 *  }
 * }
 * 
 * function selective_listener(data) {
 *  if (data.type == "C") {
 *    return Promise.resolve({ msg: "I should answer only type C" });
 *  }
 * }
 * 
 * When all 3 listeners are registered for the runtime.onMessage event,
 * the dominant listener will always respond, even for data.type != "A" requests,
 * because it is always returning a Promise (async function). The return value of 
 * the silent listener is ignored, and the selective listener returns a value just
 * for data.type == "C". But since the dominant listener also returns <null> for
 * these requests, the actual return value depends on which listener is faster
 * and/or was registered first.
 * 
 * All notifyTools listener however ignore <null> return values (so null can actually
 * never be returned). The above dominant listener will only respond to type == "A"
 * requests, the silent listener will only respond to type == "B" requests and the
 * selective listener will respond only to type == "C" requests.
 * 
 */

// This listener returns a dominant Promise and is therefore NOT compatible with
// runtime messaging.
async function onBeforeSend_listener(data) {
  if (data.msg == "onBeforeSend") {
    return { msg: `onBeforeSend: ${data.msg}` };
  }
}

// This listener does not return a Promise and is therefore NOT compatible with
// runtime messaging.
function onMessageDisplayed_listener(data) {
  if (data.msg == "onMessageDisplayed") {
    return { msg: `onMessageDisplayed: ${data.msg}` };
  }
}

// -- Implement onLoad and onUnload of the WindowListener API

async function onLoad(activatedWhileWindowOpen) {

  window.notifyTools.setAddOnId(ADDON_ID);
  window.notifyTools.addListener(onBeforeSend_listener);
  window.notifyTools.addListener(onMessageDisplayed_listener);

  // Brute force test.
  for (let i = 0; i < 100; i++) {
    let time = Date.now();
    await window.notifyTools.notifyBackground({ command: "setPref", time });
    response = await window.notifyTools.notifyBackground({ command: "getPref" });
    if (time != response.time) console.log("ERR", { time, response: response.time });
    else console.log("OK");
  }
}

function onUnload(deactivatedWhileWindowOpen) {
  window.notifyTools.removeAllListeners();
}
